/***************************************************************
 *                 Mathematical Object Library                 *
 *      class tensor3 : declarations for 3rd order tensors     *
 *                    simula+@metz.ensam.fr                    *
 *                   GNU/linux version 0.5.2                   *
 *            software under General Public License            *
 ***************************************************************
 * copyright  2004,2005,2006 COLLARD Christophe
 * copyright  2004,2005,2006 Laboratoire de Physique et Mcanique des Matriaux (LPMM - UMR 7554)
 * copyright  2004,2005,2006 Laboratoire de Mathmatiques et ses Applications de Valenciennes (LAMAV - EA 4015)
 ***************************************************************/

/*! \class tensor3
    \brief 3rd order tensor library \n

    \htmlonly 
    <FONT color="#838383">

    tensor3 belongs to Mathematical Object Libraries (MOL++) </br>
    MOL++ is part of Simula+ <br><br>

    Simula+ is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version. <br><br>

    Simula+ is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details. <br><br>

    You should have received a copy of the GNU General Public License
    along with Simula+; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

    </FONT>
    \endhtmlonly

    \author copyright \htmlonly &#169; \endhtmlonly 2004, 2005, 2006 Christophe COLLARD \n
	    copyright \htmlonly &#169; \endhtmlonly 2004, 2005, 2006 Laboratoire de Physique et Mcanique des Matriaux (LPMM - UMR 7554) \n
	    copyright \htmlonly &#169; \endhtmlonly 2004, 2005, 2006 Laboratoire de Mathmatiques et ses Applications de Valenciennes (LAMAV - EA 4015)
    \version 0.5.2
    \date 2004-2006
    \bug none
    \warning none
*/

#ifndef __cplusplus
#error Must use C++ for the type tensor3
#endif

#if !defined(_TENSORS3_H)
#define _TENSORS3_H


#if !defined(__VECTORS_H)
#include "vectors.h"
#endif

#if !defined(__TENSORS2_H)
#include "tensors2.h"
#endif

using namespace std;


//==============================
template <class T> class tensor3
//==============================
{
  private :
    // tensor3 are stored using tensor2
    tensor2<T> *tsr;
    int size;

  public :
    tensor3 ();                   // default constructor
    tensor3 (int, bool=true, T=0);           // this constructor allocates memory
    tensor3 (int, int, int, bool=true, T=0); // this constructor allocates memory
    tensor3 (const tensor3<T>&);  // copy constructor
    tensor3 (tensor3<T>*);        // copy constructor for temporary objects
    ~tensor3 ();                  // destructor

    virtual int dim1 () const {return size;}             // returns the tensor 1st component size : k
    virtual int dim2 () const {return tsr[0].dim1();}    // returns the tensor 2nd component size : i
    virtual int dim3 () const {return tsr[0].dim2();}    // returns the tensor 3rd component size : j
    virtual void create (int);          // partial memory allocation (don't use this function except if you know what you're doing !!)
    virtual void assign (int,int,int);  // allocates memory without initilizing the tensor components to zero (so use it with great care)
    operator int () const {return size;}    // cast operator : tensor3 -> int
    friend bool operator ! (const tensor3<T>& t) {return !t.size;}

    virtual T&  operator () (int,int,int) const;        // returns the element Pk,ij : P(tensor2,line,column)
    tensor2<T>& operator [] (int) const;                // returns a tensor2 Pk,ij -> Mij
    virtual tensor3<T>& operator = (const tensor3<T>&); // defines equal operator for 3rd order tensors
    virtual tensor3<T>& operator = (tensor3<T>*);  // allocates the data from the right hand temporary object to the left hand object - no copy (deletes temporary object)
    virtual tensor3<T>& operator &= (const tensor3<T>&); // copies tensor3 without size check (use this operator with great care)
    virtual tensor3<T>& operator &= (tensor3<T>*); // allocates the data from the right hand temporary object to the left hand object (deletes temporary object) - no size check - no copy (use this operator with great care)
    template <class M> friend tensor3<M>  operator + (const tensor3<M>&, const tensor3<M>&); // overloads operator + for 3rd order tensors
    template <class M> friend tensor3<M>  operator - (const tensor3<M>&, const tensor3<M>&); // overloads operator - for 3rd order tensors
    template <class M> friend M           operator | (const tensor3<M>&, const tensor3<M>&);  // overloads operator | for 3rd order tensors T|P = Tijk Pijk
    tensor3<T>& operator += (const tensor3<T>&);  // overloads += operator for tensors
    tensor3<T>& operator -= (const tensor3<T>&);  // overloads -= operator for tensors
    template <class M> friend bool operator == (const tensor3<M>&, const tensor3<M>&);  // compares 2 tensors
    template <class M> friend bool operator != (const tensor3<M>&, const tensor3<M>&);
    // operations with scalars
    tensor3<T>& operator *= (const T&);
    template <class M> friend tensor3<M> operator * (const tensor3<M>&, const M&);
    template <class M> friend tensor3<M> operator * (const M&, const tensor3<M>&);
    // operations with vectors
    template <class M> friend tensor2<M> operator * (const tensor3<M>&, const vector<M>&);   // M_ki = P_kij * V_j
    template <class M> friend tensor2<M> operator * (const vector<M>&, const tensor3<M>&);   // M_ij = P_kij * V_k
    template <class M> friend tensor2<M> operator | (const tensor3<M>&, const vector<M>&);   // M_kj = P_kij * V_i
    template <class M> friend tensor2<M> operator | (const vector<M>&, const tensor3<M>&);   // M_kj = P_kij * V_i
    template <class M> friend tensor2<M> tensor_sum (const tensor3<M>&, const vector<M>&, int);  // M = P : v
    // operations with tensor2
    template <class M> friend vector<M> operator || (const tensor3<M>&, const tensor2<M>&);   // v_p = T_p,ij M_ij
    template <class M> friend tensor3<M> operator * (const tensor2<M>&, const tensor3<M>&);   // T_pij = M_pq * T_qij
    // overloads iostreams
    template <class M> friend ostream& operator << (ostream&, const tensor3<M>&);  // overloads output stream for tensors
    template <class M> friend istream& operator >> (istream&, const tensor3<M>&);  // overloads output stream for tensors
    virtual void save (const char*); // writes to disk
    virtual void read (const char*); // reads from disk
    tensor3<T>& approximation ();
};


//=====Private methods for tensor3=============================================


//=====Public methods for tensor3==============================================


//---------------------------------------
template <class T> tensor3<T>::tensor3 ()
//---------------------------------------
{
  size = 0;
}


//-----------------------------------------------------------------
template <class T> tensor3<T>::tensor3 (int sz, bool init, T value)
//-----------------------------------------------------------------
{
  assert (sz>0);

  tsr = new tensor2<T> [size=sz];
  for (int i=0; i<size; i++)
    tsr[i] =& matrix<T> (sz,sz,init,value);
}


//------------------------------------------------------------------------------------------------
template <class T> tensor3<T>::tensor3 (int ntensor2, int nrows, int ncolumns, bool init, T value)
//------------------------------------------------------------------------------------------------
{
  assert (nrows>0 && ncolumns>0 && ntensor2>0);

  tsr = new tensor2<T> [size=ntensor2];
  for (int i=0; i<size; i++)
    tsr[i] =& matrix<T> (nrows,ncolumns,init,value);
}


//-----------------------------------------------------------
template <class T> tensor3<T>::tensor3 (const tensor3<T>& tn)
//-----------------------------------------------------------
{
  assert (tn.dim2() && tn.dim3() && tn.size);

  tsr = new tensor2<T> [size=tn.size];
  for (int i=0; i<tn.size; i++)
      tsr[i] = tn.tsr[i];
}


//-----------------------------------------------------
template <class T> tensor3<T>::tensor3 (tensor3<T>* tn)
//-----------------------------------------------------
{
  assert ((*tn).dim2() && (*tn).dim3() && (*tn).size);

  size = (*tn).size;
  tsr = (*tn).tsr;
  (*tn).size = 0;
}


//----------------------------------------
template <class T> tensor3<T>::~tensor3 ()
//----------------------------------------
{
  // destruction of the memory allocated for the tensor

  if (size) delete [] tsr; // free memory only if it's been affected
  size = 0;
}


//------------------------------------------------------
template <class T> void tensor3<T>::create (int ntensor)
//------------------------------------------------------
{
  assert (ntensor>0);
  assert (!size);

  tsr = new tensor2<T> [size = ntensor];
}


//-------------------------------------------------------------------------------
template <class T> void tensor3<T>::assign (int ntensor, int nrows, int ncolumns)
//-------------------------------------------------------------------------------
{
  assert (ntensor>0 && !size);

  tsr = new tensor2<T> [size = ntensor];
  for (int i=0; i<size; i++)
    tsr[i] =& matrix<T> (nrows,ncolumns,false);
}


//-----------------------------------------------------------------------
template <class T> T& tensor3<T>::operator () (int k, int i, int j) const
//-----------------------------------------------------------------------
{
  assert (size);
  assert ((k>0)&&(k<=size));

  return tsr[k-1](i,j);
}


//------------------------------------------------------------------
template <class T> tensor2<T>& tensor3<T>::operator [] (int k) const
//------------------------------------------------------------------
{
  assert ((k>0)&&(k<=size));
  return tsr[k-1];
}


//--------------------------------------------------------------------------
template <class T> tensor3<T>& tensor3<T>::operator = (const tensor3<T>& tn)
//--------------------------------------------------------------------------
{
  assert (tn.dim2() && tn.dim3() && tn.size);

  if (!size) tsr = new tensor2<T> [size=tn.size];
  else assert (size==tn.size && dim2()==tn.dim2() && dim3()==tn.dim3());
  for (int i=0; i<size; i++) 
      tsr[i] = tn.tsr[i];

  return *this;
}


//--------------------------------------------------------------------
template <class T> tensor3<T>& tensor3<T>::operator = (tensor3<T>* tn)
//--------------------------------------------------------------------
{
  assert ((*tn).dim2() && (*tn).dim3() && (*tn).size);

  if (!size) size = (*tn).size;
  else
    { assert (size==(*tn).size && dim2()==(*tn).dim2() && dim3()==(*tn).dim3());
      delete [] tsr;
    }
  tsr = (*tn).tsr;
  (*tn).size = 0;

  return *this;
}


//---------------------------------------------------------------------------
template <class T> tensor3<T>& tensor3<T>::operator &= (const tensor3<T>& tn)
//---------------------------------------------------------------------------
{
  if (!tn)
    { if (size) delete [] tsr;
      size = 0;
    }

  else
    { if (size != tn.size)
	{ if (size) delete [] tsr;
	  tsr = new tensor2<T> [size = tn.size];  // allocates memory with the = operator for tensor2 (no initialization)
	}
      for (int i=1; i<=size; i++)
	(*this)[i] &= tn[i];
    }

  return *this;
}


//---------------------------------------------------------------------
template <class T> tensor3<T>& tensor3<T>::operator &= (tensor3<T>* tn)
//---------------------------------------------------------------------
{
  if (!(*tn).size)
    { if (size) delete [] tsr;
      size = 0;
    }

  else
    { if (size)
	delete [] tsr;
      size = (*tn).size;
      tsr = (*tn).tsr;
      (*tn).size = 0;
    }

  return *this;
}


//-----------------------------------------------------------------------------------
template <class T> tensor3<T> operator + (const tensor3<T>& t1, const tensor3<T>& t2)
//-----------------------------------------------------------------------------------
{
  assert (t1.dim1()==t2.dim1()); // tensors must have the same size
  assert(t1.dim1());
  tensor3<T> t = t1;
  return t+=t2;
}


//-----------------------------------------------------------------------------------
template <class T> tensor3<T> operator - (const tensor3<T>& t1, const tensor3<T>& t2)
//-----------------------------------------------------------------------------------
{
  assert (t1.dim1()==t2.dim1()); // tensors must have the same size
  assert(t1.dim1());
  tensor3<T> t = t1;
  return t-=t2;
}


//--------------------------------------------------------------------------
template <class T> T operator | (const tensor3<T>& t1, const tensor3<T>& t2)
//--------------------------------------------------------------------------
{
  assert (t1.size==t2.size); // tensors must have the same size
  assert (t1.size);

  T sum=0;
  for (int i=1; i<=t1.size; i++)
    sum += (t1[i] | t2[i]);

  return sum;
}


//---------------------------------------------------------------------------
template <class T> tensor3<T>& tensor3<T>::operator += (const tensor3<T>& ts)
//---------------------------------------------------------------------------
{
  assert (size);  // the tensor must be defined
  assert (size == ts.size);  // tensors must have the same size

  for (int i=0; i<size; i++)
    tsr[i] += ts.tsr[i];

  return *this;
}


//---------------------------------------------------------------------------
template <class T> tensor3<T>& tensor3<T>::operator -= (const tensor3<T>& ts)
//---------------------------------------------------------------------------
{
  assert (size);  // the tensor must be defined
  assert (size == ts.size);  // tensors must have the same size

  for (int i=0; i<size; i++)
    tsr[i] -= ts.tsr[i];

  return *this;
}


//------------------------------------------------------------------------------
template <class T> bool operator == (const tensor3<T>& t1, const tensor3<T>& t2)
//------------------------------------------------------------------------------
{
  assert (t1.size);

  int result=(t1.size==t2.size);
  for (int i=1; (i<=t1.size)&&(result); i++)
    result *= (t1[i]==t2[i]);

  return result;
}


//------------------------------------------------------------------------------
template <class T> bool operator != (const tensor3<T>& t1, const tensor3<T>& t2)
//------------------------------------------------------------------------------
{
  return !(t1 == t2);
}


//-------------------------------------------------------------------
template <class T> tensor3<T>& tensor3<T>::operator *= (const T& elt)
//-------------------------------------------------------------------
{
  assert (size);
  for (int i=0; i<size; i++)
    tsr [i] *= elt;
  return *this;
}


//---------------------------------------------------------------------------
template <class T> tensor3<T> operator * (const tensor3<T>& ts, const T& elt)
//---------------------------------------------------------------------------
{
  assert (ts.size);
  tensor3<T> tr = ts;
  return tr*=elt;
}


//---------------------------------------------------------------------------
template <class T> tensor3<T> operator * (const T& elt, const tensor3<T>& ts)
//---------------------------------------------------------------------------
{
  assert (ts.size);
  return ts*elt;
}


//---------------------------------------------------------------------------------
template <class T> tensor2<T> operator * (const tensor3<T>& ts, const vector<T>& v)
//---------------------------------------------------------------------------------
{
  assert (v.dim());
  assert (v.dim() == ts.dim3());

  tensor2<T> m(ts.dim1(),ts.dim2());
  for (int k=1; k<=ts.size; k++)
    m[k] += ts[k] * v;

  return m;
}


//---------------------------------------------------------------------------------
template <class T> tensor2<T> operator * (const vector<T>& v, const tensor3<T>& ts)
//---------------------------------------------------------------------------------
{
  assert (v.dim());
  assert (v.dim() == ts.dim1());

  tensor2<T> m(ts.dim2(),ts.dim3());
  for (int k=1; k<=ts.size; k++)
    m += ts[k] * v[k];

  return m;
}


//---------------------------------------------------------------------------------
template <class T> tensor2<T> operator | (const tensor3<T>& ts, const vector<T>& v)
//---------------------------------------------------------------------------------
{
  assert (v.dim());
  assert (v.dim() == ts.dim2());

  tensor2<T> m(ts.dim1(),ts.dim3());
  for (int k=1; k<=ts.size; k++)
    m[k] += v * ts[k];

  return m;
}


//---------------------------------------------------------------------------------
template <class T> tensor2<T> operator | (const vector<T>& v, const tensor3<T>& ts)
//---------------------------------------------------------------------------------
{
  return ts|v;
}


//----------------------------------------------------------------------------------------
template <class T> tensor2<T> tensor_sum (const tensor3<T>& ts, const vector<T>& v, int i)
//----------------------------------------------------------------------------------------
{
  assert (i>=1 && i<=3);

  tensor2<T> m;
  if (i==1) m =& (v*ts);
  else if (i==2) m =& (ts|v);
       else m =& (ts*v);

  return m;
}


//------------------------------------------------------------------------------------
template <class T> vector<T> operator || (const tensor3<T>& ts, const tensor2<T>& mat)
//------------------------------------------------------------------------------------
{
  assert (mat.Rows());
  assert (ts.size);

  vector<T> v(ts.size,false);
  for (int i=1; i<=ts.size; i++)
    v[i] = (ts[i] | mat);

  return v;
}


//------------------------------------------------------------------------------------
template <class T> tensor3<T> operator * (const tensor2<T>& mat, const tensor3<T>& ts)
//------------------------------------------------------------------------------------
{
  assert (mat.Rows());
  assert (mat.Columns() == ts.dim1());

  tensor3<T> tsr(mat.Rows(),ts.dim2(),ts.dim3());
  for (int p=1; p<=mat.Rows(); p++)
    for (int q=1; q<=ts.dim1(); q++)
      for (int i=1; i<=ts.dim2(); i++)
	for (int j=1; j<=ts.dim3(); j++)
	  tsr (p,i,j) += mat(p,q) * ts(q,i,j);

  return tsr;
}


//------------------------------------------------------------------------
template <class T> ostream& operator << (ostream& s, const tensor3<T>& tn)
//------------------------------------------------------------------------
{
  assert (tn.size);

  for (int i=1; i<=tn.size; i++)
      s << tn[i];
  s << endl;

  return s;
}


//------------------------------------------------------------------------
template <class T> istream& operator >> (istream& s, const tensor3<T>& tn)
//------------------------------------------------------------------------
{
  assert (tn.size);

  for (int i=1; i<=tn.size; i++)
      s >> tn[i];

  return s;
}


//---------------------------------------------------------
template <class T> void tensor3<T>::save (const char* path)
//---------------------------------------------------------
{
  assert (size);

  ofstream file(path, ios::out);
  assert (!file.fail());
  file << *this;
  file.close();
}


//---------------------------------------------------------
template <class T> void tensor3<T>::read (const char* path)
//---------------------------------------------------------
{
  assert (size);

  ifstream file(path, ios::in);
  assert (!file.fail());
  file >> *this;
  file.close();
}


//---------------------------------------------------------
template <class T> tensor3<T>& tensor3<T>::approximation ()
//---------------------------------------------------------
{
  for (int i=0; i<size; i++)
    tsr[i].approximation();

  return *this;
}


#endif
