/***************************************************************
 *                 Mathematical Object Library                 *
 *      class tensor4 : declarations for 4th order tensors     *
 *                    simula+@metz.ensam.fr                    *
 *                   GNU/linux version 0.6.0                   *
 *            software under General Public License            *
 ***************************************************************
 * copyright  2004,2005,2006 COLLARD Christophe
 * copyright  2004,2005,2006 Laboratoire de Physique et Mcanique des Matriaux (LPMM - UMR 7554)
 * copyright  2004,2005,2006 Laboratoire de Mathmatiques et ses Applications de Valenciennes (LAMAV - EA 4015)
 ***************************************************************/

/*! \class tensor4
    \brief 4th order tensor library \n

    \htmlonly 
    <FONT color="#838383">

    tensor4 belongs to Mathematical Object Libraries (MOL++) </br>
    MOL++ is part of Simula+ <br><br>

    Simula+ is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version. <br><br>

    Simula+ is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details. <br><br>

    You should have received a copy of the GNU General Public License
    along with Simula+; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

    </FONT>
    \endhtmlonly

    \author copyright \htmlonly &#169; \endhtmlonly 2004, 2005, 2006 Christophe COLLARD \n
	    copyright \htmlonly &#169; \endhtmlonly 2004, 2005, 2006 Laboratoire de Physique et Mcanique des Matriaux (LPMM - UMR 7554) \n
	    copyright \htmlonly &#169; \endhtmlonly 2004, 2005, 2006 Laboratoire de Mathmatiques et ses Applications de Valenciennes (LAMAV - EA 4015)
    \version 0.6.0
    \date 2004-2006
    \bug none
    \warning none
*/

#ifndef __cplusplus
#error Must use C++ for the type tensor4
#endif

#if !defined(_TENSORS4_H)
#define _TENSORS4_H


#if !defined(__VECTORS_H)
#include "vectors.h"
#endif

#if !defined(__MATRIX_H)
#include "matrix.h"
#endif

#if !defined(__TENSORS2_H)
#include "tensors2.h"
#endif

#if !defined(__TENSORS3_H)
#include "tensors3.h"
#endif

using namespace std;


//==============================
template <class T> class tensor4
//==============================
{
  protected :
    // tensor4 are stored using tensor3
    tensor3<T> *tsr;
    int size;

  public :
    tensor4 ();                   // default constructor
    tensor4 (int, bool=true, T=0);                // this constructor allocates memory
    tensor4 (int, int, int, int, bool=true, T=0); // this constructor allocates memory
    tensor4 (const tensor4<T>&);  // copy constructor
    tensor4 (tensor4<T>*);        // copy constructor for temporary objects
    ~tensor4 ();

    virtual int dim1 () const {return size;}             // returns the tensor 1st component size : i
    virtual int dim2 () const {return tsr[0].dim1();}    // returns the tensor 2nd component size : j
    virtual int dim3 () const {return tsr[0].dim2();}    // returns the tensor 3rd component size : k
    virtual int dim4 () const {return tsr[0].dim3();}    // returns the tensor 3rd component size : l
    virtual void create (int);              // partial memory allocation (don't use this function except if you know what you're doing !!)
    virtual void assign (int,int,int,int);  // allocates memory without initilizing the tensor components to zero
    operator int () const {return size;}         // cast operator : tensor4 -> int
    friend bool operator ! (const tensor4<T>& t) {return !t.size;}

    virtual T&   operator () (int,int,int,int) const;    // returns the element Pijkl : P(tensor3,tensor2,line,column)
    tensor3<T>& operator [] (int) const;                 // returns a 3rd order tensor Cijkl -> Tjkl
    virtual tensor4<T>& operator = (const tensor4<T>&);  // defines equal operator for 4th order tensors
    virtual tensor4<T>& operator = (tensor4<T>*);  // allocates the data from the right hand temporary object to the left hand object - no copy (deletes temporary object)
    virtual tensor4<T>& operator &= (const tensor4<T>&); // copies tensor4 without size check (use this operator with great care)
    virtual tensor4<T>& operator &= (tensor4<T>*); // allocates the data from the right hand temporary object to the left hand object (deletes temporary object) - no size check - no copy (use this operator with great care)
    template <class M> friend tensor4<M> operator +  (const tensor4<M>&, const tensor4<M>&); // overloads operator + for 4th order tensors
    template <class M> friend tensor4<M> operator -  (const tensor4<M>&, const tensor4<M>&); // overloads operator - for 4th order tensors
    template <class M> friend M          operator |  (const tensor4<M>&, const tensor4<M>&); // T_ijkl  R_ijkl
    template <class M> friend tensor4<M> operator || (const tensor4<M>&, const tensor4<M>&); // P_ijkl = T_ijpq R_pqkl
    tensor4<T>& operator += (const tensor4<T>&);     // overloads += operator for tensors
    tensor4<T>& operator -= (const tensor4<T>&);     // overloads -= operator for tensors
    template <class M> friend bool operator == (const tensor4<M>&, const tensor4<M>&);  // compares 2 tensors
    template <class M> friend bool operator != (const tensor4<M>&, const tensor4<M>&);
    // operations with scalars
    tensor4<T>& operator *= (const T&);
    template <class M> friend tensor4<M> operator * (const tensor4<M>&, const M&);
    template <class M> friend tensor4<M> operator * (const M&, const tensor4<M>&);
    // operations withs vectors
    template <class M> friend tensor3<M> operator * (const tensor4<M>&, const vector<M>&);   // P_ijk = T_ijkl * V_l
    template <class M> friend tensor3<M> operator * (const vector<M>&, const tensor4<M>&);   // P_jkl = T_ijkl * V_i
    template <class M> friend tensor3<M> operator | (const tensor4<M>&, const vector<M>&);   // P_ijl = T_ijkl * V_k
    template <class M> friend tensor3<M> operator | (const vector<M>&, const tensor4<M>&);   // P_ikl = T_ijkl * V_j
    template <class M> friend tensor3<M> tensor_sum (const tensor4<M>&, const vector<M>&, int); // P = T : v
    // operation with tensor2 (2nd order tensor)
    template <class M> friend tensor2<M> operator || (const tensor4<M>&, const tensor2<M>&);  // P_ij = T_ijkl * M_kl
    template <class M> friend tensor2<M> operator || (const tensor2<M>&, const tensor4<M>&);  // P_kl = M_ij * T_ijkl
    template <class M> friend tensor4<M> operator ^  (const tensor2<M>&, const tensor2<M>&);  // T_ijkl = M1_ij M2_kl
    // operation with 3rd order tensor
    template <class M> friend tensor4<M> operator ^  (const tensor3<M>&, const tensor3<M>&); // T_ijkl = Pn,ij * Rn,kl = P_ij[n] * R_kl[n]
    template <class M> friend tensor3<M> operator || (const tensor3<M>&, const tensor4<M>&);  // P_n,ij = R_n,kl T_klij
    template <class M> friend tensor3<M> operator || (const tensor4<M>&, const tensor3<M>&);  // P_n,ij = T_ijkl R_n,kl

    template <class Tf> friend tensor4<Tf> Id4s (int);  // returns identity tensor for symmetric tensors : Tijkl = Tjikl = Tijlk

    template <class M> friend ostream& operator << (ostream&, const tensor4<M>&);  // overloads output stream for tensors
    template <class M> friend istream& operator >> (istream&, const tensor4<M>&);  // overloads output stream for tensors
    virtual void save (const char*); // writes to disk
    virtual void read (const char*); // reads from disk
    tensor4<T>& approximation ();
};


//=====Private methods for tensor4=============================================


//=====Public methods for tensor4==============================================


//---------------------------------------
template <class T> tensor4<T>::tensor4 ()
//---------------------------------------
{
  size = 0;
}


//-----------------------------------------------------------------
template <class T> tensor4<T>::tensor4 (int sz, bool init, T value)
//-----------------------------------------------------------------
{
  assert (sz>0);
  tsr = new tensor3<T> [size=sz];
  for (int i=0; i<size; i++)
    tsr[i] =& tensor3<T> (sz,sz,sz,init,value);
}


//---------------------------------------------------------------------------------------------------------------
template <class T> tensor4<T>::tensor4 (int ntensor3, int ntensor2, int nrows, int ncolumns , bool init, T value)
//---------------------------------------------------------------------------------------------------------------
{
  assert (ntensor3>0 && ntensor2>0 && nrows>0 && ncolumns>0);
  tsr = new tensor3<T> [size=ntensor3];
  for (int i=0; i<size; i++)
    tsr[i] =& tensor3<T> (ntensor2,nrows,ncolumns,init,value);
}


//-----------------------------------------------------------
template <class T> tensor4<T>::tensor4 (const tensor4<T>& tn)
//-----------------------------------------------------------
{
  assert (tn.dim2() && tn.dim3() && tn.dim4() && tn.size);
  tsr = new tensor3<T> [size=tn.size];
  for (int i=0; i<tn.size; i++)
      tsr[i] = tn.tsr[i];
}


//-----------------------------------------------------
template <class T> tensor4<T>::tensor4 (tensor4<T>* tn)
//-----------------------------------------------------
{
  assert ((*tn).dim2() && (*tn).dim3() && (*tn).dim4() && (*tn).size);

  size = (*tn).size;
  tsr = (*tn).tsr;
  (*tn).size = 0;
}


//----------------------------------------
template <class T> tensor4<T>::~tensor4 ()
//----------------------------------------
{
  // destruction of the memory allocated for the tensor

  if (size) delete [] tsr; // free memory only if it's been affected
  size = 0;
}


//-------------------------------------------------------
template <class T> void tensor4<T>::create (int ntensor3)
//-------------------------------------------------------
{
  assert (ntensor3>0);
  assert (!size);

  tsr = new tensor3<T> [size = ntensor3];
}


//----------------------------------------------------------------------------------------------
template <class T> void tensor4<T>::assign (int ntensor3, int ntensor2, int nrows, int ncolumns)
//----------------------------------------------------------------------------------------------
{
  assert (ntensor3>0 && !size);

  tsr = new tensor3<T> [size = ntensor3];
  for (int i=0; i<size; i++)
    tsr[i] =& tensor3<T> (ntensor2,nrows,ncolumns,false);
}


//------------------------------------------------------------------------------
template <class T> T& tensor4<T>::operator () (int i, int j, int k, int l) const
//------------------------------------------------------------------------------
{
  assert (size);
  assert (i>0 && i<=size);

  return tsr[i-1](j,k,l);
}


//------------------------------------------------------------------
template <class T> tensor3<T>& tensor4<T>::operator [] (int i) const
//------------------------------------------------------------------
{
  assert ((i>0)&&(i<=size));
  return tsr[i-1];
}


//--------------------------------------------------------------------------
template <class T> tensor4<T>& tensor4<T>::operator = (const tensor4<T>& tn)
//--------------------------------------------------------------------------
{
  assert (tn.size>0);

  if (!size) tsr = new tensor3<T> [size=tn.size];
  assert (size==tn.size);
  for (int i=0; i<size; i++) 
      tsr[i] = tn.tsr[i];

  return *this;
}


//--------------------------------------------------------------------
template <class T> tensor4<T>& tensor4<T>::operator = (tensor4<T>* tn)
//--------------------------------------------------------------------
{
  assert ((*tn).dim2() && (*tn).dim3() && (*tn).dim4() && (*tn).size);

  if (!size) size = (*tn).size;
  else
    { assert (size==(*tn).size && dim2()==(*tn).dim2() && dim3()==(*tn).dim3() && dim4()==(*tn).dim4());
      delete [] tsr;
    }
  tsr = (*tn).tsr;
  (*tn).size = 0;

  return *this;
}


//---------------------------------------------------------------------------
template <class T> tensor4<T>& tensor4<T>::operator &= (const tensor4<T>& tn)
//---------------------------------------------------------------------------
{
  if (!tn)
    { if (size) delete [] tsr;
      size = 0;
    }

  else
    { if (size != tn.size)
	{ if (size) delete [] tsr;
	  tsr = new tensor3<T> [size = tn.size];  // allocates memory with the = operator for tensor2 (no initialization)
	}
      for (int i=1; i<=size; i++)
	(*this)[i] &= tn[i];
    }

  return *this;
}


//---------------------------------------------------------------------
template <class T> tensor4<T>& tensor4<T>::operator &= (tensor4<T>* tn)
//---------------------------------------------------------------------
{
  if (!tn)
    { if (size) delete [] tsr;
      size = 0;
    }

  else
    { if (size)
	delete [] tsr;
      size = (*tn).size;
      tsr = (*tn).tsr;
      (*tn).size = 0;
    }

  return *this;
}


//-----------------------------------------------------------------------------------
template <class T> tensor4<T> operator + (const tensor4<T>& t1, const tensor4<T>& t2)
//-----------------------------------------------------------------------------------
{
  assert (t1.dim1()==t2.dim1()); // tensors must have the same size
  assert(t1.dim1());
  tensor4<T> t = t1;
  return t+=t2;
}


//-----------------------------------------------------------------------------------
template <class T> tensor4<T> operator - (const tensor4<T>& t1, const tensor4<T>& t2)
//-----------------------------------------------------------------------------------
{
  assert (t1.dim1()==t2.dim1()); // tensors must have the same size
  assert(t1.dim1());
  tensor4<T> t = t1;
  return t-=t2;
}


//--------------------------------------------------------------------------
template <class T> T operator | (const tensor4<T>& t1, const tensor4<T>& t2)
//--------------------------------------------------------------------------
{
  assert (t1.size==t2.size); // tensors must have the same size
  assert(t1.size);

  T sum=0;
  for (int i=1; i<=t1.size; i++)
    sum += (t1[i] | t2[i]);

  return sum;
}


//------------------------------------------------------------------------------------
template <class T> tensor4<T> operator || (const tensor4<T>& t1, const tensor4<T>& t2)
//------------------------------------------------------------------------------------
{
  assert ( (t1.dim3() == t2.dim1()) && (t1.dim4() == t2.dim2()) );

  tensor4<T> t3 (t1.dim1(),t1.dim2(),t2.dim3(),t2.dim4());
  for (int i=1; i<=t1.dim1(); i++)
    for (int j=1; j<=t1.dim2(); j++)
      for (int p=1; p<=t1.dim3(); p++)
	for (int q=1; q<=t1.dim4(); q++)
	  for (int k=1; k<=t2.dim3(); k++)
	    for (int l=1; l<=t2.dim4(); l++)
	      t3(i,j,k,l) += t1(i,j,p,q) * t2(p,q,k,l);

  return t3;
}


//---------------------------------------------------------------------------
template <class T> tensor4<T>& tensor4<T>::operator += (const tensor4<T>& ts)
//---------------------------------------------------------------------------
{
  assert (size);  // the tensor must be defined
  assert (size == ts.size);  // tensors must have the same size
  for (int i=0; i<size; i++)
    tsr[i] += ts.tsr[i];
  return *this;
}


//---------------------------------------------------------------------------
template <class T> tensor4<T>& tensor4<T>::operator -= (const tensor4<T>& ts)
//---------------------------------------------------------------------------
{
  assert (size);  // the tensor must be defined
  assert (size == ts.size);  // tensors must have the same size
  for (int i=0; i<size; i++)
    tsr[i] -= ts.tsr[i];
  return *this;
}


//------------------------------------------------------------------------------
template <class T> bool operator == (const tensor4<T>& t1, const tensor4<T>& t2)
//------------------------------------------------------------------------------
{
  assert (t1.size);
  int result=(t1.size==t2.size);
  for (int i=1; (i<=t1.size)&&(result); i++)
    result *= (t1[i]==t2[i]);
  return result;
}


//------------------------------------------------------------------------------
template <class T> bool operator != (const tensor4<T>& t1, const tensor4<T>& t2)
//------------------------------------------------------------------------------
{
  return !(t1 == t2);
}


//-------------------------------------------------------------------
template <class T> tensor4<T>& tensor4<T>::operator *= (const T& elt)
//-------------------------------------------------------------------
{
  assert (size);
  for (int i=0; i<size; i++)
    tsr [i] *= elt;
  return *this;
}


//---------------------------------------------------------------------------
template <class T> tensor4<T> operator * (const tensor4<T>& ts, const T& elt)
//---------------------------------------------------------------------------
{
  assert (ts.size);
  tensor4<T> tr = ts;
  return  tr *= elt;
}


//---------------------------------------------------------------------------
template <class T> tensor4<T> operator * (const T& elt, const tensor4<T>& ts)
//---------------------------------------------------------------------------
{
  assert (ts.size);
  tensor4<T> tr = ts;
  return  tr *= elt;
}


//---------------------------------------------------------------------------------
template <class T> tensor3<T> operator * (const tensor4<T>& ts, const vector<T>& v)
//---------------------------------------------------------------------------------
{
  assert (v.dim());
  assert (v.dim() == ts.dim4());

  tensor3<T> ts3;
  ts3.create(ts.dim1());
  for (int i=1; i<=ts.size; i++)
    ts3[i] =& (ts[i] * v);

  return ts3;
}


//---------------------------------------------------------------------------------
template <class T> tensor3<T> operator * (const vector<T>& v, const tensor4<T>& ts)
//---------------------------------------------------------------------------------
{
  assert (v.dim());
  assert (v.dim() == ts.dim1());
  tensor3<T> tr(ts.dim2(),ts.dim3(),ts.dim4());
  for (int i=1; i<=ts.size; i++)
    tr += v[i] * ts[i];
  return tr;
}


//---------------------------------------------------------------------------------
template <class T> tensor3<T> operator | (const tensor4<T>& ts, const vector<T>& v)
//---------------------------------------------------------------------------------
{
  assert (v.dim());
  assert (v.dim() == ts.dim3());

  tensor3<T> tr(ts.dim1(),ts.dim2(),ts.dim4());
  for (int i=1; i<=ts.size; i++)
    tr[i] += ts[i] | v;

  return tr;
}


//---------------------------------------------------------------------------------
template <class T> tensor3<T> operator | (const vector<T>& v, const tensor4<T>& ts)
//---------------------------------------------------------------------------------
{
  assert (v.dim());
  assert (v.dim() == ts.dim2());

  tensor3<T> tr(ts.dim1(),ts.dim3(),ts.dim4());
  for (int i=1; i<=ts.size; i++)
    tr[i] += v * ts[i];

  return tr;
}


//----------------------------------------------------------------------------------------
template <class T> tensor3<T> tensor_sum (const tensor4<T>& ts, const vector<T>& v, int i)
//----------------------------------------------------------------------------------------
{
  assert (i>=1 && i<=4);

  tensor3<T> tr;
  if (i==1) tr =& (v*ts);
  else if (i==2) tr =& (v|ts);
       else if (i==3) tr =& (ts|v);
            else tr =& (ts*v);

  return tr;
}


//------------------------------------------------------------------------------------
template <class T> tensor2<T> operator || (const tensor4<T>& t, const tensor2<T>& mat)
//------------------------------------------------------------------------------------
{
  assert (t.dim3() == mat.Rows());
  assert (t.dim4() == mat.Columns());

  tensor2<T> mat_result(t.dim1(),t.dim2(),false);
  for (int i=1; i<=t.dim1(); i++)
    for (int j=1; j<=t.dim2(); j++)
      mat_result(i,j) = t[i][j] | mat;

  return mat_result;
}


//------------------------------------------------------------------------------------
template <class T> tensor2<T> operator || (const tensor2<T>& mat, const tensor4<T>& t)
//------------------------------------------------------------------------------------
{
  assert (t.dim1() == mat.Rows());
  assert (t.dim2() == mat.Columns());

  tensor2<T> mat_result(t.dim3(),t.dim4());
  for (int i=1; i<=t.dim1(); i++)
    for (int j=1; j<=t.dim2(); j++)
      for (int k=1; k<=t.dim3(); k++)
	for (int l=1; l<=t.dim4(); l++)
	  mat_result(k,l) += t(i,j,k,l) * mat(i,j);

  return mat_result;
}


//---------------------------------------------------------------------------------------
template <class T> tensor4<T> operator ^ (const tensor2<T>& mat1, const tensor2<T>& mat2)
//---------------------------------------------------------------------------------------
{
  assert (mat1.Rows() && mat1.Columns() && mat2.Rows() && mat2.Columns());

  tensor4<T> tsr(mat1.Rows(),mat1.Columns(),mat2.Rows(),mat2.Columns(),false);
  for (int i=1; i<=mat1.Rows(); i++)
    for (int j=1; j<=mat1.Columns(); j++)
      for (int k=1; k<=mat2.Rows(); k++)
	for (int l=1; l<=mat2.Columns(); l++)
	  tsr(i,j,k,l) = mat1(i,j) * mat2(k,l);

  return tsr;
}


//-----------------------------------------------------------------------------------
template <class T> tensor4<T> operator ^ (const tensor3<T>& t1, const tensor3<T>& t2)
//-----------------------------------------------------------------------------------
{
  assert (t1.dim1() == t2.dim1());
  tensor4<T> t4(t1.dim2(), t1.dim3(), t2.dim2(), t2.dim3());
  for (int n=1; n<=t1.dim1(); n++)
    t4 += (t1[n] ^ t2[n]);
  return t4;
}


//------------------------------------------------------------------------------------
template <class T> tensor3<T> operator || (const tensor3<T>& t3, const tensor4<T>& t4)
//------------------------------------------------------------------------------------
{
  assert (t3.dim1() && t4.dim1());
  assert ( (t3.dim2() == t4.dim1()) && (t3.dim3() == t4.dim2()) );

  tensor3<T> ts;
  ts.create(t3.dim1());

  for (int n=1; n<=t3.dim1(); n++)
    ts[n] =& (t3[n] || t4);

  return ts;
}


//------------------------------------------------------------------------------------
template <class T> tensor3<T> operator || (const tensor4<T>& t4, const tensor3<T>& t3)
//------------------------------------------------------------------------------------
{
  assert (t3.dim1() && t4.dim1());
  assert ( (t3.dim2() == t4.dim3()) && (t3.dim3() == t4.dim4()) );

  tensor3<T> ts;
  ts.create(t3.dim1());

  for (int n=1; n<=t3.dim1(); n++)
    ts[n] =& (t4 || t3[n]);

  return ts;
}


//------------------------------------------
template <class Tf> tensor4<Tf> Id4s (int n)
//------------------------------------------
{
  assert (n > 0);
  tensor4<Tf> I(n,n,n,n);

  for (int i=1; i<=n; i++)
    for (int j=1; j<=n; j++)
      { I(i,j,i,j) += .5;
	I(i,j,j,i) += .5;
      }

  return I;
}


//------------------------------------------------------------------------
template <class T> ostream& operator << (ostream& s, const tensor4<T>& tn)
//------------------------------------------------------------------------
{
  for (int i=1; i<=tn.size; i++)
      s << tn[i];
  s << endl;
  return s;
}


//------------------------------------------------------------------------
template <class T> istream& operator >> (istream& s, const tensor4<T>& tn)
//------------------------------------------------------------------------
{
  assert (tn.size);

  for (int i=1; i<=tn.size; i++)
      s >> tn[i];

  return s;
}


//---------------------------------------------------------
template <class T> void tensor4<T>::save (const char* path)
//---------------------------------------------------------
{
  assert (size);

  ofstream file(path, ios::out);
  assert (!file.fail());
  file << *this;
  file.close();
}


//---------------------------------------------------------
template <class T> void tensor4<T>::read (const char* path)
//---------------------------------------------------------
{
  assert (size);

  ifstream file(path, ios::in);
  assert (!file.fail());
  file >> *this;
  file.close();
}


//---------------------------------------------------------
template <class T> tensor4<T>& tensor4<T>::approximation ()
//---------------------------------------------------------
{
  for (int i=0; i<size; i++)
    tsr[i].approximation();
  return *this;
}


#endif
