/***************************************************************
 *                 Mathematical Object Library                 *
 *      Class Legendre : declaration for Legendre polynoms     *
 *                    simula+@metz.ensam.fr                    *
 *                   GNU/linux version 2.0.0                   *
 *            software under General Public License            *
 ***************************************************************
 * copyright  2002,2003,2004,2005,2006,2008 COLLARD Christophe
 * copyright  2002,2003,2004,2005,2006,2008 Laboratoire de Physique et Mcanique des Matriaux (LPMM - UMR 7554)
 * copyright  2002,2003,2004,2005,2006 Laboratoire de Mathmatiques et ses Applications de Valenciennes (LAMAV - EA 4015)
 ***************************************************************/

/*! \namespace mol
    \brief Mathematical Object Libraries
*/

/*! \class mol::Legendre
    \brief Legendre polynoms library \n

    \htmlonly 
    <FONT color="#838383">

    Legendre belongs to Mathematical Object Libraries (MOL++) </br>
    MOL++ is part of Simula+ <br><br>

    Simula+ is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version. <br><br>

    Simula+ is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details. <br><br>

    You should have received a copy of the GNU General Public License
    along with Simula+; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

    </FONT>
    \endhtmlonly

    \author copyright \htmlonly &#169; \endhtmlonly 2002, 2003, 2004, 2005, 2006, 2008 Christophe COLLARD \n
	    copyright \htmlonly &#169; 2002, 2003, 2004, 2005, 2006, 2008 Laboratoire de Physique et M&#233;canique des Mat&#233;riaux (LPMM - UMR 7554) \endhtmlonly \n
	    copyright \htmlonly &#169; 2002, 2003, 2004, 2005, 2006 Laboratoire de Math&#233;matiques et ses Applications de Valenciennes (LAMAV - EA 4015) \endhtmlonly
    \version 2.0.0
    \date 2002-2008
    \bug none
    \warning none
*/

#ifndef __cplusplus
#error Must use C++ for the type Legendre
#endif

#ifndef _legendre_h
#define _legendre_h


#if !defined (__IOSTREAM_H)
#include <iostream>
#endif

#if !defined(__FSTREAM_H)
#include <fstream>
#endif

#if !defined(__ASSERT_H)
#include <assert.h>
#endif

#if !defined(__MATH_H)
#include <math.h>
#endif

#if !defined(__PARAMETERS_H)
#include "../parameters.h"
#endif

#if !defined(__MATHS_H)
#include "maths.h"
#endif

using namespace std;

namespace mol
{


//===============================
template <class T> class Legendre
//===============================
{
  private :
    int deg;     // power in X for the Legendre polynom
    T p, dp, x; // store the value for the Legendre polynom and for its derivative in x

  public:
    Legendre () {deg=-1;}    // default constructor
    Legendre (int);             // constructor with the polynom degree
    ~Legendre () {deg=-1;}       // destructor

    T operator () (T);             // returns the value P(X)
    virtual T d_dx (T);            // returns the n_th derivative at point x
};


//=====Private methods for Legendre===========================================


//=====Public methods for Legendre============================================


//----------------------------------------------
template <class T> Legendre<T>::Legendre (int n)
//----------------------------------------------
{  
  assert (n>=0);
  deg = n;
  x = -1.;        // x default value
  (*this)((T) 0); // p, dp default values
}


//--------------------------------------------------
template <class T> T Legendre<T>::operator () (T xx)
//--------------------------------------------------
{
  assert (deg>=0);

  if (x != xx) // do not use this test : if (abs(x-xx) >= epsilon)
    { T p1=1, p2=0, p3;
      x = xx ;
      for (int j=1; j<=deg; j++)
	{ p3 = p2;
	  p2 = p1;
	  p1 = ( (2*j-1)*x*p2 - (j-1)*p3 ) / ((T)j);
	}
      p = p1;
      if (abs(xx)-1) dp = deg * (x*p1 - p2) / (power(x,2) - 1);
    }

  return p;
}


//------------------------------------------
template <class T> T Legendre<T>::d_dx(T xx)
//------------------------------------------
{
  assert (abs(abs(xx)-1)>epsilon); // the derivative of the Legendre polynom is not defined in -1 and 1

  if (x!=xx) (*this)(xx); // do not use this test : if (abs(x-xx) >= epsilon)

  return dp;
}


}


#endif
