/***************************************************************
 *                 Mathematical Object Library                 *
 *      class tensor4 : declarations for 4th order tensors     *
 *                    simula+@metz.ensam.fr                    *
 *                   GNU/linux version 2.4.0                   *
 *            software under General Public License            *
 ***************************************************************
 * copyright © 2004,2005,2006,2007,2008,2009,2010 COLLARD Christophe
 * copyright © 2004,2005,2006,2007,2008,2009,2010 Laboratoire de Physique et Mécanique des Matériaux (LPMM - CNRS)
 * copyright © 2004,2005,2006,2007 Laboratoire de Mathématiques et ses Applications de Valenciennes (LAMAV)
 ***************************************************************/

/*! \namespace mol
    \brief Mathematical Object Libraries
*/

/*! \class mol::tensor4
    \brief 4th order tensor library \n

    \htmlonly 
    <FONT color="#838383">

    tensor4 belongs to Mathematical Object Libraries (MOL++) </br>
    MOL++ is part of Simula+ <br><br>

    Simula+ is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version. <br><br>

    Simula+ is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details. <br><br>

    You should have received a copy of the GNU General Public License
    along with Simula+; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

    </FONT>
    \endhtmlonly

    \author copyright \htmlonly &#169; \endhtmlonly 2004, 2005, 2006, 2007, 2008, 2009, 2010 Christophe COLLARD \n
	    copyright \htmlonly &#169; 2004, 2005, 2006, 2007, 2008, 2009, 2010 Laboratoire de Physique et M&#233;canique des Mat&#233;riaux (LPMM - CNRS) \endhtmlonly \n
	    copyright \htmlonly &#169; 2004, 2005, 2006, 2007 Laboratoire de Math&#233;matiques et ses Applications de Valenciennes (LAMAV) \endhtmlonly
    \version 2.4.0
    \date 2004-2010
    \bug none
    \warning none
*/

#ifndef __cplusplus
#error Must use C++ for the type tensor4
#endif

#if !defined(_TENSORS4_H)
#define _TENSORS4_H


#if !defined(__VECTORS_H)
#include "vectors.h"
#endif

#if !defined(__MATRIX_H)
#include "matrix.h"
#endif

#if !defined(__TENSORS2_H)
#include "tensors2.h"
#endif

#if !defined(__TENSORS3_H)
#include "tensors3.h"
#endif

using namespace std;

namespace mol
{


//==============================
template <class T> class tensor4
//==============================
{
  protected :
    // tensor4 are stored using tensor3
    tensor3<T> *tsr;
    int size;

  public :
    tensor4 ();                   // default constructor
    tensor4 (int, bool=true, T=0);                // this constructor allocates memory
    tensor4 (int, int, int, int, bool=true, T=0); // this constructor allocates memory
    tensor4 (const tensor4<T>&);  // copy constructor
    tensor4 (tensor4<T>*);        // copy constructor for temporary objects
    ~tensor4 ();

    virtual int dim1 () const {return size;}             // returns the tensor 1st component size : i
    virtual int dim2 () const {return tsr[0].dim1();}    // returns the tensor 2nd component size : j
    virtual int dim3 () const {return tsr[0].dim2();}    // returns the tensor 3rd component size : k
    virtual int dim4 () const {return tsr[0].dim3();}    // returns the tensor 3rd component size : l
    virtual void create (int);              // partial memory allocation (don't use this function except if you know what you're doing !!)
    virtual void assign (int,int,int,int);  // allocates memory without initilizing the tensor components to zero
    friend bool operator ! (const tensor4<T>& t) {return !t.size;}

    virtual T&   operator () (int,int,int,int) const;    // returns the element Pijkl : P(tensor3,tensor2,line,column)
    tensor3<T>& operator [] (int) const;                 // returns a 3rd order tensor Cijkl -> Tjkl
    virtual tensor4<T>& operator = (const tensor4<T>&);  // defines equal operator for 4th order tensors
    virtual tensor4<T>& operator = (tensor4<T>*);  // allocates the data from the right hand temporary object to the left hand object - no copy (deletes temporary object)
    virtual tensor4<T>& operator &= (const tensor4<T>&); // copies tensor4 without size check (use this operator with great care)
    virtual tensor4<T>& operator &= (tensor4<T>*); // allocates the data from the right hand temporary object to the left hand object (deletes temporary object) - no size check - no copy (use this operator with great care)
    template <class Tf> friend tensor4<Tf> operator +  (const tensor4<Tf>&, const tensor4<Tf>&); // overloads operator + for 4th order tensors
    template <class Tf> friend tensor4<Tf> operator -  (const tensor4<Tf>&, const tensor4<Tf>&); // overloads operator - for 4th order tensors
    template <class Tf> friend Tf          operator |  (const tensor4<Tf>&, const tensor4<Tf>&); // T_ijkl  R_ijkl
    template <class Tf> friend tensor4<Tf> operator || (const tensor4<Tf>&, const tensor4<Tf>&); // P_ijkl = T_ijpq R_pqkl
    tensor4<T>& operator += (const tensor4<T>&);     // overloads += operator for tensors
    tensor4<T>& operator -= (const tensor4<T>&);     // overloads -= operator for tensors
    template <class Tf> friend bool operator == (const tensor4<Tf>&, const tensor4<Tf>&);  // compares 2 tensors
    template <class Tf> friend bool operator != (const tensor4<Tf>&, const tensor4<Tf>&);

    //------------------------
    // operations with scalars
    //------------------------
    tensor4<T>& operator *= (const T&);
    template <class Tf> friend tensor4<Tf> operator * (const tensor4<Tf>&, const Tf&);
    template <class Tf> friend tensor4<Tf> operator * (const Tf&, const tensor4<Tf>&);

    //-------------------------
    // operations withs vectors
    //-------------------------
    template <class Tf> friend tensor3<Tf> operator * (const tensor4<Tf>&, const vector<Tf>&);   // P_ijk = T_ijkl * V_l
    template <class Tf> friend tensor3<Tf> operator * (const vector<Tf>&, const tensor4<Tf>&);   // P_jkl = T_ijkl * V_i
    template <class Tf> friend tensor3<Tf> operator | (const tensor4<Tf>&, const vector<Tf>&);   // P_ijl = T_ijkl * V_k
    template <class Tf> friend tensor3<Tf> operator | (const vector<Tf>&, const tensor4<Tf>&);   // P_ikl = T_ijkl * V_j
    template <class Tf> friend tensor3<Tf> tensor_sum (const tensor4<Tf>&, const vector<Tf>&, int); // P = T : v

    //------------------------------------------
    // operation with tensor2 (2nd order tensor)
    //------------------------------------------
    template <class Tf> friend tensor2<Tf> operator || (const tensor4<Tf>&, const tensor2<Tf>&);  // P_ij = T_ijkl * M_kl
    template <class Tf> friend tensor2<Tf> operator || (const tensor2<Tf>&, const tensor4<Tf>&);  // P_kl = M_ij * T_ijkl
    template <class Tf> friend tensor4<Tf> operator ^  (const tensor2<Tf>&, const tensor2<Tf>&);  // T_ijkl = M1_ij M2_kl

    //--------------------------------
    // operation with 3rd order tensor
    //--------------------------------
    template <class Tf> friend tensor4<Tf> operator ^  (const tensor3<Tf>&, const tensor3<Tf>&); // T_ijkl = Pn,ij * Rn,kl = P_ij[n] * R_kl[n]
    template <class Tf> friend tensor3<Tf> operator || (const tensor3<Tf>&, const tensor4<Tf>&);  // P_n,ij = R_n,kl T_klij
    template <class Tf> friend tensor3<Tf> operator || (const tensor4<Tf>&, const tensor3<Tf>&);  // P_n,ij = T_ijkl R_n,kl

    template <class Tf> friend tensor4<Tf> Id4s (int);  // returns identity tensor for symmetric tensors : Tijkl = Tjikl = Tijlk
    template <class Tf> friend tensor4<Tf> change_basis (const matrix<Tf>&, const tensor4<Tf>&);

    template <class Tf> friend ostream& operator << (ostream&, const tensor4<Tf>&);  // overloads output stream for tensors
    template <class Tf> friend istream& operator >> (istream&, const tensor4<Tf>&);  // overloads output stream for tensors
    virtual void save (const char*);   // writes to disk
    virtual void save (const string&); // writes to disk
    virtual void read (const char*);   // reads from disk
    virtual void read (const string&);   // reads from disk
    tensor4<T>& approximation ();
};


//=====Private methods for tensor4=============================================


//=====Public methods for tensor4==============================================


/*!
  \brief Default constructor

*/

//---------------------------------------
template <class T> tensor4<T>::tensor4 ()
//---------------------------------------
{
  size = 0;
}


/*!
  \brief Constructor

  \param sz space dimensions (\f$\Bbb{R}^{sz} \times \Bbb{R}^{sz} \times \Bbb{R}^{sz} \times \Bbb{R}^{sz}\f$) for the \f$ 4^\text{th} \f$ order tensor
  \param init boolean that indicates if the constructor has to initialize the tensor coordinates (default value is true)
  \param value value for tensor coordinates (default value is 0)
*/

//-----------------------------------------------------------------
template <class T> tensor4<T>::tensor4 (int sz, bool init, T value)
//-----------------------------------------------------------------
{
  assert (sz>0);
  tsr = new tensor3<T> [size=sz];
  for (int i=0; i<size; i++)
    tsr[i] =& tensor3<T> (sz,sz,sz,init,value);
}


/*!
  \brief Constructor

  \param ntensor3 space dimension for the first component of the \f$ 4^\text{th} \f$ order tensor
  \param ntensor2 space dimension for the second component of the \f$ 4^\text{th} \f$ order tensor
  \param nrows space dimension for the third component of the \f$ 4^\text{th} \f$ order tensor
  \param ncolumns space dimension for the fourth component of the \f$ 4^\text{th} \f$ order tensor
  \param init boolean that indicates if the constructor has to initialize the tensor coordinates (default value is true)
  \param value value for tensor coordinates (default value is 0)
*/

//---------------------------------------------------------------------------------------------------------------
template <class T> tensor4<T>::tensor4 (int ntensor3, int ntensor2, int nrows, int ncolumns , bool init, T value)
//---------------------------------------------------------------------------------------------------------------
{
  assert (ntensor3>0 && ntensor2>0 && nrows>0 && ncolumns>0);
  tsr = new tensor3<T> [size=ntensor3];
  for (int i=0; i<size; i++)
    tsr[i] =& tensor3<T> (ntensor2,nrows,ncolumns,init,value);
}


/*!
  \brief Copy constructor

  \param tn \f$ 4^\text{th} \f$ order tensor to copy
*/

//-----------------------------------------------------------
template <class T> tensor4<T>::tensor4 (const tensor4<T>& tn)
//-----------------------------------------------------------
{
  assert (tn.dim2() && tn.dim3() && tn.dim4() && tn.size);
  tsr = new tensor3<T> [size=tn.size];
  for (int i=0; i<tn.size; i++)
      tsr[i] = tn.tsr[i];
}


/*!
  \brief Destructive copy constructor

  Allocates the components from the right hand side (temporary) object to the left hand side object. \n
  Since there is no copy of the coordinates of the (temporary) object, this object is destructed by this constructor.\n
  Moreover, if we define the following tensors:\n
  \f$ \textbf{tensor4$<$long double$>$ t1 (3, true, 1), t2 (5, true, 2);} \f$ \n
  the definition of the tensor \n
  \f$ \textbf{ tensor4$<$long double$>$ t3 (\&(t1 + t2));} \f$ \n
  is faster than \n
  \f$ \textbf{tensor4$<$long double$>$ t3 (t1 + t2);} \f$.

  \param tn address of the tensor to allocate
*/

//-----------------------------------------------------
template <class T> tensor4<T>::tensor4 (tensor4<T>* tn)
//-----------------------------------------------------
{
  assert ((*tn).dim2() && (*tn).dim3() && (*tn).dim4() && (*tn).size);

  size = (*tn).size;
  tsr = (*tn).tsr;
  (*tn).size = 0;
}


/*!
  \brief Destructor
*/

//----------------------------------------
template <class T> tensor4<T>::~tensor4 ()
//----------------------------------------
{
  // destruction of the memory allocated for the tensor

  if (size) delete [] tsr; // free memory only if it's been affected
  size = 0;
}


/*!
  \brief Partial memory allocation

  Allocates memory for the first component only. \n
  (don't use this function except if you know what you're doing !!)

  \param ntensor3 space dimension for the first component of the symmetric \f$ 4^\text{th} \f$ order tensor
*/

//-------------------------------------------------------
template <class T> void tensor4<T>::create (int ntensor3)
//-------------------------------------------------------
{
  assert (ntensor3>0);
  assert (!size);

  tsr = new tensor3<T> [size = ntensor3];
}


/*!
  \brief Full memory allocation without initialization of the tensor coordinates

  Allocates memory without initilizing the tensor components to zero.
  \param ntensor3 space dimension for the first component of the \f$ 4^\text{th} \f$ order tensor
  \param ntensor2 space dimension for the second component of the \f$ 4^\text{th} \f$ order tensor
  \param nrows space dimension for the third component of the \f$ 4^\text{th} \f$ order tensor
  \param ncolumns space dimension for the fourth component of the \f$ 4^\text{th} \f$ order tensor
*/

//----------------------------------------------------------------------------------------------
template <class T> void tensor4<T>::assign (int ntensor3, int ntensor2, int nrows, int ncolumns)
//----------------------------------------------------------------------------------------------
{
  assert (ntensor3>0 && !size);

  tsr = new tensor3<T> [size = ntensor3];
  for (int i=0; i<size; i++)
    tsr[i] =& tensor3<T> (ntensor2,nrows,ncolumns,false);
}


/*!
  \brief Returns the reference of a coordinate for a \f$ 4^\text{th} \f$ order tensor

  Warning : we use the mathematical notations so the tensor coordinates are from 1 to the space dimension.
  \param i first coordinate of the \f$ 4^\text{th} \f$ order tensor
  \param j second coordinate of the \f$ 4^\text{th} \f$ order tensor
  \param k third coordinate of the \f$ 4^\text{th} \f$ order tensor
  \param l fourth coordinate of the \f$ 4^\text{th} \f$ order tensor

  \return coordinate \f$ T(i,j,k,l) \f$ of the \f$ 4^\text{th} \f$ order tensor \f$ \bold{T} \f$
*/

//------------------------------------------------------------------------------
template <class T> T& tensor4<T>::operator () (int i, int j, int k, int l) const
//------------------------------------------------------------------------------
{
  assert (size);
  assert (i>0 && i<=size);

  return tsr[i-1](j,k,l);
}


/*!
  \brief Returns the reference of a set  of coordinates for a \f$ 4^\text{th} \f$ order tensor

  Warning : we use the mathematical notations so the tensor coordinates are from 1 to the space dimension.

  \param i first coordinate of the \f$ 4^\text{th} \f$ order tensor

  \return the reference of the \f$ 3^\text{rd} \f$ order tensor \f$ T(j,k,l) \f$ from the \f$ 4^\text{th} \f$ order tensor \f$ T(i,j,k,l) \f$
*/

//------------------------------------------------------------------
template <class T> tensor3<T>& tensor4<T>::operator [] (int i) const
//------------------------------------------------------------------
{
  assert ((i>0)&&(i<=size));
  return tsr[i-1];
}


/*!
  \brief Standard operator = for \f$ 4^\text{th} \f$ order tensor

  If both tensors are fully defined (size + table of coordinates), it checks that the space dimension is the same for the first coordinate and then uses the operator = of \f$ 3^\text{rd} \f$ order tensors. If the left hand side tensor is partially defined (space dimension of the first coordinate of the tensor is zero) then it allocates memory for this tensor and copies the coordinates using operator = of \f$ 3^\text{rd} \f$ order tensors. \n

  \param tn \f$ 4^\text{th} \f$ order tensor to copy

  \return reference of the left hand side tensor (for multiple equalities)

  \sa mol::tensor3::operator=(const tensor3&)
*/

//--------------------------------------------------------------------------
template <class T> tensor4<T>& tensor4<T>::operator = (const tensor4<T>& tn)
//--------------------------------------------------------------------------
{
  assert (tn.size>0);

  if (!size) tsr = new tensor3<T> [size=tn.size];
  assert (size==tn.size);
  for (int i=0; i<size; i++) 
      tsr[i] = tn.tsr[i];

  return *this;
}


/*!
  \brief Destructive operator = for \f$ 4^\text{th} \f$ order tensor

  Makes a fast copy of tensor \f$ tn \f$ data (size, address of the coordinates table) and deletes \f$ tn \f$ (turns its size to zero). Note that we do not make any duplication of the coordinates table here. \n
  This operator is generally used with temporary objects.

  \param tn \f$ 4^\text{th} \f$ order tensor address

  \return reference of the left hand side tensor (for multiple equalities)
*/

//--------------------------------------------------------------------
template <class T> tensor4<T>& tensor4<T>::operator = (tensor4<T>* tn)
//--------------------------------------------------------------------
{
  assert ((*tn).dim2() && (*tn).dim3() && (*tn).dim4() && (*tn).size);

  if (!size) size = (*tn).size;
  else
    { assert (size==(*tn).size && dim2()==(*tn).dim2() && dim3()==(*tn).dim3() && dim4()==(*tn).dim4());
      delete [] tsr;
    }
  tsr = (*tn).tsr;
  (*tn).size = 0;

  return *this;
}


/*!
  \brief Operator = for \f$ 4^\text{th} \f$ order tensors with different size

  If the size of the right hand side tensor is zero, then we delete the table of coordinates of the left hand side tensor. \n
  If both tensors are fully defined (size + table of coordinates) and have the same size then we only copy the coordinates. \n
  If both tensors are fully defined but don't have the same size, then we delete the table of coordinates of the left hand side tensor and we reallocate memory for this table with the size of the right hand side tensor, then we copy the coordinates. \n

  \param tn \f$ 4^\text{th} \f$ order tensor to copy

  \return reference of the left hand side tensor (for multiple equalities)
*/

//---------------------------------------------------------------------------
template <class T> tensor4<T>& tensor4<T>::operator &= (const tensor4<T>& tn)
//---------------------------------------------------------------------------
{
  if (!tn)
    { if (size) delete [] tsr;
      size = 0;
    }

  else
    { if (size != tn.size)
	{ if (size) delete [] tsr;
	  tsr = new tensor3<T> [size = tn.size];  // allocates memory with the = operator for tensor2 (no initialization)
	}
      for (int i=1; i<=size; i++)
	(*this)[i] &= tn[i];
    }

  return *this;
}


/*!
  \brief Destructive operator = for \f$ 4^\text{th} \f$ order tensors with different size

  Makes a fast copy of the tensor \f$ tn \f$ data (size, address of the coordinates table) and deletes \f$ tn \f$ (turns its size to zero). Note that we do not make any duplication of the coordinates table here. \n
  This operator is generally used with temporary objects.

  \param tn \f$ 4^\text{th} \f$ order tensor address

  \return reference of the left hand side tensor (for multiple equalities)
*/

//---------------------------------------------------------------------
template <class T> tensor4<T>& tensor4<T>::operator &= (tensor4<T>* tn)
//---------------------------------------------------------------------
{
  if (!tn)
    { if (size) delete [] tsr;
      size = 0;
    }

  else
    { if (size)
	delete [] tsr;
      size = (*tn).size;
      tsr = (*tn).tsr;
      (*tn).size = 0;
    }

  return *this;
}


/*!
  \brief Addition for \f$ 4^\text{th} \f$ order tensors

  Computes the addition of the tensors \f$ \boldsymbol{a} = (a_{ijkl}) \f$ and \f$ \boldsymbol{b} = (b_{ijkl}) \f$ : \n
  \f$ c_{ijkl} = a_{ijkl} + b_{ijkl} \f$, where \f$ \boldsymbol{c} = (c_{ijkl}) \f$.

  \param t1 tensor \f$ \boldsymbol{a} \f$
  \param t2 tensor \f$ \boldsymbol{b} \f$

  \return \f$ \boldsymbol{c} = \boldsymbol{a} + \boldsymbol{b} \f$
*/

//---------------------------------------------------------------------------------------
template <class Tf> tensor4<Tf> operator + (const tensor4<Tf>& t1, const tensor4<Tf>& t2)
//---------------------------------------------------------------------------------------
{
  assert (t1.size);
  assert (t1.dim1()==t2.dim1()); // tensors must have the same size
  tensor4<Tf> t = t1;
  return t+=t2;
}


/*!
  \brief Subtraction for \f$ 4^\text{th} \f$ order tensors

  Computes the subtraction of the tensors \f$ \boldsymbol{a} = (a_{ijkl}) \f$ and \f$ \boldsymbol{b} = (b_{ijkl}) \f$ : \n
  \f$ c_{ijkl} = a_{ijkl} - b_{ijkl} \f$, where \f$ \boldsymbol{c} = (c_{ijkl}) \f$.

  \param t1 tensor \f$ \boldsymbol{a} \f$
  \param t2 tensor \f$ \boldsymbol{b} \f$

  \return \f$ \boldsymbol{c} = \boldsymbol{a} - \boldsymbol{b} \f$
*/

//---------------------------------------------------------------------------------------
template <class Tf> tensor4<Tf> operator - (const tensor4<Tf>& t1, const tensor4<Tf>& t2)
//---------------------------------------------------------------------------------------
{
  assert (t1.size);
  assert (t1.dim1()==t2.dim1()); // tensors must have the same size
  tensor4<Tf> t = t1;
  return t-=t2;
}


/*!
  \brief Contracted product for \f$ 4^\text{th} \f$ order tensors

  Computes the contracted product of the tensors \f$ \boldsymbol{a} = (a_{ijkl}) \f$ and \f$ \boldsymbol{b} = (b_{ijkl}) \f$ : \n
  \f$ c = a_{ijkl} b_{ijkl} \f$, where the result \f$ c \f$ is a scalar.

  \param t1 \f$ 4^\text{th} \f$ order tensor \f$ \boldsymbol{a} \f$
  \param t2 \f$ 4^\text{th} \f$ order tensor \f$ \boldsymbol{b} \f$

  \return \f$ c = \boldsymbol{a} | \boldsymbol{b} \f$
*/

//------------------------------------------------------------------------------
template <class Tf> Tf operator | (const tensor4<Tf>& t1, const tensor4<Tf>& t2)
//------------------------------------------------------------------------------
{
  assert (t1.size);
  assert (t1.size==t2.size); // tensors must have the same size

  Tf sum=0;
  for (int i=1; i<=t1.size; i++)
    sum += (t1[i] | t2[i]);

  return sum;
}


/*!
  \brief Contracted product for \f$ 4^\text{th} \f$ order tensors

  Computes the contracted product of the tensors \f$ \boldsymbol{a} = (a_{ijkl}) \f$ and \f$ \boldsymbol{b} = (b_{ijkl}) \f$ : \n
  \f$ c_{ijkl} = a_{ijpq} b_{pqkl} \f$, where \f$ \boldsymbol{c} = (c_{ijkl}) \f$.

  \param t1 \f$ 4^\text{th} \f$ order tensor \f$ \boldsymbol{a} \f$
  \param t2 \f$ 4^\text{th} \f$ order tensor \f$ \boldsymbol{b} \f$

  \return \f$ \boldsymbol{c} = \boldsymbol{a} :: \boldsymbol{b} \f$
*/

//----------------------------------------------------------------------------------------
template <class Tf> tensor4<Tf> operator || (const tensor4<Tf>& t1, const tensor4<Tf>& t2)
//----------------------------------------------------------------------------------------
{
  assert (t1.size && t2.size);
  assert ( (t1.dim3() == t2.dim1()) && (t1.dim4() == t2.dim2()) );

  tensor4<Tf> t3 (t1.dim1(),t1.dim2(),t2.dim3(),t2.dim4());
  for (int i=1; i<=t1.dim1(); i++)
    for (int j=1; j<=t1.dim2(); j++)
      for (int p=1; p<=t1.dim3(); p++)
	for (int q=1; q<=t1.dim4(); q++)
	  for (int k=1; k<=t2.dim3(); k++)
	    for (int l=1; l<=t2.dim4(); l++)
	      t3(i,j,k,l) += t1(i,j,p,q) * t2(p,q,k,l);

  return t3;
}


/*!
  \brief Add a \f$ 4^\text{th} \f$ order tensor

  Adds the \f$ 4^\text{th} \f$ order tensor \f$ \boldsymbol{R} = (R_i) \f$ to the current \f$ 4^\text{th} \f$ order tensor \f$ \boldsymbol{T} = (T_i) \f$ : \n
  \f$ T_i += R_i \f$  (which is equivalent to \f$ T_i = T_i + R_i \f$ but faster).

  \param tn \f$ 4^\text{th} \f$ order tensor \f$ \boldsymbol{R} \f$

  \return \f$  \boldsymbol{T} =  \boldsymbol{T} +  \boldsymbol{R} \f$
*/

//---------------------------------------------------------------------------
template <class T> tensor4<T>& tensor4<T>::operator += (const tensor4<T>& tn)
//---------------------------------------------------------------------------
{
  assert (size);  // the tensor tn must be defined
  assert (size == tn.size);  // tensors must have the same size

  for (int i=0; i<size; i++)
    tsr[i] += tn.tsr[i];

  return *this;
}


/*!
  \brief Subtract a \f$ 4^\text{th} \f$ order tensor

  Subtracts the \f$ 4^\text{th} \f$ order tensor \f$ \boldsymbol{R} = (R_i) \f$ to the current \f$ 4^\text{th} \f$ order tensor \f$ \boldsymbol{T} = (T_i) \f$ : \n
  \f$ T_i -= R_i \f$  (which is equivalent to \f$ T_i = T_i - R_i \f$ but faster).

  \param tn \f$ 4^\text{th} \f$ order tensor \f$ \boldsymbol{R} \f$

  \return \f$  \boldsymbol{T} =  \boldsymbol{T} -  \boldsymbol{R} \f$
*/

//---------------------------------------------------------------------------
template <class T> tensor4<T>& tensor4<T>::operator -= (const tensor4<T>& tn)
//---------------------------------------------------------------------------
{
  assert (size);  // the tensor tn must be defined
  assert (size == tn.size);  // tensors must have the same size

  for (int i=0; i<size; i++)
    tsr[i] -= tn.tsr[i];

  return *this;
}


/*!
  \brief Compares two \f$ 4^\text{th} \f$ order tensors

  First compare tensor spaces and then compare its coordinates.
  \param t1 \f$ 4^\text{th} \f$ order tensor \f$ t_1 \f$
  \param t2 \f$ 4^\text{th} \f$ order tensor \f$ t_2 \f$

  \return TRUE if \f$ t_1 = t_2 \f$ and FALSE if \f$ t_1 \neq t_2 \f$
*/

//---------------------------------------------------------------------------------
template <class Tf> bool operator == (const tensor4<Tf>& t1, const tensor4<Tf>& t2)
//---------------------------------------------------------------------------------
{
  assert (t1.size);
  int result=(t1.size==t2.size);
  for (int i=1; (i<=t1.size)&&(result); i++)
    result *= (t1[i]==t2[i]);
  return result;
}


/*!
  \brief Compares two  \f$ 4^\text{th} \f$ order tensors

  First compare tensor spaces and then compare its coordinates.
  \param t1 \f$ 4^\text{th} \f$ order tensor \f$ t_1 \f$
  \param t2 \f$ 4^\text{th} \f$ order tensor \f$ t_2 \f$

  \return TRUE if \f$ t_1 \neq t_2 \f$ and FALSE if \f$ t_1 = t_2 \f$
*/

//---------------------------------------------------------------------------------
template <class Tf> bool operator != (const tensor4<Tf>& t1, const tensor4<Tf>& t2)
//---------------------------------------------------------------------------------
{
  return !(t1 == t2);
}


//=====Public methods for tensor4 and scalars==================================


/*!
  \brief Multiplication by a scalar

  Multiply the \f$ 4^\text{th} \f$ order tensor \f$ \boldsymbol{T} = (t_{ijkl}) \f$ by a scalar a:
  \f$ t_{ijkl} *= a \f$  (which is equivalent to \f$ t_{ijkl} = t_{ijkl} * a \f$ but faster).

  \param elt scalar \f$ a \f$

  \return \f$  \boldsymbol{T} = a \boldsymbol{T} \f$
*/

//-------------------------------------------------------------------
template <class T> tensor4<T>& tensor4<T>::operator *= (const T& elt)
//-------------------------------------------------------------------
{
  assert (size);
  for (int i=0; i<size; i++)
    tsr [i] *= elt;
  return *this;
}


/*!
  \brief Multiplication for \f$ 4^\text{th} \f$ order tensor with scalar

  Computes the multiplication of the tensor \f$ \boldsymbol{T} = (t_{ijkl}) \f$ by a scalar \f$ a \f$ : \n
  \f$ \boldsymbol{T} * a = t_{ijkl} * a \f$.

  \param ts  \f$ 4^\text{th} \f$ order tensor \f$ \boldsymbol{T} \f$
  \param elt scalar \f$ a \f$

  \return \f$ \boldsymbol{T} * a \f$
*/

//-------------------------------------------------------------------------------
template <class Tf> tensor4<Tf> operator * (const tensor4<Tf>& ts, const Tf& elt)
//-------------------------------------------------------------------------------
{
  assert (ts.size);
  tensor4<Tf> tr = ts;
  return  tr *= elt;
}


/*!
  \brief Multiplication for \f$ 4^\text{th} \f$ order tensor with scalar

  Computes the multiplication of the tensor \f$ \boldsymbol{T} = (t_{ijkl}) \f$ by a scalar \f$ a \f$ : \n
  \f$ a * \boldsymbol{T} = a * t_{ijkl} \f$.

  \param elt scalar \f$ a \f$
  \param ts  \f$ 4^\text{th} \f$ order tensor \f$ \boldsymbol{T} \f$

  \return \f$ a * \boldsymbol{T} \f$
*/

//-------------------------------------------------------------------------------
template <class Tf> tensor4<Tf> operator * (const Tf& elt, const tensor4<Tf>& ts)
//-------------------------------------------------------------------------------
{
  assert (ts.size);
  tensor4<Tf> tr = ts;
  return  tr *= elt;
}


//=====Public methods for tensor4 and vectors==================================


/*!
  \brief Multiplication for \f$ 4^\text{th} \f$ order tensor with vector

  Computes the multiplication of the tensor \f$ \boldsymbol{T} = (t_{ijkl}) \f$ with the vector \f$ \boldsymbol{v} = v_i \f$ : \n
  \f$ \boldsymbol{T} * \boldsymbol{v} = t_{ijkl} * v_l \f$.

  \param ts  \f$ 4^\text{th} \f$ order tensor \f$ \boldsymbol{T} \f$
  \param v vector \f$ \boldsymbol{v} \f$

  \return \f$ \boldsymbol{T} * \boldsymbol{v} \f$
*/

//-------------------------------------------------------------------------------------
template <class Tf> tensor3<Tf> operator * (const tensor4<Tf>& ts, const vector<Tf>& v)
//-------------------------------------------------------------------------------------
{
  assert (v.dim());
  assert (v.dim() == ts.dim4());

  tensor3<Tf> ts3;
  ts3.create(ts.dim1());
  for (int i=1; i<=ts.size; i++)
    ts3[i] =& (ts[i] * v);

  return ts3;
}


/*!
  \brief Multiplication for vector with \f$ 4^\text{th} \f$ order tensor

  Computes the multiplication of the vector \f$ \boldsymbol{v} = v_i \f$ with the tensor \f$ \boldsymbol{T} = (t_{ijkl}) \f$ : \n
  \f$ \boldsymbol{v} * \boldsymbol{T} = v_i * t_{ijkl} \f$.

  \param v vector \f$ \boldsymbol{v} \f$
  \param ts  \f$ 4^\text{th} \f$ order tensor \f$ \boldsymbol{T} \f$

  \return \f$ \boldsymbol{v} * \boldsymbol{T} \f$
*/

//-------------------------------------------------------------------------------------
template <class Tf> tensor3<Tf> operator * (const vector<Tf>& v, const tensor4<Tf>& ts)
//-------------------------------------------------------------------------------------
{
  assert (v.dim());
  assert (v.dim() == ts.dim1());
  tensor3<Tf> tr(ts.dim2(),ts.dim3(),ts.dim4());
  for (int i=1; i<=ts.size; i++)
    tr += v[i] * ts[i];
  return tr;
}


/*!
  \brief Multiplication for \f$ 4^\text{th} \f$ order tensor with vector

  Computes the multiplication of the tensor \f$ \boldsymbol{T} = (t_{ijkl}) \f$ with the vector \f$ \boldsymbol{v} = v_i \f$ : \n
  \f$ \boldsymbol{T} | \boldsymbol{v} = t_{ijkl} * v_k \f$.

  \param ts  \f$ 4^\text{th} \f$ order tensor \f$ \boldsymbol{T} \f$
  \param v vector \f$ \boldsymbol{v} \f$

  \return \f$ \boldsymbol{T} | \boldsymbol{v} \f$
*/

//-------------------------------------------------------------------------------------
template <class Tf> tensor3<Tf> operator | (const tensor4<Tf>& ts, const vector<Tf>& v)
//-------------------------------------------------------------------------------------
{
  assert (v.dim());
  assert (v.dim() == ts.dim3());

  tensor3<Tf> tr(ts.dim1(),ts.dim2(),ts.dim4());
  for (int i=1; i<=ts.size; i++)
    tr[i] += ts[i] | v;

  return tr;
}


/*!
  \brief Multiplication for vector with \f$ 4^\text{th} \f$ order tensor

  Computes the multiplication of the vector \f$ \boldsymbol{v} = v_i \f$ with the tensor \f$ \boldsymbol{T} = (t_{ijkl}) \f$ : \n
  \f$ \boldsymbol{v} | \boldsymbol{T} = v_j * t_{ijkl} \f$.

  \param v vector \f$ \boldsymbol{v} \f$
  \param ts  \f$ 4^\text{th} \f$ order tensor \f$ \boldsymbol{T} \f$

  \return \f$ \boldsymbol{v} | \boldsymbol{T} \f$
*/

//-------------------------------------------------------------------------------------
template <class Tf> tensor3<Tf> operator | (const vector<Tf>& v, const tensor4<Tf>& ts)
//-------------------------------------------------------------------------------------
{
  assert (v.dim());
  assert (v.dim() == ts.dim2());

  tensor3<Tf> tr(ts.dim1(),ts.dim3(),ts.dim4());
  for (int i=1; i<=ts.size; i++)
    tr[i] += v * ts[i];

  return tr;
}


/*!
  \brief Multiplication for \f$ 4^\text{th} \f$ order tensor and vector

  Computes the multiplication of the tensor \f$ \boldsymbol{T} = (t_{ijkl}) \f$ with the vector \f$ \boldsymbol{v} = v_i \f$ for a given component \f$ n \f$ of the tensor.

  \param ts  \f$ 4^\text{th} \f$ order tensor \f$ \boldsymbol{T} \f$
  \param v vector \f$ \boldsymbol{v} \f$
  \param i component n

  \return \f$ \boldsymbol{v} * \boldsymbol{T} \f$ (if n=1), \f$ \boldsymbol{v} | \boldsymbol{T} \f$ (if n=2), \f$ \boldsymbol{T} | \boldsymbol{v} \f$ (if n=3), \f$ \boldsymbol{T} * \boldsymbol{v} \f$ (if n=4)
*/

//--------------------------------------------------------------------------------------------
template <class Tf> tensor3<Tf> tensor_sum (const tensor4<Tf>& ts, const vector<Tf>& v, int i)
//--------------------------------------------------------------------------------------------
{
  assert (i>=1 && i<=4);

  tensor3<Tf> tr;
  if (i==1) tr =& (v*ts);
  else if (i==2) tr =& (v|ts);
       else if (i==3) tr =& (ts|v);
            else tr =& (ts*v);

  return tr;
}


//=====Public methods for tensor4 and tensor2==================================


/*!
  \brief Contracted product for \f$ 4^\text{th} \f$ order tensor with \f$ 2^\text{nd} \f$ order tensor

  Computes the contracted product of the tensors \f$ \boldsymbol{a} = (a_{ijkl}) \f$ and \f$ \boldsymbol{b} = (b_{kl}) \f$ : \n
  \f$ c_{ij} = a_{ijkl} b_{kl} \f$, where \f$ \boldsymbol{c} = (c_{ij}) \f$.

  \param t \f$ 4^\text{th} \f$ order tensor \f$ \boldsymbol{a} \f$
  \param t2 \f$ 2^\text{nd} \f$ order tensor \f$ \boldsymbol{b} \f$

  \return \f$ \boldsymbol{c} = \boldsymbol{a} :: \boldsymbol{b} \f$
*/

//---------------------------------------------------------------------------------------
template <class Tf> tensor2<Tf> operator || (const tensor4<Tf>& t, const tensor2<Tf>& t2)
//---------------------------------------------------------------------------------------
{
  assert (t.size);
  assert (t.dim3() == t2.Rows());
  assert (t.dim4() == t2.Columns());

  tensor2<Tf> t2_result(t.dim1(),t.dim2(),false);
  for (int i=1; i<=t.dim1(); i++)
    for (int j=1; j<=t.dim2(); j++)
      t2_result(i,j) = t[i][j] | t2;

  return t2_result;
}


/*!
  \brief Contracted product for \f$ 2^\text{nd} \f$ order tensor with \f$ 4^\text{th} \f$ order tensor

  Computes the contracted product of the tensors \f$ \boldsymbol{a} = (a_{ijkl}) \f$ and \f$ \boldsymbol{b} = (b_{ij}) \f$ : \n
  \f$ c_{kl} = b_{ij} a_{ijkl} \f$, where \f$ \boldsymbol{c} = (c_{kl}) \f$.

  \param t2 \f$ 2^\text{nd} \f$ order tensor \f$ \boldsymbol{b} \f$
  \param t \f$ 4^\text{th} \f$ order tensor \f$ \boldsymbol{a} \f$

  \return \f$ \boldsymbol{c} = \boldsymbol{b} :: \boldsymbol{a} \f$
*/

//---------------------------------------------------------------------------------------
template <class Tf> tensor2<Tf> operator || (const tensor2<Tf>& t2, const tensor4<Tf>& t)
//---------------------------------------------------------------------------------------
{
  assert (t.size);
  assert (t.dim1() == t2.Rows());
  assert (t.dim2() == t2.Columns());

  tensor2<Tf> t2_result(t.dim3(),t.dim4());
  for (int i=1; i<=t.dim1(); i++)
    for (int j=1; j<=t.dim2(); j++)
      for (int k=1; k<=t.dim3(); k++)
	for (int l=1; l<=t.dim4(); l++)
	  t2_result(k,l) += t(i,j,k,l) * t2(i,j);

  return t2_result;
}


/*!
  \brief Tensorial product for \f$ 2^\text{nd} \f$ order tensors

  Computes the tensorial product for the tensors \f$ \boldsymbol{a} = (a_{ij}) \f$ and \f$ \boldsymbol{b} = (b_{kl}) \f$ : \n
  \f$ c_{ijkl} = a_{ij} b_{kl} \f$, where \f$ \boldsymbol{c} = (c_{ijkl}) \f$.

  \param t1 \f$ 2^\text{nd} \f$ order tensor \f$ \boldsymbol{a} \f$
  \param t2 \f$ 2^\text{nd} \f$ order tensor \f$ \boldsymbol{b} \f$

  \return \f$ \boldsymbol{c} = \boldsymbol{a} \times \boldsymbol{b} \f$
*/

//---------------------------------------------------------------------------------------
template <class Tf> tensor4<Tf> operator ^ (const tensor2<Tf>& t1, const tensor2<Tf>& t2)
//---------------------------------------------------------------------------------------
{
  assert (t1.Rows() && t1.Columns() && t2.Rows() && t2.Columns());

  tensor4<Tf> tsr (t1.Rows(), t1.Columns(), t2.Rows(), t2.Columns(), false);
  for (int i=1; i<=t1.Rows(); i++)
    for (int j=1; j<=t1.Columns(); j++)
      for (int k=1; k<=t2.Rows(); k++)
	for (int l=1; l<=t2.Columns(); l++)
	  tsr(i,j,k,l) = t1(i,j) * t2(k,l);

  return tsr;
}


//=====Public methods for tensor4 and tensor3==================================


/*!
  \brief Product for \f$ 3^\text{rd} \f$ order tensors

  Computes the product of the tensors \f$ \boldsymbol{a} = (a_{n,ij}) \f$ and \f$ \boldsymbol{b} = (b_{n,kl}) \f$ : \n
  \f$ c_{ijkl} = a_{n,ij} b_{n,kl} \f$, where \f$ \boldsymbol{c} = (c_{ijkl}) \f$.

  \param t1 \f$ 3^\text{rd} \f$ order tensor \f$ \boldsymbol{a} \f$
  \param t2 \f$ 3^\text{th} \f$ order tensor \f$ \boldsymbol{b} \f$

  \return \f$ \boldsymbol{c} = \boldsymbol{a} \times \boldsymbol{b} \f$
*/

//---------------------------------------------------------------------------------------
template <class Tf> tensor4<Tf> operator ^ (const tensor3<Tf>& t1, const tensor3<Tf>& t2)
//---------------------------------------------------------------------------------------
{
  assert (t1.dim1());
  assert (t1.dim1() == t2.dim1());
  tensor4<Tf> t4(t1.dim2(), t1.dim3(), t2.dim2(), t2.dim3());
  for (int n=1; n<=t1.dim1(); n++)
    t4 += (t1[n] ^ t2[n]);
  return t4;
}


/*!
  \brief Contracted product for \f$ 3^\text{rd} \f$ order tensor with \f$ 4^\text{th} \f$ order tensor

  Computes the contracted product of the tensors \f$ \boldsymbol{a} = (a_{n,ij}) \f$ and \f$ \boldsymbol{b} = (b_{ijkl}) \f$ : \n
  \f$ c_{n,kl} = a_{n,ij} b_{ijkl} \f$, where \f$ \boldsymbol{c} = (c_{n,kl}) \f$.

  \param t3 \f$ 3^\text{rd} \f$ order tensor \f$ \boldsymbol{a} \f$
  \param t4 \f$ 4^\text{th} \f$ order tensor \f$ \boldsymbol{b} \f$

  \return \f$ \boldsymbol{c} = \boldsymbol{a} :: \boldsymbol{b} \f$
*/

//----------------------------------------------------------------------------------------
template <class Tf> tensor3<Tf> operator || (const tensor3<Tf>& t3, const tensor4<Tf>& t4)
//----------------------------------------------------------------------------------------
{
  assert (t3.dim1() && t4.dim1());
  assert ( (t3.dim2() == t4.dim1()) && (t3.dim3() == t4.dim2()) );

  tensor3<Tf> ts;
  ts.create(t3.dim1());

  for (int n=1; n<=t3.dim1(); n++)
    ts[n] =& (t3[n] || t4);

  return ts;
}


/*!
  \brief Contracted product for \f$ 4^\text{th} \f$ order tensor with \f$ 3^\text{rd} \f$ order tensor

  Computes the contracted product of the tensors \f$ \boldsymbol{a} = (a_{ijkl}) \f$ and \f$ \boldsymbol{b} = (b_{n,kl}) \f$ : \n
  \f$ c_{n,ij} = a_{ijkl} b_{n,kl} \f$, where \f$ \boldsymbol{c} = (c_{n,ij}) \f$.

  \param t4 \f$ 4^\text{th} \f$ order tensor \f$ \boldsymbol{a} \f$
  \param t3 \f$ 3^\text{rd} \f$ order tensor \f$ \boldsymbol{b} \f$

  \return \f$ \boldsymbol{c} = \boldsymbol{a} :: \boldsymbol{b} \f$
*/

//----------------------------------------------------------------------------------------
template <class Tf> tensor3<Tf> operator || (const tensor4<Tf>& t4, const tensor3<Tf>& t3)
//----------------------------------------------------------------------------------------
{
  assert (t3.dim1() && t4.dim1());
  assert ( (t3.dim2() == t4.dim3()) && (t3.dim3() == t4.dim4()) );

  tensor3<Tf> ts;
  ts.create(t3.dim1());

  for (int n=1; n<=t3.dim1(); n++)
    ts[n] =& (t4 || t3[n]);

  return ts;
}


//=====Public methods for tensor4==============================================


/*!
  \brief Computes symmetric \f$ 4^\text{th} \f$ order identity tensor

  \param n space dimension

  \return Identity tensor
*/

//------------------------------------------
template <class Tf> tensor4<Tf> Id4s (int n)
//------------------------------------------
{
  assert (n > 0);
  tensor4<Tf> I(n,n,n,n);

  for (int i=1; i<=n; i++)
    for (int j=1; j<=n; j++)
      { I(i,j,i,j) += .5;
	I(i,j,j,i) += .5;
      }

  return I;
}


/*!
  \brief Computes the change of basis for a \f$ 4^{th} \f$ order tensor.

  \f$ \displaystyle T^\prime_{pqmn} = P_{pi} P_{qj} P_{mk} P_{nl} T_{ijkl} \f$. \n
  \param P change of basis matrix
  \param tsr tensor T in the old basis
  \return \f$ T^\prime \f$ tensor in the new basis
*/

//----------------------------------------------------------------------------------------
template <class Tf> tensor4<Tf> change_basis (const matrix<Tf>& P, const tensor4<Tf>& tsr)
//----------------------------------------------------------------------------------------
{
  assert ((P.Rows() ==3) && (P.Columns() == 3));
  assert (tsr.size == 3 && tsr.dim2()==3 && tsr.dim3()==3 && tsr.dim4()==3);

  tensor4<Tf> Tsr(3);
  for (int p=1; p<=3; p++)
    for (int q=1; q<=3; q++)
      for (int m=1; m<=3; m++)
	for (int n=1; n<=3; n++)
	  for (int i=1; i<=3; i++)
	    for (int j=1; j<=3; j++)
	      for (int k=1; k<=3; k++)
		for (int l=1; l<=3; l++)
		  Tsr(p,q,m,n) += P(p,i) * P(q,j) * P(m,k) * P(n,l) * tsr(i,j,k,l);

  return Tsr;
}


//--------------------------------------------------------------------------
template <class Tf> ostream& operator << (ostream& s, const tensor4<Tf>& tn)
//--------------------------------------------------------------------------
{
  assert (tn.size);
  for (int i=1; i<=tn.size; i++)
      s << tn[i];
  s << endl;
  return s;
}


//--------------------------------------------------------------------------
template <class Tf> istream& operator >> (istream& s, const tensor4<Tf>& tn)
//--------------------------------------------------------------------------
{
  assert (tn.size);

  for (int i=1; i<=tn.size; i++)
      s >> tn[i];

  return s;
}


/*!
  \brief Writes \f$ 4^\text{th} \f$ order tensor to hard disk.

  \param filename path and filename
*/

//---------------------------------------------------------
template <class T> void tensor4<T>::save (const char* path)
//---------------------------------------------------------
{
  assert (size);

  ofstream file(path, ios::out);
  assert (!file.fail());
  file << *this;
  file.close();
}


/*!
  \brief Writes \f$ 4^\text{th} \f$ order tensor to hard disk.

  \param filename path and filename
*/

//---------------------------------------------------------------
template <class T> void tensor4<T>::save (const string& filename)
//---------------------------------------------------------------
{
  const char *path = filename.c_str();
  save (path);
}


/*!
  \brief Reads \f$ 4^\text{th} \f$ order tensor from hard disk.

  \param filename path and filename
*/

//---------------------------------------------------------
template <class T> void tensor4<T>::read (const char* path)
//---------------------------------------------------------
{
  assert (size);

  ifstream file(path, ios::in);
  assert (!file.fail());
  file >> *this;
  file.close();
}


/*!
  \brief Reads \f$ 4^\text{th} \f$ order tensor from hard disk.

  \param filename path and filename
*/

//---------------------------------------------------------------
template <class T> void tensor4<T>::read (const string& filename)
//---------------------------------------------------------------
{
  const char *path = filename.c_str();
  read (path);
}


//---------------------------------------------------------
template <class T> tensor4<T>& tensor4<T>::approximation ()
//---------------------------------------------------------
{
  for (int i=0; i<size; i++)
    tsr[i].approximation();
  return *this;
}


}


#endif
