/***************************************************************
 *                 Mathematical Object Library                 *
 *           class vector : declaration for vectors            *
 *                    simula+@metz.ensam.fr                    *
 *	             GNU/linux version 2.4.0	               *
 *            software under General Public License            *
 ***************************************************************
 * copyright © 1994,1995,1996,1999,2000,2001,2002,2003,2004,2005,2006,2007,2008,2009 COLLARD Christophe
 * copyright © 2002,2003,2004 CREUSE Emmanuel
 * copyright © 2003,2004 MERCIER Denis
 * copyright © 2002,2003,2004,2005,2006,2007,2008,2009 Laboratoire de Physique et Mécanique des Matériaux (LPMM - CNRS)
 * copyright © 2002,2003,2004,2005,2006,2007 Laboratoire de Mathématiques et ses Applications de Valenciennes (LAMAV)
 ***************************************************************/

/*! \namespace mol
    \brief Mathematical Object Libraries
*/

/*! \class mol::vector
    \brief vector library \n

    \htmlonly 
    <FONT color="#838383">

    vector belongs to Mathematical Object Libraries (MOL++) </br>
    MOL++ is part of Simula+ <br><br>

    Simula+ is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version. <br><br>

    Simula+ is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details. <br><br>

    You should have received a copy of the GNU General Public License
    along with Simula+; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

    </FONT>
    \endhtmlonly

    A vector is defined by its size and a pointer on the table of its coordinates. \n
    When the size of a vector is zero there is no table of coordinates attached to the pointer.

    \authors copyright \htmlonly &#169; \endhtmlonly 1994, 1995, 1996, 1999, 2000, 2001, 2002, 2003, 2004, 2005, 2006, 2007, 2008, 2009 Christophe COLLARD \n
             copyright \htmlonly &#169; \endhtmlonly 2002, 2003, 2004 Emmanuel CREUSE \n
	     copyright \htmlonly &#169; \endhtmlonly 2003, 2004 Denis MERCIER \n
	     copyright \htmlonly &#169; 2002, 2003, 2004, 2005, 2006, 2007, 2008, 2009 Laboratoire de Physique et M&#233;canique des Mat&#233;riaux (LPMM - CNRS) \endhtmlonly \n
	     copyright \htmlonly &#169; 2002, 2003, 2004, 2005, 2006, 2007 Laboratoire de Math&#233;matiques et ses Applications de Valenciennes (LAMAV) \endhtmlonly
    \version 2.4.0
    \date 1994-2010
    \bug none
    \warning none
*/

#ifndef __cplusplus
#error Must use C++ for the type vector
#endif

#ifndef _vectors_h
#define _vectors_h


#if !defined(__IOMANIP_H)
#include <iomanip>
#endif

#if !defined (__IOSTREAM_H)
#include <iostream>
#endif

#if !defined(__FSTREAM_H)
#include <fstream>
#endif

#if !defined(__ASSERT_H)
#include <assert.h>
#endif

#if !defined(__MATH_H)
#include <math.h>
#endif

#if !defined(__STRING)
#include <string>
#endif

#if !defined(__STRING_H)
#include<string.h>
#endif

#if !defined(__PARAMETERS_H)
#include "parameters.h"
#endif

#if !defined(__MATHS_H)
#include "maths.h"
#endif

using namespace std;

namespace mol
{


//=============================
template <class T> class vector
//=============================
{
  protected:
    int size;
    T* p;

  public:
    vector () {size = 0;}  	    // default constructor
    vector (int, bool=true, T=0);   // constructor - allocates memory
    vector (const vector<T>&);      // copy constructor
    vector (vector<T>*);            // copy constructor with temporary objects
    ~vector ();                     // destructor

    void assign (int);  // allocates memory for the vector but doen't initilize to zero its components (use it with great care)
    int dim () const {return size;}       // returns the vector size
    friend bool operator ! (const vector<T>& v) {return !v.size;}

    T* operator () () {return p;}
    T& operator [] (int) const;     // returns the address of an element
    virtual int operator = (int);   // allocates memory for the vector but doen't initilize to zero its components (use it with great care)
    virtual vector<T>& operator  = (const vector<T>&);
    virtual vector<T>& operator  = (vector<T>*); // allocates the data from the right hand temporary object to the left hand object - no copy (deletes temporary object)
    virtual vector<T>& operator &= (const vector<T>&); // copies vector without size check (use this operator with great care)
    virtual vector<T>& operator &= (vector<T>*); // allocates the data from the right hand temporary object to the left hand object (deletes temporary object) - no size check - no copy (use this operator with great care)
    template <class Tf> friend bool operator == (const vector<Tf>&, const vector<Tf>&);
    template <class Tf> friend bool operator != (const vector<Tf>&, const vector<Tf>&);
    template <class Tf> friend vector<Tf> operator + (const vector<Tf>&, const vector<Tf>&); // overloads operator + for vectors
    template <class Tf> friend vector<Tf> operator - (const vector<Tf>&, const vector<Tf>&); // overloads operator - for vectors
    template <class Tf> friend Tf         operator * (const vector<Tf>&, const vector<Tf>&); // defines the scalar product
    template <class Tf> friend vector<Tf> operator / (const vector<Tf>&, const vector<Tf>&); // exists only if b.size=1
    vector<T>& operator += (const vector<T>&);
    vector<T>& operator -= (const vector<T>&);
// operations between scalars and vectors
    template <class Tf> friend vector<Tf> operator * (const vector<Tf>&, const Tf&); // overloads vector * scalar operator
    template <class Tf> friend vector<Tf> operator * (const Tf&, const vector<Tf>&); // overloads scalar * vector operator
    template <class Tf> friend vector<Tf> operator / (const vector<Tf>&, const Tf&); // overloads vectors / scalars operator
    vector<T>& operator *=  (const T&);
    vector<T>& operator /=  (const T&);
// overload for the outstream
    template <class Tf> friend ostream& operator << (ostream&, const vector<Tf>&);
    template <class Tf> friend istream& operator >> (istream&, const vector<Tf>&);
// i/o disk
    void save (const char*);
    void save (const string&);
    void read (const char*);
    void read (const string&);
// vector operations
    virtual long double norm () const;  // computes the norm of a vector
    virtual T norminf () const;         // computes the infinite norm
    vector<T>& approximation ();        // set to 0 an element < epsilon (machine precision)
    virtual vector<T> ortho2D();        // computes the orthogonal vector
    virtual void permute (int,int);     // elements permutation
    template <class Tf> friend void permute (vector<Tf>&, vector<Tf>&);  // vectors permutation
    template <class Tf> friend vector<Tf> abs (const vector<Tf>&);       // give the absolute value of a vector
    template <class Tf> friend vector<int> order (vector<Tf>&);          // vectors ordering from smallest to biggest
    template <class Tf> friend vector<Tf> subvector (const vector<Tf>&, int, int);      // extracts a sub-vector
    template <class Tf> friend vector<Tf> merge (const vector<Tf>&, const vector<Tf>&); // merges 2 vectors
};


//=====Private methods for vectors===========================================



//=====Public methods for vectors============================================


/*!
  \brief Constructor
  \param n vector size
  \param init boolean that indicates if the constructor has to initialize the vector coordinates (default value is true)
  \param value value for vector coordinates (default value is 0)
  \n\n
  \f$ \textbf{Examples :} \f$ \n
  \f$ \textbf{vector$<$long double$>$ v(5);} \; \textit{Allocates memory for a vector with type long double and with 5 coordinates initialized to zero.}  \f$ \n
  \f$ \textbf{vector$<$double$>$ w;} \; \textit{Creates a vector with type double but does not allocate memory.} \f$ \n
  \f$ \textbf{vector$<$float$>$ z(3,true,-1.23);} \; \textit{Allocates memory for a vector with type float and with 3 coordinates initialized to -1.23 .} \f$ \n
*/

//-------------------------------------------------------------
template <class T> vector<T>::vector(int n, bool init, T value)
//-------------------------------------------------------------
{
  assert (n>0);

  p = new T [size=n];
  if (init)
    for (int i=0; i<n; i++)
      p[i] = value;
}


/*!
  \brief Copy constructor
  \param v vector to duplicate
*/

//------------------------------------------------------
template <class T> vector<T>::vector(const vector<T>& v)
//------------------------------------------------------
{
  assert(v.size);
  p = new T [size=v.size];
  memcpy (p, v.p, size*sizeof(T));
}


/*!
  \brief Destructive constructor

  Makes a fast copy of vector v parameters (size, address of the coordinates table "p") and deletes v (turns its size to zero). Note that we do not make any duplication of the coordinates table here. \n
  This constructor is generally used with temporary objects.
  \param v vector address
  \n\n
  \f$ \textbf{Examples :} \f$ \n
  \f$ \textbf{vector$<$double$>$ u(5), v(5);} \; \textit{Allocates memory for two vectors with type double and with 5 coordinates initialized to zero.}  \f$ \n
  \f$ \textbf{vector$<$double$>$ w(\&(u+v));} \; \textit{Calls the destructive construtor (creates w and kill the temporary object "u+v")} \f$ \n
  \f$ \textbf{vector$<$double$>$ z(\&u);} \; \textit{Calls the destructive construtor (note that u coordinates do not exist anymore after this instruction : u.size $\equiv$ 0)} \f$ \n
*/

//------------------------------------------------
template <class T> vector<T>::vector(vector<T>* v)
//------------------------------------------------
{
  size = (*v).size; // copy the size of the temporary objet
  p = (*v).p;       // copy the address of the temporary objet
  (*v).size = 0;    // set temporary object size to zero (so the data stored at p won't be delete by the destructor)
}


/*!
  \brief Destructor

  Deletes vector coordinates table if vector size is not null.
*/

//-------------------------------------
template <class T> vector<T>::~vector()
//-------------------------------------
{
  if (size) delete [] p; // desallocates memory only if vector size is not null
  size = 0;
}


/*!
  \brief Allocates memory for a vector

  Warning : this method does not initialize the vector coordinates and can only be used with null sized vectors. \n
  \param n vector size
  \n\n
  \f$ \textbf{Example :} \f$ \n
  \f$ \textbf{vector$<$double$>$ v;} \; \textit{Defines a vector with type double but does not allocate memory.} \f$ \n
  \f$ \textbf{v.assign(4);} \; \textit{Allocates memory for 4 coordinates but does not initialize them.} \f$ \n
*/

//-----------------------------------------------
template <class T> void vector<T>::assign (int n)
//-----------------------------------------------
{
  assert (!size); // checks if the vector isn't already defined
  assert (n > 0);   // checks if the value for the size is right
  p = new T [size=n]; // allocates memory for the vector (no initialization)
}


/*!
  \brief Returns the reference of a vector coordinate

  Warning : we use the mathematical notations so the vector coordinates are in the set [1;n], where n is the vector size.
  \param i vector coordinate
  \n\n
  \f$ \textbf{Examples :} \f$ \n
  \f$ \textbf{vector$<$double$>$ v(5);} \; \textit{Allocates memory for a vector with type double and with 5 coordinates.} \f$ \n
  \f$ \textbf{v[2] = 1.245;} \; \textit{Stores 1.245 for the 2nd vector coordinate.} \f$ \n
  \f$ \textbf{cout $<<$ v[2];} \; \textit{Prints out the 2nd vector coordinate.} \f$ \n

  \return \f$ i^\text{th} \f$ coordinate of the vector
*/

//--------------------------------------------------------
template <class T> T& vector<T>::operator [] (int i) const
//--------------------------------------------------------
{
  assert ((i>0) && (i<=size));
  return p[i-1];
}


/*!
  \brief Allocates memory for a vector

  Warning : this method does not initialize the vector coordinates and can only be used with null sized vectors. \n
  \param n vector size
  \return vector size (for multiple allocations)
  \n\n
  \f$ \textbf{Example :} \f$ \n
  \f$ \textbf{vector$<$double$>$ v, w;} \; \textit{Defines two vectors with type double.} \f$ \n
  \f$ \textbf{v = w = 3;} \; \textit{Allocates memory for the vectors v and w with 3 coordinates but does not initialize the coordinates.} \f$ \n
*/

//--------------------------------------------------
template <class T> int vector<T>::operator = (int n)
//--------------------------------------------------
{
  assert (!size); // checks if the vector isn't already defined
  assert (n > 0);   // checks if the value for the size is right
  assign (n);
  return n;
}


/*!
  \brief Standard operator = for vectors

  If both vectors are fully defined (size + table of coordinates), it checks that they have the same size and copies the coordinates. If the left hand side vector is partially defined (the size of the vector is zero) then it allocates memory for this vector and copies the coordinates. \n
  \param v vector to copy
  \return reference of the left hand side vector (for multiple equalities)
  \n\n
  \f$ \textbf{Examples :} \f$ \n
  \f$ \textbf{vector$<$float$>$ v, w(4), z(4);} \; \textit{Defines 3 vectors with type float and allocates memory for "w" and "z" with 4 coordinates initialized to zero.} \f$ \n
  \f$ \textbf{z[1]=1.23; z[2]=-3.4; z[4]=2.34;} \; \textit{Stores the coordinates of "z" (see} \f$ \ref vector:operator[](int) \f$\textit{).} \f$ \n
  \f$ \textbf{w = z;} \; \textit{Uses standard operator = for vectors to copy the coordinates of "z" in "w".} \f$ \n
  \f$ \textbf{v = z;} \; \textit{Allocates memory for "v" with 4 coordinates and copies the coordinates of "z" in "v".} \f$ \n
*/

//----------------------------------------------------------------------
template <class T> vector<T>& vector<T>::operator = (const vector<T>& v)
//----------------------------------------------------------------------
{
  assert(v.size);

  if (!size) p = new T [size=v.size]; // allocates memory if the left vector is 0
  else assert (size == v.size);      // checks vectors size
  memcpy (p, v.p, v.size*sizeof(T));
  return *this;
}


/*!
  \brief Destructive operator = for vectors

  Makes a fast copy of vector v data (size, address of the coordinates table "p") and deletes v (turns its size to zero). Note that we do not make any duplication of the coordinates table here. \n
  This operator is generally used with temporary objects.
  \param v vector address
  \return reference of the left hand side vector (for multiple equalities)
  \n\n
  \f$ \textbf{Examples :} \f$ \n
  \f$ \textbf{vector$<$float$>$ v, w(4), z(4);} \; \textit{Defines 3 vectors with type float and allocates memory for "w" and "z" with 4 coordinates initialized to zero.} \f$ \n
  \f$ \textbf{z[1]=1.23; z[2]=-3.4; z[4]=2.34;} \; \textit{Stores the coordinates of "z" (see} \f$ \ref vector:operator[](int) \f$\textit{).} \f$ \n
  \f$ \textbf{w =\& (z+z);} \; \left\{ \begin{aligned} & \textit{Creates a temporary object as the result of the summation of z+z.} \\ & \textit{Delete the table of coordinates of w.} \\ & \textit{Then, copies the pointer address of "z+z" coordinates table in "w" and turns the size of the temporary object to zero.} \end{aligned} \right. \f$ \n
  \f$ \textbf{w =\& z;} \; \left\{\begin{aligned} & \textit{Deletes the table of coordinates of w.} \\ & \textit{Copies the pointer address of the coordinates table of "z" in "w". Then turns "z" size to zero.} \\ & \textit{This implies that "z" has no more coordinates table. } \end{aligned} \right. \f$ \n
  \f$ \textbf{v =\& w;} \; \left\{ \begin{aligned} & \textit{Copies the size and the pointer address of the coordinates table of "w" in "v" (no memory allocation is needed for "v").} \\ & \textit{Then informs "w" of the lost of its coordinates table by turning its size to zero.} \end{aligned} \right.\f$ \n
  \f$ \textbf{vector$<$float$>$ a, b, c(4);} \f$ \n
  \f$ \textbf{a =\& (b =\& c);} \; \textit{Multiple equalities - after this instruction b and c have no more coordinates table (can also be used with temporary objects).} \f$
*/

//----------------------------------------------------------------
template <class T> vector<T>& vector<T>::operator = (vector<T>* v)
//----------------------------------------------------------------
{
  assert ((*v).size);

  if (size)
    { assert ((*v).size == size);
      delete [] p;
    }
  else size = (*v).size; // copy the size of the temporary objet

  p = (*v).p;     // copy the address of the temporary objet
  (*v).size = 0;  // set temporary object size to zero (so the data stored at p won't be delete by the destructor)

  return *this;
}


/*!
  \brief Operator = for vectors with different size

  If the size of the right hand side vector is zero, then we delete the table of coordinates of the left hand side vector. \n
  If both vectors are fully defined (size + table of coordinates) and have the same size then we only copy the coordinates. \n
  If both vectors are fully defined but don't have the same size, then we delete the table of coordinates of the left hand side vector and we reallocate memory for this table with the size of the right hand side vector, then we copy the coordinates. \n

  \param v vector to copy
  \return reference of the left hand side vector (for multiple equalities)
  \n\n
  \f$ \textbf{Examples :} \f$ \n
  \f$ \textbf{vector$<$float$>$ v, y, w(2), z(4);} \; \textit{Defines 4 vectors with type float and allocates memory for "w" and "z" coordinates.} \f$ \n
  \f$ \textbf{z[1]=1.23; z[2]=-3.4; z[4]=2.34;} \; \textit{Stores the coordinates of "z" (see} \f$ \ref vector:operator[](int) \f$\textit{).} \f$ \n
  \f$ \textbf{w \&= z;} \; \textit{Deletes the table of coordinates of w, allocates memory for "w" with 4 coordinates and copies the coordinates of "z" in "w".} \f$ \n
  \f$ \textbf{v \&= z;} \; \textit{Allocates memory for "v" with 4 coordinates and copies the coordinates of "z" in "v".} \f$ \n
  \f$ \textbf{w \&= y;} \; \textit{Deletes the table of coordinates of "w".} \f$ \n
*/

//-----------------------------------------------------------------------
template <class T> vector<T>& vector<T>::operator &= (const vector<T>& v)
//-----------------------------------------------------------------------
{
  if (!v.size)
    { if (size) delete [] p;
      size = 0;
    }

  else
    { if (v.size != size) 
	{ if (size) delete [] p;
	  p = new T [size=v.size]; // allocates memory
	}
      memcpy (p, v.p, v.size*sizeof(T));
    }

  return *this;
}


/*!
  \brief Destructive operator = for vectors with different size

  Makes a fast copy of the vector v data (size, address of the coordinates table "p") and deletes v (turns its size to zero). Note that we do not make any duplication of the coordinates table here. \n
  This operator is generally used with temporary objects.
  \param v vector address
  \return reference of the left hand side vector (for multiple equalities)
  \n\n
  \f$ \textbf{Examples :} \f$ \n
  \f$ \textbf{vector$<$float$>$ v, w(2), z(4);} \; \textit{Defines 3 vectors with type float and allocates memory for "w" and "z" coordinates.} \f$ \n
  \f$ \textbf{z[1]=1.23; z[2]=-3.4; z[4]=2.34;} \; \textit{Stores the coordinates of "z" (see} \f$ \ref vector:operator[](int) \f$\textit{).} \f$ \n
  \f$ \textbf{w \&=\& (z+z);} \; \left\{ \begin{aligned} & \textit{Creates a temporary object as the result of the summation of z+z.} \\ & \textit{Delete the table of coordinates of w.} \\ & \textit{Then, copies the pointer address of "z+z" coordinates table in "w" and turns the size of the temporary object to zero.} \end{aligned} \right. \f$ \n
  \f$ \textbf{w \&=\& z;} \; \left\{\begin{aligned} & \textit{Deletes the table of coordinates of w.} \\ & \textit{Copies the size and the pointer address of the coordinates table of "z" in "w". Then turns "z" size to zero.} \\ & \textit{This implies that "z" has no more coordinates table. } \end{aligned} \right. \f$ \n
  \f$ \textbf{v \&=\& w;} \; \left\{ \begin{aligned} & \textit{Copies the size and the pointer address of the coordinates table of "w" in "v" (no memory allocation is needed for "v").} \\ & \textit{Then informs "w" of the lost of its coordinates table by turning its size to zero.} \end{aligned} \right.\f$ \n
  \f$ \textbf{vector$<$float$>$ a, b(2), c(4);} \f$ \n
  \f$ \textbf{a \&=\& (b \&=\& c);} \; \textit{Multiple equalities - after this instruction b and c have no more coordinates table (can also be used with temporary objects).} \f$
*/

//-----------------------------------------------------------------
template <class T> vector<T>& vector<T>::operator &= (vector<T>* v)
//-----------------------------------------------------------------
{
  if (!(*v).size)
    { if (size) delete [] p;
      size = 0;
    }

  else
    { if (size) delete [] p;
      size = (*v).size; // copy the size of the temporary objet
      p = (*v).p;       // copy the address of the temporary objet
      (*v).size = 0;    // set temporary object size to zero (so the data stored at p won't be delete by the destructor)
    }

  return *this;
}


/*!
  \brief Compares two vectors

  First compare vectors size and then compare vectors coordinates.
  \param vector a
  \param vector b
  \return TRUE if \f$ a = b \f$ and FALSE if \f$ a \neq b \f$
  \n\n
  \f$ \textbf{Examples :} \f$ \n
  \f$ \textbf{vector$<$double$>$ v(2), w(4), z(4);} \; \textit{Defines 3 vectors with type double and allocates memory for their coordinates.} \f$ \n
  \f$ \textbf{w[1]=z[1]=1.23; w[2]=z[2]=-3.4; w[4]=z[4]=2.34;} \; \textit{Stores the coordinates of "w" and "z" (see} \f$ \ref vector:operator[](int) \f$\textit{).} \f$ \n
  \f$ \textbf{cout $<<$ (w==z);} \; \textit{Since w = z, prints 1 (TRUE) on screen.} \f$ \n
  \f$ \textbf{cout $<<$ (v==z);} \; \textit{Since v $ \neq $ z, prints 0 (FALSE) on screen.} \f$
*/

//-----------------------------------------------------------------------------
template <class Tf> bool operator == (const vector<Tf>& a, const vector<Tf>& b)
//-----------------------------------------------------------------------------
{
  assert (a.size);		// a must exists
  bool Bool = (a.size==b.size);	// a & b must be in the same space

  for (int i=0; i<a.size && Bool; i++)
    if (abs (a.p[i]) < epsilon) Bool *= abs (b.p[i]) < epsilon;
    else Bool *= abs ((a.p[i]-b.p[i]) / a.p[i]) < epsilon;

  return Bool;
}


/*!
  \brief Compares two vectors

  First compare vectors size and then compare vectors coordinates.
  \param vector a
  \param vector b
  \return TRUE if \f$ a \neq b \f$ and FALSE if \f$ a = b \f$
  \n\n
  \f$ \textbf{Examples :} \f$ \n
  \f$ \textbf{vector$<$double$>$ v(2), w(4), z(4);} \; \textit{Defines 3 vectors with type double and allocates memory for their coordinates.} \f$ \n
  \f$ \textbf{w[1]=z[1]=1.23; w[2]=z[2]=-3.4; w[4]=z[4]=2.34;} \; \textit{Stores the coordinates of "w" and "z" (see} \f$ \ref vector:operator[](int) \f$\textit{).} \f$ \n
  \f$ \textbf{cout $<<$ (w!=z);} \; \textit{Since w = z, prints 0 (FALSE) on screen.} \f$ \n
  \f$ \textbf{cout $<<$ (v!=z);} \; \textit{Since v $ \neq $ z, prints 1 (TRUE) on screen.} \f$
*/

//----------------------------------------------------------------------------
template<class Tf> bool operator != (const vector<Tf>& a, const vector<Tf>& b)
//----------------------------------------------------------------------------
{
  return !(a==b);
}


/*!
  \brief Addition for vectors.

  Computes the addition of the vectors \f$ \boldsymbol{a} = (a_i) \f$ and \f$ \boldsymbol{b} = (b_i) \f$ :
  \f$ c_i = a_i + b_i \f$, where \f$ \boldsymbol{c} = (c_i) \f$.
  \param vector a
  \param vector b

  \return \f$ c = a+b \f$
  \n\n
  \f$ \textbf{Example :} \f$ \n
  \f$ \textbf{vector$<$double$>$ v, w(4), z(4);} \; \textit{Defines 3 vectors with type double and allocates memory for "w" and "z" coordinates.} \f$ \n
  \f$ \textbf{z[1]=1.23; z[2]=-3.4; z[4]=2.34;} \; \textit{Stores the coordinates of "z" (see} \f$ \ref vector:operator[](int) \f$\textit{).} \f$ \n
  \f$ \textbf{w[1]=-1.45; w[3]=0.436; w[4]=5.75;} \; \textit{Stores the coordinates of "w" (see} \f$ \ref vector:operator[](int) \f$\textit{).} \f$ \n
  \f$ \textbf{v = w + z;} \; \textit{Allocates memory for "v" and stores the coordinates of "w+z" in v (see} \f$ \ref vector:operator=(const vector&) \f$\textit{).} \f$ \n
*/

//----------------------------------------------------------------------------------
template <class Tf> vector<Tf> operator + (const vector<Tf>& a, const vector<Tf>& b)
//----------------------------------------------------------------------------------
{
  assert (a.size);
  assert (a.size == b.size);

  vector<Tf> v = a;

  return v += b;
}


/*!
  \brief Subtraction for vectors.

  Computes the subtraction of the vectors \f$ \boldsymbol{a} = (a_i) \f$ and \f$ \boldsymbol{b} = (b_i) \f$ :
  \f$ c_i = a_i - b_i \f$, where \f$ \boldsymbol{c} = (c_i) \f$.
  \param vector a
  \param vector b
  \return \f$ c = a-b \f$
  \n\n
  \f$ \textbf{Example :} \f$ \n
  \f$ \textbf{vector$<$double$>$ v, w(4), z(4);} \; \textit{Defines 3 vectors with type double and allocates memory for "w" and "z" coordinates.} \f$ \n
  \f$ \textbf{z[1]=1.23; z[2]=-3.4; z[4]=2.34;} \; \textit{Stores the coordinates of "z" (see} \f$ \ref vector:operator[](int) \f$\textit{).} \f$ \n
  \f$ \textbf{w[1]=-1.45; w[3]=0.436; w[4]=5.75;} \; \textit{Stores the coordinates of "w" (see} \f$ \ref vector:operator[](int) \f$\textit{).} \f$ \n
  \f$ \textbf{v = w - z;} \; \textit{Allocates memory for "v" and stores the coordinates of "w-z" in v (see} \f$ \ref vector:operator=(const vector&) \f$\textit{).} \f$ \n
*/

//----------------------------------------------------------------------------------
template <class Tf> vector<Tf> operator - (const vector<Tf>& a, const vector<Tf>& b)
//----------------------------------------------------------------------------------
{
  assert (a.size);
  assert (a.size == b.size);

  vector<Tf> v = a;

  return v -= b;
}


/*!
  \brief Scalar product.

  Computes the scalar product of the vectors \f$ \boldsymbol{a} = (a_i) \f$ and \f$ \boldsymbol{b} = (b_i) \f$ :
  \f$ c = a_i * b_i \f$, where c is a scalar.
  \param vector a
  \param vector b
  \return \f$ c = a*b \f$
  \n\n
  \f$ \textbf{Example :} \f$ \n
  \f$ \textbf{vector$<$double$>$ w(4), z(4);} \; \textit{Defines 2 vectors with type double and allocates memory for their coordinates.} \f$ \n
  \f$ \textbf{double v;} \; \textit{Defines a scalar with type double.} \f$ \n
  \f$ \textbf{z[1]=1.23; z[2]=-3.4; z[4]=2.34;} \; \textit{Stores the coordinates of "z" (see} \f$ \ref vector:operator[](int) \f$\textit{).} \f$ \n
  \f$ \textbf{w[1]=-1.45; w[3]=0.436; w[4]=5.75;} \; \textit{Stores the coordinates of "w" (see} \f$ \ref vector:operator[](int) \f$\textit{).} \f$ \n
  \f$ \textbf{v = w * z;} \; \textit{Stores the scalar "w*z" in v.} \f$ \n
*/

//--------------------------------------------------------------------------
template <class Tf> Tf operator * (const vector<Tf>& a, const vector<Tf>& b)
//--------------------------------------------------------------------------
{
  assert (a.size);
  assert (a.size == b.size);

  Tf sp=0;
  for (int i=0; i<a.size; i++)
       sp += a.p[i] * b.p[i];

  return sp;
}


/*!
  \brief Division by a vector of size 1.

  Computes the division of a vector \f$ \boldsymbol{a} = (a_i) \f$ by a scalar stored as a vector \f$ \boldsymbol{b} = (b_1) \f$ :
  \f$ c_i = a_i / b_1 \f$, where c is a vector.
  \param vector a
  \param vector b
  \return \f$ c = a/b_1 \f$
  \n\n
  \f$ \textbf{Example :} \f$ \n
  \f$ \textbf{vector$<$double$>$ w(4), z(1);} \; \textit{Defines 2 vectors with type double and allocates memory for their coordinates.} \f$ \n
  \f$ \textbf{vector$<$double$>$ v;} \; \textit{Defines a vector with type double.} \f$ \n
  \f$ \textbf{z[1]=1.23;} \; \textit{Stores the coordinates of "z" (see} \f$ \ref vector:operator[](int) \f$\textit{).} \f$ \n
  \f$ \textbf{w[1]=-1.45; w[3]=0.436; w[4]=5.75;} \; \textit{Stores the coordinates of "w" (see} \f$ \ref vector:operator[](int) \f$\textit{).} \f$ \n
  \f$ \textbf{v = w / z;} \; \textit{Stores "w/z[1]" in v.} \f$ \n
*/

//----------------------------------------------------------------------------------
template <class Tf> vector<Tf> operator / (const vector<Tf>& a, const vector<Tf>& b)
//----------------------------------------------------------------------------------
{
  assert (a.size && (b.size==1)); // exists only if the size of b = 1
  assert (b.p[0]);
  vector<Tf> v = a;

  return v /= b.p[0];
}


/*!
  \brief Add a vector.

  Adds the vector \f$ \boldsymbol{v} = (v_i) \f$ to the current vector \f$ \boldsymbol{a} = (a_i) \f$ :
  \f$ a_i += v_i \f$  (which is equivalent to \f$ a_i = a_i + v_i \f$ but faster).

  \param vector v

  \return \f$  \boldsymbol{a} =  \boldsymbol{a} +  \boldsymbol{v} \f$
  \n\n
  \f$ \textbf{Example :} \f$ \n
  \f$ \textbf{vector$<$double$>$ w(4), z(4);} \; \textit{Defines 2 vectors with type double and allocates memory for "w" and "z" coordinates.} \f$ \n
  \f$ \textbf{z[1]=1.23; z[2]=-3.4; z[4]=2.34;} \; \textit{Stores the coordinates of "z" (see} \f$ operator[](int)const \f$\textit{).} \f$ \n
  \f$ \textbf{w[1]=-1.45; w[3]=0.436; w[4]=5.75;} \; \textit{Stores the coordinates of "w" (see} \f$ operator[](int)const \f$\textit{).} \f$ \n
  \f$ \textbf{w += z;} \; \textit{Adds to "w" coordinates the ones of "z"} \textit{).} \f$ \n
*/

//-----------------------------------------------------------------------
template <class T> vector<T>& vector<T>::operator += (const vector<T>& v)
//-----------------------------------------------------------------------
{
  assert (size);
  assert (size==v.size);

  for (int i=0; i<size; i++)
    p[i] += v.p[i];

  return *this;
}


/*!
  \brief Subtract a vector.

  Subtracts the vector \f$ \boldsymbol{v} = (v_i) \f$ to the current vector \f$ \boldsymbol{a} = (a_i) \f$ :
  \f$ a_i -= v_i \f$  (which is equivalent to \f$ a_i = a_i - v_i \f$ but faster).

  \param vector v

  \return \f$  \boldsymbol{a} =  \boldsymbol{a} -  \boldsymbol{v} \f$
  \n\n
  \f$ \textbf{Example :} \f$ \n
  \f$ \textbf{vector$<$double$>$ w(4), z(4);} \; \textit{Defines 2 vectors with type double and allocates memory for "w" and "z" coordinates.} \f$ \n
  \f$ \textbf{z[1]=1.23; z[2]=-3.4; z[4]=2.34;} \; \textit{Stores the coordinates of "z" (see} \f$ operator[](int)const \f$\textit{).} \f$ \n
  \f$ \textbf{w[1]=-1.45; w[3]=0.436; w[4]=5.75;} \; \textit{Stores the coordinates of "w" (see} \f$ operator[](int)const \f$\textit{).} \f$ \n
  \f$ \textbf{w -= z;} \; \textit{Subtracts to "w" coordinates the ones of "z"} \textit{).} \f$ \n
*/

//-----------------------------------------------------------------------
template <class T> vector<T>& vector<T>::operator -= (const vector<T>& v)
//-----------------------------------------------------------------------
{
  assert (size);
  assert (size==v.size);

  for (int i=0; i<size; i++)
    p[i] -= v.p[i];

  return *this;
}


//--------------------------------------------------------------------------
template <class Tf> vector<Tf> operator * (const vector<Tf>& v, const Tf& s)
//--------------------------------------------------------------------------
{
  assert (v.size);
  vector<Tf> w = v;

  return w *= s;
}


//--------------------------------------------------------------------------
template <class Tf> vector<Tf> operator * (const Tf& s, const vector<Tf>& v)
//--------------------------------------------------------------------------
{
  return v * s;
}


//--------------------------------------------------------------------------
template <class Tf> vector<Tf> operator / (const vector<Tf>& v, const Tf& s)
//--------------------------------------------------------------------------
{
  assert (v.size);
  assert (abs(s) > epsilon); // can't divide by zero
  vector<Tf> w = v;
  w /= s;
  return w;
}


//---------------------------------------------------------------
template <class T> vector<T>& vector<T>::operator *= (const T& s)
//---------------------------------------------------------------
{
  assert (size);

  for (int i=0; i<size; i++)
    p[i] *= s;

  return *this;
}


//---------------------------------------------------------------
template <class T> vector<T>& vector<T>::operator /= (const T& s)
//---------------------------------------------------------------
{
  assert (size);

  for (int i=0; i<size; i++)
    p[i] /= s;

  return *this;
}


//------------------------------------------------------------------------
template <class Tf> ostream& operator << (ostream& s, const vector<Tf>& v)
//------------------------------------------------------------------------
{
  assert (v.size);

  for (int i=0; i<v.size; i++)
    s << (v.p)[i] << " ";
  s << endl;

  return s;
}


//------------------------------------------------------------------------
template <class Tf> istream& operator >> (istream& s, const vector<Tf>& v)
//------------------------------------------------------------------------
{
  assert (v.size);

  for (int i=0; i<v.size; i++)
      s >> (v.p)[i];

  return s;
}


/*!
  \brief Writes vector to hard disk.

  \param path path and filename
  \n\n
  \f$ \textbf{Example :} \f$ \n
  \f$ \textbf{vector$<$double$>$ z(4);} \; \textit{Defines a vector with type double and with 4 coordinates initialized to 0.} \f$ \n
  \f$ \textbf{z.save("vector.res");} \; \textit{Writes vector to disk. The filename is "vector.res".} \f$ \n
*/

//--------------------------------------------------------
template <class T> void vector<T>::save (const char* path)
//--------------------------------------------------------
{
  assert (size);
  ofstream file(path, ios::out);
  assert (!file.fail());
  file << *this;
  file.close();
}


/*!
  \brief Writes vector to hard disk.

  \param filename path and filename
  \n\n
  \f$ \textbf{Example :} \f$ \n
  \f$ \textbf{vector$<$double$>$ z(4);} \; \textit{Defines a vector with type double and with 4 coordinates initialized to 0.} \f$ \n
  \f$ \textbf{string name="vector";} \; \textit{Defines and initializes a string to "vector".} \f$ \n
  \f$ \textbf{name += ".res";} \; \textit{Adds ".res" to string.} \f$ \n
  \f$ \textbf{z.save(name);} \; \textit{Writes vector to disk. The filename is "vector.res".} \f$ \n
*/

//--------------------------------------------------------------
template <class T> void vector<T>::save (const string& filename)
//--------------------------------------------------------------
{
  const char *path = filename.c_str();
  save (path);
}


/*!
  \brief Reads vector from hard disk.

  \param path path and filename
  \n\n
  \f$ \textbf{Example :} \f$ \n
  \f$ \textbf{vector$<$double$>$ z(4);} \; \textit{Defines a vector with type double and with 4 coordinates initialized to 0.} \f$ \n
  \f$ \textbf{z.read("vector.res");} \; \textit{Reads vector coordinates from disk. The filename is "vector.res".} \f$ \n
*/

//--------------------------------------------------------
template <class T> void vector<T>::read (const char* path)
//--------------------------------------------------------
{
  assert (size);
  ifstream file(path, ios::in);
  assert (!file.fail());
  file >> *this;
  file.close();
}


/*!
  \brief Reads vector from hard disk.

  \param filename path and filename
  \n\n
  \f$ \textbf{Example :} \f$ \n
  \f$ \textbf{vector$<$double$>$ z(4);} \; \textit{Defines a vector with type double and with 4 coordinates initialized to 0.} \f$ \n
  \f$ \textbf{string name="vector";} \; \textit{Defines and initializes a string to "vector".} \f$ \n
  \f$ \textbf{name += ".res";} \; \textit{Adds ".res" to string.} \f$ \n
  \f$ \textbf{z.read(name);} \; \textit{Reads vector coordinates from disk. The filename is "vector.res".} \f$ \n
*/

//--------------------------------------------------------------
template <class T> void vector<T>::read (const string& filename)
//--------------------------------------------------------------
{
  const char *path = filename.c_str();
  read (path);
}


//-----------------------------------------------------
template <class T> long double vector<T>::norm () const
//-----------------------------------------------------
{
  assert (size);
  long double sqr_norm = pow (p[0], 2);

  for (int i=1; i<size; i++)
    sqr_norm += pow (p[i], 2);

  return sqrt (sqr_norm);
}


//----------------------------------------------
template <class T> T vector<T>::norminf () const
//----------------------------------------------
{
  assert (size);
  T norm = abs(p[0]);

  for (int i=1; i<size; i++)
      if (abs(p[i]) > norm) norm = abs(p[i]);

  return norm;
}


//-------------------------------------------------------
template <class T> vector<T>& vector<T>::approximation ()
//-------------------------------------------------------
{
  assert (size);
  T elt;
  T norm = sup ((T)(*this).norm(), (T) 1);

  for (int i=0; i<size; i++)
    if (abs(p[i]) < epsilon * norm) p[i]=0;

  return *this;
}


//------------------------------------------------
template <class T> vector<T> vector<T>::ortho2D ()
//------------------------------------------------
{
  assert (size);
  vector<T> v(2,false);

  v[1] = -(*this)[2];
  v[2] = (*this)[1];

  return v;
}


//-------------------------------------------------------
template <class T> void vector<T>::permute (int i, int j)
//-------------------------------------------------------
{
  assert (size);
  assert (i>0 && i<=size);
  assert (j>0 && j<=size);

  if (i!=j)
    { T elt;
      elt = p[i-1];
      p[i-1] = p[j-1];
      p[j-1] = elt;
    }
}


//-------------------------------------------------------------
template <class Tf> void permute (vector<Tf>& a, vector<Tf>& b)
//-------------------------------------------------------------
{
  assert (a.size);
  assert (a.size==b.size);
  Tf* elt;

  elt = a.p;
  a.p = b.p;
  b.p = elt;
}


//------------------------------------------------------
template <class Tf> vector<Tf> abs (const vector<Tf>& v)
//------------------------------------------------------
{ 
  assert (v.size);
  vector<Tf> u(v.size,false);  // allocates memory without initializing the vector components to zero

  for (int i=0; i<v.size; i++)
    u.p[i] = abs(v.p[i]);

  return u;
}


//---------------------------------------------------
template <class Tf> vector<int> order (vector<Tf>& v)
//---------------------------------------------------
{
  // A REVOIR et OPTIMISER
  assert (v.size);

  Tf elt;
  int k;
  vector<int> positions (v.size, false);
  for (int i=0; ++i<=v.size; positions[i] = i);

  for (int i=1; i<v.size; i++)
    for (int j=i+1; j<=v.size; j++)
      if (v[i] > v[j])
	{ elt  = v[i];
          v[i] = v[j];
          v[j] = elt;
	  k = positions[i];
	  positions[i] = positions[j];
	  positions[j] = k;
        }

  return positions;
}


//--------------------------------------------------------------------------
template <class Tf> vector<Tf> subvector (const vector<Tf>& v, int i, int j)
//--------------------------------------------------------------------------
{
  assert (v.size);
  assert (i>0 && i<=j && j<=v.size);
  vector<Tf> u(j-i+1,false);

  for (int k=i; k<=j; k++)
    u.p[k-i] = v.p[k-1];

  return u;
}


//-----------------------------------------------------------------------------
template <class Tf> vector<Tf> merge (const vector<Tf>& a, const vector<Tf>& b)
//-----------------------------------------------------------------------------
{
  assert (a.size && b.size);
  vector<Tf> v (a.size + b.size,false);

  for (int i=0; i<a.size; i++)
    v.p[i] = a.p[i];
  for (int i=0; i<b.size; i++)
    v.p[a.size+i] = b.p[i];

  return v;
}


}


#endif
