/***************************************************************
 *                 Mathematical Object Library                 *
 *       class spmatrix : declarations for sparse matrix       *
 *                    simula.plus@cemes.fr                     *
 *		     GNU/linux version 3.4.0                   *
 *            software under General Public License            *
 ***************************************************************
 * copyright © 2004,2005 LAGOUCHE Sophie
 * copyright © 2004,2005 CREUSE Emmanuel
 * copyright © 2006 FACHE Damien
 * copyright © 2006,2007,2008,2009,2011,2012 COLLARD Christophe
 * copyright © 2004,2005,2006,2007,2008,2009,2011,2012 Centre National de la Recherche Scientifique
 * copyright © 2004,2005,2006,2007,2008,2009,2011 Arts et Métiers ParisTech
 * copyright © 2004,2005,2006,2007 Université de Valenciennes et du Hainaut Cambrésis
 * copyright © 2004,2005,2006,2007,2008,2009 Laboratoire de Physique et Mécanique des Matériaux (LPMM - CNRS)
 * copyright © 2004,2005,2006,2007 Laboratoire de Mathématiques et ses Applications de Valenciennes (LAMAV)
 * copyright © 2011 Laboratoire d'Etude des Microstructures et de Mécanique des Matériaux (LEM3 - CNRS)
 * copyright © 2011,2012 Centre d'Elaboration de Matériaux et d'Etudes Structurales (CEMES - CNRS)
 ***************************************************************/

/*! \namespace mol
    \brief Mathematical Object Libraries
*/

/*! 
  \class mol::spmatrix
  \brief sparse matrix library

  \htmlonly 
  <FONT color="#838383">

  spmatrix belongs to Mathematical Object Libraries (MOL++) </br>
  MOL++ is part of Simula+ <br><br>

  Simula+ is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version. <br><br>

  Simula+ is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details. <br><br>

  You should have received a copy of the GNU General Public License
  along with Simula+; if not, write to the Free Software
  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

  </FONT>
  \endhtmlonly

  A sparse matrix is used to store a matrix with many not elements per row in a dense format. \n
  To do so, we define 3 tables : \n
  mrows : stores the non zero elements (table defined in class matrix), \n
  columns_position : stores the columns position of non zero elements, \n
  nbRowsElements : number of non zero elements in each row. \n
  We also define some variables : \n
  nbRows : number of rows of sparse matrix (defined in class matrix), \n
  nbColumns : number of columns of sparse matrix (defined in class matrix), \n
  rows_max_elements : maximum number of non zero elements per row. \n
  For example, we will store the matrix \n
  \f$ \begin{pmatrix}
      1 &0 &0 &2 &0 &0 \\
      0 &2 &0 &0 &0 &0 \\
      0 &4 &0 &0 &0 &1 \\
      2 &0 &0 &0 &0 &0 \\
      0 &1 &2 &0 &0 &0
      \end{pmatrix}
  \f$ \n
  as a sparse matrix in the following way : \n
  \f$ mrows = \begin{pmatrix}
              1 &2 \\
	      2 &* \\
	      4 &1 \\
	      2 &* \\
	      1 &2
              \end{pmatrix}
  \f$
  \f$ columns\_positions = \begin{pmatrix}
                          1 &4 \\
			  2 &* \\
			  2 &6 \\
			  1 &* \\
			  2 &3
			  \end{pmatrix}
  \f$
  \f$ nbRowsElements = \begin{pmatrix}
                       2 \\ 1 \\ 2 \\ 1 \\ 2
		       \end{pmatrix}
  \f$
  \f$ \left\{ \begin{aligned}
        &nbRows = 5 \\
        &nbColumns = 6 \\
        &rows\_max\_elements = 2
      \end{aligned} \right.
  \f$
  \n
  where \f$ * \f$ is a random value. \n \n
  This way to store a matrix induces a problem when we define \f$ T\& \; operator () \; (int,int) \; const \f$. \n
  This operator is used to store and get the value of an element of the matrix. But how to return the reference of a not element (which is not stored) ? To solve this problem, we define a buffer with 3 variables : \n
  row : row position of an element, \n
  column : column position of an element, \n
  value : the value of the element located at (i,j). \n
  Now, if there is an element stored at position (i,j), then we return the reference of this element (stored in mrows). Elsewhere, we return the reference of 'value'. The idea is to store this buffer the next time we will call a method of spmatrix in order to update the tables mrows, columns_position and nbRowsElements. In the following example :\n
  \f$ spmatrix<double> M(5,6,2); \f$ \n
  \f$ M(1,2) = 2; \f$ \n
  \f$ M(1,2) \f$ is not yet defined, so we return the reference of 'value' and we store 1 in 'row' and 2 in 'column'. So the tables mrow, columns_position and nbRowsElements are not up to date. But this is not very important since we don't need at this time to use \f$ M \f$. But the next time we call a method of spmatrix (which means that we use \f$ M \f$), we will first have to update the tables (with a method named \f$ void \; update() \; const \f$). Another problem is when store a not element : \n
  \f$ M(1,2) = 0; \f$ \n
  Here the element already exists, so the update method will have to check and delete the value stored at (i,j) if it is null. \n \n
  We have also some problems with \f$ vector<T>\& \; operator [] \; (int) \; const \f$ which is used to store and get a matrix row. \n
  Since we cannot store a reduce vector using this operator (because we don't know the columns position associated with the reduced vector), we do replace the reduced vector stored at row \f$ i \f$ in mrows by its expanded expression and return it by reference. So we are able to get or allocate a full matrix row, since we don't need at this time that the tables mrow, columns_position and nbRowsElements are not up to date. We will have to update them the next time we call a sparse matrix method (with the method \f$ void \; update() \; const \f$), and we'll reduce all the expanded rows (which are tagged by turning the value of nbRowsElements[i] to -1 in \f$ operator [] \; (int \; i) \f$).

  \authors copyright \htmlonly &#169; \endhtmlonly 2004, 2005 Sophie LAGOUCHE \n
           copyright \htmlonly &#169; \endhtmlonly 2004, 2005 Emmanuel CREUSE \n
           copyright \htmlonly &#169; \endhtmlonly 2006 Damien FACHE \n
           copyright \htmlonly &#169; \endhtmlonly 2006, 2007, 2008, 2009, 2011, 2012 Christophe COLLARD \n
	   copyright \htmlonly &#169; 2004, 2005, 2006, 2007, 2008, 2009, 2011, 2012 Centre National de la Recherche Scientifique \endhtmlonly \n
	   copyright \htmlonly &#169; 2004, 2005, 2006, 2007, 2008, 2009, 2011 Arts et M&#233;tiers ParisTech \endhtmlonly \n
	     copyright \htmlonly &#169; 2004, 2005, 2006, 2007 Universit&#233; de Valenciennes et du Hainaut Cambr&#233;sis \endhtmlonly \n
	   copyright \htmlonly &#169; 2004, 2005, 2006, 2007, 2008, 2009 Laboratoire de Physique et M&#233;canique des Mat&#233;riaux (LPMM - CNRS) \endhtmlonly \n
	   copyright \htmlonly &#169; 2004, 2005, 2006, 2007 Laboratoire de Math&#233;matiques et ses Applications de Valenciennes (LAMAV) \endhtmlonly \n
	   copyright \htmlonly &#169; 2011 Laboratoire d'Etude des Microstructures et de M&#233;canique des Mat&#233;riaux (LEM3 - CNRS) \endhtmlonly \n
	   copyright \htmlonly &#169; 2011, 2012 Centre d'Elaboration de Mat&#233;riaux et d'Etudes Structurales (CEMES - CNRS) \endhtmlonly \n
  \version 3.4.0
  \date 2004-2019
  \bug multiple allocation does not work with g++ compiler (ex. A(2,4) = A(3,3) = A(5,5) = 2;)
  \warning none
*/

#ifndef __cplusplus
#error Must use C++ for the type sparse matrix
#endif

#ifndef __spmatrix_hpp
#define __spmatrix_hpp


#ifndef __stdlib_h
#include <stdlib.h>
#endif

#ifndef __iostream
#include <iostream>
#endif

#ifndef __fstream
#include <fstream>
#endif

#ifndef __assert_h
#include <assert.h>
#endif

#ifndef __parameters_h
#include "parameters.h"
#endif

#ifndef __maths_hpp
#include "maths.hpp"
#endif

#ifndef __vectors_hpp
#include "vectors.hpp"
#endif

#ifndef __matrix_hpp
#include "matrix.hpp"
#endif

using namespace std;

namespace mol
{


//==================================================
template <class T> class spmatrix : public matrix<T>
//==================================================
{
  using matrix<T>::nbRows;
  using matrix<T>::nbColumns;
  using matrix<T>::mrows;

  private:
    // buffered values - Mutables because modified by operator() or by operator []
    mutable int row;
    mutable int column;
    mutable T value;
    mutable bool update_rows;

    matrix<int> columns_position; // positions matrix
    int rows_max_elements; // max. non zero coefficients per row
    vector<int> nbRowsElements; // stores the number of non zero elements per row

    void set (int, int) const;        // sets the value at the specified coordonates
    void add (int, int, T);           // adds a value to a specified element
    T get (int, int&) const;          // returns the element at the specified coordonates
    void resize_row (int) const;      // sets the vector at the specified row
    vector<T> get_vector (int) const; // Returns the vector at the specified row
    void erase_zero(int) const;       // delete not elements in a row
    void update () const;

  public:
    spmatrix ();                    // default constructor
    spmatrix (int, int, int);       // this constructor allocates memory
    spmatrix (const spmatrix<T>&);  // copy constructor
    spmatrix (spmatrix<T>*);        // copy constructor for temporary objects (use it with great care)
    ~spmatrix ();                   // destructor

    virtual void assign (int, int, int);  // allocates memory without initilizing the matrix components to zero (so use it with great care)
    spmatrix<T> resize (int);       // resizes the number of not elements stored
    virtual int Row_Elements () const {return rows_max_elements;}  // returns the number of max. non zero coefficients per row

    operator matrix<T>  () const;   // cast operator : spmatrix -> matrix
    virtual T& operator () (int, int) const;
    virtual vector<T>& operator [] (int) const;  // returns a whole row
    virtual spmatrix<T>& operator = (const spmatrix<T>&);    // defines operator = for sparse matrices with same size or null size
    virtual spmatrix<T>& operator = (spmatrix<T>*); 	     // allocates the data from the right hand temporary object to the left hand object - no copy (deletes temporary object)
    virtual spmatrix<T>& operator &= (const spmatrix<T>&);   // copies matrix without size check (use this operator with great care)
    virtual spmatrix<T>& operator &= (spmatrix<T>*);	     // allocates the data from the right hand temporary object to the left hand object (deletes temporary object) - no size check - no copy (use this operator with great care)
    template <class Tf> friend spmatrix<Tf> operator + (const spmatrix<Tf>&, const spmatrix<Tf>&); // overloads sparse matrix + sparse matrix
    template <class Tf> friend spmatrix<Tf> operator - (const spmatrix<Tf>&, const spmatrix<Tf>&); // overloads sparse matrix - sparse matrix
    template <class Tf> friend spmatrix<Tf> operator * (const Tf&, const spmatrix<Tf>&); // overloads scalar * sparse matrix
    template <class Tf> friend spmatrix<Tf> operator * (const spmatrix<Tf>&, const Tf&); // overloads sparse matrix * scalar
    template <class Tf> friend vector<Tf>   operator * (const spmatrix<Tf>&, const vector<Tf>&);   // overloads sparse matrix * vector
    template <class Tf> friend vector<Tf>   operator * (const vector<Tf>&, const spmatrix<Tf>&);   // overloads vector * sparse matrix
    template <class Tf> friend matrix<Tf>   operator * (const spmatrix<Tf>&, const spmatrix<Tf>&); // overloads sparse matrix * sparse matrix
    template <class Tf> friend spmatrix<Tf> operator / (const spmatrix<Tf>&, const Tf&); // overloads sparse matrix / scalar
    template <class Tf> friend Tf operator | (const spmatrix<Tf>&, const spmatrix<Tf>&); // overloads sparse matrix | sparse matrix
    spmatrix<T>& operator += (const spmatrix<T>&); // overloads += operator for matrix
    spmatrix<T>& operator -= (const spmatrix<T>&); // overloads -= operator for matrix
    spmatrix<T>& operator *= (const T&);           // overloads *= operator between matrix and scalar
    spmatrix<T>& operator /= (const T&);           // overloads /= operator between matrix and scalar
    template <class Tf> friend bool operator == (const spmatrix<Tf>&, const spmatrix<Tf>&); // compares 2 sparse matrices
    template <class Tf> friend bool operator != (const spmatrix<Tf>&, const spmatrix<Tf>&);

    template <class Tf> friend ostream& operator << (ostream&, const spmatrix<Tf>&); // overloads output stream for sparse matrix
    template <class Tf> friend istream& operator >> (istream&, spmatrix<Tf>&);       // overloads input stream for sparse matrix

    template <class Tf> friend spmatrix<Tf> t (const spmatrix<Tf>&);      // defines tranposition operator for sparse matrix
    template <class Tf> friend spmatrix<Tf> abs (const spmatrix<Tf>&);    // gives the absolute value of a sparse matrix
    template <class Tf> friend Tf tr (const spmatrix<Tf>&);               // gives the trace of a sparse matrix
    template <class Tf> friend Tf max (const spmatrix<Tf>&);              // gives the greatest element of a sparse matrix
    template <class Tf> friend vector<Tf> diagonal (const spmatrix<Tf>&); // returns the diagonal of a sparse matrix
    template <class Tf> friend spmatrix<Tf> Id2sp (int);                  // returns the identity matrix

    void save (const char* path);
    void save (const string&);
    void read (const char* path);
    void read (const string&);

  //operators = spmat/mat and = spmat/symmat are not yet implemented, due to the apparent difference between a matrix and a sparse matrix.
};


//=====Private methods for spmatrix===========================================


/*!
  \brief  Stores the temporary element into the sparse matrix.

  Let \f$ A \f$ be a sparse matrix. We store the temporary element \f$ value \f$ (see detailed description) in the value table and \f$ j \f$ in the coordinates table.
  \param i sparse matrix row number
  \param j sparse matrix column number
*/

//-----------------------------------------------------------
template <class T> void spmatrix<T>::set (int i, int j) const
//-----------------------------------------------------------
{
  assert (i>0 && i<=nbRows);
  assert (j>0 && j<=nbColumns);
  assert (i==row && j==column);

  int r = nbRowsElements[i];

  if (abs(value) > epsilon)
    { bool done = false;
      for (int k=1; k<=r && !done; k++)
	if (columns_position(i,k) == j)
	  { mrows[i-1][k] = value;
	    done = true;
	  }

      if (!done)
	{ if (r+1 > rows_max_elements)
	    { cout << "Fatal error : row " << i << " already contains " << rows_max_elements << " non-zero elements, that's the maximum !\n";
	      assert (false);
	      exit(1);
	    }
	  columns_position(i,r+1) = j;
	  mrows[i-1][r+1] = value;
	  nbRowsElements[i]++;
    }       }

  else
    { int nbmax = 1;
      for (int n=1; n<=r; n++)
	if (abs(mrows[i-1][n]) > epsilon)
	  nbmax = n;

      for (int k=1; k<=nbmax; k++)
	if (columns_position(i,k) == j)
	  { mrows[i-1][k] = mrows[i-1][nbmax];
	    mrows[i-1][nbmax] = 0;
	    columns_position(i,k) = columns_position(i,nbmax);
	    columns_position(i,nbmax) = 0;
	    nbRowsElements[i]--;
	    nbmax --;
	  }
    }
}


/*!
  \brief Adds a coefficient to the element (i,j) of a sparse matrix.

  \param i sparse matrix row number
  \param j sparse matrix column number
  \param value value to add
*/

//--------------------------------------------------------------
template <class T> void spmatrix<T>::add (int i, int j, T value)
//--------------------------------------------------------------
{
  assert (i>0 && i<=nbRows);
  assert (j>0 && j<=nbColumns);

  int r = nbRowsElements[i];

  if (abs(value) > epsilon)
    { bool done = false;
      for (int k=1; k<=r; k++)
	if (columns_position(i,k) == j)
	  { if (abs(mrows[i-1][k] + value) < epsilon)
	      { mrows[i-1][k] = mrows[i-1][r];
		columns_position(i,k) = columns_position(i,r);
		mrows[i-1][r] = 0;
		columns_position(i,r) = 0;
		nbRowsElements[i] -= 1;
		done = true;
	      }
	    else
	      { mrows[i-1][k] += value;
		done = true;
	      }
	  }

      if (!done)
	{ if (r+1 > rows_max_elements)
	    { cout << "Fatal error : it's impossible to add this non-zero element because row " << i << " already contains " << rows_max_elements << " non-zero elements, that's the maximum !\n";
	      assert(false);
	      exit(1);
	    }
	  columns_position(i,r+1) = j;
	  mrows[i-1][r+1] = value;
	  nbRowsElements[i] = r+1;
    }       }
}


/*!
  \brief Returns a copy of a sparse matrix element.

  Let A be a sparse matrix. If \f$ A(i,j) \neq 0 \f$ then we replace \f$ j \f$ by its position in the coordinates table and we return a copy of the A(i,j) value (see detailed description). Elsewhere, we return \f$ 0 \f$.
  \param i sparse matrix row number
  \param j sparse matrix column number (returns the position where j is stored in the coordinates table)
  \return sparse matrix (i,j) value

*/

//---------------------------------------------------------
template <class T> T spmatrix<T>::get (int i, int& j) const
//---------------------------------------------------------
{
  assert (i>0 && i<=nbRows);
  assert (j>0 && j<=nbColumns);

  T scalar = 0;
  bool exist = false;

  int r = nbRowsElements[i];

  for (int k=1; k<=r && !exist; k++){
    exist = (columns_position(i,k) == j);
    if (exist)
      scalar = mrows[i-1][j=k];
  }

  return scalar;
}


/*!
  \brief  Resizes a sparse matrix row.

  Deletes not values at the \f$ i^\text{th} \f$ (full) row of a sparse matrix.
  \param i sparse matrix row number
*/

//-----------------------------------------------------------
template <class T> void spmatrix<T>::resize_row (int i) const
//-----------------------------------------------------------
{
  assert(i>0 && i<=nbRows);

  vector<T> v =& mrows[i-1];
  assert(v.dim() == nbColumns);

  int nb = 0;
  mrows[i-1] = rows_max_elements;

  // v is scanned to enter the non-zero values in mrows and columns_position
  for (int j=1; j<=nbColumns; j++)
    { if (abs(v[j]) > epsilon)
	{ if (++nb > rows_max_elements)
	    { cout << "Fatal error : there are too many non-zero elements in this vector" << endl;
	      exit(1);
	    }
	  mrows[i-1][nb] = v[j];
	  columns_position(i,nb) = j;
	}
    }

  nbRowsElements[i] = nb;
}


/*!
  \brief Returns a copy of a sparse matrix (full) row.

  \param i sparse matrix row number
  \return sparse matrix \f$ i^\text{th} \f$ row (including zero elements)
*/

//----------------------------------------------------------------
template <class T> vector<T> spmatrix<T>::get_vector (int i) const
//----------------------------------------------------------------
{
  assert (i>0 && i<=nbRows);

  vector<T> v(nbColumns);

  int r = nbRowsElements[i];
  for (int j=1; j<=r; j++)
    v[columns_position(i,j)] = mrows[i-1][j];

  return v;
}


/*!
  \brief Deletes not elements in a (reduced) row of a sparse matrix.

  \param i sparse matrix row number
*/

//-----------------------------------------------------------
template <class T> void spmatrix<T>::erase_zero (int i) const
//-----------------------------------------------------------
{
  assert (mrows[i-1].dim() == rows_max_elements);

  int r = nbRowsElements[i];
  bool exist = false;

  for (int k=1; k<=r && !exist; k++)
    if (abs(mrows[i-1][k]) < epsilon)
      { exist = true;
	mrows[i-1][k] = mrows[i-1][r];
	columns_position(i,k) = columns_position(i,r);
	columns_position(i,r) = 0;
	nbRowsElements[i] -= 1;
      }
  row = 0;
}


/*!
  \brief Updates sparse matrix.

  We update the (i,j) sparse matrix value or (exclusive or) the \f$ i^\text{th} \f$ sparse matrix row.
*/

//--------------------------------------------------
template <class T> void spmatrix<T>::update () const
//--------------------------------------------------
{
  assert (!(row*update_rows)); // can't update sparse matrix element and row at the same time

  if (row)
    if (!column) erase_zero(row);
    else set(row,column);
  else if (update_rows)
    { for (int i=1; i<=nbRows; i++)
	if (nbRowsElements[i] == -1) resize_row(i);
    }
  row = update_rows = 0;
}


//=====Public methods for spmatrix============================================


/*!
  \brief Default constructor.
*/

//-----------------------------------------
template <class T> spmatrix<T>::spmatrix ()
//-----------------------------------------
{
  nbRows = rows_max_elements = nbColumns = row = column = update_rows = 0;
}


/*!
  \brief Constructor.

  \param rows number of rows of the sparse matrix
  \param columns number of columns of the sparse matrix
  \param non_zero_elements maximal number of non-zero elements per row
*/

//----------------------------------------------------------------------------------------------------------------------------------
template <class T> spmatrix<T>::spmatrix (int rows, int columns, int non_zero_elements) : matrix<T> (rows, non_zero_elements, false)
//----------------------------------------------------------------------------------------------------------------------------------
{
  assert (rows>0 && columns>0 && columns>=non_zero_elements);

  if (columns == non_zero_elements) cout << "WARNING : you are using a sparse matrix to store a full matrix \n";

  nbColumns = columns;
  rows_max_elements = non_zero_elements;
  columns_position.assign (rows, non_zero_elements);
  nbRowsElements =& vector<int>(rows);  // the number of elements per row must be initialized

  // initialize buffer
  row = update_rows = 0;
}


/*!
  \brief Copy constructor.

  \param mat sparse matrix to duplicate
*/

//---------------------------------------------------------------
template <class T> spmatrix<T>::spmatrix (const spmatrix<T>& mat)
//---------------------------------------------------------------
{
  assert (mat.nbRows && mat.nbColumns && mat.rows_max_elements);

  mat.update();

  nbColumns = mat.nbColumns;
  rows_max_elements = mat.rows_max_elements;

  mrows = new vector<T> [nbRows=mat.nbRows];
  for (int i=0; i<nbRows; i++)
    mrows[i] = mat.mrows[i];

  columns_position = mat.columns_position;
  nbRowsElements = mat.nbRowsElements;

  row = update_rows = 0;
}


/*!
  \brief Destructive constructor.

  Makes a fast copy of sparse matrix 'mat' parameters and deletes 'mat' (turns its size to zero). Note that we neither duplicate the coordinates table nor the value table here.
  This constructor is generally used with temporary objects.

  \param mat sparse matrix address
*/

//---------------------------------------------------------
template <class T> spmatrix<T>::spmatrix (spmatrix<T>* mat)
//---------------------------------------------------------
{
  assert ((*mat).nbRows);

  (*mat).update();

  nbRows = (*mat).nbRows;
  nbColumns = (*mat).nbColumns;
  rows_max_elements = (*mat).rows_max_elements;
  mrows = (*mat).mrows;	       	                  // copy the address of the temporary objet

  columns_position =& ((*mat).columns_position);  // copy the address of the temporary objet
  nbRowsElements =& ((*mat).nbRowsElements);      // copy the address of the temporary objet

  // set temporary objects size to zero (so the data stored at p won't be delete by the destructor)
  (*mat).nbRows = (*mat).nbColumns = (*mat).rows_max_elements = 0;

  row = update_rows = 0;
}


/*!
  \brief Destructor.
*/

//------------------------------------------
template <class T> spmatrix<T>::~spmatrix ()
//------------------------------------------
{
  nbRows = rows_max_elements = nbColumns = row = column = update_rows = 0;
}


/*!
  \brief Allocates memory for a sparse matrix without initializing its components.

  \param nrows number of rows of the sparse matrix
  \param ncolumns number of columns of the sparse matrix
  \param nrow_elements maximum number of non-zero elements on a sparse matrix row
*/

//--------------------------------------------------------------------------------------
template <class T> void spmatrix<T>::assign (int nrows, int ncolumns, int nrow_elements)
//--------------------------------------------------------------------------------------
{
  assert (nrows>0 && ncolumns>0 && nrow_elements>0);
  assert (nrow_elements <= ncolumns);
  assert (!nbRows);

  if (ncolumns == nrow_elements) cout << "WARNING : you are using a sparse matrix to store a full matrix \n";

  nbColumns = ncolumns;
  rows_max_elements = nrow_elements;

  mrows = new vector<T> [nbRows = nrows];  // allocates memory for rows (no initialization)
  for (int i=0; i<nbRows; i++)
    mrows[i].assign (rows_max_elements);

  columns_position =& matrix<int> (nbRows, rows_max_elements, false);
  nbRowsElements =& vector<int> (nbRows);  // the number of elements per row must be initialized

  // initialize buffer
  row = update_rows = 0;
}


/*!
  \brief Changes the number of not elements stored of a sparse matrix.

  Resizes the values table and the coordinates table.

  \param n new number of not elements per row
  \return resized sparse matrix
*/

//--------------------------------------------------------
template <class T> spmatrix<T> spmatrix<T>::resize (int n)
//--------------------------------------------------------
{
  assert (n>0);

  update ();

  spmatrix<T> mat (nbRows, nbColumns, n);

  for (int i=1; i<=nbRows; i++)
    { assert (nbRowsElements[i] <= n);
      mat.nbRowsElements[i] = nbRowsElements[i];
      for (int j=1; j<=n; j++)
	{ mat.columns_position(i,j) = columns_position(i,j);
	  mat.mrows[i-1][j] = mrows[i-1][j];
	}
    }

  return mat;
}


/*!
  \brief Cast operator : spmatrix -> matrix.

  Converts a sparse matrix into a matrix.

  \return matrix
*/

//---------------------------------------------------------
template <class T> spmatrix<T>::operator matrix<T> () const
//---------------------------------------------------------
{
  assert (nbRows);

  update (); //updating the buffer

  matrix<T> mat (nbRows, nbColumns);

  for (int i=1; i<=nbRows; i++)
    { int r = nbRowsElements[i];
      for (int j=1; j<=r; j++)
	mat(i,columns_position(i,j)) = mrows[i-1][j];
    }

  return mat;
}


/*!
  \brief Operator () for sparse matrix.

  Let \f$ A \f$ be a sparse matrix. This operator allows to store a matrix element. But, in order to allow multiple allocations (for example \f$ A(1,1) = A(1,2) = A(2,1) \f$, and also because this operator can be used to get a sparse matrix element (for example \f$ x = A(1,1) + A(1,2) \f$), we have to return \n
  the reference of \f$ A(i,j) \f$ when \f$ A(i,j) \neq 0 \f$, \n
  the buffer \f$ value \f$ when \f$ A(i,j) = 0 \f$. \n
 
  \param i sparse matrix row
  \param j sparse matrix column
  \return \f$ A(i,j) \f$ or buffered \f$ value \f$
*/

//-----------------------------------------------------------------
template <class T> T& spmatrix<T>::operator () (int i, int j) const
//-----------------------------------------------------------------
{
  assert (i>0 && i<=nbRows);
  assert (j>0 && j<=nbColumns);

  update (); // Updating the good old buffer

  row = i;
  column = j;

  if (get(i,j))  // get value and position j in the value table (=> j != column)
    { column = 0;
      return mrows[i-1][j]; // returns stored value
    }
  else
    return value=0; // returns buffer
}


/*!
  \brief Operator [] for sparse matrix.

  Let M be a sparse matrix, and \f$ v \f$ be a vector. In order to allow multiple row allocation (\f$ M[1] = M[2] = M[3] = v; \f$) we replace the \f$ i^\text{th} \f$ reduced row (without not elements) of \f$ M \f$ by its full row (with not elements) and we return this full row by reference. Then, when the uptade method is called (the next time a sparse matrix method is called), we reduce this row by storing only not elements.

  \param i sparse matrix row
  \return full row (with zero elements) of a sparse matrix
*/

//------------------------------------------------------------------
template <class T> vector<T>& spmatrix<T>::operator [] (int i) const
//------------------------------------------------------------------
{
  assert (i>0 && i<=nbRows);

  if (row) update (); // Updating the good old buffer for operator (,) - never update buffer for operator [] !

  if (nbRowsElements[i] != -1)
    { mrows[i-1] &=& get_vector(i);
      nbRowsElements[i] = -1;
      update_rows = true;
    }

  return mrows[i-1];
}


/*!
  \brief Operator = for sparse matrix.

  \param mat sparse matrix to copy
  \return left hand side sparse matrix
  \n\n
  \f$ \textbf{Examples :} \f$ \n
  \f$ \textbf{spmatrix$<$long double$>$ M(5,6,2), N(5,6,2);} \; \textit{Allocates memory for sparse matrices}  \f$ \n
  \f$ \textbf{spmatrix$<$long double$>$ P;} \; \textit{Defines sparse matrix but doen't allocate memory.} \f$ \n
  \f$ \textbf{M = N;} \; \textit{Checks that M and N have the same number of rows, columns and max. non zero elements per row, then copies N in M} \f$ \n
  \f$ \textbf{P = N;} \; \textit{Allocates memory for P, then copies N in P} \f$
*/

//------------------------------------------------------------------------------
template <class T> spmatrix<T>& spmatrix<T>::operator = (const spmatrix<T>& mat)
//------------------------------------------------------------------------------
{
  assert (mat.nbRows && mat.rows_max_elements && mat.nbColumns);

  if (!nbRows) assign(mat.nbRows, mat.nbColumns, mat.rows_max_elements);
  else assert ( (nbRows==mat.nbRows) && (rows_max_elements==mat.rows_max_elements) && (nbColumns==mat.nbColumns) );

  for (int i=0; i<mat.nbRows; i++)
    mrows[i] = mat.mrows[i];

  columns_position = mat.columns_position;
  nbRowsElements = mat.nbRowsElements;

  // update buffer
  row = mat.row;
  column = mat.column;
  value = mat.value;
  update_rows = mat.update_rows;

  return *this;
}


/*!
  \brief Destructive operator = for sparse matrix.

  Makes a fast copy of sparse matrix 'mat' parameters and tables, then deletes mat (turns its size to zero). Note that here, we do not make any duplication of the tables mrows, columns_position and nbRowsElements (see detailed description). \n
  This operator is generally used with temporary objects.
  \param mat sparse matrix address
  \return left hand side sparse matrix
*/

//------------------------------------------------------------------------
template <class T> spmatrix<T>& spmatrix<T>::operator = (spmatrix<T>* mat)
//------------------------------------------------------------------------
{
  assert(mat->nbRows && mat->nbColumns && mat->rows_max_elements);

  if (nbRows)
    { assert((nbRows == mat->nbRows) && (nbColumns == mat->nbColumns) && (rows_max_elements == mat->rows_max_elements));
      delete [] mrows;  //no need to delete columns_position and nbRowElements. It's done with the operator &=
    }
  else
    { nbRows = mat->nbRows;
      nbColumns = mat->nbColumns;
      rows_max_elements = mat->rows_max_elements;
    }
  //at this point they inevitably have the same size
  mrows = mat->mrows;
  columns_position =& mat->columns_position;
  nbRowsElements =& mat->nbRowsElements;

  mat->nbRows = mat->nbColumns = mat->rows_max_elements = 0;

  // update buffer
  row = mat->row;
  column = mat->column;
  value = mat->value;
  update_rows = mat->update_rows;

  return *this;
}


/*!
  \brief Operator = for sparse matrices with different sizes.

  \param mat sparse matrix to copy
  \return left hand side sparse matrix
  \sa vector::operator&=
*/

//-------------------------------------------------------------------------------
template <class T> spmatrix<T>& spmatrix<T>::operator &= (const spmatrix<T>& mat)
//-------------------------------------------------------------------------------
{
  // Copying the buffer state.
  row = mat.row;
  column = mat.column;
  value = mat.value;

  update_rows = mat.update_rows;

  if (!mat)
    { if (nbRows)
	{ delete [] mrows;
	  columns_position.~matrix();
	  nbRowsElements.~vector();
	}
      nbRows = nbColumns = rows_max_elements = 0;
    }

  else
    { if ((nbRows != mat.nbRows) || (nbColumns != mat.nbColumns) || (rows_max_elements != mat.rows_max_elements))
	{ if (nbRows)
	    delete [] mrows;  // no need to delete columns_position and nbRowElements. It's done with the operator &=
	  mrows = new vector<T> [nbRows = mat.nbRows];
	  nbColumns = mat.nbColumns;
	  rows_max_elements = mat.rows_max_elements;
	}

      for (int i=0; i<mat.nbRows; i++)
	mrows[i] = mat.mrows[i];

      columns_position &= mat.columns_position;
      nbRowsElements &= mat.nbRowsElements;
    }

  return *this;
}


/*!
  \brief Destructive operator = for sparse matrices with different sizes.

  This operator is generally used with temporary objects.

  \param mat matrix address
  \return left hand side sparse matrix
  \sa vector::operator&=(vector<T>*)
*/

//-------------------------------------------------------------------------
template <class T> spmatrix<T>& spmatrix<T>::operator &= (spmatrix<T>* mat)
//-------------------------------------------------------------------------
{
  // Getting the buffer state for furthr access.
  row = mat->row;
  column = mat->column;
  value = mat->value;

  update_rows = mat->update_rows;

  if (!(*mat).nbRows)
    { if (nbRows)
	{ delete [] mrows;
	  columns_position.~matrix();
	  nbRowsElements.~vector();
	}
      nbRows = nbColumns = rows_max_elements = 0;
    }

  else
    { if (nbRows)
	delete [] mrows;  // no need to delete columns_position and nbRowElements. It's done with the operator &=&
      nbRows = (*mat).nbRows;
      nbColumns = (*mat).nbColumns;
      rows_max_elements = (*mat).rows_max_elements;
      mrows = (*mat).mrows;
      columns_position &=& (*mat).columns_position;
      nbRowsElements &=& (*mat).nbRowsElements;
      (*mat).nbRows = (*mat).nbColumns = (*mat).rows_max_elements = 0;
    }

  return *this;
}


/*!
  \brief Addition for sparse matrices

  \param mat1 sparse matrix
  \param mat2 sparse matrix
  \return sparse matrix \f$ mat1 + mat2 \f$
*/

//----------------------------------------------------------------------------------------------
template <class Tf> spmatrix<Tf> operator + (const spmatrix<Tf>& mat1, const spmatrix<Tf>& mat2)
//----------------------------------------------------------------------------------------------
{
  assert( (mat1.nbRows == mat2.nbRows)  &&  (mat1.nbColumns == mat2.nbColumns) );

  mat1.update (); // Updating the good old buffer
  mat2.update (); // Updating the good old buffer

  if (mat1.rows_max_elements < mat2.rows_max_elements) return mat2 + mat1;

  int size = min (mat1.rows_max_elements+mat2.rows_max_elements, mat1.nbColumns);
  spmatrix<Tf> res(mat1.nbRows, mat1.nbColumns, size);
  for (int i=1; i<=mat1.nbRows; i++)
    { for (int j=1; j<=mat1.rows_max_elements; j++)
	{ res.mrows[i-1][j] = mat1.mrows[i-1][j];
	  res.columns_position(i,j) = mat1.columns_position(i,j);
	}
      res.nbRowsElements[i] = mat1.nbRowsElements[i];
    }

  for (int i=1; i<=mat1.nbRows; i++)
    { for (int j=1; j<=mat2.nbRowsElements[i]; j++)
	res.add(i, mat2.columns_position(i,j), mat2.mrows[i-1][j]);
      if (res.nbRowsElements[i] > size) size = res.nbRowsElements[i];
    }

  if (size) return res.resize(size);
  else return res;
}


/*!
  \brief Subtraction for sparse matrices

  \param mat1 sparse matrix
  \param mat2 sparse matrix
  \return sparse matrix \f$ mat1 - mat2 \f$
*/

//----------------------------------------------------------------------------------------------
template <class Tf> spmatrix<Tf> operator - (const spmatrix<Tf>& mat1, const spmatrix<Tf>& mat2)
//----------------------------------------------------------------------------------------------
{
  assert( (mat1.nbRows == mat2.nbRows)  &&  (mat1.nbColumns == mat2.nbColumns) );

  mat1.update ();
  mat2.update ();

  int size = min (mat1.rows_max_elements+mat2.rows_max_elements, mat1.nbColumns);
  spmatrix<Tf> res;

  if (mat1.rows_max_elements > mat2.rows_max_elements)
    { res.assign (mat1.nbRows, mat1.nbColumns, size);
      for (int i=1; i<=mat1.nbRows; i++)
	{ for (int j=1; j<=mat1.rows_max_elements; j++)
	    { res.mrows[i-1][j] = mat1.mrows[i-1][j];
	      res.columns_position(i,j) = mat1.columns_position(i,j);
	    }
	  res.nbRowsElements[i] = mat1.nbRowsElements[i];
	}

      for (int i=1; i<=mat1.nbRows; i++)
	{ for (int j=1; j<=mat2.nbRowsElements[i]; j++)
	    res.add(i, mat2.columns_position(i,j), -mat2.mrows[i-1][j]);
	  if (res.nbRowsElements[i] > size) size = res.nbRowsElements[i];
	}
    }

  else
    { res.assign (mat2.nbRows, mat2.nbColumns, size);
      for (int i=1; i<=mat2.nbRows; i++)
	{ for (int j=1; j<=mat2.rows_max_elements; j++)
	    { res.mrows[i-1][j] = -mat2.mrows[i-1][j];
	      res.columns_position(i,j) = mat2.columns_position(i,j);
	    }
	  res.nbRowsElements[i] = mat2.nbRowsElements[i];
	}

      for (int i=1; i<=mat2.nbRows; i++)
	{ for (int j=1; j<=mat1.nbRowsElements[i]; j++)
	    res.add(i, mat1.columns_position(i,j), mat1.mrows[i-1][j]);
	  if (res.nbRowsElements[i] > size) size = res.nbRowsElements[i];
	}
    }

  if (size) return res.resize(size);
  else return res;
}


/*!
  \brief Product between scalar and sparse matrix.

  \param elt scalar
  \param mat sparse matrix
  \return sparse matrix \f$ elt*mat \f$
*/

//----------------------------------------------------------------------------------
template <class Tf> spmatrix<Tf> operator * (const Tf& elt, const spmatrix<Tf>& mat)
//----------------------------------------------------------------------------------
{
  assert (mat.nbRows && mat.rows_max_elements);

  spmatrix<Tf> sp;

  if (abs(elt) > epsilon)
    { mat.update ();
      sp = mat;
      for (int i=1; i<=sp.nbRows; i++)
	sp.mrows[i-1] *= elt;
    }

  return sp;
}


/*!
  \brief Product between sparse matrix and scalar.

  \param mat sparse matrix
  \param elt scalar
  \return sparse matrix \f$ mat * elt \f$
*/

//----------------------------------------------------------------------------------
template <class Tf> spmatrix<Tf> operator * (const spmatrix<Tf>& mat, const Tf& elt)
//----------------------------------------------------------------------------------
{
  return elt * mat;
}


/*!
  \brief Product between sparse matrix and vector.

  \param mat sparse matrix A
  \param v vector
  \return \f$ A * v = A_{ij} v_j \f$
*/

//--------------------------------------------------------------------------------------
template <class Tf> vector<Tf> operator * (const spmatrix<Tf>& mat, const vector<Tf>& v)
//--------------------------------------------------------------------------------------
{
  assert (mat.nbColumns == v.dim()); // checks the validity of the operation

  mat.update ();

  vector<Tf> u(mat.nbRows);
  vector<Tf> *line;
  vector<int> *positions;
  int r;

  for (int i=1; i<=mat.nbRows; i++)
    { r = mat.nbRowsElements[i];
      line = &mat.mrows[i-1];
      positions = &mat.columns_position[i];

      for (int j=1; j<=r; j++)
	u[i] += (*line)[j] * v[(*positions)[j]];
    }

  return u;
}


/*!
  \brief Product between vector and sparse matrix.

  \param v vector
  \param mat sparse matrix A
  \return \f$ v * A = A_{ij} v_i \f$
*/

//--------------------------------------------------------------------------------------
template <class Tf> vector<Tf> operator * (const vector<Tf>& v, const spmatrix<Tf>& mat)
//--------------------------------------------------------------------------------------
{
  assert(mat.nbRows == v.dim()); // checks the validity of the operation

  mat.update ();

  vector<Tf> u(mat.nbColumns);
  vector<Tf> *line;
  vector<int> *positions;
  int r;

  for (int i=1; i<=mat.nbRows; i++)
    { r = mat.nbRowsElements[i];
      line = &mat.mrows[i-1];
      positions = &mat.columns_position[i];

      for (int j=1; j<=r; j++)
	u[(*positions)[j]] += v[i] * (*line)[j];
    }

  return u;
}


/*!
  \brief Product for sparse matrices.

  \param mat1 sparse matrix A
  \param mat2 sparse matrix B
  \return \f$ A * B = A_{ik} B_{kj} \f$
*/

//-------------------------------------------------------------------------------------------
template <class Tf> matrix<Tf> operator * (const spmatrix<Tf>& mat1,const spmatrix<Tf>& mat2)
//-------------------------------------------------------------------------------------------
{
  assert(mat1.rows_max_elements && mat2.rows_max_elements);
  assert(mat1.nbColumns == mat2.nbRows);

  mat1.update ();
  mat2.update ();

  matrix<Tf> res(mat1.nbRows, mat2.nbColumns); //the result matrix

  for (int i=1; i<=mat1.nbRows; i++)
    for (int j=1; j<=mat1.nbRowsElements[i]; j++)
      { int n = mat1.columns_position(i,j);
	for (int k=1; k<=mat2.nbRowsElements[n]; k++)
	  res(i, mat2.columns_position(n,k)) += mat1.mrows[i-1][j] * mat2.mrows[n-1][k];
      }

  return res;
}


/*!
  \brief Division between sparse matrix and scalar.

  \param mat sparse matrix A
  \param elt scalar k
  \return \f$ \displaystyle A / k = \frac{A_{ij}}{k} \f$
*/

//----------------------------------------------------------------------------------
template <class Tf> spmatrix<Tf> operator / (const spmatrix<Tf>& mat, const Tf& elt)
//----------------------------------------------------------------------------------
{
  assert (abs(elt) > epsilon);

  mat.update ();

  spmatrix<Tf> sp = mat;
  sp /= elt;

  return sp;
}


/*!
  \brief 'Double inner product' for sparse matrices.

  \param mat1 sparse matrix A
  \param mat2 sparse matrix B
  \return \f$ A | B = A_{ij} B_{ij} \f$
*/

//------------------------------------------------------------------------------------
template <class Tf> Tf operator | (const spmatrix<Tf>& mat1, const spmatrix<Tf>& mat2)
//------------------------------------------------------------------------------------
{
  assert (mat1.nbRows==mat2.nbRows); // matrices must have the same size
  assert (mat1.nbRows);

  if (mat2.rows_max_elements < mat1.rows_max_elements)
    return mat2 | mat1;	//so that we work on the smallest spmatrix

  mat1.update ();
  mat2.update ();

  int r1,r2;
  Tf res = 0;

  for (int i=1; i<=mat1.nbRows; i++)
    { r1 = mat1.nbRowsElements[i];
      r2 = mat2.nbRowsElements[i];
      for (int j=1; j<=r1; j++)
	for (int k=1; k<=r2; k++)
	  if (mat1.columns_position(i,j) == mat2.columns_position(i,k))
	    res += mat1.mrows[i-1][j] * mat2.mrows[i-1][k];
    }

  return res;
}


/*!
  \brief Operator += for sparse matrices.

  \param mat sparse matrix to add
*/

//-------------------------------------------------------------------------------
template <class T> spmatrix<T>& spmatrix<T>::operator += (const spmatrix<T>& mat)
//-------------------------------------------------------------------------------
{
  assert ( (nbRows == mat.nbRows)  &&  (nbColumns == mat.nbColumns) );

  return (*this) &= (*this) + mat;	//operator&= is used because the number of non-zero elements per row may change
}


/*!
  \brief Operator -= for sparse matrices.

  \param mat sparse matrix to substract
*/

//-------------------------------------------------------------------------------
template <class T> spmatrix<T>& spmatrix<T>::operator -= (const spmatrix<T>& mat)
//-------------------------------------------------------------------------------
{
  assert ((nbRows == mat.nbRows) && (nbColumns == mat.nbColumns));

  return (*this) &= (*this) - mat;	//operator&= is used because the size of non-zero elements per row may change
}


/*!
  \brief Operator *= between sparse matrix and scalar.

 \param elt scalar
*/

//---------------------------------------------------------------------
template <class T> spmatrix<T>& spmatrix<T>::operator *= (const T& elt)
//---------------------------------------------------------------------
{
  assert (nbRows && rows_max_elements);

  if (abs(elt) < epsilon)
    { delete [] mrows;
      columns_position.~matrix();
      nbRowsElements.~vector();
      nbRows = nbColumns = rows_max_elements = 0;
    }

  else
    { update ();
      for (int i=1; i<=nbRows; i++)
	mrows[i-1] *= elt;
    }

  return *this;
}


/*!
  \brief Operator /= between sparse matrix and scalar.

  \param elt scalar
*/

//---------------------------------------------------------------------
template <class T> spmatrix<T>& spmatrix<T>::operator /= (const T& elt)
//---------------------------------------------------------------------
{
  assert (nbRows && rows_max_elements);
  assert(abs(elt) > epsilon);

  update ();

  for (int i=1; i<=nbRows; i++)
    mrows[i-1] /= elt;

  return *this;
}


/*!
  \brief Compares two sparses matrices.

  First compares sparse matrices sizes and then compares their values.

  \param mat1 sparse matrix A
  \param mat2 sparse matrix B
  \return TRUE if \f$ A = B \f$ and FALSE if \f$ A \neq B \f$
*/

//---------------------------------------------------------------------------------------
template <class Tf> bool operator == (const spmatrix<Tf>& mat1, const spmatrix<Tf>& mat2)
//---------------------------------------------------------------------------------------
{
  bool result = (mat1.nbRows==mat2.nbRows && mat1.rows_max_elements==mat2.rows_max_elements && mat1.nbColumns==mat2.nbColumns);
  int r;

  if (result)
    { mat1.update ();
      mat2.update ();

      for (int i=1; (i<=mat1.nbRows) && result; i++)
	if (mat1.nbRowsElements[i] == mat2.nbRowsElements[i])
	  { r = mat1.nbRowsElements[i];
	    for (int j=1; j<=r && result; j++)
	      if (abs (mat1.mrows[i-1][j]) < epsilon) result *= abs (mat2(i,mat1.columns_position(i,j))) < epsilon;
	      else result *= ( abs ((mat1.mrows[i-1][j] - mat2(i,mat1.columns_position(i,j))) / mat1.mrows[i-1][j]) < epsilon );
	  }
	else result = false;
    }

  return result;
}


/*!
  \brief Compares two sparse matrices.

  First compares sparse matrices sizes and then compares their values.

  \param sparse matrix A
  \param sparse matrix B
  \return TRUE if \f$ A \neq B \f$ and FALSE if \f$ A = B \f$
*/

//---------------------------------------------------------------------------------------
template <class Tf> bool operator != (const spmatrix<Tf>& mat1, const spmatrix<Tf>& mat2)
//---------------------------------------------------------------------------------------
{
  return !(mat1==mat2);
}


/*!
  \brief Overloads output stream for sparse matrices.
*/

//----------------------------------------------------------------------------
template <class Tf> ostream& operator << (ostream& s, const spmatrix<Tf>& mat)
//----------------------------------------------------------------------------
{
  assert (mat.nbRows>0 && mat.rows_max_elements>0);

  mat.update ();

  int r;

  if (mat.nbRows<=15 && mat.nbColumns<=15)
    { for (int i=1; i<=mat.nbRows; i++, s<<endl)
	for (int j=1; j<=mat.nbColumns; j++)
	  s << mat(i,j) << " ";
      s << endl;
    }

  else
    { for (int i=1; i<=mat.nbRows; i++)
	{ r = mat.nbRowsElements[i];
	  for (int j=1; j<=r; j++)
	    { int k = mat.columns_position[i][j];
	      s << i << " " << k << "  " << mat(i,k) << endl;
	    }
	}
    }

  return s;
}


/*!
  \brief Overloads input stream for sparse matrices.
*/

//-----------------------------------------------------------------------
template <class Tf> istream& operator >> (istream& is, spmatrix<Tf>& mat)
//-----------------------------------------------------------------------
{
  assert (mat.nbRows && mat.nbColumns);

  if ((mat.nbRows <= 15) && (mat.nbColumns <= 15))
    { for (int i=1; i<=mat.nbRows; i++)
	for (int j=1; j<=mat.nbColumns; j++)
	  { is >> mat.value;
	    mat.set (mat.row=i, mat.column=j);
	  }
    }

  else
    { int r, c;
      Tf val;
      for (int i=1; i<=mat.nbRows; i++)
	{ is >> r >> c >> mat.value;
	  mat.set (mat.row=r, mat.column=c);
	}
    }

  return is;
}


/*!
  \brief Transposition for sparse matrix.

  \param mat sparse matrix
  \return sparse matrix transpose
*/

//----------------------------------------------------------
template <class Tf> spmatrix<Tf> t (const spmatrix<Tf>& mat)
//----------------------------------------------------------
{

  mat.update ();

  int r;

  //number of non-zero elements per Columns
  vector<int> c(mat.nbColumns);

  for (int i=1; i<=mat.nbRows; i++)
    { r = mat.nbRowsElements[i];
      for (int j=1; j<=r; j++) 
	c[mat.columns_position(i,j)] += 1;
    }

  //number max of non-zero elements per Columns
  int maxcol = 0;
  for (int k=1; k<=mat.nbColumns; k++)
    if (maxcol < c[k]) maxcol = c[k];

  spmatrix<Tf> tmat (mat.nbColumns, mat.nbRows, maxcol);

  int nbEleMax = 0;

  for (int i=1; i<=mat.nbRows; i++)
    { r=mat.nbRowsElements[i];
      for (int j=1; j<=r; j++) 
	{ tmat.mrows[mat.columns_position(i,j)-1][tmat.nbRowsElements[mat.columns_position(i,j)]+1] = mat.mrows[i-1][j];
	  tmat.columns_position(mat.columns_position(i,j),tmat.nbRowsElements[mat.columns_position(i,j)]+1) = i;
	  tmat.nbRowsElements[mat.columns_position(i,j)]+=1;
	}
    }

  return tmat;
}


/*!
  \brief Absolute value function for sparse matrix.

  \param mat sparse matrix
  \return sparse matrix with the absolute value of all the elements of the original sparse matrix
*/

//------------------------------------------------------------
template <class Tf> spmatrix<Tf> abs (const spmatrix<Tf>& mat)
//------------------------------------------------------------
{
  assert (mat.nbColumns);

  mat.update (); //updating the buffer

  spmatrix<Tf> mtx = mat;

  for (int i=1; i<=mat.nbRows; i++)
    mtx.mrows[i-1] =& abs(mat.mrows[i-1]);

  //  mtx.row = mtx.column = mtx.vectrow = 0;

  return mtx;
}


/*!
  \brief Computes the trace of a square sparse matrix.

  \param mat square sparse matrix M
  \return sparse matrix trace \f$ ( \underset{i}{\sum} M_{ii} ) \f$

*/

//-------------------------------------------------
template <class Tf> Tf tr (const spmatrix<Tf>& mat)
//-------------------------------------------------
{
  assert (mat.nbRows);
  assert (mat.nbRows==mat.nbColumns); // matrix must be squared

  mat.update (); //updating the buffer

  int r;
  Tf tr = 0;

  for (int i=1; i<=mat.nbRows; i++)
    { r = mat.nbRowsElements[i];
      for (int j=1; j<=r; j++)
	if (i == mat.columns_position(i,j))
	  { tr += mat.mrows[i-1][j];
	    j = r;
	  }
    }

  return tr;
}


/*!
  \brief Gives the greatest element of a sparse matrix.
*/

//--------------------------------------------------
template <class Tf> Tf max (const spmatrix<Tf>& mat)
//--------------------------------------------------
{
  assert(mat.nbRows && mat.nbColumns);

  mat.update (); //updating the buffer

  int r;
  vector<Tf> *line;
  Tf max = mat.mrows[0][1];

  for (int i=1; i<=mat.nbRows; i++)
    { r = mat.nbRowsElements[i];
      line = &mat.mrows[i-1];
      for (int j=1; j<=r; j++)
	if (max < (*line)[j])  max =(*line)[j];
    }

  return max;
}


/*!
  \brief Returns the diagonal of a square sparse matrix.

  Let \f$ M \f$ be a square sparse matrix. We define the vector \f$ \mathbf{v} \f$ through \f$ M \f$ diagonal elements such as \n
  \f$ \mathbf{v} =
      \begin{pmatrix}
        M_{11} \\
	M_{22} \\
	\vdots \\
	M_{nn} \\
      \end{pmatrix}
  \f$

  \param mat sparse matrix \f$ M \f$
  \return \f$ \mathbf{v} \f$
*/

//---------------------------------------------------------------
template <class Tf> vector<Tf> diagonal (const spmatrix<Tf>& mat)
//---------------------------------------------------------------
{
  assert (mat.nbRows);
  assert (mat.Rows()==mat.Columns());

  mat.update (); //updating the buffer

  int r;
  vector<Tf> d (mat.nbRows);

  for (int i=1; i<=mat.nbRows; i++)
    { r = mat.nbRowsElements[i];
      for (int j=1; j<=r; j++)
	if (i == mat.columns_position(i,j))
	  { d[i] = mat.mrows[i-1][j];
	    j = r;
	  }
    }

  return d;
}


/*!
  \brief Returns the Identity matrix.

  \param n size of the identity matrix
  \return sparse identity matrix \f$ I_{n,n} \f$
*/

//--------------------------------------------
template <class Tf> spmatrix<Tf> Id2sp (int n)
//--------------------------------------------
{
  assert (n > 0);
  spmatrix<Tf> I(n,n,1);

  I.value = 1;
  for (int i=1; i<=n; i++)
    I.set (I.row=i, I.column=i);

  return I;
}


/*!
  \brief Writes sparse matrix to a file.

  \param path path and filename
*/

//----------------------------------------------------------
template <class T> void spmatrix<T>::save (const char* path)
//----------------------------------------------------------
{
  ofstream file(path, ios::out);
  assert (!file.fail());
  file << *this;
  file.close();
}


/*!
  \brief Writes sparse matrix to a file.

  \param filename path and filename
*/

//----------------------------------------------------------------
template <class T> void spmatrix<T>::save (const string& filename)
//----------------------------------------------------------------
{
  const char *path = filename.c_str();
  save (path);
}


/*!
  \brief Reads sparse matrix from a file.

  \param path path and filename
*/

//----------------------------------------------------------
template <class T> void spmatrix<T>::read (const char* path)
//----------------------------------------------------------
{
  ifstream file(path, ios::in);
  assert (!file.fail());
  file >> *this;
  file.close();
}


/*!
  \brief Reads sparse matrix from a file.

  \param filename path and filename
*/

//----------------------------------------------------------------
template <class T> void spmatrix<T>::read (const string& filename)
//----------------------------------------------------------------
{
  const char *path = filename.c_str();
  read (path);
}


}


#endif
