;; morph-mode for emacs
;; a mode for creating morphologically forced connections
;; version 0.5 (initial release)
;; 
;; Copyright (c) 2010 by Andrew Fabbro 
;; (andrew@fabbro.org or andrew.fabbro@gmail.com)
;;
;;      This program is free software: you can redistribute it and/or modify
;;      it under the terms of the GNU General Public License as published by
;;      the Free Software Foundation, either version 3 of the License, or
;;      (at your option) any later version.
;;
;;      This program is distributed in the hope that it will be useful,
;;      but WITHOUT ANY WARRANTY; without even the implied warranty of
;;      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;;      GNU General Public License for more details.
;;
;;      You should have received a copy of the GNU General Public License
;;      along with this program.  If not, see <http://www.gnu.org/licenses/>.
;;
;; INTRODUCTION
;;
;; Morph mode allows you to make morphologically forced connections.
;; MFCs are sentences or phrases in which various words are randomly
;; chosen from lists you've defined.  It's primarily used as a 
;; brainstorming tool.
;;
;; Let's say you're trying to think up some ideas for a fantasy world
;; you want to write a novel about.  You create the following lists:
;;
;; government: despotic, democratic, anarchic, plutocratic
;; environment: tropical, desert, rainforest, ocean, polar
;; leader trait: heroic, paranoid, fiendish, weak-willed
;; mood of the people: slavish, rebellious, hedonistic, distracted
;;  
;; Now you create a pattern like this:
;;
;; A @government@ government in a @environment@ region where the \
;; @leader trait@ ruler leads a nation of @mood of the people@ people.
;;
;; When you create a morph, it replaces each list-name (betwen the @
;; signs) with a randomly chosen word from each list.  For example:
;;
;; "A despotic government in a polar region where the fiendish
;;  ruler leads a nation of hedonistic people."
;; "A democratic government in a tropical region where the paranoid
;;  ruler leads a nation of rebellious people."
;; "A plutocratic government in a desert region where the heroic
;;  ruler leads a nation of distracted people."
;;
;; Of course, you can use MFCs for anything: new invention ideas, 
;; names of offspring, website names, new religions, etc.
;;
;; Reference: O'Reilly's _Mind Performance Hacks_ 
;;
;; HOW TO USE THIS MODE
;;
;; First, load the package with load-file if you have not done
;; so ("M-x load-file /path/to/morph-mode.el")
;;
;; Second, switch to morph mode ("M-x morph-mode")
;;
;; Third, create your morph file.  You will need to define two things:
;;    lists
;;    morph patterns
;; 
;; Then press "Control-J" (or M-x morph-gen) and your morphs will 
;; appear in a separate buffer (*Morph Output*).
;;
;; FORMAT
;;   
;; Lines containing '@' are considered patterns and lines containing
;; ':' are considered lists.  You should consider '@' and ':' special
;; characters for the purpose of morph-mode.
;;
;; Any line not having a '@' or ':' is ignored.
;; 
;; LISTS
;;
;;   Lists are in the form:
;;
;;       list name : element,element,element,...
;;
;;   The label always comes first.  Preceding/trailing whitespace is trimmed.
;;   Elements are comma-delimited.  Preceding/trailing whitespace is trimmed.
;;   Lists cannot span multiple lines.
;; 
;;   Examples:
;;      text editors: emacs, jove, pico, nano
;;      cities:Paris,Berlin,New York
;;      colors : red , blue , yellow , pink,orange , green , white
;;
;;   Notes on lists:
;;      - lists do not check for uniqueness.  Duplicating an item 
;;        simply gives it a higher chance of being chosen.
;;      - if you have more than one list with the same name, the
;;        later list will replace the earlier one.
;;
;; MORPH PATTERNS
;;
;;   Morph patterns are in the form:
;;      A @foo@ consisting of @bar@ with a @baz@
;;
;;   Morph patterns cannot span multiple lines.
;;
;;   In patterns, list references always begin and end with an @ sign.  
;;   This allows for the use of white space and other characters in the 
;;   list name.
;;
;; OUTPUT
;;
;;   To "run the morph," press Control-J while in morph-mode.  The 
;;   interpreter will ask you how many morphs you want (how many 
;;   combinations of your morph pattern).  If you have more than one 
;;   pattern, you'll get that many morphs for each pattern.
;;
;;   The interpreter will output all your morphs in a new buffer 
;;   (*Morph Output*) 
;;
;; ---------------------------------------------------------------------
;; END OF DOCUMENTATION
;; ---------------------------------------------------------------------

;; ---------------------------------------------------------------------
;; USE CL
;;  needed for dolist
;; ---------------------------------------------------------------------

(require 'cl)

;; ---------------------------------------------------------------------
;; VARIABLES
;; ---------------------------------------------------------------------

;; "global to this package" local variables
;; morph-line - the current line being parsed
(make-local-variable 'morph-line)
;; lists - hash for the lists
(make-local-variable 'lists)
;; patterns - hash for the patterns
(make-local-variable 'patterns)
;; pattern counter.  since patterns don't have names like
;; lists, we store them by number in the patterns hash
(make-local-variable 'pattern-counter)
;; how many morphs to generate for each pattern (default is 1)
(make-local-variable 'num-morphs-wanted)

;; ---------------------------------------------------------------------
;; KEYMAP
;; ---------------------------------------------------------------------
;; define local keymap - just adds Control-J to run the generator
;; (like lisp mode's Control-J to evaluate)

(defvar morph-mode-map nil)

;; ---------------------------------------------------------------------
;; MODE SETUP
;; ---------------------------------------------------------------------

(defun morph-mode()
  (interactive)
  (kill-all-local-variables)
  (setq major-mode 'morph-mode)
  (setq mode-name "Morph Mode")
  (use-local-map morph-mode-map)  
  (if morph-mode-map nil
    ( setq morph-mode-map (make-sparse-keymap))
    ( define-key morph-mode-map "\C-j" 'morph-gen)
  )
)

;; -------------------------------------------------------------------
;; UTILITY FUNCTIONS
;; -------------------------------------------------------------------

;; -------------------------------------------------------------------
;;   all-keys(): evaluates to all keys in a huash
;;     example: (all-keys myhash)
;; -------------------------------------------------------------------

(defun all-keys (hashtable)
  "Return all keys in hashtable."
  (let (allkeys)
    (maphash (lambda (kk vv) (setq allkeys (cons kk allkeys))) hashtable)
    allkeys
  )
)

;; -------------------------------------------------------------------
;;   random-element(): return a random element from a list
;;     example: (random-element mylist)
;; -------------------------------------------------------------------

(defun random-element (list)
  "Return a random element from a list"
  (nth (random (length list)) list)
)

;; -------------------------------------------------------------------
;;   trim-whitespace(): trims whitespace (thank you emacswiki)
;;     example: (setq test-string (trim-whitespace test-string))
;; -------------------------------------------------------------------

(defun trim-whitespace (str)
  "trims leading and tailing whitespace from string"
  (let ((s (if (symbolp str) (symbol-name str) str)))
    (replace-regexp-in-string "\\(^[[:space:]\n]*\\|[[:space:]\n]*$\\)" "" s)))

;; -------------------------------------------------------------------
;; HANDLERS FOR PARSING
;;   These are called from the main parsing loop in morph-gen
;; -------------------------------------------------------------------

;; handle-pattern
(defun handle-pattern () 
  (setq pattern-counter (+ 1 pattern-counter))
  (puthash (number-to-string pattern-counter) (trim-whitespace morph-line) patterns)
)

;; handle-list
(defun handle-list () 
  (setq list-name (trim-whitespace (replace-regexp-in-string ":.*$" "" morph-line)))
  (setq word-line (trim-whitespace (replace-regexp-in-string "^.*:" "" morph-line)))
  (setq word-list ( split-string word-line ","))
  (puthash list-name word-list lists)
)

;; -------------------------------------------------------------------
;; MAIN PARSING/MORPH GENERATOR
;; -------------------------------------------------------------------

;;evaluation function
(defun morph-gen ( )
  "Generate morphs"
  (interactive)

  ;; find out how many morphs the user wants
  (setq num-morphs-wanted (string-to-number (read-from-minibuffer "Number of morphs to make? ")))

  ;; go to beginning of buffer
  (goto-char (point-min))

  ;; when parsing, do we have more lines to process?  parsing
  ;; stops when this is zero
  (make-local-variable 'have-more-lines)
  (setq have-more-lines t)

  ;; seed the RNG
  (setq morph-line (random t))

  ;; reset all the vars
  (setq morph-line nil)
  (setq patterns (make-hash-table :test 'equal))
  (setq lists (make-hash-table :test 'equal))
  (setq pattern-counter 0)

  (while have-more-lines
    ;; get current line
    (setq morph-line (buffer-substring-no-properties (line-beginning-position) (line-end-position)))

    (if ( string-match ":" morph-line ) 
      (handle-list)
    )
    (if ( string-match "@" morph-line ) 
      (handle-pattern)
    )

    (setq have-more-lines (= 0 (forward-line 1)))
  )

  ;; switch to the output buffer
  (pop-to-buffer "*Morph Output*" nil)
  (erase-buffer)
  (goto-char (point-min))

  ;; check for errors
  ;; do we have at least one pattern?
  (if ( < (length (all-keys patterns)) 1)
      (error ("ERROR: no patterns defined.  Aborting output."))
  )

  ;; do we have at least one list?
  (if ( < (length (all-keys lists)) 1)
      (error ("ERROR: no lists defined.  Aborting output."))
  )

  ;; don't worry about lists we don't have (@color@ in the pattern but no 
  ;; "color" list).  those will show up in the output and be obvious enough.

  ;; for each pattern...
  (dolist (pattern (reverse (all-keys patterns)))
    (setq morphs-counter num-morphs-wanted)
    (while (> morphs-counter 0)
      (setq ptext (gethash pattern patterns))
      (dolist (l (all-keys lists))
	(setq sm (format "@%s@" l))
	(if ( string-match sm ptext )
	  (progn
	  (setq re (trim-whitespace (random-element (gethash l lists))))
	  (setq ptext (replace-regexp-in-string sm re ptext))
	  )
	)
      )
      (insert ptext "\n")
      (setq morphs-counter (1- morphs-counter))
      
    )
    (insert "\n")
  )
)
    

