//  Copyright (C) 2011 Ben Asselstine
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 3 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU Library General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 
//  02110-1301, USA.

#include <config.h>

#include <glibmm.h>
#include "main.h"
#include "mouse-graph.h"
#include "timing.h"
#include "ucompose.hpp"

MouseGraph* MouseGraph::create()
{
  return new MouseGraph();
}
MouseGraph::MouseGraph()
{
}
void MouseGraph::start_polling()
{
  int x, y;
  Gdk::ModifierType mask;
  pixmap = Gdk::Pixmap::create
    (Gdk::Window::get_default_root_window(),
     Gdk::Display::get_default()->get_default_screen()->get_width(),
     Gdk::Display::get_default()->get_default_screen()->get_height() );
  cairo = pixmap->create_cairo_context();
  cairo->set_source_rgb (1.0, 1.0, 1.0);
  cairo->rectangle(0, 0,
                   Gdk::Display::get_default()->get_default_screen()->get_width(),
                   Gdk::Display::get_default()->get_default_screen()->get_height() );
  cairo->paint();
  Gdk::Window::get_default_root_window()->get_pointer(x, y, mask);
  cairo->move_to(x, y);
  cairo->set_source_rgb (0, 0, 0);
  cairo->set_line_width(1.00);
  poll_timer = Timing::instance().register_timer
    (sigc::mem_fun(this, &MouseGraph::poll), 50);
  draw_timer = Timing::instance().register_timer
    (sigc::mem_fun(this, &MouseGraph::draw_points), 60000);
}

void MouseGraph::stop_polling()
{
  int w= 0, h= 0;
  draw_points();
  cairo.clear();
  pixmap->get_size(w, h);
  graphed.emit(Gdk::Pixbuf::create(Glib::RefPtr<Gdk::Drawable>(pixmap), 0, 0, w, h));
  poll_timer.disconnect();
  draw_timer.disconnect();
}

int MouseGraph::is_button_pressed(Gdk::ModifierType &mask)
{
  if (mask & Gdk::BUTTON1_MASK ||
      mask & Gdk::BUTTON2_MASK ||
      mask & Gdk::BUTTON2_MASK ||
      mask & Gdk::BUTTON3_MASK ||
      mask & Gdk::BUTTON4_MASK ||
      mask & Gdk::BUTTON5_MASK)
    return 1;
  return 0;
}

bool MouseGraph::draw_points()
{
  if (points.empty() == true)
    return Timing::CONTINUE;
  std::list<struct mousepoint> p = points;
  points.clear();
  for (std::list<struct mousepoint>::iterator i = p.begin(); i != p.end(); i++)
    draw_point(&*i);
  cairo->stroke();
  cairo->move_to(p.back().x, p.back().y);

  return Timing::CONTINUE;
}

void MouseGraph::draw_line(int x1, int y1)
{
  cairo->line_to(x1, y1);
}

void MouseGraph::draw_circle(int x1, int y1)
{
  cairo->stroke();
  cairo->save();
  int fill = 0;
  if (rand() % 10 == 0)
    {
      fill = 1;
      cairo->arc(x1, y1, (rand() %100) + 4 , 0, 2.0 * M_PI);
    }
  else if (rand() % 20 == 0)
      cairo->arc(x1, y1, (rand() %1000) + 4 , 0, 2.0 * M_PI);
  else
      cairo->arc(x1, y1, (rand() %10) + 4 , 0, 2.0 * M_PI);
  if (fill)
    {
      if (rand() % 2 == 0)
        cairo->set_source_rgba(0.0, 0.0, 0.0, 0.05);
      else
        cairo->set_source_rgba(1.0, 1.0, 1.0, 0.05);
      cairo->fill_preserve();
    }
  cairo->restore();
  cairo->stroke();
  cairo->set_source_rgba(0.0, 0.0, 0.0, 1.0);
}

void MouseGraph::draw_point(struct mousepoint *point)
{
  return draw (point->x, point->y, point->mask);
}

void MouseGraph::draw (int x1, int y1, Gdk::ModifierType &mask)
{
  prev_x = x1;
  prev_y = y1;
  prev_mask =mask;
  draw_line(x1, y1);
  int button_pressed = is_button_pressed(mask);
  if (button_pressed != prev_button_pressed)
    draw_circle(x1, y1);
  prev_button_pressed = button_pressed;
}

bool MouseGraph::poll()
{
  int x, y;
  Gdk::ModifierType mask;
  Gdk::Window::get_default_root_window()->get_pointer(x, y, mask);
  if (x == prev_x && y == prev_y && mask == prev_mask)
    return Timing::CONTINUE;
  struct mousepoint point;
  point.x = x; point.y = y; point.mask = mask;
  points.push_back(point);
  //draw(x, y, mask);
  return Timing::CONTINUE;
}

MouseGraph::~MouseGraph()
{
}

void MouseGraph::toggle_polling()
{
  if (poll_timer.connected())
    stop_polling();
  else
    start_polling();
}

bool MouseGraph::is_graphing()
{
  if (poll_timer.connected())
    return true;
  else
    return false;
}

void MouseGraph::graph_it()
{
  int w= 0, h= 0;
  draw_points();
  pixmap->get_size(w, h);
  graphed.emit(Gdk::Pixbuf::create(Glib::RefPtr<Gdk::Drawable>(pixmap), 0, 0, w, h));
}

