//  Copyright (C) 2011, 2014 Ben Asselstine
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 3 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU Library General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 
//  02110-1301, USA.

#include <config.h>
#include <stdbool.h>
#include <memory>
#include <stdlib.h>
#include <iostream>
#include <assert.h>

#include "main.h"
#include <gtkmm.h>
#include <gtkmm/main.h>
#include "timing.h"
#include "ucompose.hpp"
#include "mouse-graph.h"
#include "mousegraph-window.h"

struct Main::Impl
{
    MouseGraph *mouse_graph;
    std::auto_ptr<Gtk::Main> gtk_main;
    MouseGraphWindow *window;
    std::list<std::string> temporary_files_to_delete;
    sigc::connection on_timer_registered(Timing::timer_slot s,
					 int msecs_interval);
};


static Main *singleton;

Main::Main(int &argc, char **&argv)
    : impl(new Impl)
{
    version_appears_on_the_command_line = false;
    singleton = this;

    Glib::set_prgname("Mouse Graph");
    Glib::thread_init();
    try
      {
        Glib::OptionContext context("");
        Glib::OptionEntry entry;
        Glib::OptionGroup options (PACKAGE_NAME, "Mouse Graph Options", "Command-line options for Mouse Graph");
        entry.set_long_name ("version");
        entry.set_short_name ('V');
        entry.set_description ("Show version and exit.");
        options.add_entry (entry, version_appears_on_the_command_line);
        context.add_group(options);
        context.set_summary("This program records mouse activity in picture form.");

        try
          {
            context.parse(argc, argv);
          }
        catch(const Glib::Error& ex)
          {
            std::cout << ex.what() << std::endl;
            std::cout << "Try `" PACKAGE_NAME 
              " --help' for more information." << std::endl;
            exit(0);
          }

        if (version_appears_on_the_command_line)
          {
            std::cout << PACKAGE_NAME << " " << PACKAGE_VERSION << std::endl;
            std::cout << 
              "This is free software: " <<
              "you are free to change and redistribute it." << std::endl;
            std::cout << 
              "There is NO WARRANTY, to the extent permitted by law." << 
              std::endl;
            exit(0);
          }

	Timing::instance().timer_registered.connect(
	    sigc::mem_fun(*impl, &Main::Impl::on_timer_registered));

	impl->gtk_main.reset(new Gtk::Main(argc, argv, context));

    }
    catch (const Glib::Error &ex) {
	std::cerr << ex.what() << std::endl;
    }
}

Main::~Main()
{
    delete impl;
    singleton = 0;
}

Main &Main::instance()
{
    assert(singleton != 0);
    return *singleton;
}

bool Main::iterate_main_loop()
{
    try
    {
	impl->gtk_main->iteration(false);
    }
    catch (const Glib::Error &ex) {
	std::cerr << ex.what() << std::endl;
    }

    return true;
}

void Main::start_main_loop()
{
    try
      {
        impl->window = new MouseGraphWindow();
        impl->mouse_graph = MouseGraph::create();
        impl->mouse_graph->graphed.connect(sigc::mem_fun(*this, &Main::on_mouse_graphed));

        impl->gtk_main->run(impl->window->get_window());
      }
    catch (const Glib::Error &ex) {
	std::cerr << ex.what() << std::endl;
    }
}
        
void Main::on_mouse_graphed(Cairo::RefPtr<Cairo::ImageSurface> surface)
{
  std::string temp;
  int fd = Glib::file_open_tmp (temp);
  close (fd);
  Glib::RefPtr<Gdk::Pixbuf> pixbuf = Gdk::Pixbuf::create(surface, 0, 0, surface->get_width(), surface->get_height());

  pixbuf->save(temp, "png");
  impl->temporary_files_to_delete.push_back(temp);
  Glib::ustring cmd = Glib::find_program_in_path("xdg-open") + " '" + temp +"'";
  Glib::spawn_command_line_async(cmd);
}

void Main::remove_all_temporary_files()
{
  for (std::list<std::string>::iterator i = impl->temporary_files_to_delete.begin(); i != impl->temporary_files_to_delete.end(); i++)
    remove ((*i).c_str());
}


void Main::stop_main_loop()
{
    try
    {
	impl->gtk_main->quit();
        delete impl->window;
        remove_all_temporary_files();
    }
    catch (const Glib::Error &ex) {
	std::cerr << ex.what() << std::endl;
    }
}

std::string Main::get_data_path()
{
  return MOUSE_GRAPH_DATADIR;
}

sigc::connection Main::Impl::on_timer_registered(Timing::timer_slot s,
						 int msecs_interval)
{
    return Glib::signal_timeout().connect(s, msecs_interval);
}

void Main::toggle_mouse_graphing()
{
  impl->mouse_graph->toggle_polling();
}
    
bool Main::is_graphing()
{
  return impl->mouse_graph->is_graphing();
}

void Main::graph_it()
{
  return impl->mouse_graph->graph_it();
}
