// -*- mode: c++; indent-tabs-mode: nil; c-basic-offset: 4 -*-
// $Header: /home/pgavin/cvsroot/mpak/include/mpak/build/environment.hh,v 1.4 2004/07/01 22:45:37 pgavin Exp $
// mpak - the advanced package manager
// Copyright (C) 2003 Peter Gavin
// 
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

#ifndef __MPAK__BUILTINS__ENVIRONMENT_HH__
#define __MPAK__BUILTINS__ENVIRONMENT_HH__

#include <mpak/spec/fwd.hh>
#include <mpak/builtins/fwd.hh>

#include <boost/shared_ptr.hpp>

#include <ostream>
#include <iostream>

namespace mpak
{
    namespace build
    {
        /** \class environment
         *  \brief Encapsulates the environment for executing a build.
         *  
         *  The \c environment class encapsulates all necessary auxiliary
         *  data for executing a build.
         */ 
        class environment
        {
        private:
            const boost::shared_ptr<const builtins::config_node> config_root_;
            const boost::shared_ptr<spec::context> config_context_;
            const boost::shared_ptr<const builtins::category_node> package_tree_root_;
            const boost::shared_ptr<spec::context> package_tree_context_;
            const boost::shared_ptr<builtins::category_node> installed_tree_root_;
            const boost::shared_ptr<spec::context> installed_tree_context_;
            boost::shared_ptr<std::ostream> msg_stream_;
            
        public:
            /** \brief constructs an \c environment object
             *  \arg \c config_root the root \c node object for this
             *          <tt>environment</tt>'s configuration
             *  \arg \c config_context the \c context object used to
             *          build \c config_root
             *  \arg \c package_tree_root the root \c node object for
             *          this <tt>environment</tt>'s tree of tree of packages
             *  \arg \c package_tree_context the \c context object
             *          used to build \c package_tree_root
             *  \arg \c installed_tree_root the root \c node object
             *          for this <tt>environment</tt>'s tree of installed
             *          packages
             *  \arg \c installed_tree_context the \c context object
             *          used to build \c installed_tree_root
             */
            environment (const boost::shared_ptr<const builtins::config_node> &config_root,
                         const boost::shared_ptr<spec::context> &config_context,
                         const boost::shared_ptr<const builtins::category_node> &package_tree_root,
                         const boost::shared_ptr<spec::context> &package_tree_context,
                         const boost::shared_ptr<builtins::category_node> &installed_tree_root,
                         const boost::shared_ptr<spec::context> &installed_tree_context)
                : config_root_ (config_root),
                  config_context_ (config_context),
                  package_tree_root_ (package_tree_root),
                  package_tree_context_ (package_tree_context),
                  installed_tree_root_ (installed_tree_root),
                  installed_tree_context_ (installed_tree_context),
                  msg_stream_ (new std::ostream (std::cerr.rdbuf ()))
            {
            }
            
            /** \brief returns the root \c node for this
             *         <tt>environment</tt>'s configuration
             *  \return the root \c node for this
             *          <tt>environment</tt>'s configuration
             */
            const boost::shared_ptr<const builtins::config_node> &
            get_config_root (void)
                const
            {
                return this->config_root_;
            }
            
            /** \brief returns the \c context object used to build
             *         this <tt>environment</tt>'s configuration
             *  \return returns the \c context object used to build
             *          this <tt>environment</tt>'s configuration
             */
            const boost::shared_ptr<spec::context> &
            get_config_context (void)
                const
            {
                return this->config_context_;
            }
            
            /** \brief returns the root \c node for this
             *         <tt>environment</tt>'s tree of packages
             *  \return the root \c node for this
             *          <tt>environment</tt>'s tree of packages
             */
            const boost::shared_ptr<const builtins::category_node> &
            get_package_tree_root (void)
                const
            {
                return this->package_tree_root_;
            }
            
            /** \brief returns the \c context object used to build
             *         this <tt>environment</tt>'s tree of packages
             *  \return the \c context object used to build this
             *          <tt>environment</tt>'s tree of packages
             */
            const boost::shared_ptr<spec::context> &
            get_package_tree_context (void)
                const
            {
                return this->package_tree_context_;
            }
            
            /** \brief returns the root \c node for the
             *         <tt>environment</tt>'s tree of installed
             *         packages
             *  \return the root \c node for the
             *          <tt>environment</tt>'s tree of installed packages
             */
            const boost::shared_ptr<builtins::category_node> &
            get_installed_tree_root (void)
                const
            {
                return this->installed_tree_root_;
            }
            
            /** \brief returns the \c context object used to build
             *         this <tt>environment</tt>'s tree of installed
             *         packages
             *  \return the \c context object used to build this
             *          <tt>environment</tt>'s tree of installed packages
             */
            const boost::shared_ptr<spec::context> &
            get_installed_tree_context (void)
                const
            {
                return this->installed_tree_context_;
            }
            
            const boost::shared_ptr<std::ostream> &
            get_msg_stream (void)
                const
            {
                return this->msg_stream_;
            }
            
            void
            set_msg_stream (const boost::shared_ptr<std::ostream> &msg_stream)
            {
                this->msg_stream_ = msg_stream;
            }
        };
    }
}

#endif /* ifndef __MPAK__BUILTINS__ENVIRONMENT_HH__ */
