// -*- mode: c++; indent-tabs-mode: nil; c-basic-offset: 4 -*-
// $Header: /home/pgavin/cvsroot/mpak/include/mpak/builtins/sources_node_data.hh,v 1.4 2004/07/01 22:45:37 pgavin Exp $
// mpak - the advanced package manager
// Copyright (C) 2003 Peter Gavin
// 
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

#ifndef __MPAK__BUILTINS__SOURCES_NODE_DATA_HH__
#define __MPAK__BUILTINS__SOURCES_NODE_DATA_HH__

#include <mpak/defs.hh>
#include <mpak/spec/node_data.hh>
#include <mpak/util/checksummer.hh>

#include <boost/weak_ptr.hpp>

#include <map>
#include <vector>

namespace mpak
{
    namespace builtins
    {
        class sources_node_data
            : public spec::node_data
        {
            typedef std::multimap<std::string, std::string> source_map_type_; // key is filename, value is url or @node_path
            source_map_type_ source_map_;
            
        public:
            typedef std::pair<util::checksummer::algorithm, util::checksummer::checksum_type> checksum_pair_type;
            
        private:
            typedef std::map<std::string, checksum_pair_type> checksum_map_type_;
            checksum_map_type_ checksum_map_;
            
        public:
            sources_node_data (void)
                : spec::node_data (),
                  source_map_ (),
                  checksum_map_ ()
            {
            }
            
        protected:
            sources_node_data (const sources_node_data &that)
                : spec::node_data (that),
                  source_map_ (that.source_map_),
                  checksum_map_ (that.checksum_map_)
            {
            }
            
        public:
            virtual
            ~sources_node_data (void);
            
            virtual sources_node_data *
            clone (void)
                const;
            
            virtual void
            merge (const boost::shared_ptr<node_data> &that_node_data)
                const;
            
            void add_source (const std::string &filename, const std::string &location)
            {
                this->source_map_.insert (source_map_type_::value_type (filename, location));
            }
            
            const std::string &
            get_source (const std::string &filename)
                const
            {
                source_map_type_::const_iterator i (this->source_map_.find (filename));
                if (i == this->source_map_.end ())
                    throw failure ("could not find location for source file " + filename);
                return i->second;
            }
            
            typedef source_map_type_::value_type sources_value_type;
            typedef source_map_type_::const_iterator sources_iterator;
            
            sources_iterator
            begin_sources (void)
                const
            {
                return this->source_map_.begin ();
            }
            
            sources_iterator
            end_sources (void)
                const
            {
                return this->source_map_.end ();
            }
            
            bool
            has_checksum (const std::string &filename)
                const
            {
                return this->checksum_map_.find (filename) != this->checksum_map_.end ();
            }
            
            const checksum_pair_type &
            get_checksum (const std::string &filename)
                const
            {
                checksum_map_type_::const_iterator i (this->checksum_map_.find (filename));
                if (i == this->checksum_map_.end ())
                    throw failure ("could not find checksum for source file " + filename);
                return i->second;
            }
            
            void add_checksum (const std::string &filename, const checksum_pair_type &checksum_pair)
            {
                this->checksum_map_.insert (checksum_map_type_::value_type (filename, checksum_pair));
            }
            
            typedef checksum_map_type_::const_iterator checksum_iterator;
            
            checksum_iterator
            begin_checksums (void)
                const
            {
                return this->checksum_map_.begin ();
            }
            
            checksum_iterator
            end_checksums (void)
                const
            {
                return this->checksum_map_.end ();
            }            
        };
    }
}

#endif // ifndef __MPAK__BUILTINS__SOURCES_NODE_DATA_HH__
