// -*- mode: c++; indent-tabs-mode: nil; c-basic-offset: 4 -*-
// $Header: /home/pgavin/cvsroot/mpak/include/mpak/spec/command.hh,v 1.2 2004/07/01 22:45:37 pgavin Exp $
// mpak - the advanced package manager
// Copyright (C) 2003 Peter Gavin
// 
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

#ifndef __MPAK__SPEC__COMMAND_HH__
#define __MPAK__SPEC__COMMAND_HH__

#include <mpak/defs.hh>
#include <mpak/spec/fwd.hh>
#include <mpak/spec/command_info.hh>

#include <boost/function.hpp>
#include <boost/shared_ptr.hpp>
#include <boost/optional.hpp>

#include <stdexcept>
#include <string>
#include <set>

namespace mpak
{
    namespace spec
    {
        typedef void init_func (spec::context &context);
        typedef boost::function<boost::optional<std::string> (const boost::shared_ptr<spec::node> &,
                                                              const boost::shared_ptr<const spec::argument_vector> &,
                                                              spec::context &) > command;
        typedef boost::function<boost::shared_ptr<command_info_list> (const boost::shared_ptr<const spec::node> &) > node_command_gen;
        typedef boost::function<boost::shared_ptr<command_info_list> (const boost::shared_ptr<const spec::node_data> &) > node_data_command_gen;
        
        class bad_argument
            : public std::runtime_error
        {
        public:
            inline bad_argument (const std::string &what)
                : runtime_error (what)
            {
            }
        };
        
        class bad_command
            : public std::runtime_error
        {
        public:
            inline bad_command (const std::string &what)
                : runtime_error (what)
            {
            }
        };
        
        template<typename node_type_>
        class create_node_command
        {
        private:
            const std::string node_type_name_;
            std::set<std::string> parent_types_;
            const bool allow_serialized_init_;
            
        public:
            create_node_command (const std::string &node_type_name, bool allow_serialized_init = true)
                : node_type_name_ (node_type_name),
                  parent_types_ (),
                  allow_serialized_init_ (allow_serialized_init)
            {
            }
            
            boost::optional<std::string>
            operator () (const boost::shared_ptr<spec::node> &node,
                         const boost::shared_ptr<const spec::argument_vector> &arguments,
                         spec::context &context)
                const;
            
            void add_parent_type (const std::string &parent_type)
            {
                this->parent_types_.insert (parent_type);
            }
        };
        
        template<typename node_type_>
        boost::optional<std::string>
        create_node_command<node_type_>::
        operator () (const boost::shared_ptr<spec::node> &node,
                     const boost::shared_ptr<const spec::argument_vector> &arguments,
                     spec::context &context)
            const
        {
            if (!arguments ||
                (arguments->size () == 0) ||
                (arguments->size () > 2))
                throw spec::bad_argument ("incorrect number of arguments passed to category call");
            
            if (this->parent_types_.find (node->get_type ()) == this->parent_types_.end ()) {
                throw spec::bad_command (this->node_type_name_ + " cannot be added to node of type " + node->get_type ());
            }
            
            boost::shared_ptr<spec::command_info_list> command_infos;
            bool serialized_init (this->allow_serialized_init_);
            if (arguments->size () == 2) {
                // block given
                try {
                    command_infos = boost::get<boost::shared_ptr<spec::command_info_list> > (arguments->at (1));
                } catch (boost::bad_get &e) {
                    throw spec::bad_argument ("incorrect argument type passed to category call");
                } catch (...) {
                    throw;
                }
                serialized_init = false;
            } // else load info from file
            
            const std::string node_name (*context.reduce_argument (arguments->at (0), node));
            boost::shared_ptr<spec::node> child_node (new node_type_ (this->node_type_name_,
                                                                      node_name,
                                                                      context,
                                                                      serialized_init));
            node->add_child (child_node);
            
            if (command_infos) {
                context.execute_command_list (command_infos, child_node);
            }
            
            return boost::optional<std::string> ();
        }
        
        boost::shared_ptr<spec::command_info_list>
        default_node_command_gen (const boost::shared_ptr<const spec::node> &node);
    }
}

#endif // ifndef __MPAK__SPEC__COMMAND_HH__
