// -*- mode: c++; indent-tabs-mode: nil; c-basic-offset: 4 -*-
// $Header: /home/pgavin/cvsroot/mpak/include/mpak/util/cow_vector.hh,v 1.1 2004/07/07 01:54:57 pgavin Exp $
// mpak - the advanced package manager
// Copyright (C) 2003 Peter Gavin
// 
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

#ifndef __MPAK__UTIL_COW_SET_HH__
#define __MPAK__UTIL_COW_SET_HH__

#include <boost/shared_ptr.hpp>

#include <set>
#include <new>

namespace mpak
{
    namespace util
    {
        /** \class cow_set
         *  \brief Implements a copy-on-write set.
         *  
         *  Note if !cow_set->unique () then all non-const
         *  operations will invalidate iterators. In addition, all
         *  non-const operations automatically execute
         *  cow_set->make_unique (), so 2 or more successive
         *  non-const operations (with no copies in between) are safe,
         *  provided there are no race conditions. Thus, you can
         *  safely do cow_set->begin () immediately followed by
         *  cow_set->end ().
         *
         *  This could probably be made more efficient by not
         *  implementing it in terms of std::set, but I'm lazy.
         */
        template<typename value_type_,
                 typename compare_type_ = typename std::set<value_type_>::key_compare,
                 typename allocator_type_ = typename std::set<value_type_>::allocator_type>
        class cow_set
        {
        private:
            typedef std::set<value_type_, compare_type_, allocator_type_> set_type_;
            boost::shared_ptr<set_type_> set_;
            
        public:
            typedef typename set_type_::key_type key_type;
            typedef typename set_type_::value_type value_type;
            typedef typename set_type_::size_type size_type;
            typedef typename set_type_::reference reference;
            typedef typename set_type_::const_reference const_reference;
            typedef typename set_type_::pointer pointer;
            typedef typename set_type_::const_pointer const_pointer;
            typedef typename set_type_::iterator iterator;
            typedef typename set_type_::const_iterator const_iterator;
            typedef typename set_type_::reverse_iterator reverse_iterator;
            typedef typename set_type_::const_reverse_iterator const_reverse_iterator;
            typedef typename set_type_::allocator_type allocator_type;
            typedef typename set_type_::key_compare key_compare;
            typedef typename set_type_::value_compare value_compare;
            
            cow_set (void)
                : set_ (new set_type_)
            {
            }
            
            explicit cow_set (const compare_type_ &compare, const allocator_type_ &allocator = allocator_type_ ())
                : set_ (new set_type_ (compare, allocator))
            {
            }
            
            cow_set (const cow_set &that)
                : set_ (that.set_)
            {
            }
            
            template<typename input_iterator_type_>
            cow_set (const input_iterator_type_ &first, const input_iterator_type_ &last,
                     const compare_type_ &compare, const allocator_type_ &allocator = allocator_type_ ())
                : set_ (new set_type_ (first, last, compare))
            {
            }
            
            ~cow_set (void)
            {
            }
            
            key_compare
            key_comp (void)
                const
            {
                return this->get_set_ ().key_comp ();
            }
            
            value_compare
            value_comp (void)
                const
            {
                return this->get_set_ ().value_comp ();
            }
            
            allocator_type
            get_allocator (void)
                const
            {
                return this->get_set_ ().get_allocator ();
            }
            
            template<typename input_iterator_type_>
            void
            assign (const input_iterator_type_ &first, const input_iterator_type_ &last)
            {
                cow_set new_cow_set (first, last);
                this->swap (new_cow_set);
            }
            
            void
            assign (size_type size, const value_type &value)
            {
                cow_set new_cow_set (size, value);
                this->swap (new_cow_set);
            }
            
            iterator begin (void)
            {
                return this->get_set_ ().begin ();
            }
            
            iterator end (void)
            {
                return this->get_set_ ().end ();
            }
            
            const_iterator begin (void)
                const
            {
                return this->get_set_ ().begin ();
            }
            
            const_iterator end (void)
                const
            {
                return this->get_set_ ().end ();
            }
            
            reverse_iterator rbegin (void)
            {
                return this->get_set_ ().rbegin ();
            }
            
            reverse_iterator rend (void)
            {
                return this->get_set_ ().rend ();
            }
            
            const_reverse_iterator rbegin (void)
                const
            {
                return this->get_set_ ().rbegin ();
            }
            
            const_reverse_iterator rend (void)
                const
            {
                return this->get_set_ ().rend ();
            }
            
            bool
            empty (void)
                const
            {
                return this->get_set_ ().empty ();
            }
            
            size_type
            size (void)
                const
            {
                return this->get_set_ ().size ();
            }
            
            size_type
            max_size (void)
                const
            {
                return this->get_set_ ().max_size ();
            }
            
            void swap (cow_set &that)
            {
                this->get_set_ ().swap (that.get_set_ ());
            }
            
            void clear (void)
            {
                cow_set new_nodes;
                this->swap (new_nodes);
            }
            
            std::pair<iterator, bool>
            insert (const value_type &value)
            {
                return this->get_set_ ().insert (value);
            }
            
            iterator
            insert (const iterator &position, const value_type &value)
            {
                return this->get_set_ ().insert (position, value);
            }
            
            template<typename input_iterator_type_>
            iterator
            insert (input_iterator_type_ first, input_iterator_type_ last)
            {
                return this->get_set_ ().insert (position, value);
            }
            
            void
            erase (iterator position)
            {
                return this->get_set_ ().erase (position);
            }
            
            size_type
            erase (const key_type &key)
            {
                return this->get_set_ ().erase (key);
            }
            
            void erase (iterator first, iterator last)
            {
                return this->get_set_ ().erase (first, last);
            }
            
            size_type
            count (const key_type &key)
                const
            {
                return this->get_set_ ().count (key);
            }
            
            iterator
            find (const key_type &key)
            {
                return this->get_set_ ().find (key);
            }
            
            const_iterator
            find (const key_type &key)
                const
            {
                return this->get_set_ ().find (key);
            }
            
            iterator
            lower_bound (const key_type &key)
            {
                return this->get_set_ ().lower_bound (key);
            }            
            
            const_iterator
            lower_bound (const key_type &key)
                const
            {
                return this->get_set_ ().lower_bound (key);
            }            
            
            iterator
            upper_bound (const key_type &key)
            {
                return this->get_set_ ().upper_bound (key);
            }            
            
            const_iterator
            upper_bound (const key_type &key)
                const
            {
                return this->get_set_ ().upper_bound (key);
            }            
            
            std::pair<iterator, iterator>
            equal_range (const key_type &key)
            {
                return this->get_set_ ().equal_range (key);
            }
            
            std::pair<const_iterator, const_iterator>
            equal_range (const key_type &key)
                const
            {
                return this->get_set_ ().equal_range (key);
            }
            
            void
            unique (void)
            const
            {
                return this->set_.unique ();
            }
            
            void
            make_unique (void)
            {
                if (!this->set_.unique ()) {
                    boost::shared_ptr<set_type_> new_nodes (new set_type_(*this->set_));
                    this->set_.swap (new_nodes);
                }
            }
            
            cow_set &
            operator= (const cow_set &that)
            {
                cow_set new_cow_set (that);
                this->swap (new_cow_set);
                return *this;
            }
            
        private:
            const set_type_ &
            get_set_ (void)
                const
            {
                return *this->set_;
            }
            
            set_type_ &
            get_set_ (void)
            {
                this->make_unique ();
                return *this->set_;
            }
        };
    }
}

#endif // ifndef __MPAK__UTIL_COW_SET_HH__
