// -*- mode: c++; indent-tabs-mode: nil; c-basic-offset: 4 -*-
// $Header: /home/pgavin/cvsroot/mpak/libmpak/mpak/util/node_path.cc,v 1.3 2004/07/01 22:45:37 pgavin Exp $
// mpak - the advanced package manager
// Copyright (C) 2003 Peter Gavin
// 
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

#include <config.h>

#include <mpak/util/node_path.hh>
#include <mpak/util/node_path_grammar.hh>
#include <mpak/spec/node.hh>

#include <boost/spirit.hpp>
#include <boost/spirit/phoenix.hpp>
#include <boost/shared_ptr.hpp>

#include <string>
#include <sstream>

namespace mpak
{
    namespace util
    {
        void
        node_path::
        parse_string (const std::string &node_path_string)
        {
            if (node_path_string.empty ()) {
                this->clear ();
                return;
            }
            
            node_path path;
            node_path_grammar node_path_g;
            
            BOOST_SPIRIT_DEBUG_NODE(node_path_g);
            
            boost::spirit::parse_info<std::string::const_iterator> info =
                boost::spirit::parse (node_path_string.begin (),
                                      node_path_string.end (),
                                      node_path_g[phoenix::var (path) = phoenix::arg1]);
            
            if (!info.full) {
                throw failure ("invalid node_path string: " + node_path_string);
            }
            
            this->swap (path);
        }
        
        const std::string
        node_path::
        get_string (void)
            const
        {
            if (this->elements_.empty ())
                return std::string ();
            
            std::ostringstream oss;
            
            element_vector_::const_iterator i (this->elements_.begin ());
            goto skip_first;
            for (; i != this->elements_.end (); ++i) {
                oss << ':';
            skip_first:
                oss << *i;
            }
            
            return oss.str ();
        }
        
        const boost::shared_ptr<const spec::node>
        node_path::
        match (const boost::shared_ptr<const spec::node> &root_node,
               const std::string &node_type)
            const
        {
            if (!this->elements_.empty ())
                return this->match_helper_ (root_node, 0, node_type);
            else {
                if (root_node->get_type () == node_type) {
                    return root_node;
                } else {
                    return boost::shared_ptr<const spec::node> ();
                }
            }
        }
        
        const boost::shared_ptr<const spec::node>
        node_path::
        match_helper_ (const boost::shared_ptr<const spec::node> &parent_node,
                       const node_path::element_vector_::size_type element_index,
                       const std::string &node_type)
            const
        {
            if (element_index < (this->elements_.size () - 1)) {
                for (spec::node::child_const_iterator child_node (parent_node->begin_children ());
                     child_node != parent_node->end_children (); ++child_node) {
                    if ((*child_node)->get_name () == this->elements_[element_index]) {
                        return this->match_helper_ (*child_node, element_index + 1, node_type);
                    }
                }
            } else if (element_index == (this->elements_.size () - 1)) {
                // last element
                if (parent_node->has_child (node_type, this->elements_[element_index])) {
                    return parent_node->get_child (node_type, this->elements_[element_index]);
                }
            }
            return boost::shared_ptr<const spec::node> ();
        }
        
        const boost::shared_ptr<spec::node>
        node_path::
        match_nc (const boost::shared_ptr<spec::node> &root_node,
               const std::string &node_type)
            const
        {
            if (!this->elements_.empty ())
                return this->match_helper_nc_ (root_node, 0, node_type);
            else {
                if (root_node->get_type () == node_type) {
                    return root_node;
                } else {
                    return boost::shared_ptr<spec::node> ();
                }
            }
        }
        
        const boost::shared_ptr<spec::node>
        node_path::
        match_helper_nc_ (const boost::shared_ptr<spec::node> &parent_node,
                       const node_path::element_vector_::size_type element_index,
                       const std::string &node_type)
            const
        {
            if (element_index < (this->elements_.size () - 1)) {
                for (spec::node::child_iterator child_node (parent_node->begin_children ());
                     child_node != parent_node->end_children (); ++child_node) {
                    if ((*child_node)->get_name () == this->elements_[element_index]) {
                        return this->match_helper_nc_ (*child_node, element_index + 1, node_type);
                    }
                }
            } else if (element_index == (this->elements_.size () - 1)) {
                // last element
                if (parent_node->has_child (node_type, this->elements_[element_index])) {
                    return parent_node->get_child (node_type, this->elements_[element_index]);
                }
            }
            return boost::shared_ptr<spec::node> ();
        }
        
        void
        node_path::
        read_node (const boost::shared_ptr<const spec::node> &node)
        {
            assert (node);
            node_path new_node_path;
            new_node_path.read_node_helper_ (node);
            this->swap (new_node_path);
        }
        
        void
        node_path::
        read_node_helper_ (const boost::shared_ptr<const spec::node> &node)
        {
            const boost::optional<const boost::weak_ptr<const spec::node> > weak_parent (node->get_parent ());
            if (weak_parent) {
                const boost::shared_ptr<const spec::node> parent (*weak_parent);
                this->read_node_helper_ (parent);
                this->elements_.push_back (node->get_name ());
            }
        }
    }
}
