// -*- mode: c++; indent-tabs-mode: nil; c-basic-offset: 4 -*-
// $Header: /home/pgavin/cvsroot/mpak/libmpak/mpak/util/version_spec.cc,v 1.3 2004/06/03 12:45:37 pgavin Exp $
// mpak - the advanced package manager
// Copyright (C) 2003 Peter Gavin
// 
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

#include <config.h>

#include <mpak/util/version_spec.hh>
#include <mpak/util/version_spec_grammar.hh>

#include <boost/spirit.hpp>
#include <boost/spirit/phoenix.hpp>

#include <algorithm>
#include <sstream>
#include <map>
#include <cstdlib>
#include <cctype>

namespace mpak
{
    namespace util
    {
        bool
        version_spec::
        valid (void)
            const
        {
            if (this->version_numbers_.empty ())
                return false;
            for (version_number_vector::const_iterator i (this->version_numbers_.begin ()); i != this->version_numbers_.end (); ++i) {
                for (std::string::const_iterator j (i->begin ()); j != i->end (); ++j) {
                    if (!isdigit (*j))
                        return false;
                }
            }
            return true;
        }
        
        void
        version_spec::
        parse_string (const std::string &version_string)
        {
            using phoenix::var;
            if (version_string.empty ()) {
                this->clear ();
                return;
            }
            version_spec spec;
            detail::version_spec_grammar version_spec_g;
            
            BOOST_SPIRIT_DEBUG_NODE (version_spec_g);
            
            boost::spirit::parse_info<std::string::const_iterator> info =
                boost::spirit::parse (version_string.begin (),
                                      version_string.end (),
                                      version_spec_g[phoenix::var (spec) = phoenix::arg1]);
            
            if (!info.full) {
                throw failure (std::string ("invalid version string: ") + version_string);
            }
            
            this->swap (spec);
        }
        
        const std::string
        version_spec::
        get_string (void)
            const
        {
            if (!this->valid ()) {
                throw failure ("invalid version_spec");
            }
            
            std::ostringstream oss;
            
            version_number_vector::const_iterator i (this->version_numbers_.begin ());
            goto skip_first;
            for (; i != this->version_numbers_.end (); ++i) {
                oss << '.';
            skip_first:
                oss << *i;
            }
            
            if (this->version_letter_) {
                oss << this->version_letter_;
            }
            
            boost::optional<std::string> release_type_string (this->get_release_type_as_string ());
            if (release_type_string || this->release_number_) {
                oss << '_';
                if (release_type_string)
                    oss << *release_type_string;
                if (this->release_number_) {
                    oss << *this->release_number_;
                }
            }
            
            if (this->revision_number_) {
                oss << '-' << *this->revision_number_;
            }
            return oss.str ();
        }
        
        const std::string
        version_spec::
        get_string_no_revision (void)
            const
        {
            if (!this->valid ()) {
                throw failure ("invalid version_spec");
            }
            
            std::ostringstream oss;
            
            version_number_vector::const_iterator i (this->version_numbers_.begin ());
            goto skip_first;
            for (; i != this->version_numbers_.end (); ++i) {
                oss << '.';
            skip_first:
                oss << *i;
            }
            
            if (this->version_letter_) {
                oss << this->version_letter_;
            }
            
            boost::optional<std::string> release_type_string (this->get_release_type_as_string ());
            if (release_type_string || this->release_number_) {
                oss << '_';
                if (release_type_string)
                    oss << *release_type_string;
                if (this->release_number_) {
                    oss << *this->release_number_;
                }
            }
            
            return oss.str ();
        }
        
        bool
        version_spec::
        operator< (const version_spec &that)
            const
        {
            assert (this->valid ());
            assert (that.valid ());
            const version_number_vector::size_type this_size (this->version_numbers_.size ());
            const version_number_vector::size_type that_size (that.version_numbers_.size ());
            version_number_vector::size_type i = 0;
            for (i = 0;
                 i < std::min (this_size, that_size);
                 i++) {
                unsigned this_val (std::strtoul (this->version_numbers_[i].c_str (), 0, 10));
                unsigned that_val (std::strtoul (that.version_numbers_[i].c_str (), 0, 10));
                if (this_val < that_val)
                    return true;
                if (this_val > that_val)
                    return false;
                // if we have more leading zeros in this than in that,
                // we treat this as being a lower version. e.g. 005 is
                // lower than 05 (like in wire gauges)
                if (this->version_numbers_[i].size () > that.version_numbers_[i].size ())
                    return true;
                if (this->version_numbers_[i].size () < that.version_numbers_[i].size ())
                    return false;
            }
            if (this_size > that_size) {
                // 1.0.0 is treated as being a higher version than 1.0
                return true;
            } else if (that_size > this_size) {
                return false;
            }
            
            if (this->version_letter_ < that.version_letter_)
                return true;
            if (this->version_letter_ > that.version_letter_)
                return false;
            
            if ((this->release_type_ ? *this->release_type_ : 0) <
                (that.release_type_ ? *that.release_type_ : 0))
                return true;
            if ((this->release_type_ ? *this->release_type_ : 0) >
                (that.release_type_ ? *that.release_type_ : 0))
                return false;
            
            if ((this->release_number_ ? *this->release_number_ : 0) <
                (that.release_number_ ? *that.release_number_ : 0))
                return true;
            if ((this->release_number_ ? *this->release_number_ : 0) >
                (that.release_number_ ? *that.release_number_ : 0))
                return false;
            
            if ((this->revision_number_ ? *this->revision_number_ : 0) <
                (that.revision_number_ ? *that.revision_number_ : 0))
                return true;
            if ((this->revision_number_ ? *this->revision_number_ : 0) >
                (that.revision_number_ ? *that.revision_number_ : 0))
                return false;
            
            return false;
        }
        
        bool
        version_spec::
        operator== (const version_spec &that)
            const
        {
            const version_number_vector::size_type this_size (this->version_numbers_.size ());
            const version_number_vector::size_type that_size (that.version_numbers_.size ());
            version_number_vector::size_type i;
            if (this_size != that_size)
                return false;
            for (i = 0;
                 i != this_size;
                 i++) {
                if (this->version_numbers_[i] != that.version_numbers_[i])
                    return false;
            }
            if (this->version_letter_ != that.version_letter_) {
                return false;
            }
            
            if ((this->release_type_ ? *this->release_type_ : 0) !=
                (that.release_type_ ? *that.release_type_ : 0))
                return false;
            
            if ((this->release_number_ ? *this->release_number_ : 0) !=
                (that.release_number_ ? *that.release_number_ : 0))
                return false;
            
            if ((this->revision_number_ ? *this->revision_number_ : 0) !=
                (that.revision_number_ ? *that.revision_number_ : 0))
                return false;
            
            return true;
        }
        
        namespace {
            typedef std::map<std::string, version_spec::release_types> release_type_map_by_string_;
            typedef std::map<version_spec::release_types, std::string> release_type_map_by_int_;
            
            static const release_type_map_by_string_ &
            get_release_types_by_string_ (void)
            {
                static release_type_map_by_string_ release_types_by_string;
                static bool init = true;
                
                if (init) {
                    release_types_by_string.insert (release_type_map_by_string_::value_type ("pl",    version_spec::release_pl));
                    release_types_by_string.insert (release_type_map_by_string_::value_type ("",      version_spec::release_normal));
                    release_types_by_string.insert (release_type_map_by_string_::value_type ("rc",    version_spec::release_rc));
                    release_types_by_string.insert (release_type_map_by_string_::value_type ("pre",   version_spec::release_pre));
                    release_types_by_string.insert (release_type_map_by_string_::value_type ("test",  version_spec::release_test));
                    release_types_by_string.insert (release_type_map_by_string_::value_type ("beta",  version_spec::release_beta));
                    release_types_by_string.insert (release_type_map_by_string_::value_type ("alpha", version_spec::release_alpha));
                    init = false;
                }
                
                return release_types_by_string;
            }
            
            static const release_type_map_by_int_ &
            get_release_types_by_int_ (void)
            {
                static release_type_map_by_int_ release_types_by_int;
                static bool init = true;
                
                if (init) {
                    for (std::map<std::string, version_spec::release_types>::const_iterator i = get_release_types_by_string_ ().begin ();
                         i != get_release_types_by_string_ ().end (); i++) {
                        release_types_by_int.insert (std::map<version_spec::release_types, std::string>::value_type (i->second, i->first));
                    }
                    init = false;
                }
                
                return release_types_by_int;
            }
            
        }
        
        const boost::optional<std::string>
        version_spec::
        get_release_type_as_string (void)
            const
        {
            if (this->release_type_) {
                release_type_map_by_int_::const_iterator i (get_release_types_by_int_ ().find (*this->release_type_));
                return i->second;
            } else {
                return boost::optional<std::string> ();
            }
        }
        
        void
        version_spec::
        set_release_type (const boost::optional<std::string> &release_type)
        {
            if (!release_type) {
                this->release_type_ = boost::optional<release_types> ();
                return;
            }
            release_type_map_by_string_::const_iterator i (get_release_types_by_string_ ().find (*release_type));
            if (i != get_release_types_by_string_ ().end ())
                this->release_type_ = i->second;
            else
                throw failure ("invalid release type string");
        }
        
        void
        version_spec::
        set_release_type (const boost::optional<release_types> &release_type)
        {
            if (!release_type ||
                (get_release_types_by_int_ ().find (*release_type) != get_release_types_by_int_ ().end ()))
                this->release_type_ = release_type;
            else
                throw failure ("invalid release type integer");
        }
    }
}
