// -*- mode: c++; indent-tabs-mode: nil; c-basic-offset: 4 -*-
// $Header: /home/pgavin/cvsroot/mpak/mpak/mpak.cc,v 1.4 2004/07/06 17:20:31 pgavin Exp $ */
// mpak - the advanced package manager
// Copyright (C) 2003 Peter Gavin
// 
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

#include <config.h>

#include <mpak/spec/context.hh>
#include <mpak/spec/node.hh>
#include <mpak/spec/pickler.hh>
#include <mpak/spec/file_pickler.hh>
#include <mpak/spec/tree_pickler.hh>
#include <mpak/spec/database_pickler.hh>
#include <mpak/builtins/init.hh>
#include <mpak/builtins/config_node.hh>
#include <mpak/builtins/category_node.hh>
#include <mpak/builtins/version_node.hh>
#include <mpak/build/dependency_calc.hh>
#include <mpak/build/environment.hh>
#include <mpak/util/dependency.hh>
#include <mpak/util/node_path.hh>
#include <mpak/util/node_path_list.hh>

#include <boost/shared_ptr.hpp>
#include <boost/optional.hpp>
#include <boost/filesystem/path.hpp>
#include <boost/filesystem/operations.hpp>
#include <boost/filesystem/convenience.hpp>
#include <boost/filesystem/exception.hpp>

#include <iostream>
#include <string>
#include <cstdlib>
#include <cstring>
#include <exception>
#include <new>

#include <getopt.h>

namespace
{
    static bool verbose (false);
    static const std::string mpak_config_dir (SYSCONFDIR "/mpak");
    static bool config_file_set (false);
    static boost::optional<std::string> config_file_string;
    static boost::optional<std::string> arg_root_dir_string (std::string ("/"));
    
    int do_build (int argc, char *argv[]);
    int do_install (int argc, char *argv[]);
    int do_print (int argc, char *argv[]);
    int do_query (int argc, char *argv[]);
    int do_init (int argc, char *argv[]);
}

int main (int argc, char *argv[])
{
    try {
        boost::filesystem::path::default_name_check (boost::filesystem::native);
        (void) boost::filesystem::initial_path ();
        
        const char shortopts[] = "+vc:r:";
        const struct option longopts[] = {
            { "verbose", no_argument, 0, 'v' },
            { "config", required_argument, 0, 'c' },
            { "root", required_argument, 0, 'r' },
            { 0 }
        };
        
        while (1) {
            int c (getopt_long (argc, argv, shortopts, longopts, 0));
            
            if (c == -1) // no more global options
                break;
            
            switch (c) {
            case 'v':
                std::cerr << "running verbosely" << std::endl;
                verbose = true;
                break;
            case 'c':
                try {
                    config_file_set = true;
                    config_file_string = optarg;
                } catch (boost::filesystem::filesystem_error &e) {
                    std::cerr << "error setting config file path: " << e.what () << std::endl;
                    exit (EXIT_FAILURE);
                }
                break;
            case 'r':
                try {
                    arg_root_dir_string = std::string (optarg);
                } catch (boost::filesystem::filesystem_error &e) {
                    std::cerr << "error setting root path: " << e.what () << std::endl;
                    exit (EXIT_FAILURE);
                }
                {
                    if (arg_root_dir_string) {
                        boost::filesystem::path root_dir (*arg_root_dir_string);
                        if (!root_dir.is_complete ()) {
                            std::cerr << "root " << root_dir.native_directory_string () << " is not absolute" << std::endl;
                            exit (EXIT_FAILURE);
                        }
                        if (!exists (root_dir)) {
                            std::cerr << "root " << root_dir.native_directory_string () << " does not exist" << std::endl;
                            exit (EXIT_FAILURE);
                        }
                        if (!is_directory (root_dir)) {                    
                            std::cerr << "root " << root_dir.native_directory_string () << " is not a directory" << std::endl;
                            exit (EXIT_FAILURE);
                        }
                    }
                }
                break;
            default:
                std::cerr << "error parsing options" << std::endl;
                exit (EXIT_FAILURE);
            }
        }
        
        if (!config_file_string) {
            config_file_string = (boost::filesystem::path (*arg_root_dir_string) / mpak_config_dir / "mpak.conf").native_file_string ();
        }
        
        if (optind == argc) {
            std::cerr << "no command given" << std::endl;
        } else {
            if (verbose)
                std::cerr << "command: " << argv[optind] << std::endl;
            if (!std::strcmp (argv[optind], "build")) {
                ++optind;
                return do_build (argc, argv);
            } else if (!std::strcmp (argv[optind], "install")) {
                ++optind;
                return do_install (argc, argv);
            } else if (!std::strcmp (argv[optind], "print")) {
                ++optind;
                return do_print (argc, argv);
            } else if (!std::strcmp (argv[optind], "query")) {
                ++optind;
                return do_query (argc, argv);
            } else if (!std::strcmp (argv[optind], "init")) {
                ++optind;
                return do_init (argc, argv);
            } else {
                std::cerr << "invalid command: " << argv[optind] << std::endl;
                exit (EXIT_FAILURE);
            }
        }
    } catch (std::exception &e) {
        std::cerr << "caught an unhandled exception: " << e.what () << std::endl;
        std::exit (EXIT_FAILURE);
    } catch (...) {
        std::cerr << "caught an unhandled unknown exception" << std::endl;
        std::exit (EXIT_FAILURE);
    }
}

namespace
{
    int do_build (int argc, char *argv[])
    {
        std::string stage ("merge");
    
        const char shortopts[] = "+s:";
        const struct option longopts[] = {
            { "stage", required_argument, 0, 's' },
            { 0 }
        };
    
        while (1) {
            int c (getopt_long (argc, argv, shortopts, longopts, 0));
        
            if (c == -1) // no more options
                break;
        
            switch (c) {
            case 's':
                stage = optarg;
                break;
            default:
                std::cerr << "error parsing options" << std::endl;
                exit (EXIT_FAILURE);
            }
        }
        
        std::string node_path_string;
        if (optind == argc) {
            std::cerr << "no node_path given" << std::endl;
            exit (EXIT_FAILURE);
        } else {
            node_path_string = argv[optind];
            ++optind;
            if (optind != argc) {
                std::cerr << "too many arguments" << std::endl;
                exit (EXIT_FAILURE);
            }
        }
        
        // initialize configuration
        boost::shared_ptr<mpak::spec::pickler> config_pickler (new mpak::spec::file_pickler (*config_file_string));
        
        boost::shared_ptr<mpak::spec::context> config_context (new mpak::spec::context (config_pickler));
        mpak::builtins::init_config_context (*config_context);
        
        boost::shared_ptr<mpak::builtins::config_node> config_root (new mpak::builtins::config_node ("mpak:config", "config_root", *config_context));
        
        config_root->set_root_dir ("/");
        config_root->set_package_tree_dir (mpak_config_dir + "/packages");
        config_root->set_statedb_dir (LOCALSTATEDIR "/db/mpak");
        config_root->set_mpaktmp_dir ("/tmp/mpak");
        config_root->set_source_dir (LOCALSTATEDIR "/cache/mpak/sources");
        
        config_context->execute_command_list (config_pickler->read (), config_root);
        
        if (arg_root_dir_string) {
            config_root->set_root_dir (*arg_root_dir_string);
        }
        
        // initialize package tree structures
        boost::shared_ptr<mpak::spec::pickler> package_tree_pickler (new mpak::spec::tree_pickler (config_root->get_root_dir () / config_root->get_package_tree_dir ()));
        boost::shared_ptr<mpak::spec::context> package_tree_context (new mpak::spec::context (package_tree_pickler));
        mpak::builtins::init_package_tree_context (*package_tree_context, config_root);
        
        boost::shared_ptr<mpak::builtins::category_node> package_tree_root (new mpak::builtins::category_node ("mpak:category", "package_tree_root",
                                                                                                               *package_tree_context));
        package_tree_context->execute_command_list (package_tree_pickler->read (), package_tree_root);
        
        // initialize installed tree structures
        
        // TODO: make this use the root we're installing to, not the current one
        boost::shared_ptr<mpak::spec::pickler> installed_tree_pickler (new mpak::spec::database_pickler (config_root->get_root_dir () / config_root->get_statedb_dir () / "install.db"));
        boost::shared_ptr<mpak::spec::context> installed_tree_context (new mpak::spec::context (installed_tree_pickler));
        mpak::builtins::init_installed_tree_context (*installed_tree_context);
        
        boost::shared_ptr<mpak::builtins::category_node> installed_tree_root (new mpak::builtins::category_node ("mpak:category", "installed_tree_root",
                                                                                                                 *installed_tree_context));
        installed_tree_context->execute_command_list (installed_tree_pickler->read (), installed_tree_root);
        
        // environment
        mpak::build::environment env (config_root,
                                      config_context,
                                      package_tree_root,
                                      package_tree_context,
                                      installed_tree_root,
                                      installed_tree_context);
        
        // build manager
        mpak::build::manager manager;
        mpak::builtins::init_build_manager (manager);
        
        mpak::util::node_path node_path (node_path_string);
        manager.execute_stage (stage, node_path, env);
        
        return EXIT_SUCCESS;
    }
    
    int do_install (int argc, char *argv[])
    {
        const char shortopts[] = "+p";
        const struct option longopts[] = {
            { "print", no_argument, 0, 'p' },
            { 0 }
        };
        
        bool print (false);
        
        while (1) {
            int c (getopt_long (argc, argv, shortopts, longopts, 0));
        
            if (c == -1) // no more options
                break;
        
            switch (c) {
            case 'p':
                print = true;
                break;
            default:
                std::cerr << "error parsing options" << std::endl;
                exit (EXIT_FAILURE);
            }
        }
        
        mpak::build::dependency_calc depcalc;
        
        if (optind == argc) {
            std::cerr << "no dependencies given" << std::endl;
            exit (EXIT_FAILURE);
        } else {
            while (optind != argc) {
                std::string dep_string (argv[optind]);
                mpak::util::dependency dep (dep_string);
                depcalc.add_dependency (dep);
                optind++;
            }
        }
        
        // initialize configuration
        boost::shared_ptr<mpak::spec::pickler> config_pickler (new mpak::spec::file_pickler (*config_file_string));
        
        boost::shared_ptr<mpak::spec::context> config_context (new mpak::spec::context (config_pickler));
        mpak::builtins::init_config_context (*config_context);
        
        boost::shared_ptr<mpak::builtins::config_node> config_root (new mpak::builtins::config_node ("mpak:config", "config_root", *config_context));
        
        config_root->set_root_dir ("/");
        config_root->set_package_tree_dir (mpak_config_dir + "/packages");
        config_root->set_statedb_dir (LOCALSTATEDIR "/db/mpak");
        config_root->set_mpaktmp_dir ("/tmp/mpak");
        config_root->set_source_dir (LOCALSTATEDIR "/cache/mpak/sources");
        
        config_context->execute_command_list (config_pickler->read (), config_root);
        
        if (arg_root_dir_string) {
            config_root->set_root_dir (*arg_root_dir_string);
        }
        
        // initialize package tree structures
        boost::shared_ptr<mpak::spec::pickler> package_tree_pickler (new mpak::spec::tree_pickler (config_root->get_root_dir () / config_root->get_package_tree_dir ()));
        boost::shared_ptr<mpak::spec::context> package_tree_context (new mpak::spec::context (package_tree_pickler));
        mpak::builtins::init_package_tree_context (*package_tree_context, config_root);
        
        boost::shared_ptr<mpak::builtins::category_node> package_tree_root (new mpak::builtins::category_node ("mpak:category", "package_tree_root",
                                                                                                               *package_tree_context));
        package_tree_context->execute_command_list (package_tree_pickler->read (), package_tree_root);
        
        // initialize installed tree structures
        
        // TODO: make this use the root we're installing to, not the current one
        boost::shared_ptr<mpak::spec::pickler> installed_tree_pickler (new mpak::spec::database_pickler (config_root->get_root_dir () / config_root->get_statedb_dir () / "install.db"));
        boost::shared_ptr<mpak::spec::context> installed_tree_context (new mpak::spec::context (installed_tree_pickler));
        mpak::builtins::init_installed_tree_context (*installed_tree_context);
        
        boost::shared_ptr<mpak::builtins::category_node> installed_tree_root (new mpak::builtins::category_node ("mpak:category", "installed_tree_root",
                                                                                                                 *installed_tree_context));
        installed_tree_context->execute_command_list (installed_tree_pickler->read (), installed_tree_root);
        
        // environment
        mpak::build::environment env (config_root,
                                      config_context,
                                      package_tree_root,
                                      package_tree_context,
                                      installed_tree_root,
                                      installed_tree_context);
        
        mpak::util::node_path_list result (depcalc.calculate (env));
        
        if (print) {
            for (mpak::util::node_path_list::const_iterator i (result.begin ()); i != result.end (); ++i) {
                std::cout << i->get_string () << std::endl;
            }
        } else {
            mpak::build::manager manager;
            mpak::builtins::init_build_manager (manager);
            
            for (mpak::util::node_path_list::const_iterator i (result.begin ()); i != result.end (); ++i) {
                manager.execute_stage ("clean", *i, env);
                manager.execute_stage ("merge", *i, env);
                manager.execute_stage ("clean", *i, env);
            }
        }
        
        return EXIT_SUCCESS;
    }
    
    void print_tree (const boost::shared_ptr<const mpak::spec::node> &node, std::size_t indent)
    {
        for (mpak::spec::node::child_const_iterator i (node->begin_children ()); i != node->end_children (); ++i) {
            std::cout << std::string (indent, '\t') << (*i)->get_type () << ' ' << (*i)->get_name ();
            if ((*i)->begin_children () == (*i)->end_children ()) {
                std::cout << std::endl;
            } else {
                std::cout << " {" << std::endl;
                print_tree ((*i), indent + 1);
                std::cout << std::string (indent, '\t') << '}' << std::endl;
            }
        }
    }
    
    int do_print (int argc, char *argv[])
    {
        const char shortopts[] = "+pi";
        const struct option longopts[] = {
            { "package", no_argument, 0, 'p' },
            { "installed", no_argument, 0, 'i' },
            { 0 }
        };
        
        enum {
            print_package_tree,
            print_installed_tree,
        } tree_to_print;
        
        tree_to_print = print_package_tree;
        
        while (1) {
            int c (getopt_long (argc, argv, shortopts, longopts, 0));
            
            if (c == -1) // no more options
                break;
        
            switch (c) {
            case 'p':
                tree_to_print = print_package_tree;
                break;
            case 'i':
                tree_to_print = print_installed_tree;
                break;
            default:
                std::cerr << "error parsing options" << std::endl;
                exit (EXIT_FAILURE);
            }
        }
        
        if (optind != argc) {
            std::cerr << "too many arguments" << std::endl;
            exit (EXIT_FAILURE);
        }
        
        // initialize configuration
        boost::shared_ptr<mpak::spec::pickler> config_pickler (new mpak::spec::file_pickler (*config_file_string));
        
        boost::shared_ptr<mpak::spec::context> config_context (new mpak::spec::context (config_pickler));
        mpak::builtins::init_config_context (*config_context);
        
        boost::shared_ptr<mpak::builtins::config_node> config_root (new mpak::builtins::config_node ("mpak:config", "config_root", *config_context));
        
        config_root->set_root_dir ("/");
        config_root->set_package_tree_dir (mpak_config_dir + "/packages");
        config_root->set_statedb_dir (LOCALSTATEDIR "/db/mpak");
        config_root->set_mpaktmp_dir ("/tmp/mpak");
        config_root->set_source_dir (LOCALSTATEDIR "/cache/mpak/sources");
        
        config_context->execute_command_list (config_pickler->read (), config_root);
        
        if (arg_root_dir_string) {
            config_root->set_root_dir (*arg_root_dir_string);
        }
        
        // initialize package tree structures
        boost::shared_ptr<mpak::spec::node> root_node;
        boost::shared_ptr<mpak::spec::pickler> pickler;
        boost::shared_ptr<mpak::spec::context> context;
        
        std::string root_name;
        
        switch (tree_to_print) {
        case print_package_tree:
            {
                pickler.reset (new mpak::spec::tree_pickler (config_root->get_root_dir () / config_root->get_package_tree_dir ()));
                context.reset (new mpak::spec::context (pickler));
                mpak::builtins::init_package_tree_context (*context, config_root);
                root_name = "package_tree_root";
            }
            break;
        case print_installed_tree:
            {
                pickler.reset (new mpak::spec::database_pickler (config_root->get_root_dir () / config_root->get_statedb_dir () / "install.db"));
                context.reset (new mpak::spec::context (pickler));
                mpak::builtins::init_installed_tree_context (*context);
                root_name = "installed_tree_root";
            }
            break;
        };
        
        root_node.reset (new mpak::builtins::category_node ("mpak:category", root_name,
                                                            *context));
        context->execute_command_list (pickler->read (), root_node);
        print_tree (root_node, 0);
        
        return EXIT_SUCCESS;
    }
    
    void print_command_infos (const boost::shared_ptr<const mpak::spec::command_info_list> &command_infos, unsigned indent = 0);
    
    class argument_printer
        : public boost::static_visitor<void>
    {
        const unsigned indent_;
        
    public:
        argument_printer (unsigned indent)
            : indent_ (indent)
        {
        }
        
        void operator () (const std::string &argument) const
        {
            std::cout << " [" << argument << "]";
        }
        
        void operator () (const boost::shared_ptr<mpak::spec::command_info_list> &argument) const
        {
            std::cout << " {" << std::endl;
            print_command_infos (argument, this->indent_ + 1);
            std::cout << std::string (this->indent_, '\t') << "}";
        }
    };
    
    void print_command_infos (const boost::shared_ptr<const mpak::spec::command_info_list> &command_infos, unsigned indent)
    {
        for (mpak::spec::command_info_list::const_iterator cii = command_infos->begin (); cii != command_infos->end (); cii++) {
            std::cout << std::string (indent, '\t') << cii->identifier;
            if (!cii->arguments)
                continue;
            for (mpak::spec::argument_vector::const_iterator ali = cii->arguments->begin (); ali != cii->arguments->end (); ali++) {
                boost::apply_visitor (argument_printer (indent), *ali);
            }
            std::cout << std::endl;
        }
    }
    
    int do_query (int argc, char *argv[])
    {
        const char shortopts[] = "+pi";
        const struct option longopts[] = {
            { "package", no_argument, 0, 'p' },
            { "installed", no_argument, 0, 'i' },
            { "node-type", required_argument, 0, 't' },
            { 0 }
        };
        
        std::string node_type ("mpak:version");
        
        enum {
            print_package_tree,
            print_installed_tree,
        } tree_to_print;
        
        tree_to_print = print_package_tree;
        
        while (1) {
            int c (getopt_long (argc, argv, shortopts, longopts, 0));
            
            if (c == -1) // no more options
                break;
            
            switch (c) {
            case 'p':
                tree_to_print = print_package_tree;
                break;
            case 'i':
                tree_to_print = print_installed_tree;
                break;
            case 't':
                node_type = optarg;
                break;
            default:
                std::cerr << "error parsing options" << std::endl;
                exit (EXIT_FAILURE);
            }
        }
        
        std::string node_path_string;
        if (optind == argc) {
            std::cerr << "no node_path given" << std::endl;
            exit (EXIT_FAILURE);
        } else {
            node_path_string = argv[optind];
            ++optind;
            if (optind != argc) {
                std::cerr << "too many arguments" << std::endl;
                exit (EXIT_FAILURE);
            }
        }
        
        // initialize configuration
        boost::shared_ptr<mpak::spec::pickler> config_pickler (new mpak::spec::file_pickler (*config_file_string));
        
        boost::shared_ptr<mpak::spec::context> config_context (new mpak::spec::context (config_pickler));
        mpak::builtins::init_config_context (*config_context);
        
        boost::shared_ptr<mpak::builtins::config_node> config_root (new mpak::builtins::config_node ("mpak:config", "config", *config_context));
        
        config_root->set_root_dir ("/");
        config_root->set_package_tree_dir (mpak_config_dir + "/packages");
        config_root->set_statedb_dir (LOCALSTATEDIR "/db/mpak");
        config_root->set_mpaktmp_dir ("/tmp/mpak");
        config_root->set_source_dir (LOCALSTATEDIR "/cache/mpak/sources");
        
        config_context->execute_command_list (config_pickler->read (), config_root);
        
        if (arg_root_dir_string) {
            config_root->set_root_dir (*arg_root_dir_string);
        }
        
        // initialize package tree structures
        
        boost::shared_ptr<mpak::spec::pickler> pickler;
        boost::shared_ptr<mpak::spec::context> context;
        std::string root_name;
        
        switch (tree_to_print) {
        case print_package_tree:
            {
                pickler.reset (new mpak::spec::tree_pickler (config_root->get_root_dir () / config_root->get_package_tree_dir ()));
                context.reset (new mpak::spec::context (pickler));
                mpak::builtins::init_package_tree_context (*context, config_root);
                root_name = "package_tree_root";
            }
            break;
        case print_installed_tree:
            {
                pickler.reset (new mpak::spec::database_pickler (config_root->get_root_dir () / config_root->get_statedb_dir () / "install.db"));
                context.reset (new mpak::spec::context (pickler));
                mpak::builtins::init_installed_tree_context (*context);
                root_name = "installed_tree_root";
            }
            break;
        };
        
        boost::shared_ptr<mpak::spec::node> root_node (new mpak::builtins::category_node ("mpak:category", root_name,
                                                                                          *context));
        context->execute_command_list (pickler->read (), root_node);
        
        mpak::util::node_path node_path (node_path_string);
        boost::shared_ptr<const mpak::spec::node> node (node_path.match (root_node, node_type));
        if (node) {
            if (verbose)
                std::cerr << "found " << node_path_string << std::endl;
        } else {
            std::cerr << "could not find " << node_path_string << std::endl;
            exit (EXIT_FAILURE);
        }
        
        print_command_infos (context->generate_command_infos_with_data (node));
        
        return EXIT_SUCCESS;
    }

    int do_init (int argc, char *argv[])
    {
#if 0
        const char shortopts[] = "+";
        const struct option longopts[] = {
            { 0 }
        };
        
        while (1) {
            int c (getopt_long (argc, argv, shortopts, longopts, 0));
        
            if (c == -1) // no more options
                break;
        
            switch (c) {
            default:
                std::cerr << "error parsing options" << std::endl;
                exit (EXIT_FAILURE);
            }
        }
#endif
        
        if (optind != argc) {
            std::cerr << "too many arguments" << std::endl;
            exit (EXIT_FAILURE);
        }
        
        // initialize configuration
        boost::shared_ptr<mpak::spec::pickler> config_pickler (new mpak::spec::file_pickler (*config_file_string));
        
        boost::shared_ptr<mpak::spec::context> config_context (new mpak::spec::context (config_pickler));
        mpak::builtins::init_config_context (*config_context);
        
        boost::shared_ptr<mpak::builtins::config_node> config_root (new mpak::builtins::config_node ("mpak:config", "config", *config_context));
        
        config_root->set_root_dir ("/");
        config_root->set_package_tree_dir (mpak_config_dir + "/packages");
        config_root->set_statedb_dir (LOCALSTATEDIR "/db/mpak");
        config_root->set_mpaktmp_dir ("/tmp/mpak");
        config_root->set_source_dir (LOCALSTATEDIR "/cache/mpak/sources");
        
        config_context->execute_command_list (config_pickler->read (), config_root);
        
        if (arg_root_dir_string) {
            config_root->set_root_dir (*arg_root_dir_string);
        }
        
        boost::filesystem::create_directories (config_root->get_root_dir ());
        boost::filesystem::create_directories (config_root->get_root_dir () / config_root->get_package_tree_dir ());
        boost::filesystem::create_directories (config_root->get_root_dir () / config_root->get_statedb_dir ());
        boost::filesystem::create_directories (config_root->get_root_dir () / config_root->get_mpaktmp_dir ());
        boost::filesystem::create_directories (config_root->get_root_dir () / config_root->get_source_dir ());
        
        return EXIT_SUCCESS;
    }
}
