// -*- mode: c++; indent-tabs-mode: nil; c-basic-offset: 4 -*-
// $Header: /home/pgavin/cvsroot/mpak/libmpak/mpak/util/file_operations.cc,v 1.6 2004/07/01 22:45:37 pgavin Exp $
// mpak - the advanced package manager
// Copyright (C) 2003 Peter Gavin
// 
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

#include <config.h>

#include <mpak/util/file_operations.hh>

#include <boost/filesystem/path.hpp>
#include <boost/filesystem/operations.hpp>
#include <boost/filesystem/exception.hpp>
#include <boost/function.hpp>

#include <map>
#include <fstream>
#include <cstdlib>
#include <cstdio>
#include <cstring>

#include <unistd.h>
#include <sys/stat.h>
#include <time.h>
#include <sys/time.h>

namespace mpak
{
    namespace util
    {
        void traverse_tree (const boost::filesystem::path &path,
                            const traverse_tree_function &function,
                            traverse_tree_flags flags)
        {
            using boost::filesystem::exists;
            using boost::filesystem::symbolic_link_exists;
            using boost::filesystem::is_directory;
            if (exists (path) && !symbolic_link_exists (path)) {
                if ((flags & traverse_tree_type_) == traverse_tree_preorder) {
                    function (path);
                }
                if (is_directory (path)) {
                    for (boost::filesystem::directory_iterator i (path); i != boost::filesystem::directory_iterator (); ++i) {
                        traverse_tree (*i, function, flags);
                    }
                }
                if ((flags & traverse_tree_type_) == traverse_tree_postorder) {
                    function (path);
                }
            } else if (symbolic_link_exists (path)) {
                function (path);
            }
        }
        
        // just like boost::filesystem::remove, but actually removes bad symlinks
        bool remove (const boost::filesystem::path &path)
        {
            if (exists (path) || symbolic_link_exists (path)) {
                if (std::remove (path.native_file_string ().c_str ()) != 0) {
                    int error (errno);
                    // POSIX says "If the directory is not an empty directory, rmdir()
                    // shall fail and set errno to EEXIST or ENOTEMPTY."
                    // Linux uses ENOTEMPTY, Solaris uses EEXIST.
                    if (error == EEXIST)
                        error = ENOTEMPTY;
                    throw boost::filesystem::filesystem_error ("mpak::util::remove", path, error );
                }
                return true;
            }
            return false;
        }
        
        const std::string
        read_symbolic_link (const boost::filesystem::path &path)
        {
            int size (128);
            char *buffer (0);
            
            while (1) {
                // to hell with deprecation, we need realloc here
                buffer = (char *) std::realloc (buffer, size);
                if (!buffer) {
                    throw std::bad_alloc ();
                }
                
                int nchars = readlink (path.native_file_string ().c_str (), buffer, size);
                if (nchars < 0) {
                    free (buffer);
                    throw file_operation_error ("could not read symbolic link");
                }
                if (nchars < size) {
                    try {
                        std::string ret (buffer, buffer + nchars);
                        free (buffer);
                        buffer = 0;
                        return ret;
                    } catch (...) {
                        if (buffer)
                            free (buffer);
                        throw;
                    }
                }
                size *= 2;
            }
        }
        
        void make_symbolic_link (const std::string &to, const boost::filesystem::path &dest)
        {
            if (symlink (to.c_str (), dest.native_file_string ().c_str ())) {
                char buffer[128];
                throw file_operation_error (std::string ("could not make symlink from ") + dest.native_file_string () + " to " + to +
                                            ": " + strerror_r (errno, buffer, 128));
            }
        }
        
        namespace
        {
            typedef std::map<std::pair<dev_t, ino_t>, boost::filesystem::path> copied_map_type;
            
            void do_duplicate (const boost::filesystem::path &source, const boost::filesystem::path &dest_dir, copied_map_type &copied_map)
            {
                struct stat sourcestat;
                if (lstat (source.native_file_string ().c_str (), &sourcestat)) {
                    char buffer[128];
                    throw file_operation_error (std::string ("could not stat file ") + source.native_file_string () + ": " + strerror_r (errno, buffer, 128));
                }
                
                const boost::filesystem::path dest (dest_dir / source.leaf ());
                if (S_ISDIR (sourcestat.st_mode)) {
                    struct stat deststat;
                    if (!lstat (dest.native_file_string ().c_str (), &deststat)) {
                        if (!S_ISDIR (deststat.st_mode))
                            throw file_operation_error ("file " + dest.native_file_string () + " exists and is not a directory");
                    } else {
                        if (mkdir (dest.native_directory_string ().c_str (), sourcestat.st_mode)) {
                            throw file_operation_error ("could not make directory: " + dest.native_file_string ());
                        }
                    }
                    boost::filesystem::directory_iterator end;
                    for (boost::filesystem::directory_iterator i (source); i != end; ++i) {
                        do_duplicate (*i, dest, copied_map);
                    }
                } else {
                    struct stat deststat;
                    if (!lstat (dest.native_file_string ().c_str (), &deststat)) {
                        if (S_ISDIR (deststat.st_mode)) {
                            throw file_operation_error ("file " + dest.native_file_string () + "exists and is a directory");
                        }
                        util::remove (dest);
                    }
                    if (sourcestat.st_nlink > 1) {
                        // preserve hard links by keeping track of files already copied
                        copied_map_type::const_iterator i (copied_map.find (std::make_pair (sourcestat.st_dev, sourcestat.st_ino)));
                        if (i != copied_map.end ()) {
                            if (link (i->second.native_file_string ().c_str (), dest.native_file_string ().c_str ()))
                                throw file_operation_error ("could not make hard link from " + i->second.native_file_string () +
                                                            " to " + dest.native_file_string ().c_str ());
                            return;
                        } else {
                            copied_map.insert (copied_map_type::value_type (std::make_pair (sourcestat.st_dev, sourcestat.st_ino), dest));
                        }
                    }
                    if (S_ISREG (sourcestat.st_mode)) {
                        // copy a regular file
                        // TODO: implement sparse copying
                        std::filebuf sourcefb;
                        std::ofstream destos;
                        if (!sourcefb.open (source.native_file_string ().c_str (), std::ios_base::in))
                            throw file_operation_error ("could not open source file " + source.native_file_string () + "for reading");
                        destos.open (dest.native_file_string ().c_str (), std::ios_base::out);
                        if (!destos)
                            throw file_operation_error ("could not open destination file " + dest.native_file_string () + "for writing");
                        destos << &sourcefb;
                        destos.close ();
                        sourcefb.close ();
                    } else if (S_ISCHR (sourcestat.st_mode) ||
                               S_ISBLK (sourcestat.st_mode) ||
                               S_ISSOCK (sourcestat.st_mode)) {
                        mknod (dest.native_file_string ().c_str (), sourcestat.st_mode, sourcestat.st_rdev);
                    } else if (S_ISLNK (sourcestat.st_mode)) {
                        make_symbolic_link (read_symbolic_link (source), dest);
                    } else if (S_ISFIFO (sourcestat.st_mode)) {
                        mkfifo (dest.native_file_string ().c_str (), sourcestat.st_mode);
                    } else {
                        throw file_operation_error (std::string ("file ") + source.native_file_string ().c_str () + " is of unrecognized type");
                    }
                }
                lchown (dest.native_file_string ().c_str (), sourcestat.st_uid, sourcestat.st_gid);
                if (!S_ISLNK (sourcestat.st_mode))
                    chmod (dest.native_file_string ().c_str (), sourcestat.st_mode);
                
                if (!S_ISLNK (sourcestat.st_mode)) {
                    struct timeval tv[2];
                    tv[0].tv_sec = sourcestat.st_atim.tv_sec;
                    tv[0].tv_usec = sourcestat.st_atim.tv_nsec / 1000;
                    tv[1].tv_sec = sourcestat.st_mtim.tv_sec;
                    tv[1].tv_usec = sourcestat.st_mtim.tv_nsec / 1000;
                    utimes (dest.native_file_string ().c_str (), tv);
                }
            }
        }
        
        void duplicate (const boost::filesystem::path &source, const boost::filesystem::path &dest_dir)
        {
            copied_map_type copied_map;
            do_duplicate (source, dest_dir, copied_map);
        }
        
        void duplicate_contents (const boost::filesystem::path &source_dir, const boost::filesystem::path &dest_dir)
        {
            copied_map_type copied_map;
            boost::filesystem::directory_iterator end;
            for (boost::filesystem::directory_iterator i (source_dir); i != end; ++i)
                do_duplicate (*i, dest_dir, copied_map);
        }
        
        boost::filesystem::path
        make_relative_path (const boost::filesystem::path &path, const boost::filesystem::path &base_path,
                            bool allow_ascending)
        {
            if (!path.is_complete ()) {
                throw file_operation_error ("path " + path.string () + " is not complete");
            }
            if (!base_path.is_complete ()) {
                throw file_operation_error ("path " + base_path.string () + " is not complete");
            }
            
            // skip common components
            boost::filesystem::path::iterator base_path_i (base_path.begin ());
            boost::filesystem::path::iterator path_i (path.begin ());
            
            for (; (base_path_i != base_path.end ()) && (path_i != path.end ()); ++base_path_i, ++path_i) {
            }
            
            if (!allow_ascending && (base_path_i != base_path.end ())) {
                throw file_operation_error ("path " + base_path.string () + " is not a proper base path of path " + path.string ());
            }
            
            boost::filesystem::path ret;
            for (; base_path_i != base_path.end (); ++base_path_i) {
                ret /= "..";
            }
            
            for (; path_i != path.end (); ++path_i) {
                ret /= *path_i;
            }
            
            return ret;
        }
    }
}
