#!/bin/bash

# This file is part of mpd-hits.
# Copyright (C) 2010, 2011, 2013, 2015 Dmitry Samoyloff.
#
# mpd-hits is free software: you can redistribute it and/or modify it under
# the terms of the GNU General Public License as published by the Free
# Software Foundation, either version 3 of the License, or (at your option)
# any later version.
#
# mpd-hits is distributed in the hope that it will be useful, but WITHOUT
# ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
# FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
# more details.
#
# You should have received a copy of the GNU General Public License along
# with mpd-hits. If not, see <http://www.gnu.org/licenses/>.

set -e

PACKAGE_NAME=mpd-hits
SCRIPT_DIR=$(cd `dirname $0` && pwd)
VERSION=`cd $SCRIPT_DIR && cat VERSION`
OUTPUT_DIR=/tmp/mpd-hits-build-package

print_usage() {
    echo "Usage: `basename $0` [--dont-sign] TARGET"
    echo "       `basename $0` [-h | --help]"
    echo
    echo "TARGET is one of:"
    echo "  debian"
    echo "  gentoo"
    echo "  tarball"
    echo
}

not_implemented() {
    echo "Support for $1 is not implemented yet."
}

check_for() {
    if ! which $1 >/dev/null; then
        echo "Command \"$1\" not found."
        exit 1
    fi
}

create_tarball() {
    TARBALL_DIR_REL=$PACKAGE_NAME-$VERSION
    TARBALL_NAME=$PACKAGE_NAME-${VERSION}.tar.bz2

    cd $OUTPUT_DIR

    # Prepare project directory for distribution
    cp -a $SCRIPT_DIR $TARBALL_DIR_REL
    find $TARBALL_DIR_REL -name "CVS" | xargs rm -rf
    find $TARBALL_DIR_REL -name ".*" | xargs rm -rf
    find $TARBALL_DIR_REL -name "*~" | xargs rm -rf
    rm -rf $TARBALL_DIR_REL/debian/mpd-hits/ $TARBALL_DIR_REL/build-stamp
    make -C $TARBALL_DIR_REL clean >/dev/null

    echo "Tarball contents:"
    echo "-----------------"
    tar cvjf $TARBALL_NAME $TARBALL_DIR_REL
    echo

    rm -rf $TARBALL_DIR_REL # Remove temporary stuff
}

sign_files() {
    [ -n "$DONT_SIGN_FILES" ] && return 0

    for f in "$@"; do
        if [ -f "$f" ]; then
            echo "==== Signing file: '`pwd`/$f'"
            gpg --detach-sign "$f"
        fi
    done
}

build_debian() {
    check_for dpkg-buildpackage

    DEB_DIR=$OUTPUT_DIR/mpd-hits-deb

    # Re-create build directory, removing a previous stuff.
    rm -rf $DEB_DIR
    mkdir -p $DEB_DIR

    # Prepare package directory.
    create_tarball
    cd $DEB_DIR
    mv $OUTPUT_DIR/$TARBALL_NAME .
    tar xjf $TARBALL_NAME

    # Build package.
    echo "Building deb package:"
    echo "---------------------"
    cd $TARBALL_DIR_REL
    dpkg-buildpackage -rfakeroot $DPKG_BUILDPACKAGE_OPTS
    cd ..
    echo

    # Cleanup.
    rm -rf $TARBALL_DIR_REL
    rm -f $TARBALL_NAME

    # Sign output files.
    sign_files *.deb *.tar.gz

    echo "Saved files:"
    echo "------------"
    echo "$DEB_DIR/*"
    echo
}

build_gentoo() {
    echo 'WARNING: Gentoo ebuild is obsolete and unsupported currently!'
    echo "If you want to maintain it -- you're welcome!"
    echo
    echo 'Press Enter to continue.'
    read

    create_tarball
    cp -f "$SCRIPT_DIR/gentoo/mpd-hits.ebuild" \
        "$OUTPUT_DIR/mpd-hits-${VERSION}.ebuild"

    # Sign output files.
    cd "$OUTPUT_DIR"
    sign_files *

    echo "Saved files:"
    echo "------------"
    echo "$OUTPUT_DIR/*"
}

build_tarball() {
    create_tarball

    # Sign output files.
    cd "$OUTPUT_DIR"
    sign_files *

    echo "Saved files:"
    echo "------------"
    echo "$OUTPUT_DIR/*"
}

# Parse command line.
DONT_SIGN_FILES=
if [ $# -eq 0 ]; then
    print_usage
    exit 0
else
    while [ "$#" != "0" ]; do
        if [ "$1" = "--dont-sign" ]; then
            DPKG_BUILDPACKAGE_OPTS='-us -uc'
            DONT_SIGN_FILES=1
            shift
        elif [ "$1" = "-h" ] || [ "$1" = "--help" ]; then
            print_usage
            exit 0
        else
            if [ -n "$TARGET" ]; then
                print_usage
                exit 1
            fi
            TARGET=$1
            shift
        fi
    done
fi

rm -rf "$OUTPUT_DIR"
mkdir -p "$OUTPUT_DIR"

case "$TARGET" in
    debian | \
    gentoo | \
    tarball)
        FUNC_POSTFIX=`echo $TARGET | sed 's/-/_/'`
        eval build_$FUNC_POSTFIX
        ;;
    *)
        print_usage
        exit 1
        ;;
esac

exit 0
