/***************************************************************
 *   Message Passing Interface for Simula+ Object Libraries    *
 *                 declaration for mpi receive                 *
 *                    simula+@metz.ensam.fr                    *
 *	             GNU/linux version 2.4.4	               *
 *            software under General Public License            *
 ***************************************************************
 * copyright © 2009,2010 COLLARD Christophe
 * copyright © 2009,2010 Centre National de la Recherche Scientifique
 * copyright © 2009,2010 Arts et Métiers ParisTech
 * copyright © 2009,2010 Laboratoire de Physique et Mécanique des Matériaux
 ***************************************************************/

/*! \namespace mpisol
    \brief Message Passing Interface for Simula+ Object Libraries
*/

/*! \defgroup receive MPI receive library for MOL++

    \brief receive library for MOL++ \n

    \htmlonly 
    <FONT color="#838383">

    receive belongs to Message Passing Interface for Simula+ Object Libraries (MPISOL++) </br>
    MPISOL++ is part of Simula+ <br><br>

    Simula+ is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version. <br><br>

    Simula+ is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details. <br><br>

    You should have received a copy of the GNU General Public License
    along with Simula+; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

    </FONT>
    \endhtmlonly

    \authors copyright \htmlonly &#169; \endhtmlonly 2009, 2010 Christophe COLLARD \n
	     copyright \htmlonly &#169; \endhtmlonly 2009, 2010 Centre National de la Recherche Scientifique \n
	     copyright \htmlonly &#169; 2009, 2010 Arts et M&#233;tiers ParisTech \endhtmlonly \n
	     copyright \htmlonly &#169; 2009, 2010 Laboratoire de Physique et M&#233;canique des Mat&#233;riaux \endhtmlonly
    \version 2.4.4
    \date 2009-2010
    \bug none
    \warning none
*/

/*@{*/

#ifndef __cplusplus
#error Must use C++ for the type receive
#endif

#ifndef _receive_h
#define _receive_h

#if !defined (__MPI_H)
#include <mpi.h>
#endif

#if !defined(__VECTORS_H)
#include "MOL++/vectors.h"
#endif

#if !defined(__MATRIX_H)
#include "MOL++/matrix.h"
#endif

#if !defined(__MATRIX_H)
#include "MOL++/matrix.h"
#endif

#if !defined(__SYMMATRIX_H)
#include "MOL++/symmatrix.h"
#endif

#if !defined(__TENSOR2_H)
#include "MOL++/tensors2.h"
#endif

#if !defined(__SYMTENSOR2_H)
#include "MOL++/symtensors2.h"
#endif

#if !defined(__TENSOR3_H)
#include "MOL++/tensors3.h"
#endif

#if !defined(__TENSOR4_H)
#include "MOL++/tensors4.h"
#endif

#if !defined(__SYMTENSOR4_H)
#include "MOL++/symtensors4.h"
#endif

#if !defined(__POLYNOMS_H)
#include "MOL++/polynoms.h"
#endif

using namespace std;
using namespace mol;

namespace mpisol
{

/*!
  \ingroup receive
  \brief mpi receive for scalar

  \param s scalar to receive
  \param thread receiving thread
  \param tag tag value (choose the same tag value to send and receive your objet)
*/

//------------------------------------------------------------------
  template <class Tf> void receive (Tf& s, int from_thread, int tag)
//------------------------------------------------------------------
{
  int nb_threads = MPI::COMM_WORLD.Get_size();
  assert (from_thread>=0 && from_thread<nb_threads);

  Tf* elt;
  elt = &s;
  MPI::Datatype element_type = MPI::BYTE.Create_hvector(1, sizeof(Tf), 1);
  element_type.Commit();
  MPI::COMM_WORLD.Recv (elt, 1, element_type, from_thread, tag);
  element_type.Free();
}


/*!
  \ingroup receive
  \brief mpi receive for vector

  \param v vector to receive
  \param thread receiving thread
  \param tag tag value (choose the same tag value to send and receive your objet)

  \sa \ref mol::vector "vector"
*/

//------------------------------------------------------------------------
template <class Tf> void receive (vector<Tf>& v, int from_thread, int tag)
//------------------------------------------------------------------------
{
  int nb_threads = MPI::COMM_WORLD.Get_size();
  assert (from_thread>=0 && from_thread<nb_threads);

  // receive space dimension
  int size[1];
  MPI::COMM_WORLD.Recv (size, 1, MPI::INT, from_thread, tag);
  if (!v.dim()) v = size[0]; // allocates memory
  else assert (v.dim() == size[0]);

  // receive vector
  MPI::Datatype vector_type = MPI::BYTE.Create_hvector(1, v.dim()*sizeof(Tf), 1); // number of bytes: v.size * sizeof(Tf)
  vector_type.Commit();
  MPI::COMM_WORLD.Recv(v(), 1, vector_type, from_thread, tag);
  vector_type.Free();
}


/*!
  \ingroup receive
  \brief mpi receive for matrix

  \param mat matrix to receive
  \param thread receiving thread
  \param tag tag value (choose the same tag value to send and receive your objet)

  \sa \ref mol::matrix "matrix"
*/

//--------------------------------------------------------------------------
template <class Tf> void receive (matrix<Tf>& mat, int from_thread, int tag)
//--------------------------------------------------------------------------
{
  int nb_threads = MPI::COMM_WORLD.Get_size();
  assert (from_thread>=0 && from_thread<nb_threads);

  // receive space dimension
  int size[2];
  MPI::COMM_WORLD.Recv (size, 2, MPI::INT, from_thread, tag);
  if (!mat.Rows()) mat.assign (size[0], size[1]); // allocates memory
  else assert (mat.Rows() == size[0] && mat.Columns() == size[1]);

  // receive matrix
  for (int i=1; i<=mat.Rows(); i++)
    receive (mat[i], from_thread, tag+i);
}


/*!
  \ingroup receive
  \brief mpi receive for symmatrix

  \param smat symmetric matrix to receive
  \param thread receiving thread
  \param tag tag value (choose the same tag value to send and receive your objet)

  \sa \ref mol::symmatrix "symmatrix"
*/

//------------------------------------------------------------------------------
template <class Tf> void receive (symmatrix<Tf>& smat, int from_thread, int tag)
//------------------------------------------------------------------------------
{
  int nb_threads = MPI::COMM_WORLD.Get_size();
  assert (from_thread>=0 && from_thread<nb_threads);

  // receive space dimension
  int size[2];
  MPI::COMM_WORLD.Recv (size, 2, MPI::INT, from_thread, tag);
  if (!smat.Rows()) smat.assign (size[0], size[1]); // allocates memory
  else assert (smat.Rows() == size[0] && smat.Columns() == size[1]);

  //receive symmatrix
  for (int i=1; i<=smat.Rows(); i++)
    receive (smat[i], from_thread, tag-i);
}


/*!
  \ingroup receive
  \brief mpi receive for tensor2

  \param tsr \f$ 2^\text{nd} \f$ order tensor to receive
  \param thread receiving thread
  \param tag tag value (choose the same tag value to send and receive your objet)

  \sa \ref mol::tensor2 "tensor2"
*/

//---------------------------------------------------------------------------
template <class Tf> void receive (tensor2<Tf>& tsr, int from_thread, int tag)
//---------------------------------------------------------------------------
{
  int nb_threads = MPI::COMM_WORLD.Get_size();
  assert (from_thread>=0 && from_thread<nb_threads);

  // receive space dimension
  int size[2];
  MPI::COMM_WORLD.Recv (size, 2, MPI::INT, from_thread, tag);
  if (!tsr.dim1()) tsr.assign (size[0], size[1]); // allocates memory
  else assert (tsr.dim1() == size[0] && tsr.dim2() == size[1]);

  //receive symmatrix
  for (int i=1; i<=tsr.dim1(); i++)
    receive (tsr[i], from_thread, tag-i);
}


/*!
  \ingroup receive
  \brief mpi receive for symtensor2

  \param tsr symmetric \f$ 2^\text{nd} \f$ order tensor to receive
  \param thread receiving thread
  \param tag tag value (choose the same tag value to send and receive your objet)

  \sa \ref mol::symtensor2 "symtensor2"
*/

//------------------------------------------------------------------------------
template <class Tf> void receive (symtensor2<Tf>& tsr, int from_thread, int tag)
//------------------------------------------------------------------------------
{
  int nb_threads = MPI::COMM_WORLD.Get_size();
  assert (from_thread>=0 && from_thread<nb_threads);

  // receive space dimension
  int size[2];
  MPI::COMM_WORLD.Recv (size, 2, MPI::INT, from_thread, tag);
  if (!tsr.dim1()) tsr.assign (size[0], size[1]); // allocates memory
  else assert (tsr.dim1() == size[0] && tsr.dim2() == size[1]);

  //receive symmatrix
  for (int i=1; i<=tsr.dim1(); i++)
    receive (tsr[i], from_thread, tag-i);
}


/*!
  \ingroup receive
  \brief mpi receive for tensor3

  \param tsr \f$ 3^\text{rd} \f$ order tensor to receive
  \param thread receiving thread
  \param tag tag value (choose the same tag value to send and receive your objet)

  \sa \ref mol::tensor3 "tensor3"
*/

//---------------------------------------------------------------------------
template <class Tf> void receive (tensor3<Tf>& tsr, int from_thread, int tag)
//---------------------------------------------------------------------------
{
  int nb_threads = MPI::COMM_WORLD.Get_size();
  assert (from_thread>=0 && from_thread<nb_threads);

  // receive space dimension
  int size[3];
  MPI::COMM_WORLD.Recv (size, 3, MPI::INT, from_thread, tag);
  if (!tsr.dim1()) tsr.assign (size[0], size[1], size[2]); // allocates memory
  else assert (tsr.dim1() == size[0] && tsr.dim2() == size[1] && tsr.dim3() == size[2]);

  //receive tensor3
  for (int i=1; i<=tsr.dim1(); i++)
    receive (tsr[i], from_thread, tag-i);
}


/*!
  \ingroup receive
  \brief mpi receive for tensor4

  \param tsr \f$ 4^\text{th} \f$ order tensor to receive
  \param thread receiving thread
  \param tag tag value (choose the same tag value to send and receive your objet)

  \sa \ref mol::tensor4 "tensor4"
*/

//---------------------------------------------------------------------------
template <class Tf> void receive (tensor4<Tf>& tsr, int from_thread, int tag)
//---------------------------------------------------------------------------
{
  int nb_threads = MPI::COMM_WORLD.Get_size();
  assert (from_thread>=0 && from_thread<nb_threads);

  // receive space dimension
  int size[4];
  MPI::COMM_WORLD.Recv (size, 4, MPI::INT, from_thread, tag);
  if (!tsr.dim1()) tsr.assign (size[0], size[1], size[2], size[3]); // allocates memory
  else assert (tsr.dim1() == size[0] && tsr.dim2() == size[1] && tsr.dim3() == size[2] && tsr.dim4() == size[3]);

  //receive tensor4
  for (int i=1; i<=tsr.dim1(); i++)
    receive (tsr[i], from_thread, tag-i);
}


/*!
  \ingroup receive
  \brief mpi receive for symtensor4

  \param tsr symmetric \f$ 4^\text{th} \f$ order tensor to receive
  \param thread receiving thread
  \param tag tag value (choose the same tag value to send and receive your objet)

  \sa \ref mol::symtensor4 "symtensor4"
*/

//------------------------------------------------------------------------------
template <class Tf> void receive (symtensor4<Tf>& tsr, int from_thread, int tag)
//------------------------------------------------------------------------------
{
  int nb_threads = MPI::COMM_WORLD.Get_size();
  assert (from_thread>=0 && from_thread<nb_threads);

  // receive space dimension
  int size[2];
  MPI::COMM_WORLD.Recv (size, 2, MPI::INT, from_thread, tag);
  if (!tsr.dim1()) tsr.assign (size[0], size[0], size[1], size[1]); // allocates memory
  else assert (tsr.dim1() == size[0] && tsr.dim3() == size[1]);

  //receive symmatrix
  for (int i=1; i<=0.5*size[0]*(size[0]+1); i++)
    receive (tsr[i], from_thread, tag-i);
}


/*!
  \ingroup receive
  \brief mpi receive for polynom

  \param Px polynom to receive
  \param thread receiving thread
  \param tag tag value (choose the same tag value to send and receive your objet)

  \sa \ref mol::polynom "polynom"
*/

//--------------------------------------------------------------------------
template <class Tf> void receive (polynom<Tf>& Px, int from_thread, int tag)
//--------------------------------------------------------------------------
{
  int nb_threads = MPI::COMM_WORLD.Get_size();
  assert (from_thread>=0 && from_thread<nb_threads);

  // receive the number of polynom coefficients (degree + 1)
  int size[1];
  MPI::COMM_WORLD.Recv (size, 1, MPI::INT, from_thread, tag);
  if (Px.degree()<0) Px = size[0]-1; // allocates memory
  else assert (Px.degree()+1 == size[0]);

  // receive vector
  MPI::Datatype polynom_type = MPI::BYTE.Create_hvector(1, size[0]*sizeof(Tf), 1); // number of bytes: v.size * sizeof(Tf)
  polynom_type.Commit();
  MPI::COMM_WORLD.Recv(Px(), 1, polynom_type, from_thread, tag);
  polynom_type.Free();
}


}


#endif


/*@}*/
