# Copyright (C) 2004, 2005  National Institute of Advanced Industrial Science and Technology
#
# This file is part of msgcab.
#
# msgcab is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# msgcab is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with msgcab; if not, write to the Free Software
# Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

require 'time'
require 'msgcab/nov'
require 'msgcab/logger'
require 'pathname'
require 'ostruct'

module MsgCab
  class Database
    private_class_method :new

    include Logging

    @@instance = nil
    def self.instance
      unless @@instance
        adapter_name = Config['database', 'adapter'] || 'DBI'
        require "msgcab/database/#{adapter_name.downcase}"
        adapter_class = MsgCab.const_get(adapter_name)
        @@instance = new(adapter_class.new)
      end
      @@instance
    end

    def initialize(adapter)
      @adapter = adapter
      load_schema(Pathname.new(__FILE__).dirname + 'schema.sql')
    end
    attr_reader :adapter

    def self.disconnect
      begin
        @@instance.disconnect
      ensure
        @@instance = nil
      end
    end

    def disconnect
      @adapter.disconnect
    end

    def load_schema(path)
      path.open do |file|
        stmt = ''
        table = nil
        loop do
          line = file.gets
          unless line
            if table && !@adapter.tables.include?(table)
              @adapter.prepare(stmt, table).execute
              log(3, "created table: #{table}")
            end
            break
          end
          if line =~ /^CREATE\s+TABLE\s+(\S+)/i
            if table && !@adapter.tables.include?(table)
              @adapter.prepare(stmt, table).execute
              log(3, "created table: #{table}")
            end
            stmt = line
            table = $1
          else
            stmt << line
          end
        end
      end
    end

    def nov(number)
      row = @adapter.select_one(<<'End', number)
SELECT nov FROM message WHERE number = ?
End
      Nov.parse(row[0]) if row
    end

    def folder(name)
      row = @adapter.select_one(<<'End', name)
SELECT MIN(folder_number), MAX(folder_number) FROM filter WHERE folder = ?
End
      return nil unless row && row[0]
      min = row[0].to_i
      max = row[1].to_i
      row = @adapter.select_one(<<'End', name)
SELECT MAX(time) FROM filter WHERE folder = ?
End
      time = Time.xmlschema(row[0]) rescue Time.parse(row[0])
      OpenStruct.new({:name => name, :min  => min, :max => max, :time => time})
    end

    # Return a list of folder names.
    def folders
      @adapter.select_all(<<'End').collect {|row| folder(row[0])}
SELECT DISTINCT folder FROM filter
End
    end

    # Return the maximum _global_ number assigned in whole archive.
    def max
      row = @adapter.select_one(<<'End')
SELECT MAX(number) FROM message
End
      return row && row[0] ? row[0].to_i : -1
    end

    def recent_summary(folder, length)
      rows = @adapter.select_all(<<'End', folder, length)
SELECT filter.folder_number, message.*
  FROM filter JOIN message ON message.number = filter.number
  WHERE filter.folder = ? ORDER BY filter.folder_number DESC LIMIT ?
End
      rows.collect {|folder_number, number, msg_id, digest, date, nov|
        date = Time.xmlschema(date) rescue Time.parse(date)
        nov = MsgCab::Nov.parse(nov)
        OpenStruct.new({:folder => folder,
                         :folder_number => folder_number.to_i,
                         :number => number.to_i,
                         :msg_id => msg_id,
                         :digest => digest,
                         :date => date,
                         :nov => nov})
      }
    end

    def number_summary(folder, min, max = min)
      rows = @adapter.select_all(<<'End', folder, min, max)
SELECT filter.folder_number, message.*
  FROM filter JOIN message ON message.number = filter.number
  WHERE filter.folder = ? AND filter.folder_number >= ? AND filter.folder_number <= ?
  ORDER BY filter.folder_number ASC
End
      rows.collect {|folder_number, number, msg_id, digest, date, nov|
        date = Time.xmlschema(date) rescue Time.parse(date)
        nov = MsgCab::Nov.parse(nov)
        OpenStruct.new({:folder => folder,
                         :folder_number => folder_number.to_i,
                         :number => number.to_i,
                         :msg_id => msg_id,
                         :digest => digest,
                         :date => date,
                         :nov => nov})
      }
    end

    def message(number)
      row = @adapter.select_one(<<'End', number)
SELECT * FROM message WHERE number = ?
End
      row.shift
      msg_id, digest, date, nov = row
      date = Time.xmlschema(date) rescue Time.parse(date)
      nov = MsgCab::Nov.parse(nov)
      folder, folder_number = to_folder_number(number)
      OpenStruct.new({:folder => folder,
                       :folder_number => folder_number,
                       :number => number,
                       :msg_id => msg_id,
                       :digest => digest,
                       :date => date,
                       :nov => nov})
    end

    def from_folder_number(folder, folder_number)
      row = @adapter.select_one(<<'End', folder, folder_number)
SELECT number FROM filter WHERE folder = ? AND folder_number = ?
End
      row[0].to_i if row
    end

    def to_folder_number(number)
      row = @adapter.select_one(<<'End', number)
SELECT folder, folder_number FROM filter WHERE number = ?
End
      row[1] = row[1].to_i if row
      row
    end

    def numbers(folder, min, max = min)
      @adapter.select_all(<<'End', folder, min, max).collect {|row| row[0].to_i}
SELECT folder_number
  FROM filter
  WHERE folder = ? AND folder_number >= ? AND folder_number <= ?
End
    end

    def numbers_by_msg_id(msg_id)
      @adapter.select_all(<<'End', msg_id).collect {|row| row[0].to_i}
SELECT number FROM message WHERE msg_id = ?
End
    end

    def ancestors(msg_id)
      @adapter.select_all(<<'End', msg_id).collect {|row| row[0]}
SELECT DISTINCT ref_id FROM reference WHERE msg_id = ?
End
    end

    def descendants(msg_id)
      @adapter.select_all(<<'End', msg_id).collect {|row| row[0]}
SELECT DISTINCT msg_id FROM reference WHERE ref_id = ?
End
    end

    def numbers_by_digest(digest)
      @adapter.select_all(<<'End', digest).collect {|row| row[0].to_i}
SELECT number FROM message WHERE digest = ?
End
    end

    def header_fields(number)
      rows = @adapter.select_all(<<'End', number)
SELECT name, body FROM header WHERE number = ?
End
      result = Hash.new
      rows.each do |name, body|
        entry = result[name] ||= Array.new
        entry << body
      end
      result
    end

    def header_field(number, name)
      row = @adapter.select_one(<<'End', number, name)
SELECT body FROM header WHERE number = ? AND name = ?
End
      row[0] if row
    end

    def thread(threads, depth, current, siblings, parents, msg_id_hash)
      result = Array.new
      return result if msg_id_hash.key?(current)
      msg_id_hash[current] = false
      row = @adapter.select_one(<<'End', current)
SELECT nov FROM message WHERE msg_id = ?
End
      nov = MsgCab::Nov.parse(row[0])
      numbers = numbers_by_msg_id(current)
      xref = numbers.collect {|number| to_folder_number(number)}
      result << OpenStruct.new({:number => numbers[0],
                                 :depth => depth,
                                 :xref => xref,
                                 :nov => nov,
                                 :parents => parents | [numbers[0]]})
      children = threads.select {|t| t[0] == current}.collect {|r| r[1]}
      unless children.empty?
        result.concat(thread(threads,depth.succ, children.shift, children,
                             parents | [numbers[0]], msg_id_hash))
      end
      siblings.each_with_index do |sibling, index|
        result.concat(thread(threads,
                             depth, sibling, [], parents, msg_id_hash))
      end
      result
    end

    def disjoint_sets(numbers)
      parent_stmt = @adapter.prepare(<<'End')
SELECT parent FROM disjoint WHERE number = ?
End
      number_stmt = @adapter.prepare(<<'End')
SELECT number FROM disjoint WHERE parent = ?
End
      result = Array.new
      number_hash = Hash.new
      numbers.each do |number|
        next if number_hash.key?(number)
        parent_stmt.execute(number)
        parent = parent_stmt.fetch[0].to_i
        next if number_hash.key?(parent)
        number_stmt.execute(parent)
        disjoint = number_stmt.fetch_all.collect {|row|
          r = row[0].to_i
          number_hash[r] = false
          r
        }
        disjoint.delete_at(disjoint.index(parent))
        disjoint.unshift(parent)
        result << disjoint
      end
      result
    end

    def recent_topics(folder, length)
      rows = @adapter.select_all(<<'End', folder, length)
SELECT number FROM filter WHERE folder = ? ORDER BY folder_number DESC LIMIT ?
End
      topic_hash = Hash.new
      result = Array.new
      rows.each do |row|
        nov, = @adapter.select_one(<<'End', row[0].to_i)
SELECT nov FROM message WHERE number = ?
End
        nov = MsgCab::Nov.parse(nov)
        canonical_subject = nov.decode.canonical_subject
        if topic = topic_hash[canonical_subject]
          topic.size += 1
          next
        end
        rows = @adapter.select_all(<<'End', row[0].to_i)
SELECT folder, folder_number FROM filter WHERE number = ?
End
        folder_name, folder_number = rows.assoc(folder) || rows[0]
        topic = OpenStruct.new({:canonical_subject => canonical_subject,
                                 :nov => nov,
                                 :folder_name => folder_name,
                                 :folder_number => folder_number.to_i,
                                 :size => 1})
        result << topic
        topic_hash[canonical_subject] = topic
      end
      result
    end

    def recent_threads(folder, length)
      rows = @adapter.select_all(<<'End', folder, length)
SELECT number FROM filter WHERE folder = ? ORDER BY folder_number DESC LIMIT ?
End
      result = Array.new
      msg_id_hash = Hash.new
      threads = disjoint_sets(rows.collect {|row| row[0].to_i}).flatten.collect do |number|
        row = @adapter.select_one(<<'End', number)
SELECT msg_id FROM message WHERE number = ?
End
        [parent_msg_id(row[0]), row[0]]
      end
      threads.each do |t|
        if t[0] == nil
          result.concat(thread(threads, 0, t[1], [], [], msg_id_hash))
        end
      end
      result
    end

    def number_threads(folder, min, max)
      rows = @adapter.select_all(<<'End', folder, min, max)
SELECT message.msg_id
  FROM filter JOIN message ON filter.number = message.number
  WHERE filter.folder = ? AND filter.folder_number >= ? AND filter.folder_number <= ?
  ORDER BY filter.folder_number ASC
End
      result = Array.new
      msg_id_hash = Hash.new
      threads = rows.collect do |row|
        [parent_msg_id(row[0]), row[0]]
      end
      threads.each do |t|
        if t[0] == nil
          result.concat(thread(threads, 0, t[1], [], [], msg_id_hash))
        end
      end
      result
    end

    def one_threads(folder, folder_number)
      number = from_folder_number(folder, folder_number)
      row = @adapter.select_one(<<'End', number)
SELECT parent FROM disjoint WHERE number = ?
End
      rows = @adapter.select_all(<<'End', row[0].to_i)
SELECT number FROM disjoint WHERE parent = ?
End
      result = Array.new
      msg_id_hash = Hash.new
      threads = rows.collect do |row|
        row = @adapter.select_one(<<'End', row[0].to_i)
SELECT msg_id FROM message WHERE number = ?
End
        [parent_msg_id(row[0]), row[0]]
      end
      threads.each do |t|
        if t[0] == nil
          result.concat(thread(threads, 0, t[1], [], [], msg_id_hash))
        end
      end
      result
    end

    def parent_msg_id(msg_id)
      row = @adapter.select_one(<<'End', msg_id)
SELECT ref_id FROM parent WHERE msg_id = ?
End
      return row[0] if row && @adapter.select_one(<<'End', row[0])
SELECT * FROM message WHERE msg_id = ?
End
      row = @adapter.select_one(<<'End', msg_id)
SELECT nov FROM message WHERE msg_id = ?
End
      if row
        parent_ids = Nov.parse(row[0]).references.reverse!
        parent_ids.each do |msg_id|
          return msg_id if @adapter.select_one(<<'End', msg_id)
SELECT * FROM message WHERE msg_id = ?
End
        end
      end
      return nil
    end

    def property(number, name)
      row = @adapter.select_one(<<'End', number, name)
SELECT value FROM property WHERE number = ? AND name = ?
End
      row[0] if row
    end
  end
end
