# Copyright (C) 2004, 2005  National Institute of Advanced Industrial Science and Technology
#
# This file is part of msgcab.
#
# msgcab is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# msgcab is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with msgcab; if not, write to the Free Software
# Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

module MsgCab
  module Logging
    def log(logging_level, str)
      unless defined? @logger and @logger
        h = Config['logger'] || Hash.new
        if defined? MsgCab::FLAVOR
          h.update(Config['logger', MsgCab::FLAVOR] || Hash.new)
        end
        @logger = Logger.new(h['device'], h['logging_level'])
      end
      @logger.log(logging_level, str)
    end
  end

  class Logger
    DefaultLoggingLevel = 2

    def initialize(device = nil, logging_level = nil)
      device = Logger.open_device(device) if device.kind_of? String
      @device = device
      @logging_level = logging_level || DefaultLoggingLevel
    end
    attr_accessor :logging_level

    def self.open_device(str)
      case str
      when 'STDOUT'
        $stdout
      when 'STDERR'
        $stderr
      when String
        Pathname.new(str).open(File::CREAT|File::APPEND|File::WRONLY, 0600)
      end
    end

    def log(logging_level, str)
      return if logging_level > @logging_level
      return unless @device
      begin
        @device.flock(File::LOCK_EX) if @device.respond_to? 'flock'
        @device << Time.now.strftime('%Y/%m/%d %H:%M:%S ')
        @device.puts(str)
        @device.flush
      ensure
        @device.flock(File::LOCK_UN) if @device.respond_to? 'flock'
      end
    end
  end
end
