# Copyright (C) 2004, 2005  National Institute of Advanced Industrial Science and Technology
#
# This file is part of msgcab.
#
# msgcab is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# msgcab is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with msgcab; if not, write to the Free Software
# Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

require 'pathname'
require 'msgcab/logger'

module MsgCab
  class MailTree
    include Logging

    def initialize
      @path = Config.absolute_path(Config['mailtree', 'path'] || './mt')
      @number_base = Config['mailtree', 'number_base'] || 100
      @number_padding = Config['mailtree', 'number_padding'] || 2
    end

    def fetch(number, default = nil, &block)
      path = number_to_path(number)
      if File.exist?(path)
        File.open(path) {|file| file.read}
      elsif default
        default
      elsif block_given?
        block.call
      else
        raise IndexError, 'not found'
      end
    end

    def store(number, data)
      path = number_to_path(number)
      begin
        safe_mkdir_p(path.dirname, @path)
        File.open(path.to_s, 'a') do |file|
          file.write(data)
        end
      rescue Exception => e
        path.unlink if path.exist?
        raise
      end
    end

    def max_entry(path)
      entries = path.entries
      entries.reject! {|entry| entry.to_s !~ /\A\d+\z/}
      entries.collect! {|entry| entry.to_s.to_i}
      entries.max
    end

    def last_number
      max_num_digits = max_entry(@path)
      path = @path + max_num_digits.to_s
      number = 0
      max_digit = 0
      max_num_digits.times do
        max_digit = max_entry(path)
        number = number * @number_base + max_digit
        path += max_digit.to_s.rjust(@number_padding, '0')
      end
      number
    end

    def number_to_path(number)
      digits = []
      until number < @number_base
        number, mod = number.divmod(@number_base)
        digits << mod
      end
      if number >= 0
        digits << number
      end
      digits.reverse!
      path = @path + digits.length.to_s
      digits.each do |digit|
        path += digit.to_s.rjust(@number_padding, '0')
      end
      path
    end

    def path_to_number(path)
      path = path.expand_path.relative_path_from(@path)
      digits = Array.new
      path.each_filename do |name|
        digits << name.to_i
      end
      length = digits.shift
      raise ArgumentError, path unless digits.length == length.to_i
      digits.inject(0) {|accu, digit| accu * @number_base + digit.to_i}
    end

    private
    def safe_mkdir_p(parent, top)
      top.mkdir unless top.directory?
      parent.relative_path_from(top).each_filename do |name|
        top += name
        top.mkdir unless top.directory?
      end
    end
  end
end
