# Copyright (C) 2004, 2005  National Institute of Advanced Industrial Science and Technology
#
# This file is part of msgcab.
#
# msgcab is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# msgcab is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with msgcab; if not, write to the Free Software
# Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

require 'pathname'

class RssPlugin < MsgCab::Plugin
  def start_plugin
    MsgCab::Callback.add_callback(:view_initialize, :add_rss_anchor) do |view|
      view.add_anchor(:header, RssAnchor, true)
    end

    @controller = attributes[:controller]
    @controller.add_action(%r|\A#{MsgCab::WebApp::FolderPat}/rss(?:/:(\d+))?\z|,
                           method(:do_recent_summary_rss).to_proc)
  end

  def do_recent_summary_rss(webapp, folder_name, max)
    view = MsgCab::WebApp::View.new(webapp)
    folder = @controller.database.folder(folder_name)
    unless folder
      raise MsgCab::WebApp::InvalidRequest, "no such folder: #{folder_name}"
    end
    recent_max = max ? max.to_i : MsgCab::Config['webapp', 'rss_recent_max'] || 20

    items = Array.new
    @controller.database.recent_summary(folder.name, recent_max).each do |message|
      entity = MsgCab::Entity.parse(@controller.mailtree.fetch(message.number))
      if entity.multipart?
        entity = entity[0]
      end
      items << [message.nov.decode(view.encoding),
        description(entity, view.encoding),
        webapp.make_absolute_uri(:path_info => "#{folder.name}/#{message.folder_number}")]
    end
    view.attributes[:folder] = folder
    view.attributes[:items] = items
    
    view.output_template(Pathname.new(__FILE__).dirname + 'rss.xml')
  end
  
  def description(entity, encoding)
    text = decode_entity(entity, encoding)
    text.gsub!(/^In (?:the )?message .*wrote:$/im, '')
    text.gsub!(/^At .*wrote:$/im, '')
    text.gsub!(/^[ \t]*[-_.a-z0-9]+>+|[ \t]*[\]>|}+].*$/, '')
    text.gsub!(/[ \n]+/, ' ')
    chars = text.split(//)
    if chars.length > 124
      chars[0 ... 124].join + "..."
    else
      chars.join
    end
  end

  def decode_entity(entity, encoding)
    while entity.multipart?
      entity = entity[0]
    end
    return "[Can't decode]" if entity.content_type !~ /text\/plain/i
    charset = entity.charset
    if encoding && charset
      begin
        Iconv.conv(encoding, charset, entity.decode)
      rescue Iconv::Failure => e
        "#{e.success} [Failed to decode: #{e.failed}]"
      rescue Exception
        "[Can't decode]"
      end
    else
      entity.decode
    end
  end

  def stop_plugin
    MsgCab::Callback.remove_callback(:view_initialize, :add_rss_anchor)
  end
end

class RssAnchor < MsgCab::WebApp::Anchor
  include HTree.compile_template(<<'End')
<div _template="replace">
  <span _if="view.attributes[:folder]" _else="next_anchor.replace">
    <a class="button" _attr_href='view.uri("#{view.attributes[:folder].name}/rss")'><img _attr_src='view.uri("/.data/rss/rss.png")' alt="[RSS]"/></a><br/>
  </span>
  <span _call="next_anchor.replace"></span>
</div>
End
end
