# Copyright (C) 2004, 2005  National Institute of Advanced Industrial Science and Technology
#
# This file is part of msgcab.
#
# msgcab is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# msgcab is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with msgcab; if not, write to the Free Software
# Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

require 'test/unit'
require 'pathname'
require 'fileutils'
require 'msgcab/source'

include MsgCab

class TestSource < Test::Unit::TestCase
  def test_s_open
    path = Pathname.new(__FILE__).dirname + 'test-source0.txt'
    begin
      path.open('w+') do |file|
        file.write('')
        file.rewind
        assert_instance_of(FileSource, Source.open(path))
        file.write('From ')
        file.rewind 
        assert_instance_of(MboxSource, Source.open(path))
      end
    ensure
      path.unlink
    end
    path = Pathname.new(__FILE__).dirname + 'test-source'
    begin
      path.mkdir
      assert_instance_of(DirSource, Source.open(path))
    ensure
      path.rmdir
    end
  end

  def test_io_source
    str = <<'End'
From aaa@bbb
From: aaa

End
    io = StringIO.new(str)
    source = IOSource.new(io)
    assert_equal(1, source.size)
    chunks = source.to_a
    assert_equal(1, chunks.size)
    assert_equal(str, chunks[0].data)
  end

  def test_file_source
    source = FileSource.new(Pathname.new(__FILE__).dirname + 'test-source1.txt')
    assert_equal(1, source.size)
    chunks = source.to_a
    assert_equal(1, chunks.size)
  end

  def test_mbox_source
    source = MboxSource.new(Pathname.new(__FILE__).dirname + 'test-source2.txt')
    assert_equal(2, source.size)
    chunks = source.to_a
    assert_equal(2, chunks.size)
    assert_equal(<<'End'.strip, chunks[1].data)
From: bbb

From
End
  end

  def test_dir_source_maildir
    path = Pathname.new(__FILE__).dirname + 'test-source'
    begin
      path.mkdir
      cur_path = path + 'cur'
      cur_path.mkdir
      new_path = path + 'new'
      new_path.mkdir
      txt_path = Pathname.new(__FILE__).dirname + 'test-source3.txt'
      (new_path + 'a').make_link(txt_path)
      (cur_path + 'b').make_link(txt_path)
      (cur_path + '.b').make_link(txt_path)
      source = DirSource.new(path)
      assert_equal(2, source.size)
      chunks = source.to_a
      assert_equal(2, chunks.size)
      assert_equal(<<'End'.strip, chunks[0].data)
From: aaa

bbb
End
      source = DirSource.new(path, false)
      source.to_a
      assert_equal(false, (new_path + 'a').exist?)
      assert_equal(true, (cur_path + 'a:2,').exist?)
    ensure
      FileUtils.rm_r(path.to_s)
    end
  end

  def test_dir_source_mh
    path = Pathname.new(__FILE__).dirname + 'test-source'
    begin
      path.mkdir
      txt_path = Pathname.new(__FILE__).dirname + 'test-source3.txt'
      (path + '0').make_link(txt_path)
      (path + '1').make_link(txt_path)
      (path + 'a').make_link(txt_path)
      source = DirSource.new(path, false)
      assert_equal(2, source.size)
      chunks = source.to_a
      assert_equal(2, chunks.size)
      assert_equal(<<'End'.strip, chunks[0].data)
From: aaa

bbb
End
    ensure
      FileUtils.rm_r(path.to_s)
    end
  end
end
