// muesli-javascript.c -*- C -*-
/* Interface to javascript evaluators / template for new language interfaces
   Copyright (C) 2008, 2009, 2010 University of Limerick

   This file is part of Muesli.
   
   Muesli is free software: you can redistribute it and/or modify it under
   the terms of the GNU General Public License as published by the Free
   Software Foundation, either version 3 of the License, or (at your
   option) any later version.
   
   Muesli is distributed in the hope that it will be useful, but WITHOUT
   ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
   FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
   for more details.
   
   You should have received a copy of the GNU General Public License
   along with Muesli.  If not, see <http://www.gnu.org/licenses/>.
*/

#ifndef _MUESLI_JAVASCRIPT_
#define _MUESLI_JAVASCRIPT_

#include "../config.h"
#include "muesli.h"

#ifdef HAVE_LIBMOZJS

#include <js/jsapi.h>
#include <stdio.h>
#include <limits.h>
#include <unistd.h>
#include <string.h>
#include <stdlib.h>
#include <string.h>
#include <getopt.h>		// todo: do I need this?

static int ambient_transient = 0;

JSClass global_class = {
        "global",0,
        JS_PropertyStub,JS_PropertyStub,JS_PropertyStub,JS_PropertyStub,
        JS_EnumerateStub,JS_ResolveStub,JS_ConvertStub,JS_FinalizeStub
    };

typedef struct javascript_state {
  JSRuntime *rt;
  JSContext *cx;
  JSObject *global; 
} javascript_state;

static evaluator_interface *javascript_interface = NULL;

static JSBool
javascript_set_parameter(JSContext *cx, JSObject *obj, uintN argc, jsval *argv, jsval *rval)
{
  char *parameter_name = NULL;
  char *parameter_value_string = NULL;
  jsdouble parameter_value_number = 0.0;
  int got_string = 0;
  int got_number = 0; 

  if (JS_ConvertArguments(cx, argc, argv, "ss", &parameter_name, &parameter_value_string)) {
    got_string = 1;
  } else if (JS_ConvertArguments(cx, argc, argv, "sd", &parameter_name, &parameter_value_number)) {
    got_number = 1;
  } else {
    return JS_FALSE;
  }

  char option_code = muesli_find_option_letter(javascript_interface->getopt_options, parameter_name);

  if (option_code != -1) {
    if (got_string) {
      char *value_copy = (char*)malloc(strlen(parameter_value_string));
      strcpy(value_copy, parameter_value_string);
      (javascript_interface->handle_option)(javascript_interface->app_params,
					    option_code,
					    muesli_malloc_copy_string(parameter_value_string),
					    0.0, 1,
					    "javascript");
    } else if (got_number) {
      (javascript_interface->handle_option)(javascript_interface->app_params,
					    option_code, NULL, parameter_value_number, 1,
					    "javascript");
    } else {
      (javascript_interface->handle_option)(javascript_interface->app_params,
					    option_code, (char*)"true", 0.0, 1,
					    "javascript");
    }
  }
  *rval = JSVAL_VOID;
  return JS_TRUE;
}

#if 0
static JSBool
javascript_set_parameters(JSContext *cx, JSObject *obj, uintN argc, jsval *argv, jsval *rval)
{
  JSObject table;

  if (!JS_ConvertArguments(cx, argc, argv, "O", &table)) {
    return JS_FALSE;
  }

#if 0
  // Fill in: use a table iterator from your language
  javascript_table table = javascript_table_arg(cs, 0);

  javascript_table_iteration_start(cs, table);
  while (javascript_table_iteration_next(cs, table) != 0) {
    javascript_set_parameter(cs,
			 javascript_table_iteration_current_key(cs, table),
			 javascript_table_iteration_current_value(cs, table));
  }

#endif
  *rval = JSVAL_VOID;
  return JS_TRUE;
}
#endif

static JSBool
muesli_to_javascript(JSContext *cx,
		     muesli_value_t mval,
		     jsval *rval)
{
  switch (mval.type) {
  case muesli_value_string:
    *rval = STRING_TO_JSVAL(JS_NewStringCopyN(cx,
					      mval.data.as_string,
					      strlen(mval.data.as_string)));
    return JS_TRUE;
  case muesli_value_float:
    return JS_NewNumberValue(cx, mval.data.as_float, rval);
  case muesli_value_integer:
    return JS_NewNumberValue(cx, (float)mval.data.as_int, rval);
  case muesli_value_boolean:
    *rval = mval.data.as_bool ? JSVAL_TRUE : JSVAL_FALSE;
    return JS_TRUE;
  default:
    return JS_FALSE;
  }
}

void
javascript_to_muesli(javascript_state *our_state,
		     jsval *pval,
		     muesli_value_t *presult)
{
  if (JSVAL_IS_STRING(*pval)) {
    fprintf(stderr, "string result\n");
    presult->data.as_string = JS_GetStringBytesZ(our_state->cx, JSVAL_TO_STRING(*pval));
    presult->type = muesli_value_string;
  } else if (JSVAL_IS_DOUBLE(*pval)) {
    jsdouble *d = JSVAL_TO_DOUBLE(*pval);
    fprintf(stderr, "double result\n");
    presult->data.as_float = (float)(*d);
    presult->type = muesli_value_float;
  } else if (JSVAL_IS_INT(*pval)) {
    fprintf(stderr, "integer result\n");
    presult->data.as_int = JSVAL_TO_INT(*pval);
    presult->type = muesli_value_integer;
  } else if (JSVAL_IS_BOOLEAN(*pval)) {
    fprintf(stderr, "boolean result\n");
    presult->data.as_bool = JSVAL_TO_BOOLEAN(*pval);
    presult->type = muesli_value_boolean;
  } else {
    fprintf(stderr, "other result\n");
    presult->type = muesli_value_none;
#if 1
    if (JSVAL_IS_VOID(*pval)) {
      fprintf(stderr, "void\n");
    }
    if (JSVAL_IS_NULL(*pval)) {
      fprintf(stderr, "null\n");
    }
    if (JSVAL_IS_SPECIAL(*pval)) {
      fprintf(stderr, "special\n");
    }
#endif
  }
}

static JSBool
javascript_get_parameter(JSContext *cx, JSObject *obj,
			 uintN argc, jsval *argv,
			 jsval *rval)
{
  char *parameter_name = NULL;

  if (!JS_ConvertArguments(cx, argc, argv, "s", &parameter_name)) {
    return JS_FALSE;
  }

  char option_code
    = muesli_find_option_letter(javascript_interface->getopt_options,
				parameter_name);

  if (option_code != -1) {

    return muesli_to_javascript(cx,
				((javascript_interface->handle_option)
				 (javascript_interface->app_params,
				  option_code,	// option
				  NULL, 0.0,	// value
				  0,	// set
				  "javascript")),
				rval);
  }
  return JS_FALSE;
}

#if 0
static JSBool
javascript_get_parameters(JSContext *cx, JSObject *obj, uintN argc, jsval *argv, jsval *rval)
{
  javascript_table *table = javascript_create_table(cs, 48);

  // todo: fix and re-instate -- I have to get long_options across to it somehow
  struct option *option_names = javascript_interface->getopt_options;

  while (option_names->name != 0) {
    
    result_type_t result_type;
    muesli_value_eval_t result;

    (javascript_interface->handle_option)(javascript_interface->app_params,
				      (option_names->val), // opt
				      NULL, 0.0,		     // value
				      0,		     // set
				      &result_type, &result,
				      "javascript");

    switch (result.type) {
    case muesli_value_string:
      javascript_set_table_string(cs, table, (char*)(option_names->name), result.data.as_string);
      break;
    case muesli_value_float:
      javascript_set_table_number(cs, table, (char*)(option_names->name), result.data.as_float);
      break;
    case muesli_value_boolean:
      javascript_set_table_boolean(cs, table, (char*)(option_names->name), result.data.as_int);
      break;
    default:
      javascript_set_table_boolean(cs, table, (char*)(option_names->name), 0);
      break;
    }
    option_names++;
  }
}
#endif

///////////////////////////////////////
// Call arbitrary evaluators by name //
///////////////////////////////////////

static int
javascript_eval_in_language(JSContext *cx, JSObject *obj,
			    uintN argc, jsval *argv,
			    jsval *rval)
{
  char *language_name = NULL;
  char *evaluand = NULL;

  if (!JS_ConvertArguments(cx, argc, argv, "ss", &language_name, &evaluand)) {
    return JS_FALSE;
  }

  return muesli_to_javascript(cx,
			      muesli_eval_in_language(language_name,
						      evaluand,
						      strlen(evaluand),
						      ambient_transient),
			      rval);
}

///////////////////////////////
// Custom built-in functions //
///////////////////////////////

static JSBool
javascript_eval_custom(JSContext *cx, JSObject *obj,
		       uintN argc, jsval *argv,
		       jsval *rval)
{
  char *evaluand = NULL;

  if (!JS_ConvertArguments(cx, argc, argv, "s", &evaluand)) {
    return JS_FALSE;
  }

  return muesli_to_javascript(cx,
			      ((javascript_interface->eval_string)
			       (javascript_interface,
				evaluand, strlen(evaluand),
				ambient_transient)),
			      rval);
}

////////////////
// Initialize //
////////////////

void
javascript_evaluator_init(evaluator_interface *interface)
{
  // Init javascript interface

  // Fill in: initialize your javascript application
  javascript_state *our_javascript_state = (javascript_state*)malloc(sizeof(javascript_state));

  our_javascript_state->rt = JS_NewRuntime(0x100000);
  our_javascript_state->cx = JS_NewContext(our_javascript_state->rt, 0x1000);
  our_javascript_state->global = JS_NewObject(our_javascript_state->cx, &global_class, NULL, NULL);
  JS_InitStandardClasses(our_javascript_state->cx, our_javascript_state->global);

  interface->state = our_javascript_state;
  javascript_interface = interface;

  javascript_interface->version = muesli_malloc_copy_string(JS_VersionToString(JS_GetVersion(our_javascript_state->cx)));

  // Extend system

  JS_DefineFunction(our_javascript_state->cx, our_javascript_state->global, (char*)"GetParameter", javascript_get_parameter, 1, 0);
  JS_DefineFunction(our_javascript_state->cx, our_javascript_state->global, (char*)"SetParameter", javascript_set_parameter, 2, 0);
#if 0
  JS_DefineFunction(our_javascript_state->cx, our_javascript_state->global, (char*)"Parameters", javascript_get_parameters, 0, 0);
  JS_DefineFunction(our_javascript_state->cx, our_javascript_state->global, (char*)"ModifyParameters", javascript_set_parameters, 1, 0);
#endif
  JS_DefineFunction(our_javascript_state->cx, our_javascript_state->global, (char*)"Custom", javascript_eval_custom, 1, 0);
  JS_DefineFunction(our_javascript_state->cx, our_javascript_state->global, (char*)"EvalInLanguage", javascript_eval_in_language, 2, 0);

}

static void
javascript_load_file(evaluator_interface *interface,
		const char *filename)
{
  int muesli_flags = interface->flags;
  if (muesli_flags & TRACE_MUESLI_LOAD) {
    fprintf(stderr, "Loading %s\n", filename);
  }
  // Fill in: load the functions file given as (char*)(filename)
  if (muesli_flags & TRACE_MUESLI_LOAD) {
    fprintf(stderr, "Loaded %s\n", filename);
  }
}

static muesli_value_t
javascript_eval_string(evaluator_interface *evaluator,
		       const char *javascript_c_string,
		       unsigned int string_length,
		       int transient)
{
  muesli_value_t result;
  ANULL_VALUE(result);

  if (javascript_c_string) {
    javascript_state *our_state = (javascript_state*)(evaluator->state);
    int old_ambient_transient = ambient_transient;
    int ok;
    jsval rval;
    ambient_transient = transient;

    // fprintf(stderr, "Javascript evaluating string \"%s\"\n", javascript_c_string);

    ok = JS_EvaluateScript(our_state->cx, our_state->global,
			   javascript_c_string, strlen(javascript_c_string),
			   "muesli API", 0, &rval);

    javascript_to_muesli(our_state, &rval, &result);

    ambient_transient = old_ambient_transient;
  }

  return result;
}

void
javascript_setup(evaluator_interface *new_javascript_interface)
{
  javascript_interface = new_javascript_interface;

  javascript_interface->eval_string = javascript_eval_string;
  javascript_interface->load_file = javascript_load_file;

#if 0
  /* We can't actually do this, the type signature is wrong */
  javascript_interface->to_muesli_value = javascript_to_muesli;
  javascript_interface->from_muesli_value = muesli_to_javascript;
#endif
  javascript_interface->version = NULL;
}

#endif
#endif

