// muesli-prolog.c -*- C -*-
/* Interface to prolog evaluators
   Copyright (C) 2008, 2009, 2010 University of Limerick

   This file is part of Muesli.

   Muesli is free software: you can redistribute it and/or modify it under
   the terms of the GNU General Public License as published by the Free
   Software Foundation, either version 3 of the License, or (at your
   option) any later version.

   Muesli is distributed in the hope that it will be useful, but WITHOUT
   ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
   FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
   for more details.

   You should have received a copy of the GNU General Public License
   along with Muesli.  If not, see <http://www.gnu.org/licenses/>.
*/

#ifndef _MUESLI_PROLOG_
#define _MUESLI_PROLOG_

#include "../config.h"
#include "muesli.h"

#ifdef HAVE_LIBPL

#include <stdio.h>
#include <limits.h>
#include <unistd.h>
#include <string.h>
#include <stdlib.h>
#include <getopt.h>		// todo: do I need this?

#include <SWI-Prolog.h>

static int ambient_transient = 0;

#if 0

prolog_state *prolog_create_prolog() { return NULL; }

// Getting args from your scripting language stack:
int prolog_count_args(prolog_state *S) { return 0; }
int prolog_number_arg(int argi) { return 0; }
char *prolog_string_arg(int argi) { return (char*)""; }
int prolog_arg_is_string(int argi) { return 0; }
int prolog_arg_is_number(int argi) { return 0; }

// Putting results back onto your scripting language stack:
void prolog_return_string(const char *string_back) {}
void prolog_return_number(float number_back) {}
void prolog_return_integer(int number_back) {}
void prolog_return_boolean(int int_back) {}
void prolog_return_void(prolog_state *S) {};

// Handling table / map / alist structures of your scripting language:
prolog_table *prolog_create_table(int size) { return NULL; }
void prolog_set_table_string(prolog_table *T, char *key, const char *value) {}
void prolog_set_table_number(prolog_table *T, char *key, float value) {}
void prolog_set_table_boolean(prolog_table *T, char *key, int value) {}

// Make a name / symbol of the scripting language:
prolog_symbol *prolog_make_symbol(const char *n) { return NULL; }

// Non-local exits in the scripting language
void prolog_throw(prolog_symbol *Y, float n) {}

// Set a global variable of the scripting language to contain a table:
void prolog_set_global_variable_table(prolog_symbol *varname, prolog_table *table) {}

// Extend the scripting language with an added built-in function:
void prolog_register_builtin_function(const char *funname, void (*function)(prolog_state *FS)) {}
#endif

// A specialized non-local exit; could either throw, or just quit the
// application run:
void prolog_error(char *message, char *arg)
{
  fprintf(stderr, "Got error \"%s\", \"%s\"\n", message, arg);
}

static evaluator_interface *prolog_interface = NULL;

static char *option_names_var_name = (char*)"option_names";

#if 0
static void
prolog_set_parameter(prolog_state *cs)
{
  if (prolog_count_args(cs) < 2) {
    fprintf(stderr, "too few args to prolog_set_parameter\n");
    prolog_error(cs, (char*)"argcount", (char*)"set_parameter");
  }

  char option_code = muesli_find_option_letter(prolog_interface->getopt_options, prolog_string_arg(cs, 0));

  if (option_code != -1) {
    if (prolog_arg_is_string(cs, 1)) {
      (prolog_interface->handle_option)(prolog_interface->app_params,
					option_code,
					muesli_malloc_copy_string(prolog_string_arg(cs, 1)),
					0.0, 1,
					"prolog");
    } else if (prolog_arg_is_number(cs, 1)) {
      (prolog_interface->handle_option)(prolog_interface->app_params,
					option_code, NULL, prolog_number_arg(cs, 2), 1,
					"prolog");
    } else {
      (prolog_interface->handle_option)(prolog_interface->app_params,
					option_code, (char*)"true", 0.0, 1,
					"prolog");
    }
  }
}

static void
prolog_set_parameters(prolog_state *cs)
{
  if (prolog_count_args(cs) < 1)  {
    fprintf(stderr, "too few args to prolog_set_parameters\n");
    prolog_error(cs, (char*)"argcount", (char*)"set_parameters");
  }

#if 0
  // Fill in: use a table iterator from your language
  prolog_table table = prolog_table_arg(cs, 0);

  prolog_table_iteration_start(cs, table);
  while (prolog_table_iteration_next(cs, table) != 0) {
    prolog_set_parameter(cs,
			 prolog_table_iteration_current_key(cs, table),
			 prolog_table_iteration_current_value(cs, table));
  }

#endif
}

static void
prolog_get_parameter(prolog_state *cs)
{
  if (prolog_count_args(cs) < 1)  {
    fprintf(stderr, "too few args to prolog_get_parameter\n");
    prolog_error(cs, (char*)"argcount", (char*)"get_parameter");
  }

  char option_code = muesli_find_option_letter(prolog_interface->getopt_options,
					       prolog_string_arg(cs, 0));

  if (option_code != -1) {

muesli_value_t result = (prolog_interface->handle_option)(prolog_interface->app_params,
							  option_code,	// option
							  NULL, 0.0,	// value
							  0,	// set
							  "prolog");

    switch (result.type) {
    case muesli_value_string:
      prolog_return_string(cs, result.data.as_string);
      break;
    case muesli_value_float:
      prolog_return_number(cs, result.data.as_float);
      break;
    case muesli_value_integer:
      prolog_return_integer(cs, result.data.as_int);
      break;
    case muesli_value_boolean:
      prolog_return_boolean(cs, result.data.as_int);
      break;
    default:
      prolog_return_boolean(cs, 0);
      break;
    }
  }
}

static void
prolog_get_parameters(prolog_state *cs)
{
  prolog_table *table = prolog_create_table(cs, 48);

  // todo: fix and re-instate -- I have to get long_options across to it somehow
  struct option *option_names = prolog_interface->getopt_options;

  while (option_names->name != 0) {

    muesli_value_t result = (prolog_interface->handle_option)(prolog_interface->app_params,
							      (option_names->val), // opt
							      NULL, 0.0, // value
							      0, // set
							      "prolog");

    switch (result.type) {
    case muesli_value_string:
      prolog_set_table_string(cs, table, (char*)(option_names->name), result.data.as_string);
      break;
    case muesli_value_float:
      prolog_set_table_number(cs, table, (char*)(option_names->name), result.data.as_float);
      break;
    case muesli_value_boolean:
      prolog_set_table_boolean(cs, table, (char*)(option_names->name), result.data.as_int);
      break;
    default:
      prolog_set_table_boolean(cs, table, (char*)(option_names->name), 0);
      break;
    }
    option_names++;
  }
}
#endif

///////////////////////////////////////
// Call arbitrary evaluators by name //
///////////////////////////////////////

#if 0
static int
muesli_prolog_eval_in_language(prolog_state *cs)
{
  if (prolog_count_args(cs) < 2) {
    fprintf(stderr, "too few args to prolog_eval_in_language\n");
    prolog_error(cs, (char*)"argcount", (char*)"eval_in_language");
    return 0;
  }

  if ((prolog_arg_is_string(cs, 1)) && (prolog_arg_is_string(cs, 2))) {
    const char *language_name = prolog_string_arg(cs, 1);
    const char *evaluand = prolog_string_arg(cs, 2);
    unsigned int evaluand_length = strlen(evaluand);

    fprintf(stderr, "In prolog_eval_in_language(\"%s\", \"%s\")\n", language_name, evaluand);

    muesli_value_t result = muesli_eval_in_language(language_name,
						    evaluand,
						    evaluand_length,
						    ambient_transient);

    switch (result.type) {
    case muesli_value_float:
      prolog_return_number(cs, result.data.as_float);
      break;
    case muesli_value_integer:
      prolog_return_integer(cs, result.data.as_int);
      break;
    case muesli_value_string:
      prolog_return_string(cs, result.data.as_string);
      break;
    case muesli_value_boolean:
      prolog_return_boolean(cs, result.data.as_bool);
      break;
    case muesli_value_none:
    case muesli_value_error:
      prolog_return_boolean(cs, 0);
      break;
    }
  } else {
    fprintf(stderr, "wrong type args to prolog_eval_in_language\n");
    prolog_error(cs, (char*)"argtype", (char*)"eval_in_language");
    return 0;
  }
  return 1;
}
#endif

///////////////////////////////
// Custom built-in functions //
///////////////////////////////

#if 0
static void
prolog_eval_custom_function(prolog_state *cs)
{
  if (prolog_arg_is_string(cs, 0)) {
    char *string_arg = prolog_string_arg(cs, 0);
    muesli_value_t result = (prolog_interface->eval_string)(prolog_interface,
							    string_arg, strlen(string_arg),
							    ambient_transient);
    switch (result.type) {
    case muesli_value_float:
      prolog_return_number(cs, result.data.as_float);
      break;
    case muesli_value_integer:
      prolog_return_integer(cs, result.data.as_int);
      break;
    case muesli_value_string:
      prolog_return_string(cs, result.data.as_string);
      break;
    case muesli_value_boolean:
      prolog_return_boolean(cs, result.data.as_int);
      break;
    case muesli_value_none:
    case muesli_value_error:
      prolog_return_void(cs);
      break;
    }
  } else {
    fprintf(stderr, "prolog_eval_function must be given a string\n");
    prolog_error(cs, (char*)"argtype", (char*)"prolog_eval");
  }
}
#endif

////////////////
// Initialize //
////////////////

static const PL_extension predicates[] =
{
/*{ "name",	arity,  function,	PL_FA_<flags> },*/

#if 0
  // Fill in: register all these language extensions (as applicable):
  Muesli_Add_Fn_1(interface, (char*)"get_parameter", prolog_get_parameter);
  Muesli_Add_Fn_2(interface, (char*)"set_parameter", prolog_set_parameter);
  Muesli_Add_Fn_0(interface, (char*)"evolution_parameters", prolog_get_parameters);
  Muesli_Add_Fn_1(interface, (char*)"modify_evolution_parameters", prolog_set_parameters);
  Muesli_Add_Fn_1(interface, (char*)"prolog", prolog_eval_function);
  Muesli_Add_Fn_1(interface, (char*)"eval_in_language", muesli_prolog_eval_in_language);
#endif

  { NULL,	0, 	NULL,		0 }	/* terminating line */
};

void
prolog_evaluator_init(evaluator_interface *interface)
{

  char *pl_argv[] = {
    "muesli",
    NULL
  };
  int pl_argc = 1;

  // Init prolog interface

  PL_register_extensions(predicates);	/* This is the only PL_ call allowed */
					/* before PL_initialise().  It */
					/* ensures the foreign predicates */
					/* are available before loading */
					/* Prolog code */

  if (!PL_initialise(pl_argc, pl_argv) )
    PL_halt(1);

  prolog_interface = interface;

#if 0
  // Set up option names

  // Fill in: Create a table of option names.  You may well not need to bother.
  prolog_table *option_names_table = prolog_create_table(our_prolog_state, 48);
  struct option *option_names;
  for (option_names = prolog_interface->getopt_options;
       (option_names != NULL) && (option_names->name != 0);
       option_names++) {
    prolog_set_table_number(our_prolog_state, option_names_table,
			    (char*)(option_names->name), (option_names->val));
  }
  prolog_set_global_variable_table(our_prolog_state,
				   prolog_make_symbol(our_prolog_state,
						      option_names_var_name),
				   option_names_table);
#endif
}

static void
prolog_load_file(evaluator_interface *interface,
		const char *filename)
{
  int muesli_flags = interface->flags;
  int buff_len = strlen((char*)"consult([''])") + strlen(filename) + 1;
  char *buff = (char*)alloca(buff_len);

  if (muesli_flags & TRACE_MUESLI_LOAD) {
    fprintf(stderr, "Loading %s\n", filename);
  }

  snprintf(buff, buff_len, "consult(['%s'])", filename);

  /* todo: run this as a command (query) */


















  /*
    Or look in the sources to find how this is implemented:
    -s file

    Load file as a script.  This option may be used from the shell to make
    Prolog load a file before entering the toplevel.  It is also used to
    turn a file into an executable Prolog script on Unix systems using the
    following first line

    #!/usr/local/bin/pl option ... -s
  */



  if (muesli_flags & TRACE_MUESLI_LOAD) {
    fprintf(stderr, "Loaded %s\n", filename);
  }
}

static muesli_value_t
prolog_eval_string(evaluator_interface *evaluator,
		   const char *prolog_c_string,
		   unsigned int string_length,
		   int transient)
{
  muesli_value_t result;
  ANULL_VALUE(result);

  if (prolog_c_string) {






    /*
      Alternatively, look in the sources to find how this is
      implemented:

       -g goal

              Goal is executed just before entering the top level. The
              default is a predicate which prints the welcome
              message. The welcome message can thus be suppressed by
              giving -g true.  goal can be a complex term. In this
              case, quotes are normally needed to protect it from
              being expanded by the Unix shell.
    */









    int old_ambient_transient = ambient_transient;
    fid_t fid = PL_open_foreign_frame();
    term_t goal  = PL_new_term_ref();
    term_t a1    = PL_new_term_ref();
    term_t a2    = PL_new_term_ref();
    functor_t s2 = PL_new_functor(PL_new_atom("statistics"), 2);

    ambient_transient = transient;
    // fprintf(stderr, "Prolog evaluating string \"%s\"\n", prolog_c_string);


    /* todo: fill in the prolog data structures */


















    PL_put_atom_chars(a1, "atoms");
    PL_cons_functor(goal, s2, a1, a2);
    PL_call(goal, NULL);         /* call it in current module */


    {
      int raw_result_type = PL_term_type(a2);
      switch (raw_result_type) {
      case PL_INTEGER:
	{
	  int number;
	  PL_get_integer(a2, &number);
	  result.data.as_int = number;
	  result.type = muesli_value_integer;
	}
	break;
      case PL_FLOAT:
	{
	  double number;
	  PL_get_float(a2, &number);
	  result.data.as_float = number;
	  result.type = muesli_value_float;
	}	break;
      case PL_ATOM:
	{
	  char *chars;
	  PL_get_atom_chars(a2, &chars);
	  result.data.as_string = muesli_malloc_copy_string(chars);
	  result.type = muesli_value_string;
	}
	break;
      case PL_STRING:
	{
	  char *chars;
	  unsigned int size;
	  PL_get_string_chars(a2, &chars, &size);
	  result.data.as_string = muesli_malloc_copy_string(chars);
	  result.type = muesli_value_string;
	}
	break;
      default:
	{
	  char *chars;
	  PL_get_chars(a2, &chars, CVT_ALL);
	  result.data.as_string = muesli_malloc_copy_string(chars);
	  result.type = muesli_value_string;
	}
      }
    }

    PL_close_foreign_frame(fid);

    ambient_transient = old_ambient_transient;
  }

  return result;
}

void
prolog_setup(evaluator_interface *new_prolog_interface)
{
#define VERSION_STRING_MAX 128
  char *prolog_version_string = malloc(VERSION_STRING_MAX);
  int pl_major_version = PLVERSION / 10000;
  int pl_minor_version = (PLVERSION % 10000) / 100;
  int pl_patch = PLVERSION % 100;

  snprintf(prolog_version_string, VERSION_STRING_MAX, "%d.%d.%d",
	   pl_major_version, pl_minor_version, pl_patch);

  prolog_interface = new_prolog_interface;

  prolog_interface->eval_string = prolog_eval_string;
  prolog_interface->load_file = prolog_load_file;

  prolog_interface->version = prolog_version_string;
}

#endif
#endif

