// muesli-R.c -*- C -*-
/* Interface to R evaluators / template for new language interfaces
   Copyright (C) 2008, 2009, 2010 University of Limerick

   This file is part of Muesli.

   Muesli is free software: you can redistribute it and/or modify it under
   the terms of the GNU General Public License as published by the Free
   Software Foundation, either version 3 of the License, or (at your
   option) any later version.

   Muesli is distributed in the hope that it will be useful, but WITHOUT
   ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
   FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
   for more details.

   You should have received a copy of the GNU General Public License
   along with Muesli.  If not, see <http://www.gnu.org/licenses/>.
*/

#ifndef _MUESLI_R_
#define _MUESLI_R_

#include "../config.h"
#include "muesli.h"
#include "muesli-internal.h"

#ifdef HAVE_LIBR

#include <stdio.h>
#include <limits.h>
#include <unistd.h>
#include <string.h>
#include <stdlib.h>
#include <getopt.h>		// todo: do I need this?

#include <R.h>
#include <Rinternals.h>
#include <Rversion.h>
#include <R_ext/Parse.h>
#include <Rembedded.h>

static int ambient_transient = 0;

static evaluator_interface *R_interface = NULL;

static char *option_names_var_name = (char*)"option_names";

muesli_value_t
R_to_muesli(SEXP Rval)
{
  muesli_value_t mval;
  ANULL_VALUE(mval);

  switch (TYPEOF(Rval)) {
  case NILSXP: /* nil = NULL */
    mval.data.as_bool = 0;
    mval.type = muesli_value_boolean;
    break;

  case SYMSXP: /* symbols */
    {
      const char *raw_string = CHAR(PRINTNAME(Rval));
      char *our_string = (char*)malloc(strlen(raw_string)+1);
      strcpy(our_string, raw_string);
      mval.data.as_string = our_string;
      mval.type = muesli_value_string;
    }
    break;

  case CHARSXP: /* "scalar" string type (internal only)*/
    {
      const char *raw_string = CHAR(Rval);
      char *our_string = (char*)malloc(strlen(raw_string)+1);
      strcpy(our_string, raw_string);
      mval.data.as_string = our_string;
      mval.type = muesli_value_string;
    }
    break;

  case LGLSXP: /* logical vectors */
    mval.data.as_bool = asLogical(Rval);
    mval.type = muesli_value_boolean;
    break;

  case INTSXP: /* integer vectors */
    mval.data.as_int = asInteger(Rval);
    mval.type = muesli_value_integer;
    break;

  case REALSXP: /* real variables */
    mval.data.as_float = asReal(Rval);
    mval.type = muesli_value_float;
    break;
  }

  return mval;
}

#if 0
static void
R_set_parameter(R_state *cs)
{
  if (R_count_args(cs) < 2) {
    fprintf(stderr, "too few args to R_set_parameter\n");
    R_error(cs, (char*)"argcount", (char*)"set_parameter");
  }

  char option_code = muesli_find_option_letter(R_interface->getopt_options, R_string_arg(cs, 0));

  if (option_code != -1) {
    if (R_arg_is_string(cs, 1)) {
      (R_interface->handle_option)(R_interface->app_params,
					option_code,
					muesli_malloc_copy_string(R_string_arg(cs, 1)),
					0.0, 1,
					"R");
    } else if (R_arg_is_number(cs, 1)) {
      (R_interface->handle_option)(R_interface->app_params,
					option_code, NULL, R_number_arg(cs, 2), 1,
					"R");
    } else {
      (R_interface->handle_option)(R_interface->app_params,
					option_code, (char*)"true", 0.0, 1,
					"R");
    }
  }
}

static void
R_set_parameters(R_state *cs)
{
  if (R_count_args(cs) < 1)  {
    fprintf(stderr, "too few args to R_set_parameters\n");
    R_error(cs, (char*)"argcount", (char*)"set_parameters");
  }

#if 0
  // Fill in: use a table iterator from your language
  R_table table = R_table_arg(cs, 0);

  R_table_iteration_start(cs, table);
  while (R_table_iteration_next(cs, table) != 0) {
    R_set_parameter(cs,
			 R_table_iteration_current_key(cs, table),
			 R_table_iteration_current_value(cs, table));
  }

#endif
}

static void
R_get_parameter(R_state *cs)
{
  if (R_count_args(cs) < 1)  {
    fprintf(stderr, "too few args to R_get_parameter\n");
    R_error(cs, (char*)"argcount", (char*)"get_parameter");
  }

  char option_code = muesli_find_option_letter(R_interface->getopt_options,
					       R_string_arg(cs, 0));

  if (option_code != -1) {

muesli_value_t result = (R_interface->handle_option)(R_interface->app_params,
							  option_code,	// option
							  NULL, 0.0,	// value
							  0,	// set
							  "R");

    switch (result.type) {
    case muesli_value_string:
      R_return_string(cs, result.data.as_string);
      break;
    case muesli_value_float:
      R_return_number(cs, result.data.as_float);
      break;
    case muesli_value_integer:
      R_return_integer(cs, result.data.as_int);
      break;
    case muesli_value_boolean:
      R_return_boolean(cs, result.data.as_int);
      break;
    default:
      R_return_boolean(cs, 0);
      break;
    }
  }
}

static void
R_get_parameters(R_state *cs)
{
  R_table *table = R_create_table(cs, 48);

  // todo: fix and re-instate -- I have to get long_options across to it somehow
  struct option *option_names = R_interface->getopt_options;

  while (option_names->name != 0) {

    muesli_value_t result = (R_interface->handle_option)(R_interface->app_params,
							      (option_names->val), // opt
							      NULL, 0.0, // value
							      0, // set
							      "R");

    switch (result.type) {
    case muesli_value_string:
      R_set_table_string(cs, table, (char*)(option_names->name), result.data.as_string);
      break;
    case muesli_value_float:
      R_set_table_number(cs, table, (char*)(option_names->name), result.data.as_float);
      break;
    case muesli_value_boolean:
      R_set_table_boolean(cs, table, (char*)(option_names->name), result.data.as_int);
      break;
    default:
      R_set_table_boolean(cs, table, (char*)(option_names->name), 0);
      break;
    }
    option_names++;
  }
}
#endif

///////////////////////////////////////
// Call arbitrary evaluators by name //
///////////////////////////////////////

#if 0
static int
muesli_R_eval_in_language(R_state *cs)
{
  if (R_count_args(cs) < 2) {
    fprintf(stderr, "too few args to R_eval_in_language\n");
    R_error(cs, (char*)"argcount", (char*)"eval_in_language");
    return 0;
  }

  if ((R_arg_is_string(cs, 1)) && (R_arg_is_string(cs, 2))) {
    const char *language_name = R_string_arg(cs, 1);
    const char *evaluand = R_string_arg(cs, 2);
    unsigned int evaluand_length = strlen(evaluand);

    fprintf(stderr, "In R_eval_in_language(\"%s\", \"%s\")\n", language_name, evaluand);

    muesli_value_t result = muesli_eval_in_language(language_name,
						    evaluand,
						    evaluand_length,
						    ambient_transient);

    switch (result.type) {
    case muesli_value_float:
      R_return_number(cs, result.data.as_float);
      break;
    case muesli_value_integer:
      R_return_integer(cs, result.data.as_int);
      break;
    case muesli_value_string:
      R_return_string(cs, result.data.as_string);
      break;
    case muesli_value_boolean:
      R_return_boolean(cs, result.data.as_bool);
      break;
    case muesli_value_none:
    case muesli_value_error:
      R_return_boolean(cs, 0);
      break;
    }
  } else {
    fprintf(stderr, "wrong type args to R_eval_in_language\n");
    R_error(cs, (char*)"argtype", (char*)"eval_in_language");
    return 0;
  }
  return 1;
}
#endif

///////////////////////////////
// Custom built-in functions //
///////////////////////////////

#if 0
static void
R_eval_function(R_state *cs)
{
  if (R_arg_is_string(cs, 0)) {
    char *string_arg = R_string_arg(cs, 0);
    muesli_value_t result = (R_interface->eval_string)(R_interface,
							    string_arg, strlen(string_arg),
							    ambient_transient);
    switch (result.type) {
    case muesli_value_float:
      R_return_number(cs, result.data.as_float);
      break;
    case muesli_value_integer:
      R_return_integer(cs, result.data.as_int);
      break;
    case muesli_value_string:
      R_return_string(cs, result.data.as_string);
      break;
    case muesli_value_boolean:
      R_return_boolean(cs, result.data.as_int);
      break;
    case muesli_value_none:
    case muesli_value_error:
      R_return_void(cs);
      break;
    }
  } else {
    fprintf(stderr, "R_eval_function must be given a string\n");
    R_error(cs, (char*)"argtype", (char*)"R_eval");
  }
}
#endif

////////////////
// Initialize //
////////////////

void
R_evaluator_init(evaluator_interface *interface)
{
  // Init R interface

  char *rargv[2] = {(char*)"muesli", NULL};
  int rargc = 1;

  R_interface = interface;

  Rf_initEmbeddedR(rargc, rargv);

#if 0
  // Extend system

  DllInfo *info = R_getEmbeddingDllInfo();
  R_registerRoutines(info, cMethods, callMethods, NULL, NULL);

  // Fill in: register all these language extensions (as applicable):
  if (muesli_extensions & MUESLI_EXTENSION_SINGLE_PARAMETER) {
    Muesli_Add_Fn_1(interface, (char*)"get_parameter", R_get_parameter);
    Muesli_Add_Fn_2(interface, (char*)"set_parameter", R_set_parameter);
  }
  if (muesli_extensions & MUESLI_EXTENSION_PARAMETER_BLOCK) {
    Muesli_Add_Fn_0(interface, (char*)"parameters", R_get_parameters);
    Muesli_Add_Fn_1(interface, (char*)"modify_parameters", R_set_parameters);
  }
  if (muesli_extensions & MUESLI_EXTENSION_EVAL_IN_LANGUAGE) {
    Muesli_Add_Fn_1(interface, (char*)"eval_in_language", muesli_R_eval_in_language);
  }
  if (muesli_extensions & MUESLI_EXTENSION_CUSTOM) {
    Muesli_Add_Fn_1(interface, (char*)"custom", R_eval_function);
  }
#endif

#if 0
  // Set up option names

  // Fill in: Create a table of option names.  You may well not need to bother.
  R_table *option_names_table = R_create_table(our_R_state, 48);
  struct option *option_names;
  for (option_names = R_interface->getopt_options;
       (option_names != NULL) && (option_names->name != 0);
       option_names++) {
    R_set_table_number(our_R_state, option_names_table,
		       (char*)(option_names->name), (option_names->val));
  }
  R_set_global_variable_table(our_R_state,
			      R_make_symbol(our_R_state,
					    option_names_var_name),
			      option_names_table);
#endif
}

void
R_evaluator_close(evaluator_interface *interface)
{
    Rf_endEmbeddedR(0);
}

static void
R_load_file(evaluator_interface *interface,
	    const char *filename)
{
  int muesli_flags = interface->flags;
  if (muesli_flags & TRACE_MUESLI_LOAD) {
    fprintf(stderr, "Loading %s\n", filename);
  }
  // Fill in: load the functions file given as (char*)(filename)
  if (muesli_flags & TRACE_MUESLI_LOAD) {
    fprintf(stderr, "Loaded %s\n", filename);
  }
}

static muesli_value_t
R_eval_string(evaluator_interface *evaluator,
		   const char *R_c_string,
		   unsigned int string_length,
		   int transient)
{
  muesli_value_t result;
  ANULL_VALUE(result);

  if (R_c_string) {
    int old_ambient_transient = ambient_transient;
    SEXP e, tmp, raw_result;
    int hadError = 0;
    ParseStatus status;
    ambient_transient = transient;

    PROTECT(tmp = mkString(R_c_string));
    PROTECT(e = R_ParseVector(tmp, 1, &status, R_NilValue));
    raw_result = R_tryEval(VECTOR_ELT(e,0), R_GlobalEnv, &hadError);
    UNPROTECT(2);

    if (!hadError) {
      result = R_to_muesli(raw_result);
    }

    ambient_transient = old_ambient_transient;
  }

  return result;
}

void
R_setup(evaluator_interface *new_R_interface)
{
  char *R_version_string = (char*)malloc(strlen(R_MAJOR) + strlen(R_MINOR) + 2);
  R_interface = new_R_interface;

  R_interface->eval_string = R_eval_string;
  R_interface->load_file = R_load_file;

  R_interface->close_evaluator = R_evaluator_close;

  sprintf(R_version_string, "%s.%s", R_MAJOR, R_MINOR);
  R_interface->version = R_version_string;
}

#endif
#endif

