// muesli-guile.c -*- C -*-
/* muesli interface to GUILE
   Copyright (C) 2008, 2009 University of Limerick

   This file is part of Muesli.
   
   Muesli is free software: you can redistribute it and/or modify it under
   the terms of the GNU General Public License as published by the Free
   Software Foundation, either version 3 of the License, or (at your
   option) any later version.
   
   Muesli is distributed in the hope that it will be useful, but WITHOUT
   ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
   FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
   for more details.
   
   You should have received a copy of the GNU General Public License
   along with Muesli.  If not, see <http://www.gnu.org/licenses/>.
*/

#ifndef _MUESLI_GUILE_
#define _MUESLI_GUILE_

#include "../config.h"

#ifdef HAVE_LIBGUILE

#include "muesli.h"
#include "muesli-internal.h"
// #include "lang-extns.h"

#include <limits.h>
#include <getopt.h>

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <alloca.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <errno.h>
#include <fcntl.h>
#include <math.h>

// #include "lang-extns-cpp.h"

#include <libguile.h>

// Forward declaration (to keep normal order of functions but make
// this available to traced file loading):
static muesli_value_t
muesli_guile_eval_string(evaluator_interface *interface,
			const char *scratch,
			unsigned int string_length,
			int transient);

// Interface from the rest of Muesli, set each time Muesli calls us
static evaluator_interface *current_interface = NULL;

static int ambient_transient = 0;

static SCM option_names_table = SCM_EOL;

#define NUMBER_BUF_SIZE 64
static char number_buf[NUMBER_BUF_SIZE];

/********************/
/* Value conversion */
/********************/

static muesli_value_t
guile_to_muesli(SCM guile_value)
{
  muesli_value_t muesli_value;
  ANULL_VALUE(muesli_value);

  if (scm_is_bool(guile_value)) {
    muesli_value.data.as_bool = scm_is_true(guile_value) ? 1 : 0;
    muesli_value.type = muesli_value_boolean;
  } else if (scm_is_integer(guile_value)) {
    muesli_value.data.as_int = scm_to_int(guile_value);
    muesli_value.type = muesli_value_integer;
  } else if (scm_is_real(guile_value)) {
    muesli_value.data.as_float = (float)scm_to_double(guile_value);
    muesli_value.type = muesli_value_float; 
  } else if (scm_is_rational(guile_value)) {
    muesli_value.data.as_float =
      ((float)scm_to_double(scm_numerator(guile_value))
       / (float)scm_to_double(scm_denominator(guile_value)));
    muesli_value.type = muesli_value_float; 
  } else if (scm_is_symbol(guile_value)) {
    muesli_value.data.as_string = scm_to_locale_string(scm_symbol_to_string(guile_value));
    muesli_value.type = muesli_value_string;
  } else if (scm_is_string(guile_value)) {
    muesli_value.data.as_string = scm_to_locale_string(guile_value);
    muesli_value.type = muesli_value_string;
  }

  return muesli_value;
}

static SCM
muesli_to_guile(muesli_value_t muesli_value)
{
  switch (muesli_value.type) {
  case muesli_value_string:
    if (muesli_value.data.as_string != NULL) {
      return scm_from_locale_stringn(muesli_value.data.as_string,
				     strlen(muesli_value.data.as_string));
    } else {
      return SCM_UNDEFINED;
    }
  case muesli_value_float:
    return scm_from_double(muesli_value.data.as_float);
  case muesli_value_integer:
    return scm_from_int(muesli_value.data.as_int);
  case muesli_value_boolean:
    return scm_from_bool(muesli_value.data.as_bool);
  default:
    return SCM_UNDEFINED;
  }
}

/**************/
/* Parameters */
/**************/

static SCM
muesli_guile_set_parameter(SCM option, SCM value)
{
  int ok = 0;

  char *option_name = scm_to_locale_string(option);

  if (scm_is_string(value)) {
    char *option_value = scm_to_locale_string(value);
    ok = muesli_set_parameter(current_interface,
			      current_interface->getopt_options,
			      option_name, option_value,
			      (const char*)"guile");
    free(option_value);
  } else if (scm_is_real(value)) {
    snprintf(number_buf, NUMBER_BUF_SIZE, "%f", (float)scm_to_double(value));
    ok = muesli_set_parameter(current_interface,
			      current_interface->getopt_options,
			      option_name, number_buf,
			      (const char*)"guile");
  } else if (scm_is_integer(value)) {
    snprintf(number_buf, NUMBER_BUF_SIZE, "%d", scm_to_int(value));
    ok = muesli_set_parameter(current_interface,
			      current_interface->getopt_options,
			      option_name, number_buf,
			      (const char*)"guile");
  } else {
    ok = muesli_set_parameter(current_interface,
			      current_interface->getopt_options,
			      option_name, (char*)"true",
			      (const char*)"guile");
  }

  free(option_name);

  return scm_from_bool(ok);
}

static SCM
muesli_guile_set_parameters(SCM changes)
{
  while (!scm_null_p(changes)) {
    SCM change = scm_car(changes);
    if (scm_is_pair(change)) {
      muesli_guile_set_parameter(scm_car(change), scm_cdr(change));
    }
    changes = scm_cdr(changes);
  }
  return SCM_EOL;
}

static SCM
muesli_guile_get_parameter(SCM option)
{
  char *option_name = scm_to_locale_string(option);
  char option_code = muesli_find_option_letter(current_interface->getopt_options, option_name);
  free(option_name);

  if (option_code != -1) {
    return muesli_to_guile((current_interface->handle_option)
			   (current_interface->app_params,
			    option_code,	// option
			    NULL, 0.0,	// value
			    0,	// set
			    (const char *)"guile"));

  }
  return SCM_UNDEFINED;
}

static SCM
muesli_guile_get_parameters()
{
  SCM names = option_names_table;
  SCM results = SCM_EOL;

  while (!scm_null_p(names)) {
    if (scm_is_pair(names)) {
      SCM pair = scm_car(names);
      SCM lisp_result;

      muesli_value_t result = (current_interface->handle_option)(current_interface->app_params,
								 (int)(scm_to_int(scm_cdr(pair))), // opt
								 NULL, 0.0,		     // value
								 0,		     // set
								 (const char *)"guile");

      lisp_result = muesli_to_guile(result);

      if (lisp_result != SCM_EOL) {
	results = scm_cons(scm_cons(scm_car(pair), lisp_result), results);
      }
    }
    names = scm_cdr(names);
  }
  return results;
}

///////////////////////////////////////
// Call arbitrary evaluators by name //
///////////////////////////////////////

static SCM
muesli_guile_eval_in_language(SCM language_name, SCM evaluand)
{
  if ((scm_is_string(language_name) || scm_is_symbol(language_name))
      && (scm_is_string(evaluand) || scm_is_symbol(evaluand))) {
    const char *language = scm_to_locale_string(scm_is_string(language_name)
						? language_name
						: scm_symbol_to_string(language_name)
						);
    const char *scratch = scm_to_locale_string(scm_is_string(evaluand)
					       ? evaluand
					       : scm_symbol_to_string(evaluand));
    unsigned int string_length = strlen(scratch);

    // fprintf(stderr, "In muesli_guile_eval_in_language(\"%s\", \"%s\")\n", language, scratch);

    return muesli_to_guile(muesli_eval_in_language(language,
						   scratch,
						   string_length,
						   ambient_transient));
  } else {
#if 0
    // todo: make this a throw, or something like that
    fprintf(stderr, "Arguments to muesli_guile_eval_in_language must be strings\n");
    exit(EXIT_USER_ERROR);
    return SCM_EOL;
#else
    return evaluand;
#endif
  }
  return SCM_EOL;
}

///////////////////////////////
// Custom built-in functions //
///////////////////////////////

// This function gives GUILE access to `custom_eval_individual' in
// muesli-custom.c, thus letting you write a custom evaluator (in C)
// that can be used from a GUILE program.

static SCM
muesli_guile_custom_eval_function(SCM eval_in)
{
  if (scm_is_string(eval_in) || scm_is_symbol(eval_in)) {
    evaluator_interface *custom_evaluator = muesli_get_named_evaluator("custom", 0);
    if (custom_evaluator == NULL) {
      fprintf(stderr, "custom evaluator not found\n");
      return eval_in;
    } else {
      char *str = scm_to_locale_string(scm_is_string(eval_in)
				       ? eval_in
				       : scm_symbol_to_string(eval_in));

      return muesli_to_guile((custom_evaluator->eval_string)(custom_evaluator,
							     str, strlen(str),
							     ambient_transient));
    }
  } else {
#if 0
    fprintf(stderr, "muesli_guile_custom_eval_function must be given a string\n");
    exit(EXIT_USER_ERROR);
#else
    return eval_in;
#endif
  }
  return eval_in;
}

///////////////////////////////////////////
// Call stack-based bytecode interpreter //
///////////////////////////////////////////

static SCM
muesli_guile_eval_bytecode_string(SCM args)
{
  SCM program = scm_car(args);
  if (scm_is_string(program)) {
    char *str = scm_to_locale_string(program);
    SCM l;
    evaluator_interface *stack_evaluator = muesli_get_named_evaluator("stack-code", 0);

    if (stack_evaluator == NULL) {
      fprintf(stderr, "stack evaluator not found\n");
      return SCM_BOOL_F;
    } else {
      (stack_evaluator->eval_clear)(stack_evaluator);

      for (l = scm_cdr(args); !scm_null_p(l); l = scm_cdr(l)) {
	SCM s = scm_car(l);
	if (scm_is_real(s)) {
	  (stack_evaluator->eval_give)(stack_evaluator, (float)scm_to_double(s));
	} else if (scm_is_integer(s)) {
	  (stack_evaluator->eval_give)(stack_evaluator, (float)scm_to_int(s));
	} else {
	  (stack_evaluator->eval_give)(stack_evaluator, 0.0);
	}
      }

      return scm_from_double((double)((stack_evaluator->eval_given)(stack_evaluator, str, strlen(str))));
    }
  }
  return SCM_EOL;
}

////////////////
// Initialize //
////////////////

static void
guile_evaluator_inner_init(evaluator_interface *interface)
{
  struct option *option_names;
  scm_init_guile();

  current_interface = interface;

  // general functions:
  if (muesli_extensions & MUESLI_EXTENSION_SINGLE_PARAMETER) {
    Muesli_Add_Fn_1(current_interface, (char*)"get-parameter", muesli_guile_get_parameter);
    Muesli_Add_Fn_2(current_interface, (char*)"set-parameter", muesli_guile_set_parameter);
  }
  if (muesli_extensions & MUESLI_EXTENSION_PARAMETER_BLOCK) {
    Muesli_Add_Fn_0(current_interface, (char*)"parameters", muesli_guile_get_parameters);
    Muesli_Add_Fn_1(current_interface, (char*)"modify-parameters", muesli_guile_set_parameters);
  }
  if (muesli_extensions & MUESLI_EXTENSION_EVAL_IN_LANGUAGE) {
    Muesli_Add_Fn_2(current_interface, (char*)"eval-in-language", muesli_guile_eval_in_language);
  }
  if (muesli_extensions & MUESLI_EXTENSION_CUSTOM) {
    Muesli_Add_Fn_1(current_interface, (char*)"custom", muesli_guile_custom_eval_function);
    Muesli_Add_Fn_V(current_interface, (char*)"bytecode", muesli_guile_eval_bytecode_string);
  }

  // set up option names
  for (option_names = current_interface->getopt_options;
       (option_names != NULL) && (option_names->name != 0);
       option_names++) {
    option_names_table = scm_cons(scm_cons(scm_from_locale_symbol((char*)(option_names->name)),
					   scm_from_int(option_names->val)),
				  option_names_table);
  }
  scm_c_define((char*)"option-names", option_names_table);

#if 0
  guile_verbose(guile_cons(guile_flocons(interface->language_verbosity),
			 SCM_BOOL_F));
#endif
}

void
guile_evaluator_init(evaluator_interface *interface)
{
  int muesli_flags = interface->flags;

  if (muesli_flags & TRACE_MUESLI_INIT) {
    fprintf(stderr, "Initializing GUILE\n");
  }

  scm_with_guile((void*)guile_evaluator_inner_init, interface);

  if (muesli_flags & TRACE_MUESLI_INIT) {
    fprintf(stderr, "Initialized GUILE\n");
  }
}

////////////
// Extend //
////////////

static void
guile_add_function(evaluator_interface *interface,
		  int arity, const char *name,
		  void *function)
{
  switch (arity) {
  case ARGS_COUNT_VARIABLE:
    scm_c_define_gsubr(name, 0, 0, 1, function);
    break;
  case ARGS_NO_EVAL:
#if 0
    guile_init_fsubr((char*)name, function);
#else
    fprintf(stderr, "Don't know how to define fsubr in Guile: %s\n", name);
#endif
    break;
  case 0:
    scm_c_define_gsubr(name, 0, 0, 0, function);
    break;
  case 1:
    scm_c_define_gsubr(name, 1, 0, 0, function);
    break;
  case 2:
    scm_c_define_gsubr(name, 2, 0, 0, function);
    break;
  case 3:
    scm_c_define_gsubr(name, 3, 0, 0, function);
    break;
  }
}

static void
guile_load_file(evaluator_interface *interface,
	       const char *filename)
{
  int muesli_flags = interface->flags;

  current_interface = interface;

  
  if (muesli_flags & TRACE_MUESLI_LOAD) {
    fprintf(stderr, "Loading %s\n", filename);
  }

  scm_c_primitive_load(filename);

  if (muesli_flags & TRACE_MUESLI_LOAD) {
    fprintf(stderr, "Loaded %s\n", filename);
  }
  
}

////////////////
// Evaluation //
////////////////

// Currently experimenting: I'd like to use the direct eval, and it
// works a bit... but it all goes horribly wrong at garbage
// collection.  However, the indirect eval (which is slower; it's the
// old eval_individual code) does the garbage collection for us, as it
// uses the REPL which included that.

// #define DIRECT_EVAL 1

#if 0
extern int trace_this_eval_guile;
int trace_this_eval_guile = 0;
#endif

static muesli_value_t
muesli_guile_eval_string(evaluator_interface *interface,
			const char *scratch,
			unsigned int string_length,
			int transient)
{
  int old_ambient_transient = ambient_transient;
  muesli_value_t result;
  
  ambient_transient = transient;
  current_interface = interface;

  result = guile_to_muesli(scm_c_eval_string(scratch));

  ambient_transient = old_ambient_transient;

  return result;
}

void
guile_setup(evaluator_interface *new_guile_interface)
{
  current_interface = new_guile_interface;

  current_interface->eval_string = muesli_guile_eval_string;
  current_interface->add_function = guile_add_function;
  current_interface->load_file = guile_load_file;
  current_interface->array_separator = (char*)" ";

  current_interface->from_muesli_value = muesli_to_guile;
  current_interface->to_muesli_value = guile_to_muesli;

  current_interface->version = scm_to_locale_string(scm_version());
}

#endif
#endif
