/* muesli-perl.c -*- C -*- */
/* muesli interface to Perl
   Copyright (C) 2008, 2009 University of Limerick

   This file is part of Muesli.
   
   Muesli is free software: you can redistribute it and/or modify it under
   the terms of the GNU General Public License as published by the Free
   Software Foundation, either version 3 of the License, or (at your
   option) any later version.
   
   Muesli is distributed in the hope that it will be useful, but WITHOUT
   ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
   FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
   for more details.
   
   You should have received a copy of the GNU General Public License
   along with Muesli.  If not, see <http://www.gnu.org/licenses/>.
*/
#ifndef _MUESLI_PERL_
#define _MUESLI_PERL_

#include "../muesli-config.h"

#ifdef HAVE_LIBPERL

#include "../muesli-config.h"
#include "muesli.h"

#include <limits.h>

#include <unistd.h>
#include <getopt.h>
#include <EXTERN.h>
#include <perl.h>
#include <XSUB.h>

#include "muesli-internal.h"

PerlInterpreter *my_perl;

static int ambient_transient = 0;

void
perl_register_builtin_function(char *funname,
			       void (*function)(PerlInterpreter *FS,
						CV *thing))
{
  newXS(funname, function, "muesli-perl.c");
}

/* Interface from the rest of the world */
static evaluator_interface *perl_interface = NULL;

static char *option_names_var_name = (char*)"option_names";

/*
From the example by John Quillan:

The code for sc_forward_row is shown in Listing 3 and found in
sc_perl.c. Before I describe this code, let me give you a quick
overview of how Perl treats scalars. Each scalar has many pieces of
information, including a double numeric value, a string value and
flags to indicate which parts are valid. For our purposes, the scalar
can hold three types of values: an integer value (IV), a double value
(NV) and a string value (PV). For any scalar value (SV), you can get
to their respective values with the macros SvIV, SvNV and SvPV.

Now, in the Listing 3 code, XS is a Perl macro that defines the
function. dXSARGS sets up some stuff for the rest of XSub, such as the
variable items that contains the number of items passed to Xsub on the
Perl argument stack. If the argument count does not equal 1,
XS_RETURN_IV returns 1 to Perl to indicate an error. Otherwise, the
top element of the Perl argument stack, ST(0), is converted to an
integer value and passed to the forwrow function.

Note that all of the XSub code was generated by hand. Some of this
work can be done with Perl's xsubpp or with a tool called swig, but in
this case, I felt it was simpler to code it myself.

Finally, tell the Perl interpreter about this Xsub with the statement:

newXS("sc_forward_row",sc_forward_row,"sc_perl.c");

The first argument is the name of the subroutine in Perl. The next
argument is the actual C routine (in this case they are the same, but
they don't have to be). The last argument is the file in which the
subroutine is defined, and is used for error messages. I chose to
create all of the newXS functions by parsing my sc_perl.c file with a
Perl script, so that I would not have to do two things every time I
added a new XSub.

XS(sc_forward_row)
{
 dXSARGS;
 if(items != 1) {
  XSRETURN_IV(1);
 }
 forwrow(SvIV(ST(0)));
}
*/

XS(set_parameter_perl)
{
  dXSARGS;
  if(items != 2) {
    XSRETURN_IV(1);
  } else {

    unsigned int name_len, val_len;

    char *option_name = SvPV(ST(0), name_len);
    char *option_value = SvPV(ST(1), val_len);

    char option_code = muesli_find_option_letter(perl_interface->getopt_options,
						 option_name);

    fprintf(stderr, "set_parameter_perl(%s,%s)\n", option_name, option_value);

    if (option_code != -1) {

      (perl_interface->handle_option)(perl_interface->app_params,
				      option_code,
				      muesli_malloc_copy_string(option_value),
				      0.0,
				      1,
				      (char *)"perl");
    }
  }
}

XS(set_parameters_perl)
{
  dXSARGS;
  if(items != 1) {
    XSRETURN_IV(1);
  } else {
#if 1
    SV *sv = ST(0);
    if (SvTYPE(sv) == SVt_PVHV) {
      HV *hv = (HV*)sv;
      HE *entry;
      (void)hv_iterinit(hv);
      fprintf(stderr, "got a SVt_PVHV\n");
      /*SUPPRESS 560*/
      while ((entry = hv_iternext(hv))) {
	I32 klen;
	char *key = hv_iterkey(entry, &klen);
	SV* valuepl = hv_iterval(hv,entry);
	unsigned int vlen;
	char *value = SvPV(valuepl, vlen);
	char option_code = muesli_find_option_letter(perl_interface->getopt_options,
						     key);


	fprintf(stderr, "key %s value %s\n", key, value);
	(perl_interface->handle_option)(perl_interface->app_params,
					option_code,	/* option */
					value, 0.0,	/* value */
					1,		/* set */
					(char *)"perl");
      }
    } else {
      fprintf(stderr, "Not a SVt_PVHV (type %d)\n", SvTYPE(sv));
      /* am in fact getting a SVt_PV when I pass an array in directly */
    }
#endif
  }
}

XS(get_parameter_perl)
{
  dXSARGS;
  if(items != 1) {
    XSRETURN_IV(1);
  } else {

    unsigned int name_len;
    char *option_name = SvPV(ST(0), name_len);
    char option_code = muesli_find_option_letter(perl_interface->getopt_options,
						 option_name);

    fprintf(stderr, "get_parameter_perl(%s)\n", option_name);

    if (option_code != -1) {

      Muesli_Perl_Return((perl_interface->handle_option)(perl_interface->app_params,
							 option_code,	/* option */
							 NULL, 0.0,	/* value */
							 0,             /* set */
							 (char *)"perl"));
    }
  }
}

XS(get_parameters_perl)
{
  dXSARGS;
  if(items != 0) {
    XSRETURN_IV(1);
  } else {
#if 0

    /*
      Possibly useful:
      avhv_store_ent
    */

    perl_table *table = perl_create_table(cs, 48);

    struct option *option_names = perl_interface->getopt_options;

    while (option_names->name != 0) {

      muesli_value_t result = (perl_interface->handle_option)(perl_interface->app_params,
							      (option_names->val), /* opt */
							      NULL, 0.0, /* value */
							      0, /* set */
							      (char *)"perl");

      switch (result.type) {
      case muesli_value_const_string:
	perl_set_table_string(cs, table, (char*)(option_names->name), result.data.as_const_string);
	break;
      case muesli_value_float:
	perl_set_table_number(cs, table, (char*)(option_names->name), result.data.as_float);
	break;
      case muesli_value_boolean:
	perl_set_table_boolean(cs, table, (char*)(option_names->name), result.data.as_int);
	break;
      default:
	perl_set_table_boolean(cs, table, (char*)(option_names->name), 0);
	break;
      }
      option_names++;
    }
#endif
  }
}

/*////////////////////////
// Call other languages //
////////////////////////*/

/*
  This function gives Perl access to `eval_in_language' in muesli.c.
*/

XS(eval_in_language_perl)
{
  dXSARGS;
  if(items != 2) {
    XSRETURN_IV(1);
  } else {
    unsigned int name_len, frag_len;

    char *language_name = SvPV(ST(0), name_len);
    char *code_fragment = SvPV(ST(1), frag_len);

    Muesli_Perl_Return(muesli_eval_in_language(language_name,
					       code_fragment,
					       frag_len,
					       ambient_transient));
    
  }
}

/*/////////////////////////////
// Custom built-in functions //
/////////////////////////////*/

/*
  This function gives Perl access to `custom_eval_individual' in
  muesli-custom.c, thus letting you write a custom evaluator (in C) that
  can be used from a Perl muesli setup program.
*/

XS(custom_eval_function_perl)
{
  dXSARGS;
  if(items != 1) {
    XSRETURN_IV(1);
  } else {
    unsigned int frag_len;
    char *code_fragment = SvPV(ST(0), frag_len);
    evaluator_interface *evaluator = muesli_get_named_evaluator("custom", 0);
    if (evaluator == NULL) {
      fprintf(stderr, "Custom evaluator not found");
      /* todo: some kind of throw? */
      XSRETURN_IV(1);
      return;
    }
    Muesli_Perl_Return((evaluator->eval_string)(evaluator,
						code_fragment,
						frag_len,
						ambient_transient));
  }
}

/*//////////////
// Initialize //
//////////////*/

EXTERN_C void boot_DynaLoader (pTHX_ CV* cv);

static void
xs_init(pTHX)
{
  char *file = "muesli-perl.c";
  dXSUB_SYS;

  newXS("DynaLoader::boot_DynaLoader", boot_DynaLoader, file);

  if (muesli_extensions & MUESLI_EXTENSION_SINGLE_PARAMETER) {
    perl_register_builtin_function((char*)"MuesliPerl::GetParameter", get_parameter_perl);
    perl_register_builtin_function((char*)"MuesliPerl::SetParameter", set_parameter_perl);
  }
#if 1
  if (muesli_extensions & MUESLI_EXTENSION_PARAMETER_BLOCK) {
    perl_register_builtin_function((char*)"MuesliPerl::Parameters", get_parameters_perl);
    perl_register_builtin_function((char*)"MuesliPerl::ModifyParameters", set_parameters_perl);
  }
#endif
  if (muesli_extensions & MUESLI_EXTENSION_EVAL_IN_LANGUAGE) {
    perl_register_builtin_function((char*)"MuesliPerl::EvalInLanguage", eval_in_language_perl);
  }
  if (muesli_extensions & MUESLI_EXTENSION_CUSTOM) {
    perl_register_builtin_function((char*)"MuesliPerl::Custom", custom_eval_function_perl);
  }
}

void
perl_load_file(evaluator_interface *interface,
	       const char *filename)
{
  int muesli_flags = interface->flags;
  Muesli_Enter_Evaluator(interface, 0);
  if (muesli_flags & TRACE_MUESLI_LOAD) {
    fprintf(stderr, "Loading %s\n", filename);
  }

  /* TODO: fill in loading the file */

  if (muesli_flags & TRACE_MUESLI_LOAD) {
    fprintf(stderr, "Loaded %s\n", filename);
  }
  Muesli_Leave_Evaluator();
}

void
perl_evaluator_init(evaluator_interface *interface)
{
  char *embedding[] = { "", "-e", "0" };


  PERL_SYS_INIT3(&interface->app_argc,&interface->app_argv,&interface->app_env);
  my_perl = perl_alloc();
  perl_construct(my_perl);
  PL_exit_flags |= PERL_EXIT_DESTRUCT_END;

  perl_parse(my_perl, xs_init, 3, embedding, NULL);
  perl_run(my_perl);
}

void
perl_shutdown(evaluator_interface *interface)
{
  /* Todo: this crashes in perl_destruct when muesli-demo quits */
  perl_destruct(my_perl);
  perl_free(my_perl);
  PERL_SYS_TERM();
}

static muesli_value_t
perl_eval_string(evaluator_interface *interface,
		 const char *perl_c_string,
		 unsigned int string_length,
		 int transient)
{
  muesli_value_t result;
  ANULL_VALUE(result);

  perl_interface = interface;

  if (perl_c_string) {
    int old_ambient_transient = ambient_transient;
    ambient_transient = transient;

#if 0
    fprintf(stderr, "Evaluating string \"%s\"\n", perl_c_string);
#endif

    SV *val = eval_pv(perl_c_string,
		      0 /* 1 */
		      );
    char *string_back = SvPV_nolen(val);

    ambient_transient = old_ambient_transient;

    if (isdigit(string_back[0])
	|| ((string_back[0] == '-')
	    && isdigit(string_back[1]))) {
      result.data.as_float = SvNV(val);
      result.type = muesli_value_float;
    } else {
      result.data.as_const_string = string_back;
      result.type = muesli_value_const_string;
    }
  }
  return result;
}

void
perl_setup(evaluator_interface *new_perl_interface)
{
  perl_interface = new_perl_interface;

  perl_interface->eval_string = perl_eval_string;
  perl_interface->load_file = perl_load_file;

  perl_interface->close_evaluator = perl_shutdown;
}

#endif
#endif
