/* muesli main header file
   Copyright (C) 2008, 2009 University of Limerick

   This file is part of muesli.
   
   muesli is free software: you can redistribute it and/or modify it under
   the terms of the GNU General Public License as published by the Free
   Software Foundation, either version 3 of the License, or (at your
   option) any later version.
   
   muesli is distributed in the hope that it will be useful, but WITHOUT
   ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
   FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
   for more details.
   
   You should have received a copy of the GNU General Public License
   along with muesli.  If not, see <http://www.gnu.org/licenses/>.
*/

#ifndef MUESLI_INT_H
#define MUESLI_INT_H

#include <stdio.h>
#include <unistd.h>

extern const char *muesli_version;

/* Data type tags for union for dynamic typing */

typedef enum {
  muesli_value_none,
  muesli_value_float,
  muesli_value_integer,
  muesli_value_malloced_string,
  muesli_value_const_string,
  muesli_value_boolean,
  muesli_value_error
} value_type_tag_t;

/* Indication of who owns a compound dynamic data value */

typedef enum {
  private_storage,
  malloced_by_language,
  malloced_by_muesli
} value_storage_t;

/* Dynamic data typing structure */

typedef struct muesli_value_t {
  union {
    float as_float;
    char *as_malloced_string;
    const char *as_const_string;
    int as_int;
    int as_bool;
  } data;
  value_type_tag_t type;
  value_type_tag_t element_type;
  value_storage_t storage;
} muesli_value_t;

#define ANULL_VALUE(_v_) \
  (_v_).type = muesli_value_none; \
  (_v_).element_type = muesli_value_none; \
  (_v_).storage = private_storage;

/* How much tracing to output */

typedef enum {
  silent,
  quiet,
  standard,
  verbose 
} verbosity_level;

/* Which of our own native extension functions to enable in each language */

#define MUESLI_EXTENSION_EVAL_IN_LANGUAGE  1
#define MUESLI_EXTENSION_CUSTOM            2
#define MUESLI_EXTENSION_SINGLE_PARAMETER  4
#define MUESLI_EXTENSION_PARAMETER_BLOCK   8
#define MUESLI_EXTENSION_ALL               0xf

extern unsigned int muesli_extensions;

/* Show the name of a type tag */

extern char *muesli_result_type_name(value_type_tag_t enumerated);

/* Pre-declare the main structure: */
struct evaluator_interface;

/* Access to `binary' (non-textual) languages.  You can't supply
   arguments to program fragments for them by substituting them in
   with sprintf etc, so we provide a generic way to give them
   arguments on a stack.
 */
typedef void (*binary_clear_args_function_t)(struct evaluator_interface *interface);
typedef void (*binary_give_arg_function_t)(struct evaluator_interface *interface,
					   float arg);
typedef float (*binary_eval_given_args_function_t)(struct evaluator_interface *interface,
						   const char *scratch,
						   unsigned int length);
typedef float (*binary_0_evaluator_function_t)(struct evaluator_interface *interface,
					       const char *scratch,
					       unsigned int length);
typedef float (*binary_1_evaluator_function_t)(struct evaluator_interface *interface,
					       const char *scratch,
					       unsigned int length,
					       float);
typedef float (*binary_2_evaluator_function_t)(struct evaluator_interface *interface,
					       const char *scratch,
					       unsigned int length,
					       float, float);

/* Types of data representation converter functions */

typedef muesli_value_t (*to_muesli_value_t)(unsigned long language_native);
typedef unsigned long (*from_muesli_value_t)(muesli_value_t muesli_value);

/* Types of functions for setting up callbacks to your application. */

typedef void (*evaluator_specializer_t)(struct evaluator_interface *interface);
typedef void (*evaluator_function_adder_t)(struct evaluator_interface *interface,
					   int arity,
					   const char *name,
					   void *function);

/* Interface to command-line-like (getopt) options */

typedef muesli_value_t (*option_handler_t)(void *params_block,
					   char opt,
					   /* const */ char *value_string, float value_number,
					   int set,
					   const char *from_language);

/* The main interface to evaluating bits of program text: */

typedef muesli_value_t (*string_evaluator_function_t)(struct evaluator_interface *evaluator_interface,
						      const char *scratch,
						      unsigned int string_length,
						      int transient);

/*****************************************************************************/
/* Our main data structure, describing the interface to a language evaluator */
/*****************************************************************************/

typedef struct evaluator_interface {
  struct evaluator_interface *next;

  /* Any language-dependent state */
  void *state;

  /* Application parameters and data */
  void *app_params;
  void *app_data;

  /* Some evaluators like to see the args of your application's `main' */
  int app_argc;
  char **app_argv;
  char **app_env;

  /* This is called when we want to initialize the interpreter itself
     (rather than just the interface that holds the interpreter): */
  void (*init_evaluator)(struct evaluator_interface *myself);

  /* This is like an extra initializer, but it filled in by the
     application; it is where the application can add its own native
     extension functions to the interpreter: */
  evaluator_specializer_t specializer;

  /* This adds a primitive / native function to the interpreter, for
     those languages where it can conveniently be genericized.
     Applications can use it through the macros Muesli_Add_Fn_0 et al
     in their specializer function, although they dont' have to; it's
     just so that application authors don't have to learn the `add
     native function' interface for so many different languages.  */
  evaluator_function_adder_t add_function;

  /* A function to shut the evaluator down: */
  void (*close_evaluator)(struct evaluator_interface *myself);

  /* The main callables: */

  string_evaluator_function_t eval_string;

  void (*load_file)(struct evaluator_interface *interface, const char *filename);

  /* Convert values between Muesli's dynamic data typing format and
     the language's.  Mostly, this will be done within the eval_string
     function of the interface. */
  to_muesli_value_t to_muesli_value;
  from_muesli_value_t from_muesli_value;

  /* The Read-Eval-Print Loop for this interpreter, if it has one: */
  void (*repl)(struct evaluator_interface *myself);

  /* How arrays are represented in text; this is so that if the
     application needs to build text representing an array into the
     string it sends to eval_string, it can do so whatever the
     language is. */
  char *array_opener;
  char *array_separator;
  char *array_closer;

  /* The application parameter handler.  Each interpreter interface
     defines functions for getting and setting parameters, either
     singly or in a group; this is the hook those functions call to
     get the parameter data back to the application: */
  option_handler_t handle_option;
  struct option *getopt_options;

  /* These are used only for binary evaluators */
  binary_0_evaluator_function_t eval_0;
  binary_clear_args_function_t eval_clear;
  binary_give_arg_function_t eval_give;
  binary_eval_given_args_function_t eval_given;

  /* For the piped evaluator (some also used by others): */
  const char *underlying_program;
  const char *underlying_startup_string;
  const char *underlying_prompt_string;
  const char *underlying_shutdown_command;
  pid_t underlying_evaluator_pid;

  /* Information for managing the list of evaluator interfaces: */
  char *language_name;
  char *implementation_name;
  char *extension;
  char *version;
  int binary;
  int initialized;
  
  /* Controlling the amount of output Muesli displays, and so on. */
  int language_verbosity;
  int flags;

  /* This is used when setting the evaluator up.  We keep it, in case
     the evaluator is copied, so we can use it to set the copy up,
     too. */
  void (*setup_func)(struct evaluator_interface *myself);
} evaluator_interface;

/* Tracing flags */

#define TRACE_MUESLI_INIT              0x0010
#define TRACE_MUESLI_LOAD              0x0020
#define TRACE_MUESLI_EVAL              0x0040
#define TRACE_MUESLI_CHILD_PROCESS     0x0080

/**************************/
/* Entry points for setup */
/**************************/

/* Create an evaluator interface structure, but don't do anything with it. */
extern evaluator_interface *muesli_make_evaluator_interface(const char* new_language_name,
							    const char* new_implementation_name,
							    const char *new_extension,
							    int new_is_binary,
							    void (*setup_func)(evaluator_interface *myself),
							    void (*init_func)(evaluator_interface *myself),
							    option_handler_t new_option_handler,
							    struct option *new_getopt_options,
							    void* params,
							    void *data);

/* Create a copy of an interface structure */
extern evaluator_interface *muesli_copy_evaluator_interface(evaluator_interface *original,
							    const char *new_languag_name,
							    const char *new_implementation_name);

/* Create an interface structure, and add it to the list of such structures. */
extern evaluator_interface *muesli_define_evaluator(const char *new_language_name,
						    const char *new_implementation_name,
						    const char *new_extension,
						    int new_is_binary,
						    void (*new_setup)(evaluator_interface *myself),
						    void (*new_init)(evaluator_interface *myself),
						    option_handler_t new_option_handler,
						    struct option *new_getopt_options,
						    void *new_params,
						    void *new_data);

/* Create a copy of an interface structure, and change a few fields,
   and add it to the list of such structures. */
extern evaluator_interface *muesli_define_variant_evaluator(evaluator_interface *original,
							    const char *new_language_name,
							    const char *new_implementation_name,
							    const char *new_underlying_program,
							    const char *new_underlying_startup_string,
							    const char *new_underlying_prompt_string,
							    const char *new_underlying_shutdown_command);

extern void muesli_initialize_evaluator(evaluator_interface *interface);

extern void binary_init();

/**********************************/
/* Tracking the current evaluator */
/**********************************/

extern evaluator_interface *current_muesli_evaluator;

#define Muesli_Enter_Evaluator(_evinf_,_istrans_)                         \
   evaluator_interface *surrounding_evaluator = current_muesli_evaluator; \
   int old_ambient_transient = ambient_transient;                         \
   current_muesli_evaluator = (_evinf_);				  \
   ambient_transient = (_istrans_);

#define Muesli_Leave_Evaluator()                    \
  ambient_transient = old_ambient_transient;        \
  current_muesli_evaluator = surrounding_evaluator;

/*****************************************/
/* Adding application-specific functions */
/*****************************************/

extern void muesli_app_specialize(char *name, evaluator_interface *interface);

extern void muesli_set_evaluator_app_specializer(char *evaluator_name,
						 evaluator_specializer_t specializer);

/* Some convenience macros for applications adding functions in their specializers: */

#define ARGS_COUNT_VARIABLE -2
#define ARGS_NO_EVAL -1

#define Muesli_Add_Fn_0(_interface_,_name_,_function_) ((_interface_)->add_function)((_interface_), 0, (char*)(_name_), (void*)(_function_))
#define Muesli_Add_Fn_1(_interface_,_name_,_function_) ((_interface_)->add_function)((_interface_), 1, (char*)(_name_), (void*)(_function_))
#define Muesli_Add_Fn_2(_interface_,_name_,_function_) ((_interface_)->add_function)((_interface_), 2, (char*)(_name_), (void*)(_function_))
#define Muesli_Add_Fn_3(_interface_,_name_,_function_) ((_interface_)->add_function)((_interface_), 3, (char*)(_name_), (void*)(_function_))
#define Muesli_Add_Fn_4(_interface_,_name_,_function_) ((_interface_)->add_function)((_interface_), 4, (char*)(_name_), (void*)(_function_))
#define Muesli_Add_Fn_5(_interface_,_name_,_function_) ((_interface_)->add_function)((_interface_), 5, (char*)(_name_), (void*)(_function_))
#define Muesli_Add_Fn_V(_interface_,_name_,_function_) ((_interface_)->add_function)((_interface_), ARGS_COUNT_VARIABLE, (char*)(_name_), (void*)(_function_))
#define Muesli_Add_Fn_N(_interface_,_name_,_function_) ((_interface_)->add_function)((_interface_), ARGS_NO_EVAL, (char*)(_name_), (void*)(_function_))

/* Customizers for muesli's built-in evaluators */
extern void muesli_machine_code_add_function(int fn_number, float (*function)(float));

/***************************************/
/* Entry points for finding evaluators */
/***************************************/

extern evaluator_interface *muesli_get_named_evaluator(const char *evaluator_name,
						       int exit_if_not_found);
extern int muesli_evaluator_is_defined(const char *evaluator_name);
extern evaluator_interface *muesli_get_evaluator_by_extension(const char *extension);
extern const char *muesli_get_language_by_extension(const char *extension);

/*******************************/
/* Entry points for evaluation */
/*******************************/

extern muesli_value_t eval_string(evaluator_interface *evaluator_interface,
				  const char *program_fragment,
				  unsigned int fragment_length,
				  int transient);

extern int muesli_load_file(const char *filename, int silent);

extern muesli_value_t muesli_eval_in_language(const char *language,
					      const char *fragment,
					      unsigned int fragment_length,
					      int transient);

extern char *muesli_eval_to_malloced_string(evaluator_interface *ev,
					    char *fragment,
					    unsigned int fragment_length,
					    int transient,
					    int *success_flag_p);
extern const char *muesli_eval_to_const_string(evaluator_interface *ev,
					       char *fragment,
					       unsigned int fragment_length,
					       int transient,
					       int *success_flag_p);
extern float muesli_eval_to_float(evaluator_interface *ev,
				  char *fragment,
				  unsigned int fragment_length,
				  int transient,
				  int *success_flag_p);
extern int muesli_eval_to_int(evaluator_interface *ev,
			      char *fragment,
			      unsigned int fragment_length,
			      int transient,
			      int *success_flag_p);
extern int muesli_eval_to_boolean(evaluator_interface *ev,
				  char *fragment,
				  unsigned int fragment_length,
				  int transient,
				  int *success_flag_p);

extern void muesli_evaluator_repl(const char* evaluator_name);

/*****************************************/
/* Managing the collection of evaluators */
/*****************************************/

extern void muesli_register_evaluators(void *new_app_params,
				       void *new_app_data,
				       option_handler_t params_handler,
				       struct option *getopt_options,
				       unsigned int extensions,
				       int app_argc, char **app_argv,
				       char **app_env);

extern void muesli_close_evaluators();

extern char* muesli_getName(evaluator_interface *interface);
extern char* muesli_getLanguageName(evaluator_interface *interface);
extern int muesli_getBinary(evaluator_interface *interface);

extern int muesli_print_interpreter_names(FILE *stream,
					  const char *format,
					  const char *selected_name,
					  const char *selected_flag,
					  unsigned int *widest_language_return,
					  unsigned int *widest_implementation_return);

typedef void *(*map_interpreter_names_function_t)(void *incoming_list,
						  char *implementation_name,
						  char *language_name);

extern void *muesli_map_interpreter_names(map_interpreter_names_function_t fn,
					  void *val);

/**********************************/
/* getopt-style parameter acccess */
/**********************************/

extern int muesli_set_parameter(evaluator_interface *interface,
				struct option *options,
				char* option,
				char *value,
				const char *language);
extern const char* muesli_get_parameter(evaluator_interface *interface,
					struct option *options,
					char* option);
extern char muesli_find_option_letter(struct option *getopt_options,
				      const char *option_name);

/**********************/
/* Testing and timing */
/**********************/

extern int muesli_test_evaluators(int seconds);

/***************************/
/* Small utility functions */
/***************************/

extern char *muesli_malloc_copy_string(const char *original);
extern char *muesli_malloc_copy_string_n(const char *original,
					 unsigned int length);


#endif

/* muesli.h ends here */
