/*
# Copyright (C) 2008-2009 
# Raffaele Granito <raffaele.granito@tiscali.it>
#
# This file is part of myTCPClient:
# Universal TCP Client
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
# General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/


/*
.--------------.
| headers LIBC |
'--------------' */
#include <stdio.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <netinet/tcp.h>
#include <netdb.h>
#include <fcntl.h>
#include <string.h>

/*
.-------------------------.
| header common           |
'-------------------------' */
#include "common.h"
#include "output.h" 

/*
.-----------------.
| header locale   |
'-----------------' */
#include "tcp.h"  
#include "tcp_messages.h"


/*
.-------------------------------------------------------------------------------------.
|                                                                                     |
|  Funzione    : TCPConnect                                                           |
|                                                                                     |
|  Descrizione :                                                                      |
|                                                                                     |
|  Return-code :  0 Ok                                                                |
|                -1 Hostname Resolv Error                                             |
|                -2 TCP Port Error                                                    |
|                -3 Create TCP Socket Error                                           |
|                -4 Connect TCP Socket Server Error                                   |
|                                                                                     |
|                                                                                     |
|                                                                                     |
|                                                                                     |
|                                                                                     |
|                                                                                     |
|                                                                                     |
'-------------------------------------------------------------------------------------'*/
extern struct tTCPConnect TCPConnect( char *host, int porta, int debug_level )
{
    /*
    .------------------------------------------.
    | Dichiarazione delle strutture TCP/IP     |
    '------------------------------------------' */
    struct sockaddr_in addr ;

    /*
    .------------------------------------------.
    | Dichiarazione delle STRUTTURE            |
    '------------------------------------------' */
    struct tTCPConnect  rTCPConnect;
    struct tTCPConnect *pTCPConnect;

    /*
    .------------------------------------------.
    | Allocazione SPAZIO                       |
    '------------------------------------------' */
    pTCPConnect = malloc(sizeof(struct tTCPConnect));

    /*
    .------------------------------------.
    | Utilizzo dello spazio allocato     |
    '------------------------------------' */
    pTCPConnect = &rTCPConnect;

    /*
    .-----------------------------.
    | Inizializzazione Strutture  |
    '-----------------------------' */
    memset((void*)&rTCPConnect,0,sizeof(rTCPConnect)); 
    rTCPConnect.return_code = 0;


    /*
    .------------------------------------------------------------------------.
    | PASSO 1. Risoluzione Hostname                                          |
    +------------------------------------------------------------------------+
    |                                                                        |
    |                                                                        |
    |                                                                        |
    |                                                                        |
    '------------------------------------------------------------------------' */

    /*
    .------------------------------------------.
    | Dichiarazione delle STRUTTURE            |
    '------------------------------------------' */
    struct tTCPGetHostByName  rTCPGetHostByName;
    struct tTCPGetHostByName *pTCPGetHostByName;

    /*
    .------------------------------------------.
    | Allocazione SPAZIO                       |
    '------------------------------------------' */
    pTCPGetHostByName = malloc(sizeof(struct tTCPGetHostByName));

    /*
    .------------------------------------.
    | Utilizzo dello spazio allocato     |
    '------------------------------------' */
    pTCPGetHostByName = &rTCPGetHostByName;

    /*
    .-----------------------------.
    | Inizializzazione Strutture  |
    '-----------------------------' */
    memset((void*)&rTCPGetHostByName, 0, sizeof(rTCPGetHostByName));

    /*
    .-------------.
    | Risoluzione |
    '-------------' */ 
    PrintMessage(debug_level, TCP_TABMSG, "TCP_RESOLVHOST_RESOLVHOST");
    rTCPGetHostByName = TCPGetHostByName ( host, debug_level );
    if ( rTCPGetHostByName.return_code < 0 )
    {
         /*
         .----------------------.
         | Messaggio di errore  |
         '----------------------' */ 
         PrintMessage(debug_level, TCP_TABMSG, "TCP_RESOLVHOST_KO");
         PrintMessage(debug_level, TCP_TABMSG, "TCP_RESOLVHOST_RESOLVHOST_KO");

         /*
         .----------------------.
         | Struttura di ritorno |
         '----------------------' */
         rTCPConnect.idmsg       = 0                                               ;
         rTCPConnect.msgDesc     = malloc(strlen("TCP.CONNECT.RESOLV.ERROR"))      ;
         rTCPConnect.msgDesc     = "TCP.CONNECT.RESOLV.ERROR"                      ;
         rTCPConnect.return_code = -1                                              ;
         return rTCPConnect;
    }
    PrintMessage(debug_level, TCP_TABMSG, "TCP_RESOLVHOST_OK");

    /*
    .--------------------------------------------------------------.
    | Muove l'IP risolto [solo il primo] nella struttura di OUTPUT |           
    '--------------------------------------------------------------' */
    rTCPConnect.ip = rTCPGetHostByName.tabAddress[0]; 

    /*
    .----------------------------------------------------------.
    | Visualizza in modalità di debug gli IP risolti           |
    '----------------------------------------------------------' */
    if ( debug_level == __INFO__ )
    {
         int indice = 0;
         while ( indice < rTCPGetHostByName.dimTabAddress )
         { 
                 printf("+Host [%s] => IP [%s]\n", host, rTCPGetHostByName.tabAddress[indice] );
                 indice++;
         };
    }


    /*
    .------------------------------------------------------------------------.
    | PASSO 2. Verifica la PORTA TCP/IP                                      |
    +------------------------------------------------------------------------+
    |                                                                        |
    |                                                                        |
    |                                                                        |
    |                                                                        |
    '------------------------------------------------------------------------' */

    PrintMessage(debug_level, TCP_TABMSG, "TCP_CONNECT_VERIFYPORTRANGE");
    if ( porta < 1 || porta > 65635 )
    {
         /*
         .----------------------.
         | Messaggio di errore  |
         '----------------------' */
         PrintMessage(debug_level, TCP_TABMSG, "TCP_CONNECT_KO");

         /*
         .----------------------.
         | Struttura di ritorno |
         '----------------------' */
         rTCPConnect.idmsg       = 0                                                   ;
         rTCPConnect.msgDesc     = malloc(strlen("TCP.CONNECT.VERIFYPORTRANGE.ERROR")) ;
         rTCPConnect.msgDesc     = "TCP.CONNECT.VERIFYPORTRANGE.ERROR"                 ;
         rTCPConnect.return_code = -2                                                  ;
         return rTCPConnect;
    }
    rTCPConnect.port = porta ;
    PrintMessage(debug_level, TCP_TABMSG, "TCP_CONNECT_OK");

 
    /*
    .------------------------------------------------------------------------.
    | PASSO 3. Creazione SOCKET                                              |
    +------------------------------------------------------------------------+
    |                                                                        |
    |                                                                        |
    |                                                                        |
    |                                                                        |
    '------------------------------------------------------------------------' */

    /*
    |
    |
    | 
    | The socket() function shall create an unbound socket in a communications domain, and return a 
    | file descriptor that can be used in later function calls that operate on sockets.
    |
    | The socket() function takes the following arguments:
    |
    | domain         | Specifies the communications domain in which a socket is to be created.
    |                |
    | type           | Specifies the type of socket to be created.
    |                |
    | protocol       | Specifies a particular protocol to be used with the socket. Specifying a protocol of 0 causes socket() 
    |                | to use an unspecified default protocol appropriate for the requested socket type.
    |
    | The domain argument specifies the address family used in the communications domain. The address families 
    | supported by the system are implementation-defined.
    |
    | Symbolic constants that can be used for the domain argument are defined in the <sys/socket.h> header.
    |
    | The type argument specifies the socket type, which determines the semantics of communication over the socket. 
    | The following socket types are defined; implementations may specify additional socket types:
    |
    | SOCK_STREAM    | Provides sequenced, reliable, bidirectional, connection-mode byte streams, 
    |                | and may provide a transmission mechanism for out-of-band data.
    |                |
    | SOCK_DGRAM     | Provides datagrams, which are connectionless-mode, unreliable messages of fixed maximum length.
    |                |
    | SOCK_SEQPACKET | Provides sequenced, reliable, bidirectional, connection-mode transmission paths for records. 
    |                | A record can be sent using one or more output operations and received using one or more input 
    |                | operations, but a single operation never transfers part of more than one record. 
    |                | Record boundaries are visible to the receiver via the MSG_EOR flag.
    |
    | If the protocol argument is non-zero, it shall specify a protocol that is supported by the address family. 
    | If the protocol argument is zero, the default protocol for this address family and type shall be used. 
    | The protocols supported by the system are implementation-defined.
    |
    | The process may need to have appropriate privileges to use the socket() function or to create some sockets.
    |
    | RETURN VALUE
    |
    | Upon successful completion, socket() shall return a non-negative integer, the socket file descriptor. 
    | Otherwise, a value of -1 shall be returned and errno set to indicate the error.
    |
    | ERRORS
    |
    | The socket() function shall fail if:
    |
    | [EAFNOSUPPORT]    | The implementation does not support the specified address family.
    | [EMFILE]          | No more file descriptors are available for this process.
    | [ENFILE]          | No more file descriptors are available for the system.
    | [EPROTONOSUPPORT] | The protocol is not supported by the address family, or the protocol 
    |                   | is not supported by the implementation.
    | [EPROTOTYPE]      | The socket type is not supported by the protocol.
    |
    | The socket() function may fail if:
    |
    | [EACCES]          | The process does not have appropriate privileges.
    | [ENOBUFS]         | Insufficient resources were available in the system to perform the operation.
    | [ENOMEM]          | Insufficient memory was available to fulfill the request.
    |
    |
    |
    */

    PrintMessage(debug_level, TCP_TABMSG, "TCP_CONNECT_SOCKETOPEN");
    if((rTCPConnect.sock=socket(AF_INET, SOCK_STREAM, IPPROTO_TCP))<0)   
    {
         rTCPConnect.idmsg = errno ;
         rTCPConnect.return_code = -3 ;

         PrintMessage(debug_level, TCP_TABMSG, "TCP_CONNECT_KO");
         PrintMessage(debug_level, TCP_TABMSG, "TCP_CONNECT_SOCKETOPEN_KO");
         switch ( rTCPConnect.idmsg )
         {
                  case EAFNOSUPPORT    : PrintMessage(debug_level, TCP_TABMSG, "EAFNOSUPPORT");  
                                         rTCPConnect.msgDesc = malloc(strlen("TCP.CONNECT.SOCKET.EAFNOSUPPORT"));
                                         rTCPConnect.msgDesc = "TCP.CONNECT.SOCKET.EAFNOSUPPORT";
                                         break;

                  case EMFILE          : PrintMessage(debug_level, TCP_TABMSG, "EMFILE"); 
                                         rTCPConnect.msgDesc = malloc(strlen("TCP.CONNECT.SOCKET.EMFILE"));
                                         rTCPConnect.msgDesc = "TCP.CONNECT.SOCKET.EMFILE";
                                         break;

                  case ENFILE          : PrintMessage(debug_level, TCP_TABMSG, "ENFILE"); 
                                         rTCPConnect.msgDesc = malloc(strlen("TCP.CONNECT.SOCKET.ENFILE"));
                                         rTCPConnect.msgDesc = "TCP.CONNECT.SOCKET.ENFILE";
                                         break;

                  case EPROTONOSUPPORT : PrintMessage(debug_level, TCP_TABMSG, "EPROTONOSUPPORT"); 
                                         rTCPConnect.msgDesc = malloc(strlen("TCP.CONNECT.SOCKET.EPROTONOSUPPORT"));
                                         rTCPConnect.msgDesc = "TCP.CONNECT.SOCKET.EPROTONOSUPPORT";
                                         break;

                  case EPROTOTYPE      : PrintMessage(debug_level, TCP_TABMSG, "EPROTOTYPE"); 
                                         rTCPConnect.msgDesc = malloc(strlen("TCP.CONNECT.SOCKET.EPROTOTYPE"));
                                         rTCPConnect.msgDesc = "TCP.CONNECT.SOCKET.EPROTOTYPE";
                                         break;

                  case EACCES          : PrintMessage(debug_level, TCP_TABMSG, "EACCES"); 
                                         rTCPConnect.msgDesc = malloc(strlen("TCP.CONNECT.SOCKET.EACCES"));
                                         rTCPConnect.msgDesc = "TCP.CONNECT.SOCKET.EACCES";
                                         break;

                  case ENOBUFS         : PrintMessage(debug_level, TCP_TABMSG, "ENOBUFS");  
                                         rTCPConnect.msgDesc = malloc(strlen("TCP.CONNECT.SOCKET.ENOBUFS"));
                                         rTCPConnect.msgDesc = "TCP.CONNECT.SOCKET.ENOBUFS";
                                         break;

                  case ENOMEM          : PrintMessage(debug_level, TCP_TABMSG, "ENOMEM");  
                                         rTCPConnect.msgDesc = malloc(strlen("TCP.CONNECT.SOCKET.ENOMEM"));
                                         rTCPConnect.msgDesc = "TCP.CONNECT.SOCKET.ENOMEM";
                                         break;

                  default              : PrintMessage(debug_level, TCP_TABMSG, "CREATE_SOCKET_UNKNOWN_ERROR"); 
                                         rTCPConnect.msgDesc = malloc(strlen("TCP.CONNECT.SOCKET.CREATE_SOCKET_UNKNOWN_ERROR"));
                                         rTCPConnect.msgDesc = "TCP.CONNECT.SOCKET.CREATE_SOCKET_UNKNOWN_ERROR";
                                         break;
         };

         return rTCPConnect;

    }
    PrintMessage(debug_level, TCP_TABMSG, "TCP_CONNECT_OK");



    /*
    .------------------------------------------------------------------------.
    | PASSO 4. Connessione SOCKET                                            |
    +------------------------------------------------------------------------+
    |                                                                        |
    |                                                                        |
    |                                                                        |
    |                                                                        |
    '------------------------------------------------------------------------' */

    /*
    |
    |
    |
    | The connect() function shall attempt to make a connection on a socket. The function takes the following arguments:
    |
    | socket      |    Specifies the file descriptor associated with the socket.
    |             |
    | address     |    Points to a sockaddr structure containing the peer address. 
    |             |    The length and format of the address depend on the address family of the socket.
    |             | 
    | address_len |    Specifies the length of the sockaddr structure pointed to by the address argument.
    |
    |
    | If the socket has not already been bound to a local address, connect() shall bind it to an address which, 
    | unless the socket's address family is AF_UNIX, is an unused local address.
    |
    | If the initiating socket is not connection-mode, then connect() shall set the socket's peer address, 
    | and no connection is made. For SOCK_DGRAM sockets, the peer address identifies where all datagrams 
    | are sent on subsequent send() functions, and limits the remote sender for subsequent recv() functions. 
    | If address is a null address for the protocol, the socket's peer address shall be reset.
    |
    | If the initiating socket is connection-mode, then connect() shall attempt to establish a connection to 
    | the address specified by the address argument. If the connection cannot be established immediately and 
    | O_NONBLOCK is not set for the file descriptor for the socket, connect() shall block for up to an unspecified 
    | timeout interval until the connection is established. If the timeout interval expires before the connection 
    | is established, connect() shall fail and the connection attempt shall be aborted. If connect() is interrupted 
    | by a signal that is caught while blocked waiting to establish a connection, connect() shall fail and 
    | set errno to [EINTR], but the connection request shall not be aborted, and the connection shall be 
    | established asynchronously.
    |
    | If the connection cannot be established immediately and O_NONBLOCK is set for the file descriptor for the socket, 
    | connect() shall fail and set errno to [EINPROGRESS], but the connection request shall not be aborted, and the 
    | connection shall be established asynchronously. Subsequent calls to connect() for the same socket, before 
    | the connection is established, shall fail and set errno to [EALREADY].
    |
    | When the connection has been established asynchronously, select() and poll() shall indicate that the file 
    | descriptor for the socket is ready for writing.
    |
    | The socket in use may require the process to have appropriate privileges to use the connect() function.
    |
    | RETURN VALUE
    |
    | Upon successful completion, connect() shall return 0; otherwise, -1 shall be returned and errno set to indicate the error.
    |
    | ERRORS
    |
    | The connect() function shall fail if:
    |
    | [EADDRNOTAVAIL]  |  The specified address is not available from the local machine.
    | [EAFNOSUPPORT]   |  The specified address is not a valid address for the address family of the specified socket.
    | [EALREADY]       |  A connection request is already in progress for the specified socket.
    | [EBADF]          |  The socket argument is not a valid file descriptor.
    | [ECONNREFUSED]   |  The target address was not listening for connections or refused the connection request.
    | [EINPROGRESS]    |  O_NONBLOCK is set for the file descriptor for the socket and the connection cannot be immediately established; 
    |                  |  the connection shall be established asynchronously.
    |                  |
    | [EINTR]          |  The attempt to establish a connection was interrupted by delivery of a signal that was caught; 
    |                  |  the connection shall be established asynchronously.
    |                  |
    | [EISCONN]        |  The specified socket is connection-mode and is already connected.
    | [ENETUNREACH]    |  No route to the network is present.
    | [ENOTSOCK]       |  The socket argument does not refer to a socket.
    | [EPROTOTYPE]     |  The specified address has a different type than the socket bound to the specified peer address.
    | [ETIMEDOUT]      |  The attempt to connect timed out before a connection was made.
    |
    |
    | If the address family of the socket is AF_UNIX, then connect() shall fail if:
    |
    | [EIO]            | An I/O error occurred while reading from or writing to the file system.
    | [ELOOP]          | A loop exists in symbolic links encountered during resolution of the pathname in address.
    | [ENAMETOOLONG]   | A component of a pathname exceeded {NAME_MAX} characters, or an entire pathname exceeded {PATH_MAX} characters.
    | [ENOENT]         | A component of the pathname does not name an existing file or the pathname is an empty string.
    | [ENOTDIR]        | A component of the path prefix of the pathname in address is not a directory.
    |
    | The connect() function may fail if:
    |
    | [EACCES]         | Search permission is denied for a component of the path prefix; or write access to the named socket is denied.
    | [EADDRINUSE]     | Attempt to establish a connection that uses addresses that are already in use.
    | [ECONNRESET]     | Remote host reset the connection request.
    | [EHOSTUNREACH]   | The destination host cannot be reached (probably because the host is down or a remote router cannot reach it).
    | [EINVAL]         | The address_len argument is not a valid length for the address family; or invalid address family in the sockaddr structure.
    | [ELOOP]          | More than {SYMLOOP_MAX} symbolic links were encountered during resolution of the pathname in address.
    | [ENAMETOOLONG]   | Pathname resolution of a symbolic link produced an intermediate result whose length exceeds {PATH_MAX}.
    | [ENETDOWN]       | The local network interface used to reach the destination is down.
    | [ENOBUFS]        | No buffer space is available.
    | [EOPNOTSUPP]     | The socket is listening and cannot be connected.
    |
    |
    */

    struct hostent *hp ;
    memset(&addr,0,sizeof(addr));

    addr.sin_addr.s_addr = inet_addr ( rTCPGetHostByName.tabAddress[0] ) ;
    addr.sin_family = AF_INET;
    addr.sin_port   = htons(porta);

    PrintMessage(debug_level, TCP_TABMSG, "TCP_CONNECT_SOCKET");
    if ( connect ( rTCPConnect.sock, (struct sockaddr *) &addr, sizeof(addr) ) < 0 ) 
    {   
         rTCPConnect.idmsg       = errno ;
         rTCPConnect.return_code = -4    ;

         PrintMessage(debug_level, TCP_TABMSG, "TCP_CONNECT_KO");
         PrintMessage(debug_level, TCP_TABMSG, "TCP_CONNECT_SOCKET_KO");
         switch ( rTCPConnect.idmsg )
         {
                  case EADDRNOTAVAIL : PrintMessage(debug_level, TCP_TABMSG, "EADDRNOTAVAIL");
                                       rTCPConnect.msgDesc = malloc(strlen("TCP.CONNECT.EADDRNOTAVAIL"));
                                       rTCPConnect.msgDesc = "TCP.CONNECT.EADDRNOTAVAIL";
                                       break;

                  case EAFNOSUPPORT  : PrintMessage(debug_level, TCP_TABMSG, "EAFNOSUPPORT");
                                       rTCPConnect.msgDesc = malloc(strlen("TCP.CONNECT.EAFNOSUPPORT"));
                                       rTCPConnect.msgDesc = "TCP.CONNECT.EAFNOSUPPORT";
                                       break;

                  case EALREADY      : PrintMessage(debug_level, TCP_TABMSG, "EALREADY");
                                       rTCPConnect.msgDesc = malloc(strlen("TCP.CONNECT.EALREADY"));
                                       rTCPConnect.msgDesc = "TCP.CONNECT.EALREADY";
                                       break;

                  case EBADF         : PrintMessage(debug_level, TCP_TABMSG, "EBADF");
                                       rTCPConnect.msgDesc = malloc(strlen("TCP.CONNECT.EBADF"));
                                       rTCPConnect.msgDesc = "TCP.CONNECT.EBADF";
                                       break;

                  case ECONNREFUSED  : PrintMessage(debug_level, TCP_TABMSG, "ECONNREFUSED");
                                       rTCPConnect.msgDesc = malloc(strlen("TCP.CONNECT.ECONNREFUSED"));
                                       rTCPConnect.msgDesc = "TCP.CONNECT.ECONNREFUSED";
                                       break;

                  case EINPROGRESS   : PrintMessage(debug_level, TCP_TABMSG, "EINPROGRESS");
                                       rTCPConnect.msgDesc = malloc(strlen("TCP.CONNECT.EINPROGRESS"));
                                       rTCPConnect.msgDesc = "TCP.CONNECT.EINPROGRESS";
                                       break;

                  case EINTR         : PrintMessage(debug_level, TCP_TABMSG, "EINTR");
                                       rTCPConnect.msgDesc = malloc(strlen("TCP.CONNECT.EINTR"));
                                       rTCPConnect.msgDesc = "TCP.CONNECT.EINTR";
                                       break;

                  case EISCONN       : PrintMessage(debug_level, TCP_TABMSG, "EISCONN");
                                       rTCPConnect.msgDesc = malloc(strlen("TCP.CONNECT.EISCONN"));
                                       rTCPConnect.msgDesc = "TCP.CONNECT.EISCONN";
                                       break;

                  case ENETUNREACH   : PrintMessage(debug_level, TCP_TABMSG, "ENETUNREACH");
                                       rTCPConnect.msgDesc = malloc(strlen("TCP.CONNECT.ENETUNREACH"));
                                       rTCPConnect.msgDesc = "TCP.CONNECT.ENETUNREACH";
                                       break;

                  case ENOTSOCK      : PrintMessage(debug_level, TCP_TABMSG, "ENOTSOCK");
                                       rTCPConnect.msgDesc = malloc(strlen("TCP.CONNECT.ENOTSOCK"));
                                       rTCPConnect.msgDesc = "TCP.CONNECT.ENOTSOCK";
                                       break;

                  case EPROTOTYPE    : PrintMessage(debug_level, TCP_TABMSG, "EPROTOTYPE");
                                       rTCPConnect.msgDesc = malloc(strlen("TCP.CONNECT.EPROTOTYPE"));
                                       rTCPConnect.msgDesc = "TCP.CONNECT.EPROTOTYPE";
                                       break;

                  case ETIMEDOUT     : PrintMessage(debug_level, TCP_TABMSG, "ETIMEDOUT");
                                       rTCPConnect.msgDesc = malloc(strlen("TCP.CONNECT.ETIMEDOUT"));
                                       rTCPConnect.msgDesc = "TCP.CONNECT.ETIMEDOUT";
                                       break;

                  case EIO           : PrintMessage(debug_level, TCP_TABMSG, "EIO");
                                       rTCPConnect.msgDesc = malloc(strlen("TCP.CONNECT.EIO"));
                                       rTCPConnect.msgDesc = "TCP.CONNECT.EIO";
                                       break;

                  case ELOOP         : PrintMessage(debug_level, TCP_TABMSG, "ELOOP");
                                       rTCPConnect.msgDesc = malloc(strlen("TCP.CONNECT.ELOOP"));
                                       rTCPConnect.msgDesc = "TCP.CONNECT.ELOOP";
                                       break;

                  case ENAMETOOLONG  : PrintMessage(debug_level, TCP_TABMSG, "ENAMETOOLONG");
                                       rTCPConnect.msgDesc = malloc(strlen("TCP.CONNECT.ENAMETOOLONG"));
                                       rTCPConnect.msgDesc = "TCP.CONNECT.ENAMETOOLONG";
                                       break;

                  case ENOENT        : PrintMessage(debug_level, TCP_TABMSG, "ENOENT");
                                       rTCPConnect.msgDesc = malloc(strlen("TCP.CONNECT.ENOENT"));
                                       rTCPConnect.msgDesc = "TCP.CONNECT.ENOENT";
                                       break;

                  case ENOTDIR       : PrintMessage(debug_level, TCP_TABMSG, "ENOTDIR");
                                       rTCPConnect.msgDesc = malloc(strlen("TCP.CONNECT.ENOTDIR"));
                                       rTCPConnect.msgDesc = "TCP.CONNECT.ENOTDIR";
                                       break;

                  case EACCES        : PrintMessage(debug_level, TCP_TABMSG, "EACCES");
                                       rTCPConnect.msgDesc = malloc(strlen("TCP.CONNECT.EACCES"));
                                       rTCPConnect.msgDesc = "TCP.CONNECT.EACCES";
                                       break;

                  case EADDRINUSE    : PrintMessage(debug_level, TCP_TABMSG, "EADDRINUSE");
                                       rTCPConnect.msgDesc = malloc(strlen("TCP.CONNECT.EADDRINUSE"));
                                       rTCPConnect.msgDesc = "TCP.CONNECT.EADDRINUSE";
                                       break;

                  case ECONNRESET    : PrintMessage(debug_level, TCP_TABMSG, "ECONNRESET");
                                       rTCPConnect.msgDesc = malloc(strlen("TCP.CONNECT.ECONNRESET"));
                                       rTCPConnect.msgDesc = "TCP.CONNECT.ECONNRESET";
                                       break;

                  case EHOSTUNREACH  : PrintMessage(debug_level, TCP_TABMSG, "EHOSTUNREACH");
                                       rTCPConnect.msgDesc = malloc(strlen("TCP.CONNECT.EHOSTUNREACH"));
                                       rTCPConnect.msgDesc = "TCP.CONNECT.EHOSTUNREACH";
                                       break;

                  case EINVAL        : PrintMessage(debug_level, TCP_TABMSG, "EINVAL");
                                       rTCPConnect.msgDesc = malloc(strlen("TCP.CONNECT.EINVAL"));
                                       rTCPConnect.msgDesc = "TCP.CONNECT.EINVAL";
                                       break;

               // case ELOOP         : PrintMessage(debug_level, TCP_TABMSG, "ELOOP");
               //                      rTCPConnect.msgDesc = malloc(strlen("TCP.CONNECT.ELOOP"));
               //                      rTCPConnect.msgDesc = "TCP.CONNECT.ELOOP";
               //                      break;

               // case ENAMETOOLONG  : PrintMessage(debug_level, TCP_TABMSG, "ENAMETOOLONG");
               //                      rTCPConnect.msgDesc = malloc(strlen("TCP.CONNECT.ENAMETOOLONG"));
               //                      rTCPConnect.msgDesc = "TCP.CONNECT.ENAMETOOLONG";
               //                      break;

                  case ENETDOWN      : PrintMessage(debug_level, TCP_TABMSG, "ENETDOWN");
                                       rTCPConnect.msgDesc = malloc(strlen("TCP.CONNECT.ENETDOWN"));
                                       rTCPConnect.msgDesc = "TCP.CONNECT.ENETDOWN";
                                       break;

                  case ENOBUFS       : PrintMessage(debug_level, TCP_TABMSG, "ENOBUFS");
                                       rTCPConnect.msgDesc = malloc(strlen("TCP.CONNECT.ENOBUFS"));
                                       rTCPConnect.msgDesc = "TCP.CONNECT.ENOBUFS";
                                       break;

                  case EOPNOTSUPP    : PrintMessage(debug_level, TCP_TABMSG, "EOPNOTSUPP");
                                       rTCPConnect.msgDesc = malloc(strlen("TCP.CONNECT.EOPNOTSUPP"));
                                       rTCPConnect.msgDesc = "TCP.CONNECT.EOPNOTSUPP";
                                       break;

                  default            : PrintMessage(debug_level, TCP_TABMSG, "CONNECT_SOCKET_UNKNOWN_ERROR");
                                       rTCPConnect.msgDesc = malloc(strlen("TCP.CONNECT.CONNECT_SOCKET_UNKNOWN_ERROR"));
                                       rTCPConnect.msgDesc = "TCP.CONNECT.CONNECT_SOCKET_UNKNOWN_ERROR";
                                       break;
         };

         return rTCPConnect;
    }
    PrintMessage(debug_level, TCP_TABMSG, "TCP_CONNECT_OK");

    return rTCPConnect;
}


/*
.-------------------------------------------------------------------------------------.
|                                                                                     |
|  Funzione    : TCPSocketSetBlocking                                                 |
|                                                                                     |
|  Descrizione :                                                                      |
|                                                                                     |
|  Return-code :  0 Ok                                                                |
|                -1 Set Blocking Socket Error                                         |
|                -2 Set Blocking Socket Error                                         |
|                                                                                     |
|                                                                                     |
|                                                                                     |
|                                                                                     |
|                                                                                     |
|                                                                                     |
|                                                                                     |
|                                                                                     |
|                                                                                     |
'-------------------------------------------------------------------------------------' */
int TCPSocketSetBlocking ( int sock, int debug_level )
{
    long arg;

    PrintMessage(debug_level, TCP_TABMSG, "TCP_SOCKETBLOCKON");

    if((arg = fcntl(sock, F_GETFL, NULL)) < 0) {
        PrintMessage(debug_level, TCP_TABMSG, "TCP_SOCKETBLOCK_KO");
        fprintf(stderr, "Error fcntl(..., F_GETFL) (%s)\n", strerror(errno));
        return -1;
    }
    
    arg &= (~O_NONBLOCK);
 
    if( fcntl(sock, F_SETFL, arg) < 0) {
        PrintMessage(debug_level, TCP_TABMSG, "TCP_SOCKETBLOCK_KO");
        fprintf(stderr, "Error fcntl(..., F_SETFL) (%s)\n", strerror(errno));
        return -2;
    }

    PrintMessage(debug_level, TCP_TABMSG, "TCP_SOCKETBLOCK_OK");

    return 0;
}


/*
.-------------------------------------------------------------------------------------.
|                                                                                     |
|  Funzione    : TCPSocketSetNotBlocking                                              |
|                                                                                     |
|  Descrizione :                                                                      |
|                                                                                     |
|  Return-code :  0 Ok                                                                |
|                -1 Set Not Blocking Socket Error                                     |
|                -2 Set Not Blocking Socket Error                                     |
|                                                                                     |
|                                                                                     |
|                                                                                     |
|                                                                                     |
|                                                                                     |
|                                                                                     |
|                                                                                     |
|                                                                                     |
|                                                                                     |
'-------------------------------------------------------------------------------------'*/
int TCPSocketSetNotBlocking ( int sock, int debug_level )
{
    long arg;

    PrintMessage(debug_level, TCP_TABMSG, "TCP_SOCKETBLOCKOFF");

    if((arg = fcntl(sock, F_GETFL, NULL)) < 0) {
        PrintMessage(debug_level, TCP_TABMSG, "TCP_SOCKETBLOCK_KO");
        fprintf(stderr, "Error fcntl(..., F_GETFL) (%s)\n", strerror(errno));
        return -1;
    }
  
    arg |= O_NONBLOCK;

    if( fcntl(sock, F_SETFL, arg) < 0) {
        PrintMessage(debug_level, TCP_TABMSG, "TCP_SOCKETBLOCK_KO");
        fprintf(stderr, "Error fcntl(..., F_SETFL) (%s)\n", strerror(errno));
        return -2;
    }

    PrintMessage(debug_level, TCP_TABMSG, "TCP_SOCKETBLOCK_OK");

    return 0;
}



/*
.-------------------------------------------------------------------------------------.
|                                                                                     |
|  Funzione    : TCPGetHostByName                                                     |
|                                                                                     |
|  Descrizione : Hostname Resolve                                                     |
|                                                                                     |
|  Return-code :  0 Ok                                                                |
|                -1 Hostname Resolve Error                                            |
|                                                                                     |
|                                                                                     |
|                                                                                     |
|                                                                                     |
|                                                                                     |
|                                                                                     |
|                                                                                     |
|                                                                                     |
|                                                                                     |
|                                                                                     |
'-------------------------------------------------------------------------------------' */
extern struct tTCPGetHostByName TCPGetHostByName ( char *host, int debug_level )
{

    struct hostent    *hp   ;

    /*
    .--------------------------------------.
    | Dichiarazione strutture utilizzate   |
    '--------------------------------------' */
    struct tTCPGetHostByName  rTCPGetHostByName ;
    struct tTCPGetHostByName *pTCPGetHostByName ;

    /*
    .------------------------------------.
    | Allocazione Spazio                 |
    '------------------------------------' */
    pTCPGetHostByName = malloc(sizeof(struct tTCPGetHostByName));

    /*
    .------------------------------------.
    | Utilizzo dello spazio allocato     |
    '------------------------------------' */
    pTCPGetHostByName = &rTCPGetHostByName;

    /*
    .-----------------------------.
    | Inizializzazione Strutture  |
    '-----------------------------' */
    memset((void*)&rTCPGetHostByName, 0, sizeof(rTCPGetHostByName));
    rTCPGetHostByName.idmsg = 0;
    rTCPGetHostByName.return_code = 0;

    /*
    .------------------------------------------------------.
    |                                                      |
    |          Risoluzione dell'hostname server            |
    |                                                      |
    '------------------------------------------------------' */
    if( !( hp = gethostbyname ( host ) ) )
    {
         rTCPGetHostByName.idmsg = 0 ;
         rTCPGetHostByName.return_code = -1 ;
         return rTCPGetHostByName;
    }
    
    /*
    .------------------------------------------------------.
    |                                                      |
    |        Carica gli IP nella struttura di ritorno      |
    |                                                      |
    '------------------------------------------------------' */
    rTCPGetHostByName.tabAddress = malloc (160) ; 
    if ( hp->h_addrtype==AF_INET )
    {
         rTCPGetHostByName.dimTabAddress = 0 ; 
         struct  in_addr sin_addr;
         while ( hp->h_addr_list[rTCPGetHostByName.dimTabAddress] )
         {
                 memcpy(&(sin_addr), hp->h_addr_list[rTCPGetHostByName.dimTabAddress], hp->h_length);
                 rTCPGetHostByName.tabAddress[rTCPGetHostByName.dimTabAddress] = inet_ntoa(sin_addr);    
                 rTCPGetHostByName.dimTabAddress++;
         }
    }

    /*
    |
    | Errore. Non valorizza bene l'array... sempre l'ultimo elemento. problema con **array (forse)
    |
    | printf("o========> [%s]\n", rTCPGetHostByName.tabAddress[0]);
    | printf("o========> [%s]\n", rTCPGetHostByName.tabAddress[1]);
    |
    */


    return rTCPGetHostByName ;
};


/*
.-------------------------------------------------------------------------------------.
|                                                                                     |
|  Funzione    : TCPSend                                                              |
|                                                                                     |
|  Descrizione : Buffer Send to Target Host                                           |
|                                                                                     |
|  Return-code :  0 Ok                                                                |
|                -1 Send Buffer Error                                                 |
|                                                                                     |
|                                                                                     |
|                                                                                     |
|                                                                                     |
|                                                                                     |
|                                                                                     |
|                                                                                     |
|                                                                                     |
|                                                                                     |
'-------------------------------------------------------------------------------------'*/
extern struct tTCPSend TCPSend (int sock, char *request, int debug_level) 
{
    /*
    .--------------------------------------.
    | Dichiarazione strutture utilizzate   |
    '--------------------------------------' */
    struct tTCPSend  rTCPSend ;
    struct tTCPSend *pTCPSend ;

    /*
    .------------------------------------.
    | Allocazione Spazio                 |
    '------------------------------------' */
    pTCPSend = malloc(sizeof(struct tTCPSend));

    /*
    .------------------------------------.
    | Utilizzo dello spazio allocato     |
    '------------------------------------' */
    pTCPSend = &rTCPSend;

    /*
    .-----------------------------.
    | Inizializzazione Strutture  |
    '-----------------------------' */
    memset((void*)&rTCPSend, 0, sizeof(rTCPSend));
    rTCPSend.idmsg = 0;
    rTCPSend.return_code = 0;

    /*
    .-------------------------------------------.
    | Calcola la lunghezza se non viene passata |
    '-------------------------------------------' */
    PrintMessage(debug_level, TCP_TABMSG, "TCP_SEND_CALCULATE_LEN_REQUEST");
    rTCPSend.requestLen = strlen(request);
    PrintMessage(debug_level, TCP_TABMSG, "TCP_SEND_OK");

    /*
    .--------------------------------------------.
    | Alloca e valorizza la richiesta da inviare |
    '--------------------------------------------' */ 
    rTCPSend.request = malloc (rTCPSend.requestLen);
    rTCPSend.request = request ;



    /*
    .--------------------------------------------.
    |                   S e n d                  |
    '--------------------------------------------' */

    /*
    |
    |
    | The send() function shall initiate transmission of a message from the specified socket to its peer. 
    | The send() function shall send a message only when the socket is connected (including when the peer of 
    | a connectionless socket has been set via connect()).
    |
    | The send() function takes the following arguments:
    |
    | socket  |    Specifies the socket file descriptor.
    | buffer  |    Points to the buffer containing the message to send.
    | length  |    Specifies the length of the message in bytes.
    | flags   |    Specifies the type of message transmission. Values of this argument are formed by logically OR'ing zero or more of the following flags:
    |         |
    |         |    MSG_EOR    Terminates a record (if supported by the protocol).
    |         |
    |         |    MSG_OOB    Sends out-of-band data on sockets that support out-of-band communications. 
    |         |               The significance and semantics of out-of-band data are protocol-specific.
    |
    | The length of the message to be sent is specified by the length argument. If the message is too long to pass 
    | through the underlying protocol, send() shall fail and no data shall be transmitted.
    |
    | Successful completion of a call to send() does not guarantee delivery of the message. 
    | A return value of -1 indicates only locally-detected errors.
    |
    | If space is not available at the sending socket to hold the message to be transmitted, 
    | and the socket file descriptor does not have O_NONBLOCK set, send() shall block until space is available. 
    | If space is not available at the sending socket to hold the message to be transmitted, 
    | and the socket file descriptor does have O_NONBLOCK set, send() shall fail. 
    | The select() and poll() functions can be used to determine when it is possible to send more data.
    |
    | The socket in use may require the process to have appropriate privileges to use the send() function.
    |
    | RETURN VALUE
    |
    | Upon successful completion, send() shall return the number of bytes sent. Otherwise, 
    | -1 shall be returned and errno set to indicate the error.
    |
    | ERRORS
    |
    | EAGAIN | EWOULDBLOCK  |  The socket's file descriptor is marked O_NONBLOCK and the requested operation would block.
    | EBADF                 |  The socket argument is not a valid file descriptor.
    | ECONNRESET            |  A connection was forcibly closed by a peer.
    | EDESTADDRREQ          |  The socket is not connection-mode and no peer address is set.
    | EINTR                 |  A signal interrupted send() before any data was transmitted.
    | EMSGSIZE              |  The message is too large to be sent all at once, as the socket requires.
    | ENOTCONN              |  The socket is not connected or otherwise has not had the peer pre-specified.
    | ENOTSOCK              |  The socket argument does not refer to a socket.
    | EOPNOTSUPP            |  The socket argument is associated with a socket that does not support one or more of the values set in flags.
    | EPIPE                 |  The socket is shut down for writing, or the socket is connection-mode and is no longer connected. 
    |                       |  In the latter case, and if the socket is of type SOCK_STREAM, the SIGPIPE signal is generated to the calling thread.
    | EACCES                |  The calling process does not have the appropriate privileges.
    | EIO                   |  An I/O error occurred while reading from or writing to the file system.
    | ENETDOWN              |  The local network interface used to reach the destination is down.
    | ENETUNREACH           |  No route to the network is present.
    | ENOBUFS               |  Insufficient resources were available in the system to perform the operation.
    | 
    |
    */

    PrintMessage(debug_level, TCP_TABMSG, "TCP_SEND_REQUEST");
    if ( debug_level == __INFO__ ) printf("(%s) ", rTCPSend.request);
    if (send(sock, rTCPSend.request, rTCPSend.requestLen, 0)<0) 
    {
         rTCPSend.idmsg = errno ;
         rTCPSend.return_code = -1 ;

         PrintMessage(debug_level, TCP_TABMSG, "TCP_SEND_KO");
         switch ( rTCPSend.idmsg )                                                   
         {
                  case EAGAIN       : PrintMessage(debug_level, TCP_TABMSG, "EAGAIN");  
                                      rTCPSend.msgDesc = malloc(strlen("TCP.SEND.EAGAIN"));
                                      rTCPSend.msgDesc = "TCP.SEND.EAGAIN";
                                      break;               

               // case EWOULDBLOCK  : PrintMessage(debug_level, TCP_TABMSG, "EWOULDBLOCK");
               //                     rTCPSend.msgDesc = malloc(strlen("TCP.SEND.EWOULDBLOCK"));
               //                     rTCPSend.msgDesc = "TCP.SEND.EWOULDBLOCK";
               //                     break;                   

                  case EBADF        : PrintMessage(debug_level, TCP_TABMSG, "EBADF");
                                      rTCPSend.msgDesc = malloc(strlen("TCP.SEND.EBADF"));                 
                                      rTCPSend.msgDesc = "TCP.SEND.EBADF";
                                      break;
                  
                  case ECONNRESET   : PrintMessage(debug_level, TCP_TABMSG, "ECONNRESET");
                                      rTCPSend.msgDesc = malloc(strlen("TCP.SEND.ECONNRESET"));                 
                                      rTCPSend.msgDesc = "TCP.SEND.ECONNRESET";
                                      break;
                  
                  case EDESTADDRREQ : PrintMessage(debug_level, TCP_TABMSG, "EDESTADDRREQ");
                                      rTCPSend.msgDesc = malloc(strlen("TCP.SEND.EDESTADDRREQ"));                 
                                      rTCPSend.msgDesc = "TCP.SEND.EDESTADDRREQ";
                                      break;
                  
                  case EINTR        : PrintMessage(debug_level, TCP_TABMSG, "EINTR");
                                      rTCPSend.msgDesc = malloc(strlen("TCP.SEND.EINTR"));                 
                                      rTCPSend.msgDesc = "TCP.SEND.EINTR";
                                      break;
                  
                  case EMSGSIZE     : PrintMessage(debug_level, TCP_TABMSG, "EMSGSIZE");
                                      rTCPSend.msgDesc = malloc(strlen("TCP.SEND.EMSGSIZE"));                 
                                      rTCPSend.msgDesc = "TCP.SEND.EMSGSIZE";
                                      break;
                  
                  case ENOTCONN     : PrintMessage(debug_level, TCP_TABMSG, "ENOTCONN");
                                      rTCPSend.msgDesc = malloc(strlen("TCP.SEND.ENOTCONN"));                 
                                      rTCPSend.msgDesc = "TCP.SEND.ENOTCONN";
                                      break;
                  
                  case ENOTSOCK     : PrintMessage(debug_level, TCP_TABMSG, "ENOTSOCK");
                                      rTCPSend.msgDesc = malloc(strlen("TCP.SEND.ENOTSOCK"));                 
                                      rTCPSend.msgDesc = "TCP.SEND.ENOTSOCK";
                                      break;
                  
                  case EOPNOTSUPP   : PrintMessage(debug_level, TCP_TABMSG, "EOPNOTSUPP");
                                      rTCPSend.msgDesc = malloc(strlen("TCP.SEND.EOPNOTSUPP"));                 
                                      rTCPSend.msgDesc = "TCP.SEND.EOPNOTSUPP";
                                      break;
                  
                  case EPIPE        : PrintMessage(debug_level, TCP_TABMSG, "EPIPE");
                                      rTCPSend.msgDesc = malloc(strlen("TCP.SEND.EPIPE"));                 
                                      rTCPSend.msgDesc = "TCP.SEND.EPIPE";
                                      break;
                  
                  case EACCES       : PrintMessage(debug_level, TCP_TABMSG, "EACCES");
                                      rTCPSend.msgDesc = malloc(strlen("TCP.SEND.EACCES")); 
                                      rTCPSend.msgDesc = "TCP.SEND.EACCES";
                                      break;
                  
                  case EIO          : PrintMessage(debug_level, TCP_TABMSG, "EIO");
                                      rTCPSend.msgDesc = malloc(strlen("TCP.SEND.EIO"));                 
                                      rTCPSend.msgDesc = "TCP.SEND.EIO";
                                      break;
                  
                  case ENETDOWN     : PrintMessage(debug_level, TCP_TABMSG, "ENETDOWN");
                                      rTCPSend.msgDesc = malloc(strlen("TCP.SEND.ENETDOWN"));                 
                                      rTCPSend.msgDesc = "TCP.SEND.ENETDOWN";
                                      break;
                  
                  case ENETUNREACH  : PrintMessage(debug_level, TCP_TABMSG, "ENETUNREACH");
                                      rTCPSend.msgDesc = malloc(strlen("TCP.SEND.ENETUNREACH"));                 
                                      rTCPSend.msgDesc = "TCP.SEND.ENETUNREACH";
                                      break;
                  
                  case ENOBUFS      : PrintMessage(debug_level, TCP_TABMSG, "ENOBUFS");
                                      rTCPSend.msgDesc = malloc(strlen("TCP.SEND.ENOBUFS"));                 
                                      rTCPSend.msgDesc = "TCP.SEND.ENOBUFS";
                                      break;
                  
                  default           : PrintMessage(debug_level, TCP_TABMSG, "SEND_TCP_UNKNOWN_ERROR");
                                      rTCPSend.msgDesc = malloc(strlen("TCP.SEND.SEND_TCP_UNKNOWN_ERROR"));                 
                                      rTCPSend.msgDesc = "TCP.SEND.SEND_TCP_UNKNOWN_ERROR";
                                      break;
                  
          };
    };
    PrintMessage(debug_level, TCP_TABMSG, "TCP_SEND_OK");

    return rTCPSend;
};


/*
.-------------------------------------------------------------------------------------.
|                                                                                     |
|  Funzione    : TCPReceive                                                           |
|                                                                                     |
|  Descrizione : Receive Buffer from Target Host                                      |
|                                                                                     |
|  Return-code :  0 Ok                                                                |
|                -1 Receive Buffer Error                                              |
|                                                                                     |
|                                                                                     |
|                                                                                     |
|                                                                                     |
|                                                                                     |
|                                                                                     |
|                                                                                     |
|                                                                                     |
|                                                                                     |
'-------------------------------------------------------------------------------------'*/
extern struct tTCPReceive TCPReceive (int sock, char *OutputFile, int debug_level)
{
    /*
    .--------------------------------------.
    | Dichiarazione strutture utilizzate   |
    '--------------------------------------' */
    struct tTCPReceive  rTCPReceive ;
    struct tTCPReceive *pTCPReceive ;

    /*
    .------------------------------------.
    | Allocazione Spazio                 |
    '------------------------------------' */
    pTCPReceive = malloc(sizeof(struct tTCPReceive));

    /*
    .------------------------------------.
    | Utilizzo dello spazio allocato     |
    '------------------------------------' */
    pTCPReceive = &rTCPReceive;

    /*
    .-----------------------------.
    | Inizializzazione Strutture  |
    '-----------------------------' */
    memset((void*)&rTCPReceive, 0, sizeof(rTCPReceive));

    rTCPReceive.idmsg       = 0;
    rTCPReceive.return_code = 0;


    /*
    .----------------------------------------------------.
    | Create|Open OUTPUT                                 |
    +----------------------------------------------------+
    | Crea il file di Output prima di iniziare a         |
    | leggere dal socket. Se non riesce ad aprire il     |
    | file esce per errore.                              |
    '----------------------------------------------------' */

    /*
    .------------------------------------------------.
    | Imposta il default se non impostato OutputFile |
    '------------------------------------------------' */
    if ( OutputFile == NULL ) {
         OutputFile = malloc(1024);
         strcpy (OutputFile, "/tmp/.myTCPClient.OutputFile.txt");
    };

    /*
    .-----------------.
    | Open Datastore  |
    '-----------------' */
    FILE *pOutputFile ;
    PrintMessage(debug_level, TCP_TABMSG, "TCP_RECEIVE_OPEN_DATASTORE");
    pOutputFile = fopen(OutputFile, "w");
    if (!pOutputFile)
    {
        PrintMessage(debug_level, TCP_TABMSG, "TCP_RECEIVE_KO");
        PrintMessage(debug_level, TCP_TABMSG, "TCP_RECEIVE_OPEN_DATASTORE_KO");
        rTCPReceive.return_code = -1;
        return rTCPReceive ;
    }
    PrintMessage(debug_level, TCP_TABMSG, "TCP_RECEIVE_OK");

    if ( debug_level == __INFO__ )
         printf("+Datastore [%s] \n", OutputFile );


    /*
    .---------------------------------------------------.
    | Receive                                           |
    +---------------------------------------------------+
    | Legge lo stream dal socket.                       |
    '---------------------------------------------------' */
    int numByteReceive = -1;
    int totByteReceive =  0; 
    char *response;
    response = malloc (__TCP_LEN_BUFFER_RECEIVE__);

    PrintMessage(debug_level, TCP_TABMSG, "TCP_RECEIVE_START");
    for (;;)
    {
          /*
          .----------------------------------------------.
          |                   R e c v                    |
          '----------------------------------------------' */ 

          /*
          | 
          |
          |
          |  The recv() function shall receive a message from a connection-mode or connectionless-mode socket. 
          |  It is normally used with connected sockets because it does not permit the application to retrieve 
          |  the source address of received data.
          |
          |  The recv() function takes the following arguments :
          |
          |  socket    Specifies the socket file descriptor.
          |  buffer    Points to a buffer where the message should be stored.
          |  length    Specifies the length in bytes of the buffer pointed to by the buffer argument.
          |  flags     Specifies the type of message reception. Values of this argument are formed by 
          |            logically OR'ing zero or more of the following values:
          |
          |            MSG_PEEK    Peeks at an incoming message. The data is treated as unread and the 
          |                        next recv() or similar function shall still return this data.
          |
          |            MSG_OOB     Requests out-of-band data. The significance and semantics of 
          |                        out-of-band data are protocol-specific.
          |
          |            MSG_WAITALL On SOCK_STREAM sockets this requests that the function block until the 
          |                        full amount of data can be returned. The function may return the smaller 
          |                        amount of data if the socket is a message-based socket, if a signal is caught, 
          |                        if the connection is terminated, if MSG_PEEK was specified, or if an 
          |                        error is pending for the socket.
          |
          |  The recv() function shall return the length of the message written to the buffer pointed to by the 
          |  buffer argument. For message-based sockets, such as SOCK_DGRAM and SOCK_SEQPACKET, the entire message 
          |  shall be read in a single operation. If a message is too long to fit in the supplied buffer, 
          |  and MSG_PEEK is not set in the flags argument, the excess bytes shall be discarded. For stream-based
          |  sockets, such as SOCK_STREAM, message boundaries shall be ignored. In this case, data shall be returned 
          |  to the user as soon as it becomes available, and no data shall be discarded.
          |
          |  If the MSG_WAITALL flag is not set, data shall be returned only up to the end of the first message.
          |
          |  If no messages are available at the socket and O_NONBLOCK is not set on the socket's file descriptor, 
          |  recv() shall block until a message arrives. If no messages are available at the socket and O_NONBLOCK 
          |  is set on the socket's file descriptor, recv() shall fail and set errno to [EAGAIN] or [EWOULDBLOCK].
          |
          |
          |  RETURN VALUE
          | 
          |  Upon successful completion, recv() shall return the length of the message in bytes. 
          |  If no messages are available to be received and the peer has performed an orderly shutdown, recv() 
          |  shall return 0. Otherwise, -1 shall be returned and errno set to indicate the error.
          |
          |  ERRORS
          |  
          |  The recv() function shall fail if:
          |
          |  [EAGAIN] ||   The socket's file descriptor is marked O_NONBLOCK and no data is waiting to be received; 
          |  [EWOULDBLOCK] or MSG_OOB is set and no out-of-band data is available and either the socket's file  
          |                descriptor is marked O_NONBLOCK or the socket does not support blocking to await out-of-band data.
          |  [EBADF]       The socket argument is not a valid file descriptor. 
          |  [ECONNRESET]  A connection was forcibly closed by a peer. 
          |  [EINTR]       The recv() function was interrupted by a signal that was caught, before any data was available.  
          |  [EINVAL]      The MSG_OOB flag is set and no out-of-band data is available. 
          |  [ENOTCONN]    A receive is attempted on a connection-mode socket that is not connected. 
          |  [ENOTSOCK]    The socket argument does not refer to a socket.  
          |  [EOPNOTSUPP]  The specified flags are not supported for this socket type or protocol. 
          |  [ETIMEDOUT]   The connection timed out during connection establishment, 
          |                or due to a transmission timeout on active connection.
          |
          |  The recv() function may fail if :
          |
          |  [EIO]         An I/O error occurred while reading from or writing to the file system.
          |  [ENOBUFS]     Insufficient resources were available in the system to perform the operation.
          |  [ENOMEM]      Insufficient memory was available to fulfill the request. 
          | 
          |
          |  APPLICATION USAGE
          |
          |  The recv() function is equivalent to recvfrom() with a zero address_len argument, and to read() 
          |  if no flags are used.
          |
          |  The select() and poll() functions can be used to determine when data is available to be received.
          |
          |
          */

          PrintMessage(debug_level, TCP_TABMSG, "TCP_RECEIVE_WAITING_RESPONSE");   
          memset(response, 0, __TCP_LEN_BUFFER_RECEIVE__);
          numByteReceive = recv (sock, response, __TCP_LEN_BUFFER_RECEIVE__, 0 );
          PrintMessage(debug_level, TCP_TABMSG, "TCP_RECEIVE_RESPONSE");
          if (debug_level == __INFO__) printf("%d", numByteReceive);
 
          /*
          .----------------------------------------------.
          |                 E r r o r e                  |
          +----------------------------------------------+
          | Se il numero di byte ricevuti è -1 significa |
          | che c'è stato un errore... l'errore è        |
          | contenuto nella variabile globale errno      |
          | dichiarata nell'header <errno.h>             |
          '----------------------------------------------' */
          if (numByteReceive == -1)
          {
              rTCPReceive.idmsg       =  errno ;
              rTCPReceive.return_code = -1     ;

              PrintMessage(debug_level, TCP_TABMSG, "TCP_RECEIVE_KO");
              PrintMessage(debug_level, TCP_TABMSG, "TCP_RECEIVE_RESPONSE_KO");
              switch ( rTCPReceive.idmsg )
              {
                       case EAGAIN      : PrintMessage(debug_level, TCP_TABMSG, "EAGAIN"); 
                                          rTCPReceive.msgDesc = malloc(strlen("TCP.RECV.EAGAIL"));
                                          rTCPReceive.msgDesc = "TCP.RECV.EAGAIL"; 
                                          break; 

                    // case EWOULDBLOCK : PrintMessage(debug_level, TCP_TABMSG, "EWOULDBLOCK");  
                    //                    rTCPReceive.msgDesc = malloc(strlen("TCP.RECV.EWOULDBLOCK"));
                    //                    rTCPReceive.msgDesc = "TCP.RECV.EWOULDBLOCK";
                    //                    break;
                    
                       case EBADF       : PrintMessage(debug_level, TCP_TABMSG, "EBADF");  
                                          rTCPReceive.msgDesc = malloc(strlen("TCP.RECV.EBADF"));
                                          rTCPReceive.msgDesc = "TCP.RECV.EBADF";
                                          break;
                    
                       case ECONNRESET  : PrintMessage(debug_level, TCP_TABMSG, "ECONNRESET");  
                                          rTCPReceive.msgDesc = malloc(strlen("TCP.RECV.ECONNRESET"));
                                          rTCPReceive.msgDesc = "TCP.RECV.ECONNRESET";
                                          break;
                    
                       case EINTR       : PrintMessage(debug_level, TCP_TABMSG, "EINTR");  
                                          rTCPReceive.msgDesc = malloc(strlen("TCP.RECV.EINTR"));
                                          rTCPReceive.msgDesc = "TCP.RECV.EINTR";
                                          break;
                    
                       case EINVAL      : PrintMessage(debug_level, TCP_TABMSG, "EINVAL");  
                                          rTCPReceive.msgDesc = malloc(strlen("TCP.RECV.EINVAL"));
                                          rTCPReceive.msgDesc = "TCP.RECV.EINVAL";
                                          break;
                    
                       case ENOTCONN    : PrintMessage(debug_level, TCP_TABMSG, "ENOTCONN");  
                                          rTCPReceive.msgDesc = malloc(strlen("TCP.RECV.ENOTCONN"));
                                          rTCPReceive.msgDesc = "TCP.RECV.ENOTCONN";
                                          break;
                    
                       case ENOTSOCK    : PrintMessage(debug_level, TCP_TABMSG, "ENOTSOCK");  
                                          rTCPReceive.msgDesc = malloc(strlen("TCP.RECV.ENOTSOCK"));
                                          rTCPReceive.msgDesc = "TCP.RECV.ENOTSOCK";
                                          break;
                    
                       case EOPNOTSUPP  : PrintMessage(debug_level, TCP_TABMSG, "EOPNOTSUPP");  
                                          rTCPReceive.msgDesc = malloc(strlen("TCP.RECV.EOPNOTSUPP"));
                                          rTCPReceive.msgDesc = "TCP.RECV.EOPNOTSUPP";
                                          break;
                    
                       case ETIMEDOUT   : PrintMessage(debug_level, TCP_TABMSG, "ETIMEDOUT");  
                                          rTCPReceive.msgDesc = malloc(strlen("TCP.RECV.ETIMEDOUT"));
                                          rTCPReceive.msgDesc = "TCP.RECV.ETIMEDOUT";
                                          break;
                    
                       case EIO         : PrintMessage(debug_level, TCP_TABMSG, "EIO"); 
                                          rTCPReceive.msgDesc = malloc(strlen("TCP.RECV.EIO"));
                                          rTCPReceive.msgDesc = "TCP.RECV.EIO";
                                          break;
                    
                       case ENOBUFS     : PrintMessage(debug_level, TCP_TABMSG, "ENOBUFS");  
                                          rTCPReceive.msgDesc = malloc(strlen("TCP.RECV.ENOBUFS"));
                                          rTCPReceive.msgDesc = "TCP.RECV.ENOBUFS";
                                          break;
                    
                       case ENOMEM      : PrintMessage(debug_level, TCP_TABMSG, "ENOMEM");  
                                          rTCPReceive.msgDesc = malloc(strlen("TCP.RECV.ENOMEM"));
                                          rTCPReceive.msgDesc = "TCP.RECV.ENOMEM";
                                          break;
                    
                       default          : PrintMessage(debug_level, TCP_TABMSG, "SEND_TCP_UNKNOWN_ERROR");  
                                          rTCPReceive.msgDesc = malloc(strlen("TCP.RECV.SEND_TCP_UNKNOWN_ERROR"));
                                          rTCPReceive.msgDesc = "TCP.RECV.SEND_TCP_UNKNOWN_ERROR";
                                          break;
              };
 
              break;
          };

          /*
          .----------------------------------------------.
          |                     E O F                    |
          +----------------------------------------------+
          | Se il numero di byte ricevuti è 0 significa  |
          | che il server non ha più nulla da            |
          | trasmettere al client.                       |
          |                                              |
          '----------------------------------------------' */
          if (numByteReceive ==  0)
          {
              PrintMessage(debug_level, TCP_TABMSG, "TCP_RECEIVE_OK");
              rTCPReceive.idmsg       =  0 ;
              rTCPReceive.return_code =  0 ;
              break;   
          };

          /*
          .----------------------------------------------.
          | Mantiene il conteggio dei Byte ricevuti      |
          '----------------------------------------------' */
          totByteReceive = totByteReceive + numByteReceive ;

          /*
          .----------------------------------------------.
          | Accoda i Byte ricevuti nel file di output    |
          '----------------------------------------------' */
          PrintMessage(debug_level, TCP_TABMSG, "TCP_RECEIVE_WRITE_DATASTORE");
          fwrite(response, 1, strlen(response), pOutputFile);

    }

    /*
    .----------------------------------------------------.
    | Close OUTPUT                                       |
    +----------------------------------------------------+
    | Chiude il file di Output dopo aver terminato la    |
    | la lettura della coda.                             |
    '----------------------------------------------------' */
    PrintMessage(debug_level, TCP_TABMSG, "TCP_RECEIVE_CLOSE_DATASTORE");
    fclose (pOutputFile);
    PrintMessage(debug_level, TCP_TABMSG, "TCP_RECEIVE_OK");

    /*
    .----------------------------------------------------.
    | Creare Output Structure                            |
    +----------------------------------------------------+
    | Prepara la struttura dati da rilasciare al         |
    | chiamante.                                         |
    '----------------------------------------------------' */
    rTCPReceive.response = malloc ( __TCP_LEN_BUFFER_RECEIVE__);    
    memset(rTCPReceive.response, 0, __TCP_LEN_BUFFER_RECEIVE__);
 // rTCPReceive.response    = response;    
    rTCPReceive.responseLen = totByteReceive; 

    return rTCPReceive;
};



/*
.-------------------------------------------------------------------------------------.
|                                                                                     |
|  Funzione    : TCPDisconnect                                                        |
|                                                                                     |
|  Descrizione : Disconnect and close session TCP                                     |
|                                                                                     |
|  Return-code :  0 Ok                                                                |
|                                                                                     |
|                                                                                     |
|                                                                                     |
|                                                                                     |
|                                                                                     |
|                                                                                     |
|                                                                                     |
|                                                                                     |
|                                                                                     |
'-------------------------------------------------------------------------------------'*/
extern struct tTCPDisconnect TCPDisconnect(int sock, int debug_level)
{

    /*
    .--------------------------------------.
    | Dichiarazione strutture utilizzate   |
    '--------------------------------------' */
    struct tTCPDisconnect      rTCPDisconnect      ;
    struct tTCPDisconnect     *pTCPDisconnect      ;

    /*
    .------------------------------------.
    | Allocazione Spazio                 |
    '------------------------------------' */
    pTCPDisconnect           = malloc(sizeof(struct tTCPDisconnect));

    /*
    .------------------------------------.
    | Utilizzo dello spazio allocato     |
    '------------------------------------' */
    pTCPDisconnect           = &rTCPDisconnect;

    /*
    .-----------------------------------------.
    | Inizializzazione                        |
    '-----------------------------------------' */
    memset((void*)&rTCPDisconnect, 0, sizeof(rTCPDisconnect));
    rTCPDisconnect.idmsg           = 0 ;
    rTCPDisconnect.return_code     = 0 ;

    /*
    .------------------------.
    | Shutdown Socket        |
    '------------------------' */
    PrintMessage(debug_level, TCP_TABMSG, "TCP_DISCONNECT_SHUTDOWN");
    shutdown(sock, SHUT_RDWR);
    PrintMessage(debug_level, TCP_TABMSG, "TCP_DISCONNECT_OK");

    /*
    .------------------------.
    | Close Socket           |
    '------------------------' */
    PrintMessage(debug_level, TCP_TABMSG, "TCP_DISCONNECT_CLOSE");
    close(sock);
    PrintMessage(debug_level, TCP_TABMSG, "TCP_DISCONNECT_OK");

    return rTCPDisconnect;
}



/*
.-------------------------------------------------------------------------------------.
|                                                                                     |
|  Funzione    : TCPClientConnect                                                     |
|                                                                                     |
|  Descrizione : Minimal Client TCP                                                   |
|                                                                                     |
|  Return-code :  0 Ok                                                                |
|                -1 TCP Connect Error                                                 |
|                -2 SSL Connect Error *NotUsed*                                       |
|                -3 SSL Authentication Server Error *NotUsed*                         |
|                -4 Send TCP|SSL Error                                                |
|                -5 Receive TCP|SSL Error                                             |
|                -6 SSL Disconnect Error *NotUsed*                                    |
|                -7 TCP Disconnect                                                    |
|                                                                                     |
'-------------------------------------------------------------------------------------'*/
extern struct tClientConnect TCPClientConnect ( struct tClientConnectParameter rClientConnectParameter )
{
    /*                    
    .--------------------------------------.
    | Dichiarazione strutture utilizzate   |
    '--------------------------------------' */
    struct tClientConnect      rClientConnect      ;
    struct tClientConnect     *pClientConnect      ;

    struct tTCPConnect         rTCPConnect         ;
    struct tTCPConnect        *pTCPConnect         ;

    struct tTCPSend            rTCPSend            ;
    struct tTCPSend           *pTCPSend            ;

    struct tTCPReceive         rTCPReceive         ;
    struct tTCPReceive        *pTCPReceive         ;

    struct tTCPDisconnect      rTCPDisconnect      ;
    struct tTCPDisconnect     *pTCPDisconnect      ;

    /*
    .------------------------------------.
    | Allocazione Spazio                 |
    '------------------------------------' */
    pClientConnect           = malloc(sizeof(struct tClientConnect));
    pTCPConnect              = malloc(sizeof(struct tTCPConnect));
    pTCPSend                 = malloc(sizeof(struct tTCPSend));
    pTCPReceive              = malloc(sizeof(struct tTCPReceive));
    pTCPDisconnect           = malloc(sizeof(struct tTCPDisconnect));

    /*
    .------------------------------------.
    | Utilizzo dello spazio allocato     |
    '------------------------------------' */
    pClientConnect           = &rClientConnect;
    pTCPConnect              = &rTCPConnect;
    pTCPSend                 = &rTCPSend;
    pTCPReceive              = &rTCPReceive;
    pTCPDisconnect           = &rTCPDisconnect;

    /*
    .-----------------------------.
    | Inizializzazione Strutture  |
    '-----------------------------' */
    memset((void*)&rClientConnect     , 0, sizeof(rClientConnect));
    memset((void*)&rTCPConnect        , 0, sizeof(rTCPConnect));
    memset((void*)&rTCPSend           , 0, sizeof(rTCPSend));
    memset((void*)&rTCPReceive        , 0, sizeof(rTCPReceive));
    memset((void*)&rTCPDisconnect     , 0, sizeof(rTCPDisconnect));


    /*
    .------------------------------------------.
    |  Determina il livello di DEBUG           |
    '------------------------------------------' */
    int debug_level = GetDebugLevel ( rClientConnectParameter.debug, "tcp" ) ; 

    /*
    .-------------.
    | TCP Connect |
    '-------------' */
    rClientConnect.return_code = 0;
    rTCPConnect = TCPConnect(rClientConnectParameter.host, rClientConnectParameter.porta, debug_level);
    rClientConnect.TCPIp             = rTCPConnect.ip ; 
    rClientConnect.TCPPort           = rTCPConnect.port ;
    rClientConnect.TCPConnectMsg     = rTCPConnect.idmsg ;
    rClientConnect.TCPConnectMsgDesc = rTCPConnect.msgDesc;
    rClientConnect.TCPConnectReturn  = rTCPConnect.return_code ;
    if ( rTCPConnect.return_code != 0) 
    {
         rClientConnect.return_code = -1 ; 
         return rClientConnect ;
    };

    /*
    .-----------------------.
    | Request and Response  |
    '-----------------------' */
    if ( rClientConnectParameter.message != NULL )
    {
         /*
         .--------------.
         | Send Request |
         '--------------' */
         rTCPSend = TCPSend (rTCPConnect.sock, rClientConnectParameter.message, debug_level);
         rClientConnect.ServerRequestBufferLen    = rTCPSend.requestLen;
         rClientConnect.ServerRequestBuffer       = rTCPSend.request;
         rClientConnect.ServerRequestMsg          = rTCPSend.idmsg;
         rClientConnect.ServerRequestMsgDesc      = rTCPSend.msgDesc;
         rClientConnect.ServerRequestReturn       = rTCPSend.return_code;
         if ( rTCPSend.return_code != 0)
         {
              rClientConnect.return_code = -4 ;
         };
  
         /*
         .------------------.
         | Receive Response |
         '------------------' */
         if ( rTCPSend.return_code == 0 ) 
         { 
              rTCPReceive = TCPReceive (rTCPConnect.sock, rClientConnectParameter.OutputFile, debug_level);
              rClientConnect.ServerResponseBufferNum   = 0; 
              rClientConnect.ServerResponseBufferLen   = rTCPReceive.responseLen;
              rClientConnect.ServerResponseBuffer      = rTCPReceive.response;
              rClientConnect.ServerResponseMsg         = rTCPReceive.idmsg;
              rClientConnect.ServerResponseMsgDesc     = rTCPReceive.msgDesc;
              rClientConnect.ServerResponseReturn      = rTCPReceive.return_code;
              if ( rTCPReceive.return_code != 0)
              {
                   rClientConnect.return_code = -5 ;
              };
         };
    };

    /*
    .----------------.
    | TCP Disconnect |
    '----------------' */
    rTCPDisconnect = TCPDisconnect(rTCPConnect.sock, debug_level);
    rClientConnect.TCPDisconnectMsg    = rTCPDisconnect.idmsg ;
    rClientConnect.TCPDisconnectReturn = rTCPDisconnect.return_code ;
    if ( rTCPDisconnect.return_code != 0)
    {        
         rClientConnect.return_code = -7 ;
    };

    return rClientConnect ; 
};


// ______EOF_
