/*-GNU-GPL-BEGIN-*
nepim - network pipemeter
Copyright (C) 2005 Everton da Silva Marques

nepim is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2, or (at your option)
any later version.

nepim is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with nepim; see the file COPYING.  If not, write to
the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
Boston, MA 02111-1307, USA.
*-GNU-GPL-END-*/


/* $Id: client.c,v 1.6 2005/04/19 03:40:33 evertonm Exp $ */

#include <stdio.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netdb.h>
#include <assert.h>
#include <errno.h>

#include "conf.h"
#include "sock.h"
#include "pipe.h"
#include "common.h"

nepim_pipe_set_t pipes;

static void *on_tcp_interval(oop_source *src, struct timeval tv, void *user);
static void tcp_pipe_stop(int sd);
static void *on_tcp_rate_delay(oop_source *src, struct timeval tv, void *user);

static void schedule_stat_interval(int sd)
{
  nepim_pipe_t *pipe = nepim_pipe_set_get(&pipes, sd);
  assert(pipe);

  {
    int result = gettimeofday(&pipe->tv_interval, 0);
    assert(!result);
  }

  pipe->tv_interval.tv_sec += pipe->stat_interval;

  nepim_global.oop_src->on_time(nepim_global.oop_src,
				pipe->tv_interval,
				on_tcp_interval, pipe);
}

static void *on_tcp_read(oop_source *src, int sd,
                         oop_event event, void *user)
{
  char buf[nepim_global.tcp_read_size];
  int rd;
  nepim_pipe_t *pipe = user;

  assert(event == OOP_READ);
  assert(sd == pipe->sd);

  rd = read(sd, buf, nepim_global.tcp_read_size);
  if (rd < 1) {
    switch (errno) {
    case EINTR:
      fprintf(stderr, "read: EINTR on TCP socket %d\n", sd);
      return OOP_CONTINUE;
    case EAGAIN:
      fprintf(stderr, "read: EAGAIN on TCP socket %d\n", sd);
      return OOP_CONTINUE;
    }

    fprintf(stderr, "read: connection lost on TCP socket %d\n", sd);

    report_broken_pipe_stat(stdout, pipe);

    tcp_pipe_stop(sd);
    close(sd);

    return OOP_CONTINUE;
  }

  pipe->byte_total_recv += rd;
  pipe->byte_interval_recv += rd;
  ++pipe->total_reads;
  ++pipe->interval_reads;

  return OOP_CONTINUE;
}

static void *on_tcp_write(oop_source *src, int sd,
                          oop_event event, void *user)
{
  char buf[nepim_global.tcp_write_size];
  int wr;
  nepim_pipe_t *pipe = user;

  assert(event == OOP_WRITE);
  assert(sd == pipe->sd);
  assert(pipe->max_bit_rate < 1);

  wr = write(sd, buf, nepim_global.tcp_write_size);
  if (wr < 1) {
    switch (errno) {
    case EINTR:
      fprintf(stderr, "write: EINTR on TCP socket %d\n", sd);
      return OOP_CONTINUE;
    case EAGAIN:
      fprintf(stderr, "write: EAGAIN on TCP socket %d\n", sd);
      return OOP_CONTINUE;
    case EPIPE:
      fprintf(stderr, "write: EPIPE on TCP socket %d\n", sd);
      break;
    }

    fprintf(stderr, "write: connection lost on TCP socket %d\n", sd);

    report_broken_pipe_stat(stdout, pipe);

    tcp_pipe_stop(sd);
    close(sd);

    return OOP_CONTINUE;
  }

  pipe->byte_total_sent += wr;
  pipe->byte_interval_sent += wr;
  ++pipe->total_writes;
  ++pipe->interval_writes;

  return OOP_CONTINUE;
}

static void *on_tcp_rate_write(oop_source *src, int sd,
			       oop_event event, void *user)
{
  char buf[nepim_global.tcp_write_size];
  int wr;
  nepim_pipe_t *pipe = user;
  int to_write;

  assert(event == OOP_WRITE);
  assert(sd == pipe->sd);
  assert(pipe->max_bit_rate > 0);

  to_write = NEPIM_MIN(pipe->rate_remaining, nepim_global.tcp_write_size);

  wr = write(sd, buf, to_write);
  if (wr < 1) {
    switch (errno) {
    case EINTR:
      fprintf(stderr, "rate_write: EINTR on TCP socket %d\n", sd);
      return OOP_CONTINUE;
    case EAGAIN:
      fprintf(stderr, "rate_write: EAGAIN on TCP socket %d\n", sd);
      return OOP_CONTINUE;
    case EPIPE:
      fprintf(stderr, "rate_write: EPIPE on TCP socket %d\n", sd);
      break;
    }

    fprintf(stderr, "rate_write: connection lost on TCP socket %d\n", sd);

    report_broken_pipe_stat(stdout, pipe);

    tcp_pipe_stop(sd);
    close(sd);

    return OOP_CONTINUE;
  }

  pipe->byte_total_sent += wr;
  pipe->byte_interval_sent += wr;
  pipe->rate_remaining -= wr;
  ++pipe->total_writes;
  ++pipe->interval_writes;

#if 0
  fprintf(stderr, 
	  "DEBUG %s %s %d: to_write=%d written=%d missing=%d\n",
	  __FILE__, __PRETTY_FUNCTION__,
	  sd, to_write, wr, pipe->rate_remaining);
#endif

  /* finished ? */
  if (pipe->rate_remaining < 1) {
    /* stop writing */
    nepim_global.oop_src->cancel_fd(nepim_global.oop_src,
				    sd, OOP_WRITE);

    /* schedule for next time saved by on_tcp_rate_delay() */
    nepim_global.oop_src->on_time(nepim_global.oop_src,
				  pipe->tv_rate,
				  on_tcp_rate_delay, pipe);
  }

  return OOP_CONTINUE;
}

static void *on_tcp_rate_delay(oop_source *src, struct timeval tv, void *user)
{
  nepim_pipe_t *pipe = user;
  long long tmp;

  assert(timercmp(&tv, &pipe->tv_rate, ==));
  assert(pipe->max_bit_rate > 0);

  /* save next scheduling time */
  {
    int result = gettimeofday(&pipe->tv_rate, 0);
    assert(!result);
  }
  pipe->tv_rate.tv_usec += pipe->write_delay;
  if (pipe->tv_rate.tv_usec >= 1000000) {
    pipe->tv_rate.tv_usec -= 1000000;
    ++pipe->tv_rate.tv_sec;
  }

  /* calculate bytes to be written from rate */
  tmp = pipe->max_bit_rate;
  tmp *= pipe->write_delay;
  tmp /= 8000000;
  pipe->rate_remaining = tmp;

#if 0
  fprintf(stderr, 
	  "DEBUG %s %s %d: "
	  "(bit_rate * delay / 8M = bytes) "
	  "%d * %ld / 8M = %d\n",
	  __FILE__, __PRETTY_FUNCTION__,
	  pipe->sd, 
	  pipe->max_bit_rate,
	  pipe->write_delay, 
	  pipe->rate_remaining);
#endif

  /* start to write */
  nepim_global.oop_src->on_fd(nepim_global.oop_src,
			      pipe->sd, OOP_WRITE,
			      on_tcp_rate_write, pipe);

  return OOP_CONTINUE;
}

static void *on_tcp_duration(oop_source *src, struct timeval tv, void *user)
{
  nepim_pipe_t *pipe = user;

  assert(timercmp(&tv, &pipe->tv_duration, ==));

  nepim_dump_stat(stdout, NEPIM_LABEL_TOTAL, pipe->sd, 
		  pipe->byte_total_recv,
		  pipe->byte_total_sent,
		  pipe->test_duration,
		  pipe->tv_start.tv_sec,
		  pipe->test_duration,
		  pipe->total_reads,
		  pipe->total_writes);

  pipe->byte_total_recv = 0;
  pipe->byte_total_sent = 0;
  pipe->total_reads     = 0;
  pipe->total_writes    = 0;

  tcp_pipe_stop(pipe->sd);
  close(pipe->sd);

  return OOP_CONTINUE;
}

static void *on_tcp_interval(oop_source *src, struct timeval tv, void *user)
{
  nepim_pipe_t *pipe = user;

  assert(timercmp(&tv, &pipe->tv_interval, ==));

  nepim_dump_stat(stdout, NEPIM_LABEL_PARTIAL, pipe->sd, 
		  pipe->byte_interval_recv,
		  pipe->byte_interval_sent,
		  pipe->stat_interval,
		  pipe->tv_start.tv_sec,
		  pipe->test_duration,
		  pipe->interval_reads,
		  pipe->interval_writes);

  pipe->byte_interval_recv = 0;
  pipe->byte_interval_sent = 0;
  pipe->interval_reads     = 0;
  pipe->interval_writes    = 0;

  schedule_stat_interval(pipe->sd);

  return OOP_CONTINUE;
}

static int greet(int sd)
{
  char buf[1024];
  int pr;
  int wr;

  pr = snprintf(buf, sizeof(buf),
		"hello server_send=%d bit_rate=%lld "
		"stat_interval=%d test_duration=%d "
		"write_delay=%ld\n", 
		nepim_global.duplex_mode,
		nepim_global.bit_rate,
		nepim_global.stat_interval,
		nepim_global.test_duration,
		nepim_global.write_delay);
  assert(pr > 0);
  assert(pr < sizeof(buf));

  fprintf(stderr, "%d: sending: %s", sd, buf);

  wr = write(sd, buf, pr);
  if (wr != pr)
    return -1;

  return 0;
}

static void tcp_pipe_start(int sd)
{
  nepim_pipe_t *pipe = nepim_pipe_set_get(&pipes, sd);
  assert(pipe);

  nepim_global.oop_src->on_fd(nepim_global.oop_src,
			      sd, OOP_READ,
			      on_tcp_read, pipe);

  if (pipe->must_send) {
    if (pipe->max_bit_rate > 0) {
      pipe->tv_rate = OOP_TIME_NOW;
      nepim_global.oop_src->on_time(nepim_global.oop_src,
				    pipe->tv_rate,
				    on_tcp_rate_delay, pipe);
    }
    else {
      nepim_global.oop_src->on_fd(nepim_global.oop_src,
				  sd, OOP_WRITE,
				  on_tcp_write, pipe);
    }
  }


  {
    int result = gettimeofday(&pipe->tv_start, 0);
    assert(!result);
    pipe->tv_duration = pipe->tv_start;
  }
  pipe->tv_duration.tv_sec += pipe->test_duration;
  nepim_global.oop_src->on_time(nepim_global.oop_src,
				pipe->tv_duration,
				on_tcp_duration, pipe);

  schedule_stat_interval(sd);
}

static void tcp_pipe_stop(int sd)
{
  nepim_pipe_t *pipe = nepim_pipe_set_get(&pipes, sd);
  assert(pipe);

  nepim_global.oop_src->cancel_fd(nepim_global.oop_src,
				  sd, OOP_READ);

  if (pipe->must_send) {
    if (pipe->max_bit_rate > 0) {
      /* stop current writing, if any */
      nepim_global.oop_src->cancel_fd(nepim_global.oop_src,
				      sd, OOP_WRITE);

      /* stop periodic write scheduler */
      nepim_global.oop_src->cancel_time(nepim_global.oop_src,
					pipe->tv_rate,
					on_tcp_rate_delay, pipe);
    }
    else {
      nepim_global.oop_src->cancel_fd(nepim_global.oop_src,
				      sd, OOP_WRITE);
    }
  }

  nepim_global.oop_src->cancel_time(nepim_global.oop_src,
				    pipe->tv_duration,
				    on_tcp_duration, pipe);

  nepim_global.oop_src->cancel_time(nepim_global.oop_src,
				    pipe->tv_interval,
				    on_tcp_interval, pipe);

  nepim_pipe_set_del(&pipes, sd);
}

static void spawn_one_tcp_client()
{
  struct addrinfo hints;
  struct addrinfo *ai_res;
  struct addrinfo *ai;
  int result;

  hints.ai_socktype = SOCK_STREAM;
  hints.ai_protocol = IPPROTO_TCP;
  hints.ai_flags = AI_CANONNAME;
  hints.ai_family = PF_UNSPEC;
  hints.ai_addrlen = 0;
  hints.ai_addr = 0;
  hints.ai_canonname = 0;

  fprintf(stderr, 
	  "TCP socket solving %s,%d\n",
	  nepim_global.hostname, nepim_global.port);

  result = getaddrinfo(nepim_global.hostname, 0, &hints, &ai_res);
  if (result) {
    fprintf(stderr, "getaddrinfo(%s): %s\n",
            nepim_global.hostname, gai_strerror(result));
    return;
  }

  for (ai = ai_res; ai; ai = ai->ai_next) {
    int sd;

    fprintf(stderr, 
	    "TCP socket trying %s,%d\n",
	    nepim_global.hostname, nepim_global.port);

    sd = nepim_connect_client_socket(ai->ai_addr, ai->ai_addrlen,
				     ai->ai_family, ai->ai_socktype, 
				     ai->ai_protocol, nepim_global.port);
    if (sd < 0) {
      fprintf(stderr, 
	      "could not connect TCP socket to %s,%d: %d\n",
	      nepim_global.hostname, nepim_global.port, sd);
      continue;
    }

    fprintf(stderr, 
	    "%d: TCP socket connected to: "
	    "host=%s,%d len=%d family=%d type=%d proto=%d\n",
	    sd, nepim_global.hostname, nepim_global.port, 
	    ai->ai_addrlen, ai->ai_family, 
	    ai->ai_socktype, ai->ai_protocol);

    nepim_pipe_set_add(&pipes, sd, 1,
		       nepim_global.bit_rate, 
		       nepim_global.stat_interval,
		       nepim_global.test_duration,
		       nepim_global.write_delay);

    break;
  }

  freeaddrinfo(ai_res);
}

static void spawn_tcp_clients()
{
  int i;

  /* create sockets */
  for (i = 0; i < nepim_global.pipes; ++i)
    spawn_one_tcp_client();

  /* activate sockets */
  for (i = 0; i < pipes.capacity; ++i)
    if (nepim_pipe_set_get(&pipes, i)) {

      if (greet(i)) {
	fprintf(stderr,
		"%d: could not greet %s,%d\n",
		i, nepim_global.hostname, nepim_global.port);

	nepim_pipe_set_del(&pipes, i);
	close(i);

	continue;
      }

      fprintf(stderr, 
	      "%d: greetings sent to %s,%d\n",
	      i, nepim_global.hostname, nepim_global.port);

      tcp_pipe_start(i);
    }
}

void nepim_udp_clients();

void nepim_client_run()
{
  void *result;

  nepim_pipe_set_init(&pipes);

  if (nepim_global.udp_mode)
    nepim_udp_clients();
  else
    spawn_tcp_clients();

  result = oop_sys_run(nepim_global.oop_sys);
}
