/*-GNU-GPL-BEGIN-*
nepim - network pipemeter
Copyright (C) 2005 Everton da Silva Marques

nepim is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2, or (at your option)
any later version.

nepim is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with nepim; see the file COPYING.  If not, write to
the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
Boston, MA 02111-1307, USA.
*-GNU-GPL-END-*/


/* $Id: sock.c,v 1.6 2005/04/19 04:11:51 evertonm Exp $ */


#include <assert.h>
#include <netdb.h>
#include <unistd.h>
#include <fcntl.h>
#include <stdio.h>
#include <errno.h>
#include <string.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <netinet/tcp.h>

#include "sock.h"

/* 
   TCP options work-around for
   Solaris' lack of SOL_TCP.
*/
#ifndef SOL_TCP
#define SOL_TCP 6
#endif

#define NEPIM_SOCK_ERR_NONE     (0)
#define NEPIM_SOCK_ERR_UNSPEC   (-1)
#define NEPIM_SOCK_ERR_SOCKET   (-2)
#define NEPIM_SOCK_ERR_BIND     (-3)
#define NEPIM_SOCK_ERR_LISTEN   (-4)
#define NEPIM_SOCK_ERR_CONNECT  (-5)
#define NEPIM_SOCK_ERR_BLOCK    (-6)
#define NEPIM_SOCK_ERR_UNBLOCK  (-7)
#define NEPIM_SOCK_ERR_UNLINGER (-8)
#define NEPIM_SOCK_ERR_REUSE    (-9)
#define NEPIM_SOCK_ERR_NODELAY  (-10)


static void set_port(struct sockaddr *addr, int family, int port)
{
  union {
    struct sockaddr_in inet;
    struct sockaddr_in6 inet6;
  } *sa = (void *) addr;

  switch(family) {
  case PF_INET:
    sa->inet.sin_port = htons(port);
    break;
  case PF_INET6:
    sa->inet6.sin6_port = htons(port);
    break;
  default:
    assert(0);
  }
}

int nepim_socket_block(int sd)
{
  long flags;

  flags = fcntl(sd, F_GETFL, 0);
  if (flags == -1)
    return NEPIM_SOCK_ERR_BLOCK;
  assert(flags >= 0);
  if (fcntl(sd, F_SETFL, flags & ~O_NONBLOCK))
    return NEPIM_SOCK_ERR_BLOCK;

  return NEPIM_SOCK_ERR_NONE;
}

int nepim_socket_nonblock(int sd)
{
  long flags;

  flags = fcntl(sd, F_GETFL, 0);
  if (flags == -1)
    return NEPIM_SOCK_ERR_UNBLOCK;
  assert(flags >= 0);
  if (fcntl(sd, F_SETFL, flags | O_NONBLOCK))
    return NEPIM_SOCK_ERR_UNBLOCK;

  return NEPIM_SOCK_ERR_NONE;
}

static int create_socket(int domain, int type, int protocol)
{
  int sd;
  int result;

  sd = socket(domain, type, protocol);
  if (sd < 0)
    return NEPIM_SOCK_ERR_SOCKET;

  if (type == SOCK_STREAM) {
    result = nepim_socket_tcp_opt(sd);
    if (result)
      return result;
  }

  result = nepim_socket_opt(sd);
  if (result)
    return result;

  return sd;
}

int nepim_create_socket(struct sockaddr *addr,
			int addr_len,
			int family,
			int type,
			int protocol,
			int port)
{
  int sd;
  int result;

  sd = create_socket(family, type, protocol);
  if (sd < 0)
    return sd;

  result = nepim_socket_nonblock(sd);
  if (result) {
    close(sd);
    return result;
  }

  set_port(addr, family, port);

  if (bind(sd, addr, addr_len)) {
    close(sd);
    return NEPIM_SOCK_ERR_BIND;
  }

  return sd;
}

int nepim_create_listener_socket(struct sockaddr *addr,
				 int addr_len,
				 int family,
				 int type,
				 int protocol,
				 int port)
{
  int sd;

  sd = nepim_create_socket(addr, addr_len, family, \
			   type, protocol, port);
  if (sd < 0)
    return sd;

  if (listen(sd, 3)) {
    close(sd);
    return NEPIM_SOCK_ERR_LISTEN;
  }

  return sd;
}

static int unlinger(int sd)
{
  struct linger opt;

  opt.l_onoff = 0;  /* active? */
  opt.l_linger = 0; /* seconds */

  return setsockopt(sd, SOL_SOCKET, SO_LINGER, &opt, sizeof(opt));
}

static int reuse(int sd)
{
  int opt = 1;

  return setsockopt(sd, SOL_SOCKET, SO_REUSEADDR, &opt, sizeof(opt));
}

static int tcp_nodelay(int sd)
{
  int opt = 1;

  return setsockopt(sd, SOL_TCP, TCP_NODELAY, &opt, sizeof(opt));
}

int nepim_socket_opt(int sd)
{
  if (unlinger(sd))
    return NEPIM_SOCK_ERR_UNLINGER;

  if (reuse(sd))
    return NEPIM_SOCK_ERR_REUSE;

  return NEPIM_SOCK_ERR_NONE;
}

int nepim_socket_tcp_opt(int sd)
{
  if (tcp_nodelay(sd))
    return NEPIM_SOCK_ERR_NODELAY;

  return NEPIM_SOCK_ERR_NONE;
}

int nepim_connect_client_socket(struct sockaddr *addr,
				int addr_len,
				int family,
				int type,
				int protocol,
				int port)
{
  int sd;
  int result;

  sd = create_socket(family, type, protocol);
  if (sd < 0)
    return sd;

  result = nepim_socket_block(sd);
  if (result) {
    close(sd);
    return result;
  }

  set_port(addr, family, port);

  fprintf(stderr, 
	  "DEBUG FIXME %s %s slow synchronous connect()\n",
	  __FILE__, __PRETTY_FUNCTION__);

  if (connect(sd, addr, addr_len)) {
    close(sd);
    return NEPIM_SOCK_ERR_CONNECT;
  }

  result = nepim_socket_nonblock(sd);
  if (result) {
    close(sd);
    return result;
  }

  return sd;
}
