/*-GNU-GPL-BEGIN-*
nepim - network pipemeter
Copyright (C) 2005 Everton da Silva Marques

nepim is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2, or (at your option)
any later version.

nepim is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with nepim; see the file COPYING.  If not, write to
the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
Boston, MA 02111-1307, USA.
*-GNU-GPL-END-*/


/* $Id: udp_server.c,v 1.21 2005/07/27 18:51:12 evertonm Exp $ */


#include <stdio.h>
#include <string.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netdb.h>
#include <errno.h>
#include <oop.h>
#include <assert.h>

#include "conf.h"
#include "sock.h"
#include "common.h"
#include "slot.h"
#include "usock.h"


/* version(1) + local_slot(2) + remote_lost(2) + seq(4) */
const int UDP_HEADER_LEN = 9; 

extern nepim_slot_set_t slots; /* from server.c */
extern nepim_usock_set_t udp_tab; /* from server.c */

static void schedule_stat_interval(int local_slot);
static void udp_slot_cancel_timers(nepim_slot_t *slot);
static void *on_udp_rate_delay(oop_source *src, struct timeval tv, void *user);

static void *on_udp_interval(oop_source *src, struct timeval tv, void *user)
{
  nepim_slot_t *slot = user;
  nepim_session_t *session = &slot->session;

  nepim_slot_stat(stdout, NEPIM_LABEL_PARTIAL, 
		  slot->udp_sd,
		  slot->index,
		  slot->index_remote,
		  session->byte_interval_recv,
		  session->byte_interval_sent,
		  session->stat_interval,
		  session->tv_start.tv_sec,
		  session->test_duration,
		  session->interval_reads,
		  session->interval_writes);

  session->byte_interval_recv = 0;
  session->byte_interval_sent = 0;
  session->interval_reads     = 0;
  session->interval_writes    = 0;

  schedule_stat_interval(slot->udp_sd);

  return OOP_CONTINUE;
}

static void schedule_stat_interval(int local_slot)
{
  nepim_slot_t *slot = nepim_slot_set_get(&slots, local_slot);
  nepim_session_t *session;
  assert(slot);

  session = &slot->session;

  {
    int result = gettimeofday(&session->tv_interval, 0);
    assert(!result);
  }

  session->tv_interval.tv_sec += session->stat_interval;

  nepim_global.oop_src->on_time(nepim_global.oop_src,
				session->tv_interval,
				on_udp_interval, slot);
}

static void *on_udp_duration(oop_source *src, struct timeval tv, void *user)
{
  nepim_slot_t *slot = user;
  nepim_session_t *session = &slot->session;
  int sd = slot->udp_sd;

  nepim_slot_stat(stdout, NEPIM_LABEL_TOTAL, 
		  sd, 
		  slot->index,
		  slot->index_remote,
		  session->byte_total_recv,
		  session->byte_total_sent,
		  session->test_duration,
		  session->tv_start.tv_sec,
		  session->test_duration,
		  session->total_reads,
		  session->total_writes);

  session->byte_total_recv = 0;
  session->byte_total_sent = 0;
  session->total_reads     = 0;
  session->total_writes    = 0;
  session->duration_done   = 1;

  udp_slot_cancel_timers(slot);

  return OOP_CONTINUE;
}

static void udp_slot_cancel_timers(nepim_slot_t *slot)
{
  nepim_session_t *session = &slot->session;

  nepim_global.oop_src->cancel_time(nepim_global.oop_src,
				    session->tv_duration,
				    on_udp_duration, slot);

  nepim_global.oop_src->cancel_time(nepim_global.oop_src,
				    session->tv_interval,
				    on_udp_interval, slot);
}

static void udp_slot_cancel_io(nepim_slot_t *slot)
{
  nepim_session_t *session = &slot->session;
  int sd = slot->udp_sd;

  nepim_global.oop_src->cancel_fd(nepim_global.oop_src,
				  sd, OOP_READ);

  if (session->must_send) {
    if (session->max_bit_rate > 0) {
      /* stop current writing, if any */
      nepim_global.oop_src->cancel_fd(nepim_global.oop_src,
				      sd, OOP_WRITE);
      
      /* stop periodic write scheduler */
      nepim_global.oop_src->cancel_time(nepim_global.oop_src,
					session->tv_rate,
					on_udp_rate_delay, slot);
    }
    else {
      nepim_global.oop_src->cancel_fd(nepim_global.oop_src,
				      sd, OOP_WRITE);
    }
  }
}

static void udp_slot_kill(nepim_slot_t *slot)
{
  int slot_index = slot->index;

  udp_slot_cancel_timers(slot);
  udp_slot_cancel_io(slot);

  nepim_sock_show_opt(stderr, slot->udp_sd);

  nepim_slot_set_del(&slots, slot_index);
}

static int udp_write(nepim_slot_t *slot, size_t len)
{
  char buf[len];
  ssize_t wr;

  assert(sizeof(buf) == len);
  assert(len >= UDP_HEADER_LEN);

  *buf                   = 0; /* version=0 */
  *(uint16_t *)(buf + 1) = htons(slot->index_remote);
  *(uint16_t *)(buf + 3) = htons(slot->index);
  *(uint32_t *)(buf + 5) = htonl(slot->seq++);

  wr = sendto(slot->udp_sd, buf, len, 0,
	      (const struct sockaddr *) &slot->remote,
	      slot->remote_len);

  return wr;
}

static void *on_udp_rate_write(oop_source *src, int sd,
			       oop_event event, void *user)
{
  int wr;
  nepim_slot_t *slot = user;
  nepim_session_t *session = &slot->session;
  int to_write;

  assert(event == OOP_WRITE);
  assert(sd == slot->udp_sd);
  assert(session->max_bit_rate > 0);

  to_write = NEPIM_MIN(session->rate_remaining, nepim_global.udp_write_size);

  wr = udp_write(slot, to_write);
  if (wr < 1) {
    switch (errno) {
    case EINTR:
      fprintf(stderr, "rate_write: EINTR on UDP socket %d\n", sd);
      return OOP_CONTINUE;
    case EAGAIN:
      fprintf(stderr, "rate_write: EAGAIN on UDP socket %d\n", sd);
      return OOP_CONTINUE;
    case EPIPE:
      fprintf(stderr, "rate_write: EPIPE on UDP socket %d\n", sd);
      break;
    }

    fprintf(stderr, "rate_write: connection lost on UDP socket %d\n", sd);

    if (!session->duration_done)
      report_broken_slot_stat(stdout, slot);

    udp_slot_kill(slot);

    return OOP_CONTINUE;
  }

  session->byte_total_sent += wr;
  session->byte_interval_sent += wr;
  session->rate_remaining -= wr;
  ++session->total_writes;
  ++session->interval_writes;

#if 0
  fprintf(stderr, 
	  "DEBUG %s %s %d: to_write=%d written=%d missing=%d\n",
	  __FILE__, __PRETTY_FUNCTION__,
	  sd, to_write, wr, session->rate_remaining);
#endif

  /* finished ? */
  if (session->rate_remaining < 1) {
    /* stop writing */
    nepim_global.oop_src->cancel_fd(nepim_global.oop_src,
				    sd, OOP_WRITE);

    /* schedule for next time saved by on_udp_rate_delay() */
    nepim_global.oop_src->on_time(nepim_global.oop_src,
				  session->tv_rate,
				  on_udp_rate_delay, slot);
  }

  return OOP_CONTINUE;
}

static void *on_udp_rate_delay(oop_source *src, struct timeval tv, void *user)
{
  nepim_slot_t *slot = user;
  nepim_session_t *session = &slot->session;
  long long tmp;

  assert(timercmp(&tv, &session->tv_rate, ==));
  assert(session->max_bit_rate > 0);

  /* save next scheduling time */
  {
    int result = gettimeofday(&session->tv_rate, 0);
    assert(!result);
  }
  session->tv_rate.tv_usec += session->write_delay;
  if (session->tv_rate.tv_usec >= 1000000) {
    session->tv_rate.tv_usec -= 1000000;
    ++session->tv_rate.tv_sec;
  }

  /* calculate bytes to be written from rate */
  tmp = session->max_bit_rate;
  tmp *= session->write_delay;
  tmp /= 8000000;
  session->rate_remaining = tmp;

#if 0
  fprintf(stderr, 
	  "DEBUG %s %s %d: "
	  "(bit_rate * delay / 8M = bytes) "
	  "%d * %ld / 8M = %d\n",
	  __FILE__, __PRETTY_FUNCTION__,
	  slot->udp_sd, 
	  session->max_bit_rate,
	  session->write_delay, 
	  session->rate_remaining);
#endif

  /* start to write */
  nepim_global.oop_src->on_fd(nepim_global.oop_src,
			      slot->udp_sd, OOP_WRITE,
			      on_udp_rate_write, slot);

  return OOP_CONTINUE;
}

static void *on_udp_write(oop_source *src, int sd,
			  oop_event event, void *user)
{
  int wr;
  nepim_slot_t *slot = user;
  nepim_session_t *session = &slot->session;

  wr = udp_write(slot, nepim_global.udp_write_size);
  if (wr < 1) {
    switch (errno) {
    case EINTR:
      fprintf(stderr, "write: EINTR on UDP socket %d\n", sd);
      return OOP_CONTINUE;
    case EAGAIN:
      fprintf(stderr, "write: EAGAIN on UDP socket %d\n", sd);
      return OOP_CONTINUE;
    case EPIPE:
      fprintf(stderr, "write: EPIPE on UDP socket %d\n", sd);
      break;
    }

    fprintf(stderr, "write: connection lost on UDP socket %d\n", sd);

    if (!session->duration_done)
      report_broken_slot_stat(stdout, slot);

    udp_slot_kill(slot);

    return OOP_CONTINUE;
  }

  session->byte_total_sent += wr;
  session->byte_interval_sent += wr;
  ++session->total_writes;
  ++session->interval_writes;

  return OOP_CONTINUE;
}

static void udp_slot_start(int local_slot)
{
  nepim_slot_t *slot = nepim_slot_set_get(&slots, local_slot);
  nepim_session_t *session;
  int sd;
  assert(slot);

  session = &slot->session;
  sd = slot->udp_sd;

  if (session->must_send) {
    if (session->max_bit_rate > 0) {
      session->tv_rate = OOP_TIME_NOW;
      nepim_global.oop_src->on_time(nepim_global.oop_src,
				    session->tv_rate,
				    on_udp_rate_delay, slot);
    }
    else {
      nepim_global.oop_src->on_fd(nepim_global.oop_src,
				  sd, OOP_WRITE,
				  on_udp_write, slot);
    }
  }

  {
    int result = gettimeofday(&session->tv_start, 0);
    assert(!result);
    session->tv_duration = session->tv_start;
  }
  session->tv_duration.tv_sec += session->test_duration;
  nepim_global.oop_src->on_time(nepim_global.oop_src,
				session->tv_duration,
				on_udp_duration, slot);

  nepim_sock_show_opt(stderr, sd);

  schedule_stat_interval(local_slot);
}

static int parse_greetings(int sd, int *local_slot, int remote_slot,
			   char *buf, const int buf_len,
			   const struct sockaddr *remote, socklen_t remote_len)
{
  nepim_greet_t opt;
  int result;
  char *eos;
  char *past_end;

  eos = memchr(buf, '\n', buf_len);
  if (eos)
    past_end = eos;
  else
    past_end = buf + buf_len;

  assert(past_end >= buf);
  assert(past_end <= (buf + buf_len));

  *past_end = '\0';

  fprintf(stderr, "DEBUG %s %s udp_buf_len=%d eos_offset=%d\n",
	  __FILE__, __PRETTY_FUNCTION__, buf_len, past_end - buf);

  result = nepim_parse_greetings(&opt, buf, past_end);
  if (result)
    return result;

  assert(*local_slot == 0xFFFF);

  *local_slot = nepim_slot_find_free(&slots);

  fprintf(stderr, 
	  "%d %d-%d: UDP incoming: send=%d max_rate=%lld "
	  "interval=%d duration=%d delay=%ld\n", 
	  sd, *local_slot, remote_slot,
	  opt.must_send, opt.bit_rate, 
	  opt.stat_interval, opt.test_duration, 
	  opt.write_delay);

  nepim_slot_set_add(&slots, sd, 
		     *local_slot, remote_slot,
		     remote, remote_len, &opt);

  return 0;
}

static void udp_consume(nepim_slot_t *slot, size_t len)
{
  fprintf(stderr, 
	  "DEBUG FIXME %s %s: %d %d-%d: %d bytes\n",
	  __FILE__, __PRETTY_FUNCTION__,
	  slot->udp_sd, slot->index, slot->index_remote,
	  len);
}

static void *on_udp_segment(oop_source *src, int sd,
                            oop_event event, void *unnused)
{
  int rd;
  char buf[nepim_global.udp_read_size];
  union {
    struct sockaddr_in inet;
    struct sockaddr_in6 inet6;
  } from;
  socklen_t fromlen = sizeof(from);
  int version;
  int local_slot;
  int remote_slot;
  uint32_t seq;

  assert(sizeof(buf) == nepim_global.udp_read_size);

  rd = recvfrom(sd, buf, nepim_global.udp_read_size, 0, 
		(struct sockaddr *) &from, &fromlen);
  if (rd < 1) {
    switch (errno) {
    case EINTR:
      fprintf(stderr, "recvfrom: EINTR on UDP socket %d\n", sd);
      return OOP_CONTINUE;
    case EAGAIN:
      fprintf(stderr, "recvfrom: EAGAIN on UDP socket %d\n", sd);
      return OOP_CONTINUE;
    }

    fprintf(stderr, "recvfrom: failure on UDP socket %d\n", sd);

    return OOP_CONTINUE;
  }

  if (rd < UDP_HEADER_LEN) {
    fprintf(stderr, 
	    "recvfrom: too short UDP segment: %d bytes\n",
	    rd);
    return OOP_CONTINUE;
  }

  assert(rd >= UDP_HEADER_LEN);
  assert(rd < sizeof(buf));

  version     = *buf;
  local_slot  = ntohs((uint16_t) *(buf + 1));
  remote_slot = ntohs((uint16_t) *(buf + 3));
  seq         = ntohl((uint32_t) *(buf + 5));

  fprintf(stderr, "DEBUG FIXME ERASEME %s %s forcing version=0 local_slot=0xFFFF\n",
	  __FILE__, __PRETTY_FUNCTION__);
  version    = 0;      /* DEBUG FIXME ERASEME */
  local_slot = 0xFFFF; /* DEBUG FIXME ERASEME */

  if (version) {
    fprintf(stderr, "recvfrom: unknown UDP header version: %d\n",
	    version);
    return OOP_CONTINUE;
  }

  /* request for new slot? */
  if (local_slot == 0xFFFF) {
    int result;

    fprintf(stderr, 
	    "DEBUG %s %s: new slot request: %d(%d-%d)\n",
	    __FILE__, __PRETTY_FUNCTION__, 
	    sd, local_slot, remote_slot);

    result = parse_greetings(sd, &local_slot, remote_slot, 
			     buf + UDP_HEADER_LEN, 
			     rd - UDP_HEADER_LEN,
			     (const struct sockaddr *) &from,
			     fromlen);
    if (result) {
      fprintf(stderr, 
	      "%s: %s: bad greetings from UDP socket %d(%d-%d): %d\n",
	      __FILE__, __PRETTY_FUNCTION__, sd, local_slot, 
	      remote_slot, result);
      return OOP_CONTINUE;
    }

    assert(local_slot < 0xFFFF);

    udp_slot_start(local_slot);

    return OOP_CONTINUE;

  } /* new slot request */

  {
    nepim_slot_t *slot = nepim_slot_set_search(&slots, local_slot);

    if (!slot) {
      fprintf(stderr, "%d: unknown UDP local slot: %d\n",
	      sd, local_slot);
      return OOP_CONTINUE;
    }

    if (slot->index_remote != remote_slot)
      fprintf(stderr, 
	      "%d %d: remote index mismatch: expected=%d got=%d\n",
	      sd, local_slot, slot->index_remote, remote_slot);

    fprintf(stderr, 
	    "DEBUG FIXME %s %s: existing slot: %d(%d-%d)\n",
	    __FILE__, __PRETTY_FUNCTION__, 
	    sd, local_slot, remote_slot);

    udp_consume(slot, rd);
  }

  return OOP_CONTINUE;
}

int nepim_udp_listener(const char *hostname, const char *portname, int mcast_join)
{
  struct addrinfo hints;
  struct addrinfo *ai_res;
  struct addrinfo *ai;
  int result;
  int udp_listeners = 0;

  hints.ai_socktype = SOCK_DGRAM;
  hints.ai_protocol = IPPROTO_UDP;
  hints.ai_flags = AI_CANONNAME;
  hints.ai_family = PF_UNSPEC;
  hints.ai_addrlen = 0;
  hints.ai_addr = 0;
  hints.ai_canonname = 0;

  result = getaddrinfo(hostname, 0, &hints, &ai_res);
  if (result) {
    fprintf(stderr, "getaddrinfo(%s, %s): %s\n",
            hostname, portname, gai_strerror(result));
    return 0;
  }

  for (ai = ai_res; ai; ai = ai->ai_next) {
    int sd;

#if 0
    fprintf(stderr, 
	    "UDP listener: host=%s,%s len=%d family=%d type=%d proto=%d\n",
	    hostname, portname, 
	    ai->ai_addrlen, ai->ai_family, 
	    ai->ai_socktype, ai->ai_protocol);
#endif

    if (nepim_global.no_inet6 && (ai->ai_family == PF_INET6))
      continue;

    sd = nepim_create_socket(ai->ai_addr, ai->ai_addrlen,
			     ai->ai_family, ai->ai_socktype, 
			     ai->ai_protocol,
			     nepim_global.pmtu_mode,
			     nepim_global.ttl,
			     mcast_join);
    if (sd < 0) {
      fprintf(stderr, 
	      "spawn_udp_listener: UDP listener socket failed for %s,%s: %d\n",
	      hostname, portname, sd);
      break;
    }

    nepim_global.oop_src->on_fd(nepim_global.oop_src,
			  	sd, OOP_READ,
				on_udp_segment, 0);

    ++udp_listeners;

    fprintf(stderr, 
	    "%d: UDP socket listening on %s,%s\n",
	    sd, hostname, portname);

    nepim_sock_show_opt(stderr, sd);
  }

  freeaddrinfo(ai_res);

  return udp_listeners;
}



