/*
 *  This file is part of Netsukuku.
 *  (c) Copyright 2011 Luca Dionisi aka lukisi <luca.dionisi@gmail.com>
 *
 *  Netsukuku is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  Netsukuku is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with Netsukuku.  If not, see <http://www.gnu.org/licenses/>.
 *
 *  BnodeTunneling address manager and correlated classes. Derived from AuxiliaryAddresManager.
 */

using Gee;
using zcd;
using Tasklets;

namespace Netsukuku
{
    /** Instance of AddressManager for BnodeTunneling addresses.
      */
    public class BnodeTunnelingAddressManager : AuxiliaryAddressManager
    {
        public PartialNIP gnode_tunneling {get; private set;}
        public NIP? peer_nip;

        public BnodeTunnelingAddressManager(int levels,
                                            int gsize,
                                            KeyPair keypair,
                                            HookReservation? hook_reservation,
                                            CreateNicDelegate create_new_nic,
                                            AddressManager autonomous_address_manager,
                                            PartialNIP gnode_tunneling,
                                            NIP? peer_nip)
        {
            this.gnode_tunneling = gnode_tunneling;
            this.peer_nip = peer_nip;
            base(levels, gsize, keypair, hook_reservation, create_new_nic, autonomous_address_manager);
            if (peer_nip != null)
            {
                ((BnodeTunnelingBorderNodesManager)border_nodes_manager).expect_tunnel();
            }
        }

        public override IncomingNodes create_IncomingNodes()
        {
            return new BnodeTunnelingIncomingNodes();
        }

        public override MapRoute create_MapRoute(int levels, int gsize, NIP nip, GNodeID[] id_myself)
        {
            return new BnodeTunnelingMapRoute(levels, gsize, nip, id_myself, this);
        }

        public override AggregatedNeighbourManager create_AggregatedNeighbourManager()
        {
            return new BnodeTunnelingAggregatedNeighbourManager(this);
        }

        public override Etp create_Etp(AggregatedNeighbourManager aggregated_neighbour_manager, MapRoute maproute)
        {
            return new BnodeTunnelingEtp(aggregated_neighbour_manager, maproute, this);
        }

        public override PeerToPeerAll create_PeerToPeerAll(AggregatedNeighbourManager aggregated_neighbour_manager, MapRoute maproute)
        {
            return new BnodeTunnelingPeerToPeerAll(aggregated_neighbour_manager, maproute);
        }

        public override Coord create_Coord(AggregatedNeighbourManager aggregated_neighbour_manager, MapRoute maproute, PeerToPeerAll peer_to_peer_all, CoordinatorKnowledgeSet coordinator_knowledge_set)
        {
            return new BnodeTunnelingCoord(aggregated_neighbour_manager, maproute, peer_to_peer_all, coordinator_knowledge_set);
        }

        public override Counter create_Counter(KeyPair keypair, AggregatedNeighbourManager aggregated_neighbour_manager, MapRoute maproute, PeerToPeerAll peer_to_peer_all)
        {
            return new BnodeTunnelingCounter(keypair, aggregated_neighbour_manager, maproute, peer_to_peer_all);
        }

        public override Andna create_Andna(KeyPair keypair, Counter counter, AggregatedNeighbourManager aggregated_neighbour_manager, MapRoute maproute, PeerToPeerAll peer_to_peer_all)
        {
            return new BnodeTunnelingAndna(keypair, counter, aggregated_neighbour_manager, maproute, peer_to_peer_all);
        }

        public override TunnelManager create_TunnelManager(string ipstr)
        {
            return new BnodeTunnelingTunnelManager(ipstr);
        }

        public override Hook create_Hook(MapRoute maproute, Coord coord)
        {
            return new BnodeTunnelingHook(maproute, coord, this);
        }

        public override BorderNodesManager create_BorderNodesManager(MapRoute maproute, AggregatedNeighbourManager aggregated_neighbour_manager, Coord coord)
        {
            return new BnodeTunnelingBorderNodesManager(this, maproute, aggregated_neighbour_manager, coord);
        }

        public override MigrationManager create_MigrationManager(MapRoute maproute, AggregatedNeighbourManager aggregated_neighbour_manager)
        {
            return new BnodeTunnelingMigrationManager(this, maproute, aggregated_neighbour_manager);
        }

        public override bool do_i_act_as_gateway_for(NIP nip, int nodeid, NetworkID netid)
        {
            // This address won't accept to be used as gateway by neighbours belonging to gnode_tunneling.
            log_debug(@"BnodeTunnelingAddressManager.do_i_act_as_gateway_for($(nip), $(nodeid), $(netid))");
            if (peer_nip == null)
            {
                log_debug(@"Won't be neighbour for anyone, since I don't know my peer yet.");
                return false;
            }
            if (peer_nip.is_equal(nip))
            {
                log_debug(@"Won't be neighbour for the peer.");
                return false;
            }
            if (nip.belongs_to(gnode_tunneling))
            {
                log_debug(@"Won't be neighbour for the tunneling gnode.");
                return false;
            }
            // else
            log_debug("As usual.");
            return base.do_i_act_as_gateway_for(nip, nodeid, netid);
        }

        public override bool do_i_act_as_gateway_for_local()
        {
            // This address won't accept to be used as gateway by other local addresses.
            return false;
        }

        public override bool do_i_participate_in_routing_tables()
        {
            // This address must not be considered in forming the kernel routing tables.
            return false;
        }
    }

    public class BnodeTunnelingIncomingNodes : AuxiliaryIncomingNodes
    {
        public BnodeTunnelingIncomingNodes()
        {
            base();
        }
    }

    public class BnodeTunnelingMapRoute : AuxiliaryMapRoute
    {
        public BnodeTunnelingMapRoute(int levels, int gsize, NIP me, GNodeID[] id_myself, AddressManager address_manager)
        {
            base(levels, gsize, me, id_myself, address_manager);
        }
    }

    public class BnodeTunnelingAggregatedNeighbourManager : AuxiliaryAggregatedNeighbourManager
    {
        public BnodeTunnelingAggregatedNeighbourManager(AddressManager address_manager)
        {
            base(address_manager);
        }

        protected override void store_add_neighbour(string key, AggregatedNeighbour val)
        {
            // This address won't accept to use as gateway a neighbour belonging to
            // gnode_tunneling.
            NIP? peer_nip = ((BnodeTunnelingAddressManager)address_manager).peer_nip;
            PartialNIP gnode_tunneling = ((BnodeTunnelingAddressManager)address_manager).gnode_tunneling;
            log_debug(@"BnodeTunnelingAggregatedNeighbourManager.store_add_neighbour($(key), $(val))");
            if (peer_nip == null)
            {
                log_debug(@"Won't be neighbour for anyone, since I don't know my peer yet.");
                return;
            }
            if (peer_nip.is_equal(val.nip))
            {
                log_debug(@"Won't be neighbour for the peer.");
                return;
            }
            if (val.nip.belongs_to(gnode_tunneling))
            {
                log_debug(@"Won't be neighbour for the tunneling gnode.");
                return;
            }
            // else
            log_debug("As usual.");
            base.store_add_neighbour(key, val);
        }
    }

    public class BnodeTunnelingEtp : AuxiliaryEtp
    {
        public BnodeTunnelingEtp(AggregatedNeighbourManager aggregated_neighbour_manager,
                        MapRoute maproute, AddressManager address_manager)
        {
            base(aggregated_neighbour_manager, maproute, address_manager);
        }

        public override RoutesSet gather_destinations(NIP nip_caller)
        {
            // This address announces only routes for itself.
            return new RoutesSet(maproute.levels);
        }

        public override void advertise_modification_to_routes(TracerPacketList? tpl=null, REM? gwrem=null)
        {
            // This address announces only routes for itself.
        }
    }

    public class BnodeTunnelingPeerToPeerAll : AuxiliaryPeerToPeerAll
    {
        public BnodeTunnelingPeerToPeerAll(AggregatedNeighbourManager aggregated_neighbour_manager, MapRoute maproute)
        {
            base(aggregated_neighbour_manager, maproute);
        }

        public override void peer_to_peer_register(PeerToPeer peer_to_peer) throws PeerToPeerError
        {
            base.peer_to_peer_register(peer_to_peer);
            if (peer_to_peer.get_type().is_a(typeof(OptionalPeerToPeer)))
            {
                ((OptionalPeerToPeer)peer_to_peer).will_participate = false;
            }
        }
    }

    public class BnodeTunnelingCoord : AuxiliaryCoord
    {
        public BnodeTunnelingCoord(AggregatedNeighbourManager aggregated_neighbour_manager,
                        MapRoute maproute, PeerToPeerAll peer_to_peer_all,
                        CoordinatorKnowledgeSet coordinator_knowledge_set)
        {
            base(aggregated_neighbour_manager, maproute, peer_to_peer_all, coordinator_knowledge_set);
        }
    }

    public class BnodeTunnelingCounter : AuxiliaryCounter
    {
        public BnodeTunnelingCounter(KeyPair keypair, AggregatedNeighbourManager aggregated_neighbour_manager,
                        MapRoute maproute, PeerToPeerAll peer_to_peer_all)
        {
            base(keypair, aggregated_neighbour_manager, maproute, peer_to_peer_all);
        }
    }

    public class BnodeTunnelingAndna : AuxiliaryAndna
    {
        public BnodeTunnelingAndna(KeyPair keypair, Counter counter, AggregatedNeighbourManager aggregated_neighbour_manager,
                        MapRoute maproute, PeerToPeerAll peer_to_peer_all)
        {
            base(keypair, counter, aggregated_neighbour_manager, maproute, peer_to_peer_all);
        }
    }

    public class BnodeTunnelingTunnelManager : AuxiliaryTunnelManager
    {
        public BnodeTunnelingTunnelManager(string my_addr)
        {
            base(my_addr);
        }
    }

    public class BnodeTunnelingHook : AuxiliaryHook
    {
        public BnodeTunnelingHook(MapRoute maproute, Coord coord, AddressManager addr_man)
        {
            base(maproute, coord, addr_man);
        }
    }

    struct struct_helper_BnodeTunnelingBorderNodesManager_make_tunnel
    {
        public BnodeTunnelingBorderNodesManager self;
    }

    struct struct_helper_BnodeTunnelingBorderNodesManager_expect_tunnel
    {
        public BnodeTunnelingBorderNodesManager self;
    }

    public class BnodeTunnelingBorderNodesManager : AuxiliaryBorderNodesManager
    {
        public BnodeTunnelingBorderNodesManager(AddressManager addr_man, MapRoute maproute,
                        AggregatedNeighbourManager aggregated_neighbour_manager,
                        Coord coord)
        {
            base(addr_man, maproute, aggregated_neighbour_manager, coord);
        }

        public override void start_operations()
        {
            // Override in BnodeTunneling, do nothing
        }

        public override void stop_operations()
        {
            // Override in BnodeTunneling, do nothing
        }

        public override bool is_border_node(int level_of_gnode)
        {
            // Override in BnodeTunneling, return false
            return false;
        }

        protected override void impl_run(int level_of_gnode) throws Error
        {
            Tasklet.declare_self("BnodeTunnelingBorderNodesManager.run");
            // Override in BnodeTunneling, do nothing
        }

        private void impl_make_tunnel() throws Error
        {
            Tasklet.declare_self("BnodeTunnelingBorderNodesManager.make_tunnel");
            // Create a virtual link in order to increase connectivity inside 'gnode'
            PartialNIP gnode = ((BnodeTunnelingAddressManager)address_manager).gnode_tunneling;
            NIP? other_nip = ((BnodeTunnelingAddressManager)address_manager).peer_nip;
            if (maproute.me.belongs_to(gnode))
            {
                log_debug(@"BorderNodesManager.make_tunnel: This auxiliary address IS member of requested gnode");
                return;
            }
            if (other_nip.belongs_to(gnode))
            {
                log_debug(@"BorderNodesManager.make_tunnel: The other auxiliary address IS member of requested gnode");
                return;
            }
            BnodeTunnelingAddressManager this_addr_man = (BnodeTunnelingAddressManager)address_manager;
            // Ok. Now the operations.
            log_debug(@"BorderNodesManager.make_tunnel: will make a tunnel between me ($(maproute.me)) and $(other_nip)");
            log_debug(@"BorderNodesManager.make_tunnel: waiting a route for $(maproute.nip_to_lvlid(other_nip))");
            // Wait that the nip is routable, and just a little more
            HCoord other_hc = maproute.nip_to_lvlid(other_nip);
            RouteNode other_rn = maproute.node_get_hcoord(other_hc);
            while (other_rn.is_free())
            {
                ms_wait(500);
                // If in the meantime too much time passed...
                if (!this_addr_man.operative)
                {
                    log_warn("BorderNodesManager.make_tunnel: The tunnel peer was not reachable; timed out");
                    return;
                }
            }
            ms_wait(1000);
            // If in the meantime too much time passed...
            if (!this_addr_man.operative)
            {
                log_warn("BorderNodesManager.make_tunnel: The tunnel peer was not reachable; timed out");
                return;
            }
            // Now, create the tunnel
            log_debug(@"BorderNodesManager.make_tunnel: found a route for $(maproute.nip_to_lvlid(other_nip))");
            log_debug(@"BorderNodesManager.make_tunnel: now creating a tunnel between me ($(maproute.me)) and $(other_nip)");
            string other_addr = nip_to_str(maproute.levels, maproute.gsize, other_nip);
            string nic_name = "";
            while (true)
            {
                try
                {
                    // keep trying
                    nic_name = this_addr_man.tunnel_manager.call_request_tunnel(other_addr);
                    break;
                }
                catch (Error e)
                {
                    log_warn(@"BorderNodesManager.make_tunnel: a tentative to contact peer failed: $(e.message)");
                    ms_wait(500);
                    if (!this_addr_man.operative)
                    {
                        log_warn("BorderNodesManager.make_tunnel: The tunnel peer was not reachable; timed out");
                        return;
                    }
                }
            }
            // Communicate new nic to autonomous_addr
            log_debug(@"BorderNodesManager.make_tunnel: tunnel created between me ($(maproute.me)) and $(other_nip)");
            this_addr_man.autonomous_address_manager.border_nodes_manager
                    .tunnel_created(this_addr_man, other_addr, nic_name, gnode.level_of_gnode());
        }

        private static void * helper_make_tunnel(void *v) throws Error
        {
            struct_helper_BnodeTunnelingBorderNodesManager_make_tunnel *tuple_p = (struct_helper_BnodeTunnelingBorderNodesManager_make_tunnel *)v;
            // The caller function has to add a reference to the ref-counted instances
            BnodeTunnelingBorderNodesManager self_save = tuple_p->self;
            // schedule back to the spawner; this will probably invalidate *v and *tuple_p.
            Tasklet.schedule_back();
            // The actual call
            self_save.impl_make_tunnel();
            // void method, return null
            return null;
        }

        public void make_tunnel()
        {
            struct_helper_BnodeTunnelingBorderNodesManager_make_tunnel arg = struct_helper_BnodeTunnelingBorderNodesManager_make_tunnel();
            arg.self = this;
            Tasklet.spawn((FunctionDelegate)helper_make_tunnel, &arg);
        }

        private void impl_expect_tunnel() throws Error
        {
            Tasklet.declare_self("BnodeTunnelingBorderNodesManager.expect_tunnel");
            // Serve a virtual link in order to increase connectivity inside 'gnode'
            PartialNIP gnode = ((BnodeTunnelingAddressManager)address_manager).gnode_tunneling;
            NIP? other_nip = ((BnodeTunnelingAddressManager)address_manager).peer_nip;
            if (maproute.me.belongs_to(gnode))
            {
                log_debug(@"BorderNodesManager.expect_tunnel: This auxiliary address IS member of requested gnode");
                return;
            }
            if (other_nip.belongs_to(gnode))
            {
                log_debug(@"BorderNodesManager.expect_tunnel: The other auxiliary address IS member of requested gnode");
                return;
            }
            BnodeTunnelingAddressManager this_addr_man = (BnodeTunnelingAddressManager)address_manager;
            // Ok. Now the operations.
            log_debug(@"BorderNodesManager.expect_tunnel: waiting to serve a tunnel between me ($(maproute.me)) and $(other_nip)");
            // Now, expect the tunnel, we keep trying for a while
            string other_addr = nip_to_str(maproute.levels, maproute.gsize, other_nip);
            string ret = "";
            while (true)
            {
                foreach (TunnelInfo tunnel_info in this_addr_man.tunnel_manager.get_active_tunnels())
                {
                    if (other_addr == tunnel_info.dest_addr)
                    {
                        ret = tunnel_info.nic_name;
                        break;
                    }
                }
                if (ret != "")
                {
                    break;
                }
                ms_wait(100);
                // If in the meantime too much time passed...
                if (!this_addr_man.operative)
                {
                    log_warn("BorderNodesManager.expect_tunnel: The tunnel was not here; timed out");
                    return;
                }
            }
            string nic_name = ret;
            // Communicate new nic to autonomous_addr
            log_debug(@"BorderNodesManager.expect_tunnel: tunnel created between me ($(maproute.me)) and $(other_nip)");
            this_addr_man.autonomous_address_manager.border_nodes_manager
                    .tunnel_created(this_addr_man, other_addr, nic_name, gnode.level_of_gnode());
        }

        private static void * helper_expect_tunnel(void *v) throws Error
        {
            struct_helper_BnodeTunnelingBorderNodesManager_expect_tunnel *tuple_p = (struct_helper_BnodeTunnelingBorderNodesManager_expect_tunnel *)v;
            // The caller function has to add a reference to the ref-counted instances
            BnodeTunnelingBorderNodesManager self_save = tuple_p->self;
            // schedule back to the spawner; this will probably invalidate *v and *tuple_p.
            Tasklet.schedule_back();
            // The actual call
            self_save.impl_expect_tunnel();
            // void method, return null
            return null;
        }

        public void expect_tunnel()
        {
            struct_helper_BnodeTunnelingBorderNodesManager_expect_tunnel arg = struct_helper_BnodeTunnelingBorderNodesManager_expect_tunnel();
            arg.self = this;
            Tasklet.spawn((FunctionDelegate)helper_expect_tunnel, &arg);
        }

        public void remove_tunnel(string peer_addr, string nic_name)
        {
            // Called in a BnodeTunneling to remove tunnel
            address_manager.tunnel_manager.call_close_tunnel(peer_addr, nic_name);
        }
    }

    public class BnodeTunnelingMigrationManager : AuxiliaryMigrationManager
    {
        public BnodeTunnelingMigrationManager(AddressManager address_manager, MapRoute maproute,
                        AggregatedNeighbourManager aggregated_neighbour_manager)
        {
            base(address_manager, maproute, aggregated_neighbour_manager);
        }
    }

    /** Derivated classes for the pseudo-nic in the autonomous address manager which we are tied to.
      */

    public class TunneledNetworkInterfaceManager : NetworkInterfaceManager
    {
        public TunneledNetworkInterfaceManager(string nic_name, NIC nic_class,
                UnicastCallbackDelegate? udp_unicast_callback=null,
                BroadcastCallbackDelegate? udp_broadcast_callback=null)
        {
            base(nic_name, nic_class, udp_unicast_callback, udp_broadcast_callback);
        }

        public override NeighbourManager create_neighbour_manager(AddressManager addr_man)
        {
            // Activate IP address on NIC
            string my_addr = nip_to_str(addr_man.maproute.levels, addr_man.maproute.gsize, addr_man.maproute.me);
            nic_class.add_address(my_addr);
            log_debug(@"'NetworkInterfaceManager: $(my_addr) activated on $(nic_name)");
            // instantiate radar and neighbour_manager
            Radar radar = new TunneledRadar(nic_name, addr_man);
            NeighbourManager neighbour_manager = new TunneledNeighbourManager(radar, Settings.MAX_NEIGHBOURS);
            // start the radar tasklet
            radar.run();
            return neighbour_manager;
        }
    }

    public class TunneledRadar : Radar
    {
        private NIP? peer_nip;
        private int peer_levels;
        private int peer_gsize;
        private int peer_nodeid;
        private NetworkID peer_netid;
        private string peer_mac;
        private bool peer_is_primary;
        private bool peer_is_auxiliary;

        public TunneledRadar(string dev_name, AddressManager address_manager)
        {
            base(dev_name, address_manager);
            peer_nip = null;
        }

        public override void radar()
        {
            // Send a broadcast packet to the tunneled peer. See if a reply comes back
            try
            {
                radar_ids = new ArrayList<int>();
                int radar_id = Random.int_range(0, ((int)Math.pow(2, 32))-1);
                radar_ids.add(radar_id);
                NIP nip = address_manager.maproute.me;
                NetworkID netid = address_manager.get_main_netid();
                int nodeid = address_manager.get_my_id();
                bcastclient.aggregated_neighbour_manager.reply(radar_id, nip, nodeid, netid);
                // Check not so often.
                ms_wait(wait_time * 5000);
                // Notify the detected neighbours
                bouquet_numb += 1;
                HashMap<string, Neighbour> detected_neighbours = get_all_avg_rtt();
                log_debug(@"radar: emits SCAN_DONE for $dev_name");
                scan_done(bouquet_numb, detected_neighbours);
            }
            catch (Error e)
            {
                log_warn(@"Exception - $(e.domain): $(e.code): $(e.message) - while doing a TunneledRadar scan. We ignore it. Soon another scan.");
                bcastclient = new AddressManagerBroadcastClient(new BroadcastID(), new string[] {this.dev_name});
                ms_wait(500);
            }
            // We're done. Reset.
            peer_nip = null;
        }

        public override void time_register(int radar_id, int levels, int gsize, NIP nip, int nodeid,
                        NetworkID netid, string mac, bool is_primary, bool is_auxiliary)
        {
            // A reply came back
            if (! radar_ids.contains(radar_id))
            {
                // drop. It isn't a reply to our current bouquet
                return;
            }

            peer_levels = levels;
            peer_gsize = gsize;
            peer_nip = nip;
            peer_nodeid = nodeid;
            peer_netid = netid;
            peer_mac = mac;
            peer_is_primary = is_primary;
            peer_is_auxiliary = is_auxiliary;
            log_debug(@"Radar: IP $(nip_to_str(levels, gsize, nip)) from network $(netid) detected");
        }

        public override HashMap<string, Neighbour> get_all_avg_rtt()
        {
            // We have one neighbour with very low REM (or no neighbours)
            HashMap<string, Neighbour> all_avg = new HashMap<string, Neighbour>();
            if (peer_nip != null)
            {
                string key = tnip_nodeid_key(peer_nip, peer_nodeid);
                ArrayList<string> macs = new ArrayList<string>();
                macs.add(peer_mac);
                all_avg[key] = new Neighbour.with_rtt(
                        peer_levels, peer_gsize,
                        peer_nip, peer_nodeid, peer_netid, peer_is_primary,
                        peer_is_auxiliary, dev_name, macs, 5000);
            }
            return all_avg;
        }
    }

    public class TunneledNeighbourManager : NeighbourManager
    {
        public TunneledNeighbourManager(Radar radar, int max_neighbours=-1)
        {
            base(radar, max_neighbours);
        }
    }
}

