/*
 *  This file is part of Netsukuku.
 *  (c) Copyright 2011 Luca Dionisi aka lukisi <luca.dionisi@gmail.com>
 *
 *  Netsukuku is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  Netsukuku is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with Netsukuku.  If not, see <http://www.gnu.org/licenses/>.
 */

using Gee;
using zcd;
using Tasklets;

namespace Netsukuku
{
    public const int DHT_DUPLICATION = 10;

    public interface IDistributedHashTable : Object
    {
        public abstract void store(DHTRecord rec, bool replicate=true) throws RPCError, PeerRefuseServiceError;
        public abstract DHTRecord? retrieve(DHTKey key) throws RPCError, PeerRefuseServiceError;
        public abstract Gee.List<DHTRecord> get_cache() throws RPCError, PeerRefuseServiceError;
    }

    public class RmtDistributedHashTablePeer : RmtPeer, IDistributedHashTable
    {
        public RmtDistributedHashTablePeer(PeerToPeer peer_to_peer_service,
                                           Object? key=null,
                                           NIP? hIP=null,
                                           AggregatedNeighbour? aggregated_neighbour=null)
        {
            base(peer_to_peer_service, key, hIP, aggregated_neighbour);
        }

        public void store(DHTRecord rec, bool replicate) throws RPCError
        {
            RemoteCall rc = new RemoteCall();
            rc.method_name = "store";
            rc.add_parameter(rec);
            rc.add_parameter(new SerializableBool(replicate));
            // NOTE: RmtPeer.rmt can throw any Error. It does already filters RemotableException classes.
            try {
                rmt(rc);
            }
            catch (RPCError e) {throw e;}
            catch (Error e) {throw new RPCError.GENERIC(@"Unexpected error $(e.domain).$(e.code) '$(e.message)'");}
        }

        public DHTRecord? retrieve(DHTKey key) throws RPCError
        {
            RemoteCall rc = new RemoteCall();
            rc.method_name = "retrieve";
            rc.add_parameter(key);
            // NOTE: RmtPeer.rmt can throw any Error. It does already filters RemotableException classes.
            try {
                ISerializable ret = rmt(rc);
                if (ret.get_type().is_a(typeof(SerializableNone))) return null;
                else return (DHTRecord)ret;
            }
            catch (RPCError e) {throw e;}
            catch (Error e) {throw new RPCError.GENERIC(@"Unexpected error $(e.domain).$(e.code) '$(e.message)'");}
        }

        public Gee.List<DHTRecord> get_cache() throws RPCError
        {
            RemoteCall rc = new RemoteCall();
            rc.method_name = "get_cache";
            // NOTE: RmtPeer.rmt can throw any Error. It does already filters RemotableException classes.
            try {
                ListISerializable ret = (ListISerializable)rmt(rc);
                return (Gee.List<DHTRecord>)ret.backed;
            }
            catch (RPCError e) {throw e;}
            catch (Error e) {throw new RPCError.GENERIC(@"Unexpected error $(e.domain).$(e.code) '$(e.message)'");}
        }
    }

    struct struct_helper_DistributedHashTable_hook_to_service
    {
        public DistributedHashTable self;
    }

    struct struct_helper_DistributedHashTable_forward_store
    {
        public DistributedHashTable self;
        public DHTRecord rec;
        Gee.List<NIP>? replica_nodes;
    }

    public class DistributedHashTable : OptionalPeerToPeer, IDistributedHashTable
    {
        public static const int mypid = 2003;

        private PeerToPeerAll peer_to_peer_all;
        private bool hooked_to_service;
        public HashMap<DHTKey, DHTRecord> cache {get; private set;}

        public DistributedHashTable(AggregatedNeighbourManager aggregated_neighbour_manager, MapRoute maproute, PeerToPeerAll peer_to_peer_all)
        {
            base(aggregated_neighbour_manager, maproute, DistributedHashTable.mypid);
            this.peer_to_peer_all = peer_to_peer_all;

            // let's register ourself in peer_to_peer_all
            will_participate = true;
            // We valorize "will_participate", which is a member of OptionalPeerToPeer, before doing
            // the registration because in the registration the manager PeerToPeerAll might change to
            // false (e.g. an auxiliary address)
            this.peer_to_peer_all.peer_to_peer_register(this);

            // Until I have a valid P2P map we don't want to answer
            //  to store/retrieve requests.
            hooked_to_service = false;
            // Start the hooking phase of the peer-to-peer service when my map is valid.
            this.map_peer_to_peer_validated.connect(hook_to_service);

            // This will contain my records.
            cache = new HashMap<DHTKey, DHTRecord>(DHTKey.hash_func, DHTKey.equal_func);
        }

        public RmtDistributedHashTablePeer peer(NIP? hIP=null, Object? key=null, AggregatedNeighbour? aggregated_neighbour=null)
        {
            assert(hIP != null || key != null);
            return new RmtDistributedHashTablePeer(this, key, hIP, aggregated_neighbour);
        }

        /** This method could be called *directly* for a dispatcher that does not need to transform
          * an exception into a remotable.
          */
        public override ISerializable _dispatch(Object? caller, RemoteCall data) throws Error
        {
            log_debug(@"$(this.get_type().name()): dispatching $data");
            string[] pieces = data.method_name.split(".");
            if (pieces[0] == "store")
            {
                if (pieces.length != 1) throw new RPCError.MALFORMED_PACKET("store is a function.");
                if (data.parameters.size != 2) throw new RPCError.MALFORMED_PACKET("store wants 2 parameter.");
                ISerializable iser0 = data.parameters.get(0);
                if (! iser0.get_type().is_a(typeof(DHTRecord)))
                    throw new RPCError.MALFORMED_PACKET("store parameter 1 is not a DHTRecord.");
                DHTRecord rec = (DHTRecord)iser0;
                ISerializable iser1 = data.parameters[1];
                if (! iser1.get_type().is_a(typeof(SerializableBool)))
                    throw new RPCError.MALFORMED_PACKET(
                        "store parameter 2 is not a bool.");
                bool replicate = ((SerializableBool)iser1).b;
                store(rec, replicate);
                return new SerializableNone();
            }
            if (pieces[0] == "retrieve")
            {
                if (pieces.length != 1) throw new RPCError.MALFORMED_PACKET("retrieve is a function.");
                if (data.parameters.size != 1) throw new RPCError.MALFORMED_PACKET("retrieve wants 1 parameters.");
                ISerializable iser0 = data.parameters.get(0);
                if (! iser0.get_type().is_a(typeof(DHTKey)))
                    throw new RPCError.MALFORMED_PACKET("retrieve parameter 1 is not a DHTKey.");
                DHTKey key = (DHTKey)iser0;
                DHTRecord? ret = retrieve(key);
                if (ret == null) return new SerializableNone();
                else return (DHTRecord)ret;
            }
            if (pieces[0] == "get_cache")
            {
                if (pieces.length != 1)
                    throw new RPCError.MALFORMED_PACKET(
                        "get_cache is a function.");
                if (data.parameters.size != 0)
                    throw new RPCError.MALFORMED_PACKET(
                        "get_cache wants no parameters.");
                return new ListISerializable.with_backer(get_cache());
            }
            return base._dispatch(caller, data);
        }

        private NIP nip_for_lvl_pos(int lvl, int pos)
        {
            int[] ret = maproute.me.get_positions();
            ret[lvl] = pos;
            return new NIP(ret);
        }

        private void impl_hook_to_service() throws Error
        {
            Tasklet.declare_self("DHT.hook_to_service");
            if (will_participate)
            {
                // TODO Handle possible errors in hooking phase. Should we try again?
                //  If we don't, we are not participating in the service.

                // clear old cache
                cache.clear();

                for (int lvl = 0; lvl < maproute.levels; lvl++)
                {
                    int? first_forward;
                    int? first_back;
                    int? last_back;
                    log_debug(@"$(this.get_type().name()): hook: $(maproute.me) finding peers at lvl $(lvl)");
                    find_hook_peers(out first_forward, out first_back, out last_back, lvl, DHT_DUPLICATION);
                    if (first_forward == null)
                    {
                        // no one in my gnode lvl+1 (except me)
                        log_debug(@"$(this.get_type().name()): hook: no one in network at this level (except me)");
                        // I need to go up one level.
                    }
                    else if (first_back == null)
                    {
                        // my gnode lvl+1 has some participants but not
                        // enough to satisfy our replica DHT_DUPLICATION.
                        // So, get all.
                        log_debug(@"$(this.get_type().name()): hook: few nodes at this level. get all from $(first_forward).");
                        NIP nip_first_forward = nip_for_lvl_pos(lvl, first_forward);
                        RmtDistributedHashTablePeer peer_first_forward = peer(nip_first_forward);
                        Gee.List<DHTRecord> cache_first_forward = peer_first_forward.get_cache();
                        foreach (DHTRecord rec in cache_first_forward)
                        {
                            DHTKey k = rec.key;
                            log_debug(@"$(this.get_type().name()): hook: there is $(k.key) in $(first_forward)");
                            if (! cache.has_key(k))
                            {
                                cache[k] = rec;
                                log_debug(@"$(this.get_type().name()): hook: got $(k.key)");
                            }
                        }
                        // and then no need to go up one level.
                        break;
                    }
                    else
                    {
                        // my gnode lvl+1 has enough participant
                        if (first_back != last_back)
                        {
                            // (lvl, first_back) is a (g)node with some participants but not
                            // enough to satisfy our replica DHT_DUPLICATION.
                            // (lvl, last_back) + ... + (lvl, first_back) has enough
                            // participant, though.
                            // So I have to get a record from first_back IFF it is not in last_back.
                            NIP nip_first_back = nip_for_lvl_pos(lvl, first_back);
                            RmtDistributedHashTablePeer peer_first_back = peer(nip_first_back);
                            NIP nip_last_back = nip_for_lvl_pos(lvl, last_back);
                            RmtDistributedHashTablePeer peer_last_back = peer(nip_last_back);
                            Gee.List<DHTRecord> recs_first = peer_first_back.get_cache();
                            Gee.List<DHTRecord> recs_last = peer_last_back.get_cache();
                            foreach (DHTRecord rec in recs_first)
                            {
                                DHTKey k = rec.key;
                                if (! cache.has_key(k))
                                {
                                    bool found = false;
                                    foreach (DHTRecord rec2 in recs_last)
                                    {
                                        DHTKey k2 = rec2.key;
                                        if (k2.equals(k))
                                        {
                                            found = true;
                                            break;
                                        }
                                    }
                                    if (!found) cache[k] = rec;
                                }
                            }
                        }
                        // Now the records that are in first forward and whose
                        // key is before me at this level.
                        NIP nip_first_forward = nip_for_lvl_pos(lvl, first_forward);
                        RmtDistributedHashTablePeer peer_first_forward = peer(nip_first_forward);
                        Gee.List<DHTRecord> recs = peer_first_forward.get_cache();
                        foreach (DHTRecord rec in recs)
                        {
                            DHTKey k = rec.key;
                            if (! cache.has_key(k))
                            {
                                int hk = h(k).position_at(lvl);
                                Gee.List<int> ids = list_ids(hk, 1);
                                if (ids.index_of(first_forward) > 
                                    ids.index_of(maproute.me.position_at(lvl)))
                                {
                                    cache[k] = rec;
                                }
                            }
                        }
                        // Then no need to go up one level.
                        break;
                    }
                }

                // Now I can answer to requests.
                hooked_to_service = true;
                log_debug(@"$(this.get_type().name()): hook: finished.");
                // Now I can participate again.
                participate();
                log_info("Distributed Hash Table service: participating.");
            }
        }

        private static void * helper_hook_to_service(void *v) throws Error
        {
            struct_helper_DistributedHashTable_hook_to_service *tuple_p = (struct_helper_DistributedHashTable_hook_to_service *)v;
            // The caller function has to add a reference to the ref-counted instances
            DistributedHashTable self_save = tuple_p->self;
            // schedule back to the spawner; this will probably invalidate *v and *tuple_p.
            Tasklet.schedule_back();
            // The actual call
            self_save.impl_hook_to_service();
            // void method, return null
            return null;
        }

        public void hook_to_service()
        {
            struct_helper_DistributedHashTable_hook_to_service arg = struct_helper_DistributedHashTable_hook_to_service();
            arg.self = this;
            Tasklet.spawn((FunctionDelegate)helper_hook_to_service, &arg);
        }

        /** This is the function h:KEY-->hIP.
          */
        public override NIP h(Object key)
        {
            // Given a DHTKey, a value for levels and a value for gsize, this
            // function should compute the same NIP in any platform
            // and any architecture.
            DHTKey _key = (DHTKey)key;
            uint hash = _key.key.hash();
            int[] positions = new int[maproute.levels];
            for (int lvl = 0; lvl < maproute.levels; lvl++)
            {
                uint pos = hash % maproute.gsize;
                positions[lvl] = (int)pos;
                hash /= maproute.gsize;
            }
            NIP ret = new NIP(positions);
            log_debug(@"$(this.get_type().name()): hash: '$(_key.key)' becomes $(ret)");
            return ret;
        }

        public void store(DHTRecord rec, bool replicate) throws PeerRefuseServiceError
        {
            if (!hooked_to_service)
            {
                if (replicate)
                {
                    // This is an original call. I am not hooked, so I should not
                    // have got this call. Anyway, I can wait a bit before throwing
                    // an error to see if I hook soon. But the client is waiting
                    // for an answer, thus after a bit I want to throw an exception
                    // in order for the client to try again.
                    Tasklets.Timer w = new Tasklets.Timer(10000); // 10 seconds
                    while (!hooked_to_service)
                    {
                        if (w.is_expired())
                            throw new PeerRefuseServiceError.GENERIC("Not hooked yet");
                        Tasklet.nap(0, 1000);
                    }
                }
                else
                {
                    // This is a replica. So the client is not waiting and will not
                    // request again. Thus I will wait for the hook whatever it takes.
                    while (!hooked_to_service) Tasklet.nap(0, 1000);
                }
            }

            // I verify that I am the hash node for the key.
            check_hash_and_start_replica(this, h(rec.key), replicate, rec, DHT_DUPLICATION,
                     /*AcceptRecordCallback*/
                     () => {
                        cache[rec.key] = rec;
                        return true;
                     },
                     /*ForwardRecordCallback*/
                     (tasklet_obj1, tasklet_replica_nodes) => {
                        DHTRecord tasklet_rec = (DHTRecord)tasklet_obj1;
                        // Here I am in a tasklet (the client has been served already)
                        if (tasklet_replica_nodes == null)
                                tasklet_replica_nodes = find_nearest_to_register(h(tasklet_rec.key), DHT_DUPLICATION);
                        // For each node of the <n> nearest except myself...
                        foreach (NIP replica_node in tasklet_replica_nodes) if (!replica_node.is_equal(maproute.me))
                        {
                            // ... in another tasklet...
                            Tasklet.tasklet_callback(
                                (tpar1, tpar2) => {
                                    NIP tonip = (NIP)tpar1;
                                    DHTRecord arec = (DHTRecord)tpar2;
                                    Tasklet.declare_self("DHT.forward");
                                    // ... forward the record to the node.
                                    try
                                    {
                                        peer(tonip).store(arec, false);
                                    }
                                    catch (RPCError e)
                                    {
                                        // report the error with some info on where it happened
                                        log_warn(@"DHT.forward: forwarding to $(tonip):"
                                            + @" got $(e.domain.to_string()) $(e.code) $(e.message)");
                                    }
                                },
                                replica_node,
                                tasklet_rec);
                        }
                     },
                     /*RefuseRecordCallback*/
                     () => {
                        // We can log the error and/or throw a specific instance
                        log_error("throw errore store: bad hash");
                        // Otherwise a generic PeerRefuseService will be thrown.
                     });
        }

        public DHTRecord? retrieve(DHTKey key) throws PeerRefuseServiceError
        {
            // I am not hooked, so I should not
            // have got this call. Anyway, I can wait a bit before throwing
            // an error to see if I hook soon. But the client is waiting
            // for an answer, thus after a bit I want to throw an exception
            // in order for the client to try again.
            Tasklets.Timer w = new Tasklets.Timer(10000); // 10 seconds
            while (!hooked_to_service)
            {
                if (w.is_expired())
                    throw new PeerRefuseServiceError.GENERIC("Not hooked yet");
                Tasklet.nap(0, 1000);
            }

            // I verify that I am the hash node for the key.
            if (search_participant(h(key)) != null)
            {
                // Throw an error, in order for the client to try again.
                throw new PeerRefuseServiceError.GENERIC("Not current best hash node");
            }

            if (cache.has_key(key))
                return cache[key];
            return null;
        }

        public Gee.List<DHTRecord> get_cache() throws PeerRefuseServiceError
        {
            // I am not hooked, so I should not
            // have got this call. Anyway, I can wait a bit before throwing
            // an error to see if I hook soon. But the client is waiting
            // for an answer, thus after a bit I want to throw an exception
            // in order for the client to try again.
            Tasklets.Timer w = new Tasklets.Timer(10000); // 10 seconds
            while (!hooked_to_service)
            {
                if (w.is_expired())
                    throw new PeerRefuseServiceError.GENERIC("Not hooked yet");
                Tasklet.nap(0, 1000);
            }

            Gee.List<DHTRecord> ret = new ArrayList<DHTRecord>(DHTRecord.key_equal_func);
            ret.add_all(cache.values);
            return ret;
        }
    }
}
