/*
 *  This file is part of Netsukuku.
 *  (c) Copyright 2011-2014 Luca Dionisi aka lukisi <luca.dionisi@gmail.com>
 *
 *  Netsukuku is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  Netsukuku is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with Netsukuku.  If not, see <http://www.gnu.org/licenses/>.
 */

/*   peerbuilder.in
errors:
 HookingError
  INEXISTENT_GNODE
  GENERIC
 QspnError
  NOT_YOUR_GATEWAY
  ALREADY_UP_TO_DATE
  GENERIC
 PeerRefuseServiceError
  GENERIC
 TunnelError
  GENERIC
 BorderNodesError
  WRONG_GNODE
  NOT_BORDER_NODE
  WILL_NOT_TUNNEL
  TIMEOUT
  GENERIC
 AndnaError
  GENERIC
serializables:
 TimeCapsule
 ParticipantNode
 PackedParticipantNodes
 OptionalServiceParticipants
 SetOptionalServiceParticipants
 PeerToPeerTracerPacketList
 PairNipDistance
 HookReservation
 BookingRecord
 Bookings
 BnodeRecord
 BnodeList
 CoordinatorKnowledge
 CoordinatorKnowledgeSet
 PairLvlNumberOfFreeNodes
 HCoord
 PartialNIP
 NIP
 REM
 NullREM
 DeadREM
 AlmostDeadREM
 RTT
 TracerPacketList
 RouteInSet
 PositionInRoutesSetPerLevel
 RoutesSetPerLevel
 RoutesSet
 ExtendedTracerPacket
 GNodeID
 NetworkID
 InfoNeighbour
 InfoRoute
 InfoNode
 QspnStats
 InfoBorderNode
 InfoCoord
 DHTKey
 DHTRecord
 PublicKey
 AndnaServiceKey
 AndnaServerRecord
 AndnaDomainRecord
 AndnaServer
 AndnaServers
 RegisterHostnameArguments
 CounterNipRecord
 CounterSetDataResponse
 CounterCheckHostnameResponse
 AndnaConfirmPubkResponse
 AndnaRegisterMainResponse
 AndnaRegisterSpreadResponse
 AndnaGetServersResponse
 AndnaGetRegistrarResponse
 AndnaGetCacheRecordsResponse
 BroadcastID
 UnicastID
peers:
 Andna
  methods:
   AndnaRegisterMainResponse register_main_for_pubk
    arguments:
     AndnaDomainRequest request
     SerializableBuffer signature
     bool replicate
    throws:
     PeerRefuseServiceError
   AndnaGetRegistrarResponse? get_registrar
    arguments:
     string hashed_domain
    throws:
     PeerRefuseServiceError
   AndnaDomainRecord? get_domain_record
    arguments:
     string hashed_domain
    throws:
     PeerRefuseServiceError
   AndnaRegisterSpreadResponse register_spread_record
    arguments:
     string hashed_domain
     int spread_number
     bool replicate
    throws:
     PeerRefuseServiceError
   AndnaGetServersResponse get_servers
    arguments:
     string hashed_domain
     AndnaServiceKey srv_key
     int spread_number
     bool chain
    throws:
     PeerRefuseServiceError
   AndnaGetCacheRecordsResponse get_cache_records
    arguments:
    throws:
     PeerRefuseServiceError
*/

using Gee;
using zcd;
using Tasklets;

namespace Netsukuku
{
    public interface IAndnaAsPeer : Object
    {
        public abstract  AndnaRegisterMainResponse
                         register_main_for_pubk
                         (AndnaDomainRequest request,
                          SerializableBuffer signature,
                          bool replicate=true)
                         throws RPCError, PeerRefuseServiceError;
        public abstract  AndnaGetRegistrarResponse?
                         get_registrar
                         (string hashed_domain)
                         throws RPCError, PeerRefuseServiceError;
        public abstract  AndnaDomainRecord?
                         get_domain_record
                         (string hashed_domain)
                         throws RPCError, PeerRefuseServiceError;
        public abstract  AndnaRegisterSpreadResponse
                         register_spread_record
                         (string hashed_domain,
                          int spread_number,
                          bool replicate=true)
                         throws RPCError, PeerRefuseServiceError;
        public abstract  AndnaGetServersResponse
                         get_servers
                         (string hashed_domain,
                          AndnaServiceKey srv_key,
                          int spread_number,
                          bool chain)
                         throws RPCError, PeerRefuseServiceError;
        public abstract  AndnaGetCacheRecordsResponse
                         get_cache_records ()
                         throws RPCError, PeerRefuseServiceError;
    }

    public class RmtAndnaPeer : RmtPeer, IAndnaAsPeer
    {
        public RmtAndnaPeer
               (PeerToPeer peer_to_peer_service,
                Object? key=null,
                NIP? hIP=null,
                AggregatedNeighbour? aggregated_neighbour=null)
        {
            base(peer_to_peer_service, key, hIP, aggregated_neighbour);
        }

        public  AndnaRegisterMainResponse
                register_main_for_pubk
                (AndnaDomainRequest request,
                 SerializableBuffer signature,
                 bool replicate)
                throws RPCError, PeerRefuseServiceError
        {
            RemoteCall rc = new RemoteCall();
            rc.method_name = "register_main_for_pubk";
            rc.add_parameter(request);
            rc.add_parameter(signature);
            rc.add_parameter(new SerializableBool(replicate));
            try {
                return (AndnaRegisterMainResponse)
                    filter_exception(
                    filter_exception_PeerRefuseServiceError(
                    this.rmt(rc))
                );
            }
            catch (RPCError e) {throw e;}
            catch (Error e)
            {
                throw new RPCError.GENERIC
                    (@"Unexpected error $(e.domain).$(e.code) '$(e.message)'");
            }
        }

        public  AndnaGetRegistrarResponse?
                get_registrar
                (string hashed_domain)
                throws RPCError, PeerRefuseServiceError
        {
            RemoteCall rc = new RemoteCall();
            rc.method_name = "get_registrar";
            rc.add_parameter(new SerializableString(hashed_domain));
            try {
                ISerializable ret =
                    filter_exception(
                    filter_exception_PeerRefuseServiceError(
                    this.rmt(rc))
                );
                if (ret.get_type().is_a(typeof(SerializableNone))) return null;
                else return (AndnaGetRegistrarResponse)ret;
            }
            catch (RPCError e) {throw e;}
            catch (Error e)
            {
                throw new RPCError.GENERIC
                    (@"Unexpected error $(e.domain).$(e.code) '$(e.message)'");
            }
        }

        public  AndnaDomainRecord?
                get_domain_record
                (string hashed_domain)
                throws RPCError, PeerRefuseServiceError
        {
            RemoteCall rc = new RemoteCall();
            rc.method_name = "get_domain_record";
            rc.add_parameter(new SerializableString(hashed_domain));
            try {
                ISerializable ret =
                    filter_exception(
                    filter_exception_PeerRefuseServiceError(
                    this.rmt(rc))
                );
                if (ret.get_type().is_a(typeof(SerializableNone))) return null;
                else return (AndnaDomainRecord)ret;
            }
            catch (RPCError e) {throw e;}
            catch (Error e)
            {
                throw new RPCError.GENERIC
                    (@"Unexpected error $(e.domain).$(e.code) '$(e.message)'");
            }
        }

        public  AndnaRegisterSpreadResponse
                register_spread_record
                (string hashed_domain,
                 int spread_number,
                 bool replicate)
                throws RPCError, PeerRefuseServiceError
        {
            RemoteCall rc = new RemoteCall();
            rc.method_name = "register_spread_record";
            rc.add_parameter(new SerializableString(hashed_domain));
            rc.add_parameter(new SerializableInt(spread_number));
            rc.add_parameter(new SerializableBool(replicate));
            try {
                return (AndnaRegisterSpreadResponse)
                    filter_exception(
                    filter_exception_PeerRefuseServiceError(
                    this.rmt(rc))
                );
            }
            catch (RPCError e) {throw e;}
            catch (Error e)
            {
                throw new RPCError.GENERIC
                    (@"Unexpected error $(e.domain).$(e.code) '$(e.message)'");
            }
        }

        public  AndnaGetServersResponse
                get_servers
                (string hashed_domain,
                 AndnaServiceKey srv_key,
                 int spread_number,
                 bool chain)
                throws RPCError, PeerRefuseServiceError
        {
            RemoteCall rc = new RemoteCall();
            rc.method_name = "get_servers";
            rc.add_parameter(new SerializableString(hashed_domain));
            rc.add_parameter(srv_key);
            rc.add_parameter(new SerializableInt(spread_number));
            rc.add_parameter(new SerializableBool(chain));
            try {
                return (AndnaGetServersResponse)
                    filter_exception(
                    filter_exception_PeerRefuseServiceError(
                    this.rmt(rc))
                );
            }
            catch (RPCError e) {throw e;}
            catch (Error e)
            {
                throw new RPCError.GENERIC
                    (@"Unexpected error $(e.domain).$(e.code) '$(e.message)'");
            }
        }

        public  AndnaGetCacheRecordsResponse
                get_cache_records ()
                throws RPCError, PeerRefuseServiceError
        {
            RemoteCall rc = new RemoteCall();
            rc.method_name = "get_cache_records";
            try {
                return (AndnaGetCacheRecordsResponse)
                    filter_exception(
                    filter_exception_PeerRefuseServiceError(
                    this.rmt(rc))
                );
            }
            catch (RPCError e) {throw e;}
            catch (Error e)
            {
                throw new RPCError.GENERIC
                    (@"Unexpected error $(e.domain).$(e.code) '$(e.message)'");
            }
        }

    }

    public class AndnaServiceHashnodeKey : Object
    {
        public AndnaServiceHashnodeKey
               (string hashed_domain, 
               int spread_num=0)
        {
            this.hashed_domain = hashed_domain;
            this.spread_num = spread_num;
        }
        public string hashed_domain {get; private set;}
        public int spread_num {get; private set;}
        public static bool equal_func(AndnaServiceHashnodeKey a,
                                      AndnaServiceHashnodeKey b)
        {
            return  a.hashed_domain == b.hashed_domain &&
                    a.spread_num == b.spread_num;
        }
    }

    class ArgumentsForDuplicationRegisterSpread : Object
    {
        public string hashed_domain;
        public int spread_number;
        public NIP hashnode;
    }

    class ArgumentsForDuplicationRegisterMain : Object
    {
        public AndnaDomainRequest request;
        public SerializableBuffer signature;
        public NIP hashnode;
    }

    class DomainWanted : Object
    {
        public string hostname;
        public TimeCapsule ttl_before_request;
        public Tasklet tasklet;
        public bool registered;
        public static bool equal_func(DomainWanted a, DomainWanted b)
        {
            return a.hostname == b.hostname;
        }
    }

    public class Andna : OptionalPeerToPeer, IAndnaAsPeer, IAndna
    {
        public static const int mypid = 3;

        // The loaded configuration of this node
        private AndnaPrivateConfigurationList _configuration;

        // get the configuration of this node
        private AndnaPrivateConfigurationList configuration {
            get {
                if (_configuration == null)
                {
                    _configuration = new AndnaPrivateConfigurationList();
                    char buffer[256];
                    string? myhostname = null;
                    if (Posix.gethostname(buffer) == 0)
                    {
                        StringBuilder sb = new StringBuilder();
                        int pos = 0;
                        while (buffer[pos] != '\0') sb.append_c(buffer[pos++]);
                        sb.append_c('\0');
                        myhostname = sb.str;
                    }
                    if (myhostname != null)
                    {
                        AndnaPrivateConfiguration cfg = new AndnaPrivateConfiguration(myhostname);
                        _configuration.lst.add(cfg);
                    }
                }
                return _configuration;
            }
        }

        private PeerToPeerAll peer_to_peer_all;
        private bool hooked_to_service;
        private Counter counter;
        private KeyPair my_keys;
        private HashMap<string, ArrayList<RegisterHostnameArguments>> request_queue;
        private HashMap<string, AndnaDomainRecord> cache;
        private ArrayList<DomainWanted> lst_domains;

        private PublicKey _mypubk;
        // get the PublicKey
        private PublicKey mypubk {
            get {
                if (_mypubk == null)
                {
                    _mypubk = my_keys.pub_key.to_pubkey();
                }
                return _mypubk;
            }
        }

        public signal void andna_hooked();
        public signal void andna_registered(string domain);

        public Andna(KeyPair keypair,
                     Counter counter,
                     AggregatedNeighbourManager aggregated_neighbour_manager,
                     MapRoute maproute,
                     PeerToPeerAll peer_to_peer_all)
        {
            base(aggregated_neighbour_manager, maproute, Andna.mypid);
            this.peer_to_peer_all = peer_to_peer_all;

            // let's register ourself in peer_to_peer_all
            will_participate = true;
            // We valorize "will_participate", which is a member of OptionalPeerToPeer, before doing
            // the registration because in the registration the manager PeerToPeerAll might change to
            // false (e.g. an auxiliary address)
            this.peer_to_peer_all.peer_to_peer_register(this);

            // Until I have a valid P2P map we don't want to answer
            //  to registration/resolution requests.
            hooked_to_service = false;
            // Start the hooking phase of the peer-to-peer service when my map is valid.
            this.map_peer_to_peer_validated.connect(hook_to_service);

            lst_domains = new ArrayList<DomainWanted>(DomainWanted.equal_func);
            this.counter = counter;
            this.my_keys = keypair;

            // The key is the hashed_domain (which is a string, so no particular hashfunc)
            //  and the value is the args to pass to register_hostname_main
            request_queue = new HashMap<string, ArrayList<RegisterHostnameArguments>>();

            // The key is the hashed_domain (which is a string, so no particular hashfunc)
            //  and the value is the registered AndnaDomainRecord
            cache = new HashMap<string, AndnaDomainRecord>();
        }

        public RmtAndnaPeer
               peer
               (NIP? hIP=null,
                Object? key=null,
                AggregatedNeighbour? aggregated_neighbour=null)
        {
            assert(hIP != null || key != null);
            return new RmtAndnaPeer(this, key, hIP, aggregated_neighbour);
        }

        /** This method could be called *directly* for a dispatcher that does not need to transform
          * an exception into a remotable.
          */
        public override ISerializable _dispatch(Object? caller, RemoteCall data) throws Error
        {
            log_debug(@"$(this.get_type().name()): dispatching $data");
            string[] pieces = data.method_name.split(".");
            if (pieces[0] == "register_main_for_pubk")
            {
                if (pieces.length != 1)
                    throw new RPCError.MALFORMED_PACKET(
                        "register_main_for_pubk is a function.");
                if (data.parameters.size != 3)
                    throw new RPCError.MALFORMED_PACKET(
                        "register_main_for_pubk wants 3 parameters.");
                ISerializable iser0 = data.parameters[0];
                if (! iser0.get_type().is_a(typeof(AndnaDomainRequest)))
                    throw new RPCError.MALFORMED_PACKET(
                        "register_main_for_pubk parameter 1 is not a AndnaDomainRequest.");
                AndnaDomainRequest request = (AndnaDomainRequest)iser0;
                ISerializable iser1 = data.parameters[1];
                if (! iser1.get_type().is_a(typeof(SerializableBuffer)))
                    throw new RPCError.MALFORMED_PACKET(
                        "register_main_for_pubk parameter 2 is not a SerializableBuffer.");
                SerializableBuffer signature = (SerializableBuffer)iser1;
                ISerializable iser2 = data.parameters[2];
                if (! iser2.get_type().is_a(typeof(SerializableBool)))
                    throw new RPCError.MALFORMED_PACKET(
                        "register_main_for_pubk parameter 3 is not a bool.");
                bool replicate = ((SerializableBool)iser2).b;
                return register_main_for_pubk(request, signature, replicate);
            }
            if (pieces[0] == "get_registrar")
            {
                if (pieces.length != 1)
                    throw new RPCError.MALFORMED_PACKET(
                        "get_registrar is a function.");
                if (data.parameters.size != 1)
                    throw new RPCError.MALFORMED_PACKET(
                        "get_registrar wants 1 parameter.");
                ISerializable iser0 = data.parameters[0];
                if (! iser0.get_type().is_a(typeof(SerializableString)))
                    throw new RPCError.MALFORMED_PACKET(
                        "get_registrar parameter 1 is not a string.");
                string hashed_domain = ((SerializableString)iser0).s;
                AndnaGetRegistrarResponse? ret = get_registrar(hashed_domain);
                if (ret == null) return new SerializableNone();
                return ret;
            }
            if (pieces[0] == "get_domain_record")
            {
                if (pieces.length != 1)
                    throw new RPCError.MALFORMED_PACKET(
                        "get_domain_record is a function.");
                if (data.parameters.size != 1)
                    throw new RPCError.MALFORMED_PACKET(
                        "get_domain_record wants 1 parameter.");
                ISerializable iser0 = data.parameters[0];
                if (! iser0.get_type().is_a(typeof(SerializableString)))
                    throw new RPCError.MALFORMED_PACKET(
                        "get_domain_record parameter 1 is not a string.");
                string hashed_domain = ((SerializableString)iser0).s;
                AndnaDomainRecord? ret = get_domain_record(hashed_domain);
                if (ret == null) return new SerializableNone();
                return ret;
            }
            if (pieces[0] == "register_spread_record")
            {
                if (pieces.length != 1)
                    throw new RPCError.MALFORMED_PACKET(
                        "register_spread_record is a function.");
                if (data.parameters.size != 3)
                    throw new RPCError.MALFORMED_PACKET(
                        "register_spread_record wants 3 parameters.");
                ISerializable iser0 = data.parameters[0];
                if (! iser0.get_type().is_a(typeof(SerializableString)))
                    throw new RPCError.MALFORMED_PACKET(
                        "register_spread_record parameter 1 is not a string.");
                string hashed_domain = ((SerializableString)iser0).s;
                ISerializable iser1 = data.parameters[1];
                if (! iser1.get_type().is_a(typeof(SerializableInt)))
                    throw new RPCError.MALFORMED_PACKET(
                        "register_spread_record parameter 2 is not a int.");
                int spread_number = ((SerializableInt)iser1).i;
                ISerializable iser2 = data.parameters[2];
                if (! iser2.get_type().is_a(typeof(SerializableBool)))
                    throw new RPCError.MALFORMED_PACKET(
                        "register_spread_record parameter 3 is not a bool.");
                bool replicate = ((SerializableBool)iser2).b;
                return register_spread_record(hashed_domain, spread_number, replicate);
            }
            if (pieces[0] == "get_servers")
            {
                if (pieces.length != 1)
                    throw new RPCError.MALFORMED_PACKET(
                        "get_servers is a function.");
                if (data.parameters.size != 4)
                    throw new RPCError.MALFORMED_PACKET(
                        "get_servers wants 4 parameters.");
                ISerializable iser0 = data.parameters[0];
                if (! iser0.get_type().is_a(typeof(SerializableString)))
                    throw new RPCError.MALFORMED_PACKET(
                        "get_servers parameter 1 is not a string.");
                string hashed_domain = ((SerializableString)iser0).s;
                ISerializable iser1 = data.parameters[1];
                if (! iser1.get_type().is_a(typeof(AndnaServiceKey)))
                    throw new RPCError.MALFORMED_PACKET(
                        "get_servers parameter 2 is not a AndnaServiceKey.");
                AndnaServiceKey srv_key = (AndnaServiceKey)iser1;
                ISerializable iser2 = data.parameters[2];
                if (! iser2.get_type().is_a(typeof(SerializableInt)))
                    throw new RPCError.MALFORMED_PACKET(
                        "get_servers parameter 3 is not a int.");
                int spread_number = ((SerializableInt)iser2).i;
                ISerializable iser3 = data.parameters[3];
                if (! iser3.get_type().is_a(typeof(SerializableBool)))
                    throw new RPCError.MALFORMED_PACKET(
                        "get_servers parameter 4 is not a bool.");
                bool chain = ((SerializableBool)iser3).b;
                return get_servers(hashed_domain, srv_key, spread_number, chain);
            }
            if (pieces[0] == "get_cache_records")
            {
                if (pieces.length != 1)
                    throw new RPCError.MALFORMED_PACKET(
                        "get_cache_records is a function.");
                if (data.parameters.size != 0)
                    throw new RPCError.MALFORMED_PACKET(
                        "get_cache_records wants no parameters.");
                return get_cache_records();
            }
            return base._dispatch(caller, data);
        }

        public static string crypto_hash(string hostname)
        {
            return AndnaUtilities.crypto_hash(hostname);
        }

        private NIP nip_for_lvl_pos(int lvl, int pos)
        {
            int[] ret = maproute.me.get_positions();
            ret[lvl] = pos;
            return new NIP(ret);
        }

        private void impl_hook_to_service() throws Error
        {
            Tasklet.declare_self("Andna.hook_to_service");
            if (will_participate)
            {
                // TODO Handle possible errors in hooking phase. Should we try again?
                //  If we don't, we are not participating in the service.

                // clear old caches
                cache.clear();
                request_queue.clear();

                for (int lvl = 0; lvl < maproute.levels; lvl++)
                {
                    int? first_forward;
                    int? first_back;
                    int? last_back;
                    log_debug(@"$(this.get_type().name()): hook: $(maproute.me) finding peers at lvl $(lvl)");
                    find_hook_peers(out first_forward,
                                    out first_back,
                                    out last_back,
                                    lvl, ANDNA_DUPLICATION);
                    if (first_forward == null)
                    {
                        // no one in my gnode lvl+1 (except me)
                        log_debug(@"$(this.get_type().name()): hook: no one in network at this level (except me)");
                        // I need to go up one level.
                    }
                    else if (first_back == null)
                    {
                        // my gnode lvl+1 has some participants but not
                        // enough to satisfy our replica ANDNA_DUPLICATION.
                        // So, get all.
                        log_debug(@"$(this.get_type().name()): hook: few nodes at this level. get all from $(first_forward).");
                        NIP nip_first_forward = nip_for_lvl_pos(lvl, first_forward);
                        RmtAndnaPeer peer_first_forward = peer(nip_first_forward);
                        // use peer_first_forward.get_cache_records() to
                        //  obtain records and save them all to my caches
                        AndnaGetCacheRecordsResponse cache_first_forward =
                                peer_first_forward.get_cache_records();
                        foreach (string hashed_domain in cache_first_forward.cache.keys)
                        {
                            log_debug(@"$(this.get_type().name()): hook: there is $(hashed_domain) in cache of $(first_forward)");
                            if (! cache.has_key(hashed_domain))
                            {
                                cache[hashed_domain] = cache_first_forward.cache[hashed_domain];
                                log_debug(@"$(this.get_type().name()): hook: got $(hashed_domain) in my cache");
                            }
                        }
                        foreach (string hashed_domain in cache_first_forward.request_queue.keys)
                        {
                            log_debug(@"$(this.get_type().name()): hook: there is $(hashed_domain) in request_queue of $(first_forward)");
                            if (! request_queue.has_key(hashed_domain))
                            {
                                request_queue[hashed_domain] = cache_first_forward.request_queue[hashed_domain];
                                log_debug(@"$(this.get_type().name()): hook: got $(hashed_domain) in my request_queue");
                            }
                        }
                        // and then no need to go up one level.
                        break;
                    }
                    else
                    {
                        // my gnode lvl+1 has enough participant
                        if (first_back != last_back)
                        {
                            // (lvl, first_back) is a (g)node with some participants but not
                            // enough to satisfy our replica ANDNA_DUPLICATION.
                            // (lvl, last_back) + ... + (lvl, first_back) has enough
                            // participant, though.
                            // So I have to get a record from first_back IFF it is not in last_back.
                            NIP nip_first_back = nip_for_lvl_pos(lvl, first_back);
                            RmtAndnaPeer peer_first_back = peer(nip_first_back);
                            NIP nip_last_back = nip_for_lvl_pos(lvl, last_back);
                            RmtAndnaPeer peer_last_back = peer(nip_last_back);
                            // use peer_first_back.get_cache_records() to
                            //  obtain records from first and peer_last_back.get_cache_records() to
                            //  obtain records from last, then use the correct equal_func to
                            //  choose the ones to store in my caches
                            AndnaGetCacheRecordsResponse recs_first =
                                peer_first_back.get_cache_records();
                            AndnaGetCacheRecordsResponse recs_last =
                                peer_last_back.get_cache_records();
                            foreach (string hashed_domain in recs_first.cache.keys)
                            {
                                log_debug(@"$(this.get_type().name()): hook: there is $(hashed_domain) in cache of $(first_back)");
                                if (! cache.has_key(hashed_domain))
                                {
                                    bool found = false;
                                    foreach (string hashed_domain2 in recs_last.cache.keys)
                                    {
                                        if (hashed_domain2 == hashed_domain)
                                        {
                                            found = true;
                                            log_debug(@"$(this.get_type().name()): hook: but also in cache of $(last_back)");
                                            break;
                                        }
                                    }
                                    if (!found)
                                    {
                                        cache[hashed_domain] =
                                            recs_first.cache[hashed_domain];
                                        log_debug(@"$(this.get_type().name()): hook: got $(hashed_domain) in my cache");
                                    }
                                }
                            }
                            foreach (string hashed_domain in recs_first.request_queue.keys)
                            {
                                log_debug(@"$(this.get_type().name()): hook: there is $(hashed_domain) in request_queue of $(first_back)");
                                if (! request_queue.has_key(hashed_domain))
                                {
                                    bool found = false;
                                    foreach (string hashed_domain2 in recs_last.request_queue.keys)
                                    {
                                        if (hashed_domain2 == hashed_domain)
                                        {
                                            found = true;
                                            log_debug(@"$(this.get_type().name()): hook: but also in request_queue of $(last_back)");
                                            break;
                                        }
                                    }
                                    if (!found)
                                    {
                                        request_queue[hashed_domain] =
                                            recs_first.request_queue[hashed_domain];
                                        log_debug(@"$(this.get_type().name()): hook: got $(hashed_domain) in my request_queue");
                                    }
                                }
                            }
                        }
                        // Now the records that are in first forward and whose
                        // key is before me at this level.
                        NIP nip_first_forward = nip_for_lvl_pos(lvl, first_forward);
                        RmtAndnaPeer peer_first_forward = peer(nip_first_forward);
                        // use peer_first_forward.get_cache_records() to
                        //  obtain records and foreach rec which is not in my cache:
                        //   use function h to compute the hashnode of rec.key;
                        //   store in hk the position at level lvl;
                        //   IF my position at level lvl comes before the
                        //      position of first_forward in list_ids(hk, 1):
                        //         save rec into my caches;
                        AndnaGetCacheRecordsResponse cache_first_forward =
                                peer_first_forward.get_cache_records();
                        foreach (string hashed_domain in cache_first_forward.cache.keys)
                        {
                            log_debug(@"$(this.get_type().name()): hook: there is $(hashed_domain) in cache of $(first_forward)");
                            if (! cache.has_key(hashed_domain))
                            {
                                bool check = false;
                                int hk = h(new AndnaServiceHashnodeKey(hashed_domain))
                                        .position_at(lvl);
                                Gee.List<int> ids = list_ids(hk, 1);
                                if (ids.index_of(first_forward) > 
                                    ids.index_of(maproute.me.position_at(lvl)))
                                {
                                    check = true;
                                }
                                if (! check)
                                {
                                    for (int spr = 1; spr <= SPREAD_ANDNA; spr++)
                                    {
                                        int hk2 = h(new AndnaServiceHashnodeKey(hashed_domain, spr))
                                            .position_at(lvl);
                                        Gee.List<int> ids2 = list_ids(hk2, 1);
                                        if (ids2.index_of(first_forward) > 
                                            ids2.index_of(maproute.me.position_at(lvl)))
                                        {
                                            check = true;
                                            break;
                                        }
                                    }
                                }
                                if (check)
                                {
                                    cache[hashed_domain] =
                                        cache_first_forward.cache[hashed_domain];
                                    log_debug(@"$(this.get_type().name()): hook: got $(hashed_domain) in my cache");
                                }
                            }
                        }
                        foreach (string hashed_domain in cache_first_forward.request_queue.keys)
                        {
                            log_debug(@"$(this.get_type().name()): hook: there is $(hashed_domain) in request_queue of $(first_forward)");
                            if (! request_queue.has_key(hashed_domain))
                            {
                                bool check = false;
                                int hk = h(new AndnaServiceHashnodeKey(hashed_domain))
                                        .position_at(lvl);
                                Gee.List<int> ids = list_ids(hk, 1);
                                if (ids.index_of(first_forward) > 
                                    ids.index_of(maproute.me.position_at(lvl)))
                                {
                                    check = true;
                                }
                                if (! check)
                                {
                                    for (int spr = 1; spr <= SPREAD_ANDNA; spr++)
                                    {
                                        int hk2 = h(new AndnaServiceHashnodeKey(hashed_domain, spr))
                                            .position_at(lvl);
                                        Gee.List<int> ids2 = list_ids(hk2, 1);
                                        if (ids2.index_of(first_forward) > 
                                            ids2.index_of(maproute.me.position_at(lvl)))
                                        {
                                            check = true;
                                            break;
                                        }
                                    }
                                }
                                if (check)
                                {
                                    request_queue[hashed_domain] =
                                        cache_first_forward.request_queue[hashed_domain];
                                    log_debug(@"$(this.get_type().name()): hook: got $(hashed_domain) in my request_queue");
                                }
                            }
                        }
                        // Then no need to go up one level.
                        break;
                    }
                }

                // Now I can answer to requests.
                hooked_to_service = true;
                log_debug(@"$(this.get_type().name()): hook: finished.");
                // Now I can participate again.
                participate();
                log_info("Andna service: participating.");
            }
        }

        public void hook_to_service()
        {
            Tasklet.tasklet_callback(
                    () => {
                        impl_hook_to_service();
                    });
        }

        /** This is the function h:KEY-->hIP.
          */
        public override NIP h(Object key)
        {
            AndnaServiceHashnodeKey _key = (AndnaServiceHashnodeKey)key;
            return AndnaUtilities.key_to_hashnode_converter(
                        _key, maproute.levels, maproute.gsize);
        }

        public AndnaDomainRequest make_domain_request(AndnaPrivateConfiguration cfg)
        {
            string hashed_domain = crypto_hash(cfg.domain);
            PublicKey pubk = my_keys.pub_key.to_pubkey();
            NIP nip = maproute.me;
            HashMap<AndnaServiceKey,
                         ArrayList<AndnaServerRequest>>
                         services;
            services = AndnaDomainRequest.make_empty_services();
            foreach (AndnaServiceKey k in cfg.services.keys)
            {
                services[k] = AndnaServerRequest.make_empty_list();
                foreach (AndnaPrivateConfigurationServer cfg_server in cfg.services[k])
                {
                    services[k].add(make_server_request(cfg_server));
                }
            }
            return new AndnaDomainRequest(hashed_domain, pubk, nip, services);
        }

        public AndnaServerRequest make_server_request(AndnaPrivateConfigurationServer cfg)
        {
            string? hashed_alias = null;
            if (cfg.alias != null)
                hashed_alias = crypto_hash(cfg.alias);
            PublicKey? pubk = cfg.pubk;
            int port_number = cfg.port_number;
            int priority = cfg.priority;
            int weight = cfg.weight;
            return new AndnaServerRequest(hashed_alias, pubk, port_number, priority, weight);
        }

        private void impl_register_name(string hostname)
        {
            Tasklet.declare_self(@"Andna.register_name(\"$(hostname)\")");
            DomainWanted dw = new DomainWanted();
            dw.tasklet = Tasklet.self();
            dw.hostname = hostname;
            dw.registered = false;
            dw.ttl_before_request = new TimeCapsule(0);
            lst_domains.add(dw);
            // This tasklet has the duty to maintain the domain 'hostname'
            // Mantenere la titolarità di un un Domain
            while (true)
            {
                try
                {
                    AndnaRegisterMainResponse resp = ask_register_main_for_pubk(hostname);
                    // if all ok
                    if (resp.response == "REGISTERED" || resp.response == "UPDATED")
                    {
                        // emit signal.
                        andna_registered(hostname);
                        dw.registered = true;
                        // spread information
                        for (int i = 1; i <= SPREAD_ANDNA; i++)
                        {
                            // in a new tasklet
                            Tasklet.tasklet_callback(
                                    (tpar1, tpar2) => {
                                        string tasklet_hostname =
                                                ((SerializableString)tpar1).s;
                                        int tasklet_i =
                                                ((SerializableInt)tpar2).i;
                                        Tasklet.declare_self("Andna.spread_registration");
                                        // request spread registration
                                        ask_register_spread_record
                                                (tasklet_hostname, tasklet_i);
                                    },
                                    new SerializableString(hostname),
                                    new SerializableInt(i));
                        }
                        // How long to wait?
                        dw.ttl_before_request = resp.expires;
                        // At most 10 minutes before expiration
                        dw.ttl_before_request = new TimeCapsule(
                                                dw.ttl_before_request.get_msec_ttl() -
                                                (int64)1000 *
                                                (int64)60 *
                                                (int64)10 /* 10 minutes in millisec */
                                              );
                        // but no more than MAX_WAIT_REFRESH_ANDNA
                        TimeCapsule maxttl = new TimeCapsule(MAX_WAIT_REFRESH_ANDNA);
                        if (dw.ttl_before_request.is_younger(maxttl))
                            dw.ttl_before_request = maxttl;
                    }
                    // if not ok
                    else
                    {
                        log_warn(@"Andna: register_name('$(hostname)')" +
                                 @" got '$(resp.response)'");
                        dw.ttl_before_request =
                                new TimeCapsule((int64)1000 *
                                                (int64)60 *
                                                (int64)2 /* 2 minutes in millisec */
                                          );
                    }
                }
                catch (Error e)
                {
                    log_warn(@"Andna: register_name('$(hostname)')" +
                             @" got $(e.domain) code $(e.code): $(e.message)'");
                    // the failure may be temporary
                    dw.ttl_before_request = new TimeCapsule(
                                                (int64)1000 *
                                                (int64)60 *
                                                (int64)2 /* 2 minutes in millisec */
                                          );
                }
                while (! dw.ttl_before_request.is_expired()) Tasklet.nap(0, 100000);
            }
        }

        private void register_name(string hostname)
        {
            Tasklet.tasklet_callback(
                    (tpar1) => {
                        string tasklet_hostname = 
                               ((SerializableString)tpar1).s;
                        impl_register_name(tasklet_hostname);
                    },
                    new SerializableString(hostname));
        }

        public void register_my_names()
        {
            foreach (AndnaPrivateConfiguration cfg in configuration.lst)
            {
                string hostname = cfg.domain;
                register_name(hostname);
            }
        }

        public void stop_register_my_names()
        {
            foreach (DomainWanted dw in lst_domains)
            {
                dw.tasklet.abort();
            }
            lst_domains.clear();
        }

        public bool register_my_names_ongoing()
        {
            return ! lst_domains.is_empty;
        }

        AndnaRegisterMainResponse ask_register_main_for_pubk
                    (string hostname)
                    throws RPCError, PeerRefuseServiceError
        {
            // Richiedere una "registrazione main"
            // Try registration or update of an hostname, record in tha ANDNA distributed database.
            log_debug(@"ANDNA: ask_register_main_for_pubk" +
                                 @"(\"$(hostname)\")");
            AndnaPrivateConfiguration? cfg = configuration.get_domain(hostname);
            assert(cfg != null);
            AndnaDomainRequest request = make_domain_request(cfg);
            SerializableBuffer signature = _sign(request, my_keys);
            return peer(null, new AndnaServiceHashnodeKey(request.hashed_domain))
                    .register_main_for_pubk(request, signature);
        }

        uchar[] prepare_msg(AndnaDomainRequest req)
        {
            uchar[] message = req.hash_for_signature();
            return message;
        }

        SerializableBuffer _sign(AndnaDomainRequest req, KeyPair key_pair)
        {
            uchar[] signed = key_pair.sign(prepare_msg(req));
            return new SerializableBuffer((uint8[])(signed));
        }

        bool _verify(AndnaDomainRequest req, PublicKey pubk, SerializableBuffer signature)
        {
            PublicKeyWrapper pkw = new PublicKeyWrapper.from_pubk(pubk);
            return pkw.verify(prepare_msg(req), (uchar[])(signature.buffer));
        }

        void ask_register_spread_record
                    (string hostname, int spread_number)
                    throws Error
        {
            // Try spreading information of an hostname in tha ANDNA SPREAD distributed database.
            log_debug(@"ANDNA: ask_register_spread_record" +
                                 @"(\"$(hostname)\", $(spread_number))");
            AndnaPrivateConfiguration? cfg = configuration.get_domain(hostname);
            assert(cfg != null);
            AndnaDomainRequest request = make_domain_request(cfg);
            string hashed_domain = request.hashed_domain;
            peer(null, new AndnaServiceHashnodeKey(hashed_domain, spread_number))
                    .register_spread_record(hashed_domain, spread_number);
        }

        public AndnaGetRegistrarResponse? ask_registrar(string hashed_domain)
        {
            // TODO cache responses?
            // TODO use spread?
            var andnapeer = peer(null,
                    new AndnaServiceHashnodeKey(hashed_domain));
            return andnapeer.get_registrar(hashed_domain);
        }

        public AndnaGetServersResponse ask_get_servers
                (string hashed_domain,
                 AndnaServiceKey srv_key=AndnaServiceKey.NULL_SERV_KEY,
                 bool chain=true)
        {
            ArrayList<PartialNIP> choose_from =
                    new ArrayList<PartialNIP>(PartialNIP.equal_func);
            ArrayList<AndnaServiceHashnodeKey> keys_from =
                    new ArrayList<AndnaServiceHashnodeKey>(AndnaServiceHashnodeKey.equal_func);
            AndnaServiceHashnodeKey k_0 =
                    new AndnaServiceHashnodeKey(hashed_domain);
            NIP nip_0 = h(k_0);
            log_debug(@"Andna.ask_get_servers: possible choice: $(nip_0)");
            PartialNIP approx_0 = search_participant_as_nip(nip_0);
            log_debug(@"                                approx: $(approx_0)");
            choose_from.add(approx_0);
            keys_from.add(k_0);
            for (int i = 1 ; i < SPREAD_ANDNA; i++)
            {
                AndnaServiceHashnodeKey k_i =
                        new AndnaServiceHashnodeKey(hashed_domain, i);
                NIP nip_i = h(k_i);
                log_debug(@"Andna.ask_get_servers: possible choice: $(nip_i)");
                PartialNIP approx_i = search_participant_as_nip(nip_i);
                log_debug(@"                                approx: $(approx_i)");
                choose_from.add(approx_i);
                keys_from.add(k_i);
            }
            PartialNIP? chosen = maproute.choose_fast(choose_from);
            if (chosen != null)
            {
                log_debug(@"Andna.ask_get_servers: winner is: $(chosen)");
                int index = choose_from.index_of(chosen);
                AndnaServiceHashnodeKey chosen_key = keys_from[index];
                int spread_number = chosen_key.spread_num;
                RmtAndnaPeer node = peer(null, keys_from[index]);
                AndnaGetServersResponse ret =
                        node.get_servers(hashed_domain, srv_key, spread_number, chain);
                // TODO save local cache
                return ret;
            }
            // no participants?
            // TODO handle this
            return null;
        }

        /** Helper methods used as a server
          */
        private void str_debug_cache()
        {
            foreach (string k in cache.keys)
            {
                log_debug(@"ANDNA:        $(k) ttl " +
                                 @"$(cache[k].expires.get_string_msec_ttl())");
            }
        }

        AndnaServerRecord make_server_record(AndnaServerRequest req)
        {
            string? hashed_alias = req.hashed_alias;
            PublicKey? pubk = req.pubk;
            int port_number = req.port_number;
            int priority = req.priority;
            int weight = req.weight;
            return new AndnaServerRecord(hashed_alias, pubk, port_number, priority, weight);
        }

        public void check_expirations_cache()
        {
            // Remove the expired entries from the ANDNA cache
            log_debug(@"ANDNA: cleaning expired entries - if any - from our ANDNA cache...");
            log_debug(@"ANDNA: cache was:");
            str_debug_cache();
            ArrayList<string> todel =
                    new ArrayList<string>();
            foreach (string hashed_domain in cache.keys)
            {
                AndnaDomainRecord record = cache[hashed_domain];
                if (record.expires.is_expired())
                {
                    // this has expired
                    todel.add(hashed_domain);
                }
            }
            foreach (string hashed_domain in todel)
            {
                AndnaDomainRecord record;
                cache.unset(hashed_domain, out record);
                log_debug(@"ANDNA: cleaned $(hashed_domain) from pubk $(record.pubk)");
                if (request_queue.has_key(hashed_domain))
                {
                    // There is something in queue
                    if (request_queue[hashed_domain].size == 0)
                    {
                        // No, it is empty, just remove.
                        request_queue.unset(hashed_domain);
                    }
                    else
                    {
                        // Take first in queue
                        RegisterHostnameArguments args =
                                request_queue[hashed_domain]
                                .remove_at(0);
                        log_debug(@"ANDNA: trying to register it to pubk $(args.request.pubk)");
                        // TODO to be tested
                        AndnaRegisterMainResponse res =
                                register_main_for_pubk
                                (args.request, args.signature, true);
                        // No need to communicate immediately to the lucky node.
                    }
                }
            }
            log_debug(@"ANDNA: cache now is:");
            str_debug_cache();
        }

        /** Remotable methods as peer()
          */
        public AndnaRegisterMainResponse register_main_for_pubk
                (AndnaDomainRequest request, SerializableBuffer signature,
                bool replicate)
                throws PeerRefuseServiceError
        {
            if (!hooked_to_service)
            {
                if (replicate)
                {
                    // This is an original call. I am not hooked, so I should not
                    // have got this call. Anyway, I can wait a bit before throwing
                    // an error to see if I hook soon. But the client is waiting
                    // for an answer, thus after a bit I want to throw an exception
                    // in order for the client to try again.
                    Tasklets.Timer w = new Tasklets.Timer(10000); // 10 seconds
                    while (!hooked_to_service)
                    {
                        if (w.is_expired())
                            throw new PeerRefuseServiceError.GENERIC("Not hooked yet");
                        Tasklet.nap(0, 1000);
                    }
                }
                else
                {
                    // This is a replica. So the client is not waiting and will not
                    // request again. Thus I will wait for the hook whatever it takes.
                    while (!hooked_to_service) Tasklet.nap(0, 1000);
                }
            }

            string response_msg = "Noooo";
            TimeCapsule response_ttl = new TimeCapsule(0);
            check_expirations_cache();
            NIP hashnode = h(new AndnaServiceHashnodeKey(request.hashed_domain));
            ArgumentsForDuplicationRegisterMain rec = new ArgumentsForDuplicationRegisterMain();
            rec.request = request;
            rec.signature = signature;
            rec.hashnode = hashnode;
            // I verify that I am the hash node for the key.
            check_hash_and_start_replica(this, hashnode, replicate, rec, ANDNA_DUPLICATION,
                    /*AcceptRecordCallback*/
                    () => {
                        // verify signature with request.pubk;
                        if (! _verify(request, request.pubk, signature))
                            throw new PeerRefuseServiceError.GENERIC(
                                        "Not your signature.");

                        // Contact counter hashnode for request.nip
                        // and ask to him for permission;
                        CounterCheckHostnameResponse checkhostname =
                            counter.ask_check_hostname(
                                request.nip,
                                request.hashed_domain,
                                request.pubk);
                        if (! checkhostname.response)
                            throw new PeerRefuseServiceError.GENERIC(
                                        "Not registered in Counter service.");

                        if (! cache.has_key(request.hashed_domain))
                        {
                            // immediately save the hostname with 0 services and MAXTTL
                            cache[request.hashed_domain] = new AndnaDomainRecord(
                                MAX_TTL_ANDNA, request.hashed_domain, request.pubk, request.nip);
                            response_msg = "REGISTERED";
                            response_ttl = new TimeCapsule(MAX_TTL_ANDNA);
                        }

                        // For any record of type 'alias', try to verify that the registrar
                        //  of the alias name is who our registrar thinks he is. Otherwise drop it.
                        HashMap<AndnaServiceKey, ArrayList<AndnaServerRecord>> services =
                                new HashMap<AndnaServiceKey, ArrayList<AndnaServerRecord>>
                                (AndnaServiceKey.hash_func, AndnaServiceKey.equal_func);
                        foreach (AndnaServiceKey srvk in request.services.keys)
                        {
                            services[srvk] = new ArrayList<AndnaServerRecord>
                                             (AndnaServerRecord.equal_func);
                            foreach (AndnaServerRequest srvrec in request.services[srvk])
                            {
                                if (srvrec.hashed_alias != null)
                                {
                                    AndnaGetRegistrarResponse? registrar_alias
                                            = ask_registrar(srvrec.hashed_alias);
                                    if (registrar_alias != null)
                                    {
                                        if (PublicKey.equal_func(registrar_alias.pubk, srvrec.pubk))
                                        {
                                            services[srvk].add(make_server_record(srvrec));
                                        }
                                    }
                                }
                                else
                                {
                                    services[srvk].add(make_server_record(srvrec));
                                }
                            }
                            if (services[srvk].size == 0) services.unset(srvk);
                        }

                        // Is a request coming from its current owner?
                        AndnaDomainRecord rec_in_cache = cache[request.hashed_domain];
                        if (PublicKey.equal_func(rec_in_cache.pubk, request.pubk))
                        {
                            // refresh data with max ttl, current nip of registrar,
                            // services records (previously epurated if necessary)
                            rec_in_cache.update(MAX_TTL_ANDNA, request.nip, services);
                            if (response_msg != "REGISTERED") response_msg = "UPDATED";
                            log_debug(@"Andna: Registered: $(rec_in_cache)");
                            response_ttl = new TimeCapsule(MAX_TTL_ANDNA);
                        }
                        else
                        {
                            // Not available. I check if I can put it in the queue.
                            if (! request_queue.has_key(request.hashed_domain))
                                request_queue[request.hashed_domain] =
                                        new ArrayList<RegisterHostnameArguments>
                                            (RegisterHostnameArguments.equal_func_for_queue);
                            // Is this registrar (check its pubk) already in the queue?
                            bool present = false;
                            foreach (RegisterHostnameArguments args
                                     in request_queue[request.hashed_domain])
                            {
                                if (PublicKey.equal_func(args.request.pubk, request.pubk))
                                {
                                
                                    present = true;
                                    // refresh request
                                    args.request = request;
                                    args.signature = signature;
                                    args.expires = new TimeCapsule(MAX_TTL_ANDNA);
                                    response_msg = "QUEUED";
                                    response_ttl = args.expires;
                                }
                            }
                            if (!present)
                            {
                                // This registrar is not yet in the queue. Is there room in the queue?
                                if (request_queue[request.hashed_domain].size < MAX_QUEUE_ANDNA)
                                {
                                    // We put the request in queue
                                    request_queue[request.hashed_domain].add(
                                            new RegisterHostnameArguments
                                                (request,
                                                 signature,
                                                 new TimeCapsule(MAX_TTL_ANDNA)));
                                }
                                else
                                {
                                    response_msg = "QUEUE_FULL";
                                    response_ttl = new TimeCapsule(0);
                                }
                            }
                        }

                        return true;
                    },
                    /*ForwardRecordCallback*/
                    (tasklet_obj1, tasklet_replica_nodes) => {
                        ArgumentsForDuplicationRegisterMain tasklet_rec =
                                (ArgumentsForDuplicationRegisterMain)tasklet_obj1;
                        // Here I am in a tasklet (the client has been served already)
                        if (tasklet_replica_nodes == null)
                                tasklet_replica_nodes =
                                        find_nearest_to_register
                                        (tasklet_rec.hashnode, ANDNA_DUPLICATION);
                        // For each node of the <n> nearest except myself...
                        foreach (NIP replica_node in tasklet_replica_nodes) if (!replica_node.is_equal(maproute.me))
                        {
                            // ... in another tasklet...
                            Tasklet.tasklet_callback(
                                (tpar1, tpar2) => {
                                    NIP tonip = (NIP)tpar1;
                                    ArgumentsForDuplicationRegisterMain arec =
                                            (ArgumentsForDuplicationRegisterMain)tpar2;
                                    Tasklet.declare_self("Andna.forward_record_main");
                                    // ... forward the record to the node.
                                    try
                                    {
                                        peer(tonip).register_main_for_pubk
                                                (arec.request,
                                                 arec.signature,
                                                 false);
                                    }
                                    catch (RPCError e)
                                    {
                                        // report the error with some info on where it happened
                                        log_warn(@"Andna.forward_record_main: forwarding to $(tonip):"
                                            + @" got $(e.domain.to_string()) $(e.code) $(e.message)");
                                    }
                                },
                                replica_node,
                                tasklet_rec);
                        }
                    });
            return new AndnaRegisterMainResponse(response_msg, response_ttl);
        }

        public AndnaGetRegistrarResponse? get_registrar
                (string hashed_domain)
                throws PeerRefuseServiceError
        {
            if (!hooked_to_service)
            {
                // I am not hooked, so I should not have got this call.
                // The client is waiting for a reply.
                // After a bit I will throw an exception.
                Tasklets.Timer w = new Tasklets.Timer(10000); // 10 seconds
                while (!hooked_to_service)
                {
                    if (w.is_expired())
                        throw new PeerRefuseServiceError.GENERIC("Not hooked yet");
                    Tasklet.nap(0, 1000);
                }
            }

            check_expirations_cache();
            // First, I check if I am the exact best participant for the hashnode
            //  for the key of this request.
            NIP hashnode = h(new AndnaServiceHashnodeKey(hashed_domain));
            if (search_participant(hashnode) != null)
            {
                // Not me.
                throw new PeerRefuseServiceError.GENERIC("Not the correct hashnode");
            }

            if (cache.has_key(hashed_domain))
            {
                AndnaDomainRecord ret = cache[hashed_domain];
                return new AndnaGetRegistrarResponse(ret.pubk,
                                                     ret.nip,
                                                     ret.expires);
            }
            else return null;
        }

        public AndnaDomainRecord? get_domain_record
                (string hashed_domain)
                throws PeerRefuseServiceError
        {
            if (!hooked_to_service)
            {
                // I am not hooked, so I should not have got this call.
                // The client is waiting for a reply.
                // After a bit I will throw an exception.
                Tasklets.Timer w = new Tasklets.Timer(10000); // 10 seconds
                while (!hooked_to_service)
                {
                    if (w.is_expired())
                        throw new PeerRefuseServiceError.GENERIC("Not hooked yet");
                    Tasklet.nap(0, 1000);
                }
            }

            check_expirations_cache();
            // First, I check if I am the exact best participant for the hashnode
            //  for the key of this request.
            NIP hashnode = h(new AndnaServiceHashnodeKey(hashed_domain));
            if (search_participant(hashnode) != null)
            {
                // Not me.
                throw new PeerRefuseServiceError.GENERIC("Not the correct hashnode");
            }

            if (! cache.has_key(hashed_domain)) return null;
            return cache[hashed_domain];
        }

        public AndnaRegisterSpreadResponse register_spread_record
                (string hashed_domain, int spread_number,
                bool replicate)
                throws PeerRefuseServiceError
        {
            if (!hooked_to_service)
            {
                // I am not hooked, so I should not have got this call.
                // The client is not waiting for a reply and will not
                // request again.
                // Anyway this is a spread registration.
                // After a bit I want to abandon.
                Tasklets.Timer w = new Tasklets.Timer(10000); // 10 seconds
                while (!hooked_to_service)
                {
                    if (w.is_expired())
                        throw new PeerRefuseServiceError.GENERIC("Not hooked yet");
                    Tasklet.nap(0, 1000);
                }
            }

            bool response_response = true;
            string response_msg = "OK";
            check_expirations_cache();
            NIP hashnode = h(new AndnaServiceHashnodeKey(hashed_domain, spread_number));
            ArgumentsForDuplicationRegisterSpread rec = new ArgumentsForDuplicationRegisterSpread();
            rec.hashed_domain = hashed_domain;
            rec.spread_number = spread_number;
            rec.hashnode = hashnode;
            // I verify that I am the hash node for the key.
            check_hash_and_start_replica(this, hashnode, replicate, rec, ANDNA_DUPLICATION,
                    /*AcceptRecordCallback*/
                    () => {
                        var an_main = peer(null, new AndnaServiceHashnodeKey(rec.hashed_domain));
                        AndnaDomainRecord? main_rec =
                                an_main.get_domain_record(rec.hashed_domain);

                        if (main_rec != null)
                        {
                            cache[rec.hashed_domain] = main_rec;
                        }
                        else
                        {
                            response_response = false;
                            response_msg = "MAIN_NOT_FOUND";
                            return false;
                        }

                        return true;
                    },
                    /*ForwardRecordCallback*/
                    (tasklet_obj1, tasklet_replica_nodes) => {
                        ArgumentsForDuplicationRegisterSpread tasklet_rec =
                                (ArgumentsForDuplicationRegisterSpread)tasklet_obj1;
                        // Here I am in a tasklet (the client has been served already)
                        if (tasklet_replica_nodes == null)
                                tasklet_replica_nodes =
                                        find_nearest_to_register
                                        (tasklet_rec.hashnode, ANDNA_DUPLICATION);
                        // For each node of the <n> nearest except myself...
                        foreach (NIP replica_node in tasklet_replica_nodes) if (!replica_node.is_equal(maproute.me))
                        {
                            // ... in another tasklet...
                            Tasklet.tasklet_callback(
                                (tpar1, tpar2) => {
                                    NIP tonip = (NIP)tpar1;
                                    ArgumentsForDuplicationRegisterSpread arec =
                                            (ArgumentsForDuplicationRegisterSpread)tpar2;
                                    Tasklet.declare_self("Andna.forward_record_spread");
                                    // ... forward the record to the node.
                                    try
                                    {
                                        peer(tonip).register_spread_record
                                                (arec.hashed_domain,
                                                 arec.spread_number,
                                                 false);
                                    }
                                    catch (RPCError e)
                                    {
                                        // report the error with some info on where it happened
                                        log_warn(@"Andna.forward_record_spread: forwarding to $(tonip):"
                                            + @" got $(e.domain.to_string()) $(e.code) $(e.message)");
                                    }
                                },
                                replica_node,
                                tasklet_rec);
                        }
                    });
            return new AndnaRegisterSpreadResponse(response_response, response_msg);
        }

        public AndnaGetServersResponse get_servers
                (string hashed_domain, AndnaServiceKey srv_key,
                int spread_number, bool chain)
                throws PeerRefuseServiceError
        {
            if (!hooked_to_service)
            {
                // I am not hooked, so I should not have got this call.
                // The client is waiting for a reply.
                // After a bit I will throw an exception.
                Tasklets.Timer w = new Tasklets.Timer(10000); // 10 seconds
                while (!hooked_to_service)
                {
                    if (w.is_expired())
                        throw new PeerRefuseServiceError.GENERIC("Not hooked yet");
                    Tasklet.nap(0, 1000);
                }
            }

            check_expirations_cache();
            // First, I check if I am the exact best participant for the hashnode
            //  for the key of this request.
            NIP hashnode = h(new AndnaServiceHashnodeKey(hashed_domain, spread_number));
            if (search_participant(hashnode) != null)
            {
                // Not me.
                throw new PeerRefuseServiceError.GENERIC("Not the correct hashnode");
            }

            TimeCapsule expires;
            AndnaServers response;
            if (cache.has_key(hashed_domain))
            {
                // I have a record (maybe for this srv_key)
                response = cache[hashed_domain].get_servers(srv_key);
                expires = response.expires;

                // Look for servers that refer other names
                ArrayList<AndnaServer> todel = new ArrayList<AndnaServer>();
                foreach (AndnaServer result in response.servers)
                {
                    if (result.alias_name != null)
                    {
                        if (! chain)
                        {
                            todel.add(result);
                        }
                        else
                        {
                            if (AndnaServiceKey.equal_func(srv_key,
                                        AndnaServiceKey.NULL_SERV_KEY))
                            {
                                // Try and resolve
                                AndnaGetServersResponse resolved_alias
                                        = ask_get_servers(hashed_domain,
                                                AndnaServiceKey.NULL_SERV_KEY,
                                                false);
                                AndnaServers alias_servers = resolved_alias.response;
                                if (alias_servers.is_not_found)
                                {
                                    todel.add(result);
                                }
                                else
                                {
                                    result.set_nip(alias_servers.servers[0].registrar_nip);
                                }
                            }
                            else
                            {
                                // it's ok to have the alias name as a response.
                            }
                        }
                    }
                }
                foreach (AndnaServer result in todel) response.servers.remove(result);
            }
            else
            {
                expires = new TimeCapsule(MAX_TTL_OF_NEGATIVE);
                response = new AndnaServers.not_found(expires);
            }

            return new AndnaGetServersResponse(response, expires);
        }

        public AndnaGetCacheRecordsResponse get_cache_records()
                throws PeerRefuseServiceError
        {
            // I am not hooked, so I should not
            // have got this call. Anyway, I can wait a bit before throwing
            // an error to see if I hook soon. But the client is waiting
            // for an answer, thus after a bit I want to throw an exception
            // in order for the client to try again.
            Tasklets.Timer w = new Tasklets.Timer(10000); // 10 seconds
            while (!hooked_to_service)
            {
                if (w.is_expired())
                    throw new PeerRefuseServiceError.GENERIC("Not hooked yet");
                Tasklet.nap(0, 1000);
            }

            return new AndnaGetCacheRecordsResponse(cache, request_queue);
        }


        /** Remotable methods as AddressManager...client
          */
        public AndnaConfirmPubkResponse confirm_pubk
                        (NIP yournip, PublicKey yourpubkey, int to_be_signed)
        {
            if (maproute.me.is_equal(yournip) && PublicKey.equal_func(mypubk, yourpubkey))
            {
                uint8[] buf = (uint8[])my_keys.sign((uchar[])(@"$(to_be_signed)".data));
                SerializableBuffer signature = new SerializableBuffer(buf);
                return new AndnaConfirmPubkResponse("OK", signature);
            }
            //else throw new Error();
            return new AndnaConfirmPubkResponse("NO", new SerializableBuffer({}));
        }

        public Gee.List<string> get_your_hostnames(NIP yournip)
        {
            ArrayList<string> ret = new ArrayList<string>();
            foreach (AndnaPrivateConfiguration c in configuration.lst)
            {
                ret.add(c.domain);
            }
            return ret;
        }

        public InfoAndna report_status()
        {
            InfoAndna ret = new InfoAndna();
            ret.will_participate = will_participate;
            ret.participating = participant;
            ret.hooked = hooked_to_service;
            ret.register_ongoing = register_my_names_ongoing();
            ret.pubk = @"$(mypubk)";
            foreach (string hashed_domain in cache.keys)
            {
                InfoAndnaCache c = new InfoAndnaCache();
                c.domain = hashed_domain;
                c.rec = cache[hashed_domain];
                ret.cache.add(c);
            }
            foreach (DomainWanted dw in lst_domains)
            {
                InfoAndnaRegistration r = new InfoAndnaRegistration();
                r.domain = dw.hostname;
                r.registered = dw.registered;
                r.ttl_before_request = dw.ttl_before_request;
                ret.registrations.add(r);
            }
            return ret;
        }

        // This remotable is only to debug a node with monitorradar
        public AndnaPrivateConfigurationList get_mynames()
        {
            // no need to make a copy of configuration since it is a remotable function
            return configuration;
        }

        // This remotable is only to debug a node with monitorradar
        public void set_mynames(AndnaPrivateConfigurationList mynames)
        {
            stop_register_my_names();
            // no need to make a copy since it is a remotable function
            _configuration = mynames;
            register_my_names();
        }

        // This remotable is only to debug a node with monitorradar
        public PublicKey? retrieve_registrar_pubk(string hashed_domain)
        {
            // get registrar
            AndnaGetRegistrarResponse? ret = ask_registrar(hashed_domain);
            if (ret == null) return null;
            return ret.pubk;
        }
    }

}
