/*
 *  This file is part of Netsukuku.
 *  (c) Copyright 2011 Luca Dionisi aka lukisi <luca.dionisi@gmail.com>
 *
 *  Netsukuku is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  Netsukuku is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with Netsukuku.  If not, see <http://www.gnu.org/licenses/>.
 */

using Gee;
using zcd;
using Tasklets;

namespace Netsukuku
{
    /** 
      */
    public class NetworkInterfaceManager : Object
    {
        public NIC nic_class {get; protected set;}
        public string nic_name {get; protected set;}
        private UnicastCallbackDelegate? udp_unicast_callback;
        private BroadcastCallbackDelegate? udp_broadcast_callback;
        public bool to_be_managed;
        public bool to_be_copied;
        public bool initialized;
        protected ArrayList<AddressManager> addrs;
        private UDPServer? udpserver;
        protected NetworkInterfaceManager.fake() {}  // to allow derivation of Glue
        public NetworkInterfaceManager(string nic_name, NIC nic_class,
                UnicastCallbackDelegate? udp_unicast_callback=null,
                BroadcastCallbackDelegate? udp_broadcast_callback=null)
        {
            this.nic_class = nic_class;
            this.nic_name = nic_name;
            this.udp_unicast_callback = udp_unicast_callback;
            this.udp_broadcast_callback = udp_broadcast_callback;
            to_be_managed = true;
            to_be_copied = true;
            initialized = false;
            addrs = new ArrayList<AddressManager>();
            udpserver = null;
        }

        public void initialize()
        {
            if (to_be_managed)
            {
                udpserver = new UDPServer(udp_unicast_callback, udp_broadcast_callback, nic_name);
                nic_class.disable_filtering();
            }
            initialized = true;
        }

        public virtual NeighbourManager create_neighbour_manager(AddressManager addr_man)
        {
            // Activate IP address on NIC
            string my_addr = nip_to_str(addr_man.maproute.levels, addr_man.maproute.gsize, addr_man.maproute.me);
            nic_class.add_address(my_addr);
            log_debug("NetworkInterfaceManager: " + my_addr + " activated on " + nic_name);
            // instantiate radar and neighbour_manager
            Radar radar = new Radar(nic_name, addr_man);
            NeighbourManager neighbour_manager = new NeighbourManager(radar);
            // start the radar tasklet
            radar.run();
            return neighbour_manager;
        }

        public virtual void dispose_neighbour_manager(AddressManager addr_man, NeighbourManager neighbour_manager)
        {
            // Dispose IP address on NIC
            Radar radar = neighbour_manager.radar;
            // stop the radar tasklet
            radar.stop();
            // This will remove the neighbours detected with this nic:
            //  give some time to get the signal from NeighbourManager, before removing it.
            Tasklet.nap(1, 0);
            // Deactivate IP address on NIC
            string my_addr = nip_to_str(addr_man.maproute.levels, addr_man.maproute.gsize, addr_man.maproute.me);
            nic_class.remove_address(my_addr);
            log_debug("NetworkInterfaceManager: " + my_addr + " deactivated on " + nic_name);
        }

        public virtual void add_address(AddressManager address_manager)
        {
            if (! initialized) initialize();
            // We start to manage a address
            addrs.add(address_manager);
            if (to_be_managed)
            {
                if (! udpserver.is_listening())
                {
                    udpserver.listen();
                }
            }
        }

        public virtual void remove_address(AddressManager address_manager)
        {
            // We cease to manage a address
            addrs.remove(address_manager);
            if (to_be_managed)
            {
                if (udpserver.is_listening() && addrs.is_empty)
                {
                    udpserver.stop();
                }
            }
        }
    }

    struct struct_helper_GlueNetworkInterfaceManager_keep_updating_netid
    {
        public GlueNetworkInterfaceManager self;
    }

    /** 
      */
    public class GlueNetworkInterfaceManager : NetworkInterfaceManager
    {
        private NeighbourManager neighbour_manager;
        public GlueNetworkInterfaceManager()
        {
            base.fake();
            Radar radar = new GlueRadar();
            neighbour_manager = new GlueNeighbourManager(radar, this);
            nic_name = "glue_nic";
            to_be_managed = false;
            to_be_copied = true;
            initialized = false;
            addrs = new ArrayList<AddressManager>();
            keep_updating_netid();
        }

        private void impl_keep_updating_netid() throws Error
        {
            Tasklet.declare_self("GlueNICManager.keep_updating_netid");
            HashMap<AddressManager, NetworkID> addr_netid = new HashMap<AddressManager, NetworkID>();
            HashMap<AddressManager, bool> addr_primary = new HashMap<AddressManager, bool>();
            while (true)
            {
                try
                {
                    ms_wait(3000);
                    // Emit event for a changed attribute for all old neighbours.
                    // If the NetworkID of one of my addresses changes, or its is_primary attribute, this must be known by
                    // the other addresses. This signal will do.
                    HashMap<AddressManager, NetworkID> new_addr_netid = new HashMap<AddressManager, NetworkID>();
                    HashMap<AddressManager, bool> new_addr_primary = new HashMap<AddressManager, bool>();
                    foreach (AddressManager address in addrs)
                    {
                        NetworkID netid = address.get_main_netid();
                        bool is_primary = address.is_primary;
                        new_addr_netid[address] = netid;
                        new_addr_primary[address] = is_primary;
                        if (addr_netid.has_key(address))
                        {
                            NetworkID old_netid = addr_netid[address];
                            bool old_is_primary = addr_primary[address];
                            if ((! old_netid.is_same_network(netid)) || old_is_primary != is_primary)
                            {
                                bool is_auxiliary = address.is_auxiliary;
                                int nodeid = address.get_my_id();
                                NIP nip = address.maproute.me;
                                int levels = address.maproute.levels;
                                int gsize = address.maproute.gsize;
                                Neighbour neighbour = new Neighbour.with_local_dispatcher(levels, gsize, nip,
                                                      nodeid, netid, is_primary, is_auxiliary, new AddressManagerDispatcher(address));
                                neighbour_manager.neighbour_rem_chged(neighbour, new NullREM());
                            }
                        }
                    }
                    addr_netid = new_addr_netid;
                    addr_primary = new_addr_primary;
                }
                catch (Error e)
                {
                    log_info(@"GlueNetworkInterfaceManager.keep_updating_netid: Got following exception: $(e.domain.to_string()) $(e.code) $(e.message), but proceeding anyway");
                }
            }
        }

        private static void * helper_keep_updating_netid(void *v) throws Error
        {
            struct_helper_GlueNetworkInterfaceManager_keep_updating_netid *tuple_p = (struct_helper_GlueNetworkInterfaceManager_keep_updating_netid *)v;
            // The caller function has to add a reference to the ref-counted instances
            GlueNetworkInterfaceManager self_save = tuple_p->self;
            // schedule back to the spawner; this will probably invalidate *v and *tuple_p.
            Tasklet.schedule_back();
            // The actual call
            self_save.impl_keep_updating_netid();
            // void method, return null
            return null;
        }

        public void keep_updating_netid()
        {
            struct_helper_GlueNetworkInterfaceManager_keep_updating_netid arg = struct_helper_GlueNetworkInterfaceManager_keep_updating_netid();
            arg.self = this;
            Tasklet.spawn((FunctionDelegate)helper_keep_updating_netid, &arg);
        }

        public override NeighbourManager create_neighbour_manager(AddressManager addr_man)
        {
            return neighbour_manager;
        }

        public override void dispose_neighbour_manager(AddressManager addr_man, NeighbourManager neighbour_manager)
        {
        }

        public override void add_address(AddressManager address)
        {
            if (address.do_i_act_as_gateway_for_local())
            {
                // base method: update this.addrs
                base.add_address(address);
                // emit event for a new neighbour
                bool is_primary = address.is_primary;
                bool is_auxiliary = address.is_auxiliary;
                NetworkID netid = address.get_main_netid();
                int nodeid = address.get_my_id();
                NIP nip = address.maproute.me;
                int levels = address.maproute.levels;
                int gsize = address.maproute.gsize;
                Neighbour neighbour = new Neighbour.with_local_dispatcher(levels, gsize, nip,
                                      nodeid, netid, is_primary, is_auxiliary, new AddressManagerDispatcher(address));
                log_debug(@"GlueNetworkInterfaceManager: emit event NEIGHBOUR_NEW for $(neighbour)");
                neighbour_manager.neighbour_new(neighbour);
            }
        }

        public override void remove_address(AddressManager address)
        {
            if (address.do_i_act_as_gateway_for_local())
            {
                // base method: update this.addrs
                base.remove_address(address);
                // emit event for a dead neighbour
                bool is_primary = address.is_primary;
                bool is_auxiliary = address.is_auxiliary;
                NetworkID netid = address.get_main_netid();
                int nodeid = address.get_my_id();
                NIP nip = address.maproute.me;
                int levels = address.maproute.levels;
                int gsize = address.maproute.gsize;
                Neighbour neighbour = new Neighbour.with_local_dispatcher(levels, gsize, nip,
                                      nodeid, netid, is_primary, is_auxiliary, new AddressManagerDispatcher(address));
                log_debug(@"GlueNetworkInterfaceManager: emit event NEIGHBOUR_DELETED for $(neighbour)");
                neighbour_manager.neighbour_deleted(neighbour);
                string str_addrs = "<List ";
                string next = "";
                foreach (AddressManager addr in addrs)
                {
                    str_addrs += next + addr.to_string();
                    next = ",";
                }
                str_addrs += ">";
                log_debug(@"'nic_manager $(nic_name): addrs: $(str_addrs)");
            }
        }

        public Gee.List<Neighbour> neighbour_list()
        {
            // Returns the list of Neighbour (and in fact their AddressManagerDispatcher)
            //  for our local addresses.
            return neighbour_manager.neighbour_list();
        }

        public ArrayList<AddressManager> get_addrs()
        {
            // to obtain the protected list from the fake NeighbourManager
            return addrs;
        }
    }
    public class GlueRadar : Radar
    {
        public GlueRadar()
        {
            base.fake();
        }

        public override void time_register(int radar_id, int levels, int gsize, NIP nip, int nodeid,
                        NetworkID netid, string mac, bool is_primary, bool is_auxiliary)
        {
            // Simulate method of real Radar. Do nothing.
        }
    }
    public class GlueNeighbourManager : NeighbourManager
    {
        private GlueNetworkInterfaceManager gluenic;
        public GlueNeighbourManager(Radar radar, GlueNetworkInterfaceManager gluenic)
        {
            base.fake();
            this.radar = radar;
            this.gluenic = gluenic;
        }

        public override void declare_dead(NIP nip, int nodeid)
        {
            // Simulate method of real NeighbourManager. Do nothing.
        }

        public override Gee.List<Neighbour> neighbour_list()
        {
            // Simulate method of NeighbourManager. Returns a neighbour for each address.
            ArrayList<AddressManager> addrs = gluenic.get_addrs();
            ArrayList<Neighbour> nlist = new ArrayList<Neighbour>(Neighbour.equal_func);
            log_debug("GlueNetworkInterfaceManager: prepare a list:");
            foreach (AddressManager address in addrs)
            {
                bool is_primary = address.is_primary;
                bool is_auxiliary = address.is_auxiliary;
                NetworkID netid = address.get_main_netid();
                int nodeid = address.get_my_id();
                NIP nip = address.maproute.me;
                int levels = address.maproute.levels;
                int gsize = address.maproute.gsize;
                Neighbour neighbour = new Neighbour.with_local_dispatcher(levels, gsize, nip,
                                      nodeid, netid, is_primary, is_auxiliary, new AddressManagerDispatcher(address));
                log_debug(@"GlueNetworkInterfaceManager: adding to list $(neighbour)");
                nlist.add(neighbour);
            }
            return nlist;
        }
    }
}

