/*
 *  This file is part of Netsukuku.
 *  (c) Copyright 2011-2014 Luca Dionisi aka lukisi <luca.dionisi@gmail.com>
 *
 *  Netsukuku is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  Netsukuku is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with Netsukuku.  If not, see <http://www.gnu.org/licenses/>.
 */

using Gee;
using Netsukuku;
using Tasklets;
using zcd;
using Ntk.Test;

namespace Netsukuku
{
#if log_tasklet
    private string tasklet_id()
    {
        return @"[$(Tasklet.self().id)] ";
    }
#else
    private string tasklet_id()
    {
        return "";
    }
#endif
    internal void log_debug(string msg) {print_out(tasklet_id()+msg+"\n");}
    internal void log_info(string msg) {print_out(tasklet_id()+msg+"\n");}
    internal void log_notice(string msg) {print_out(tasklet_id()+msg+"\n");}
    internal void log_warn(string msg) {print_out(tasklet_id()+msg+"\n");}
    internal void log_error(string msg) {print_out(tasklet_id()+msg+"\n");}
    internal void log_critical(string msg) {print_out(tasklet_id()+msg+"\n");}

    namespace AndnaConst
    {
        // 30 days in millisec
        int64 MAX_TTL_ANDNA = (int64)1000 * (int64)60 * (int64)60 * (int64)24 * (int64)30;
    }
}

namespace Ntk.Test
{
    string logger;
    const bool output = false;
    public void print_out(string s)
    {
        if (output) print(s);
    }

    public class AndnaTester : Object
    {
        public void set_up ()
        {
            logger = "";
        }

        public void tear_down ()
        {
            logger = "";
        }

        public void test_timers_boundary ()
        {
            Tasklets.Timer[] t = {};
            int64 ms = 2000;
            while (true)
            {
                print_out(@"create timer $(ms)\n");
                t += new Tasklets.Timer(ms);
                ms *= (int64)2;
                if (ms > 600000000000) break;
                // after this value the implementation details
                // of the Timer would exceed the long.MAX in a 32-bit
                // machine and go crash
            }
            Tasklet.nap(0, 10000);
            foreach (Tasklets.Timer t0 in t)
            {
                print_out(@"$(t0.get_string_msec_ttl())   $(t0.is_expired())\n");
                assert(! t0.is_expired());
            }
            Tasklets.Timer t_two = new Tasklets.Timer(2000);
            Tasklets.Timer t_decr = new Tasklets.Timer(t_two.get_msec_ttl() - 1000);
            assert(t_two.is_younger(t_decr));
            assert(!t_decr.is_expired());
            Tasklets.Timer t_expired = new Tasklets.Timer(t_two.get_msec_ttl() - 2001);
            assert(t_expired.is_expired());
        }

        public void test_ser_servkey ()
        {
            AndnaServiceKey sk0;
            {
                uchar[] orig;
                {
                    AndnaServiceKey sk = AndnaServiceKey.NULL_SERV_KEY;
                    orig = sk.serialize();
                }
                uchar []dest = new uchar[orig.length];
                for (int i = 0; i < orig.length; i++) dest[i] = orig[i];
                sk0 = (AndnaServiceKey)ISerializable.deserialize(dest);
            }
            print_out(@"sk0 $(sk0)\n");
            assert(AndnaServiceKey.equal_func(sk0, AndnaServiceKey.NULL_SERV_KEY));

            AndnaServiceKey sk1;
            {
                uchar[] orig;
                {
                    AndnaServiceKey sk = new AndnaServiceKey("imaps", "tcp");
                    orig = sk.serialize();
                }
                uchar []dest = new uchar[orig.length];
                for (int i = 0; i < orig.length; i++) dest[i] = orig[i];
                sk1 = (AndnaServiceKey)ISerializable.deserialize(dest);
            }
            assert(! AndnaServiceKey.equal_func(sk0, sk1));
            print_out(@"sk1 $(sk1)\n");

            AndnaServiceKey sk0001 = new AndnaServiceKey("a", "b");
            AndnaServiceKey sk0002 = new AndnaServiceKey("", "");
            AndnaServiceKey sk0003 = new AndnaServiceKey("a", "b");
            AndnaServiceKey sk0004 = AndnaServiceKey.NULL_SERV_KEY;
            assert(AndnaServiceKey.equal_func(sk0001, sk0003));
            assert(!AndnaServiceKey.equal_func(sk0001, sk0002));
            assert(!AndnaServiceKey.equal_func(sk0001, sk0004));
            assert(!AndnaServiceKey.equal_func(sk0002, sk0004));
            assert(!AndnaServiceKey.equal_func(sk0003, sk0004));

            AndnaServiceKey sk0005 = (AndnaServiceKey)ISerializable.deserialize(sk0001.serialize());
            assert(AndnaServiceKey.equal_func(sk0001, sk0005));
        }

        public void test_ser_publickey ()
        {
            PublicKey pk0;
            {
                uchar[] orig;
                {
                    PublicKey garbage_pk =
                        new PublicKey(new SerializableBuffer((
                            "skjflvisejkrvolseuirnlvsejrnvlisuenvlsjenrivubdd" +
                            "qwsdkrjgsblieurvnlseiunrvlosienriosveirvdqvdfbdd" +
                            "kdjahriguhisdtrhsrtjruigkolgjhdsetgsjwjvn").data));
                    orig = garbage_pk.serialize();
                }
                uchar []dest = new uchar[orig.length];
                for (int i = 0; i < orig.length; i++) dest[i] = orig[i];
                pk0 = (PublicKey)ISerializable.deserialize(dest);
            }
            PublicKey pk1;
            {
                uchar[] orig;
                {
                    PublicKey garbage_pk =
                        new PublicKey(new SerializableBuffer((
                            "skjflvisejkrvolseuirnlvsejrnvlisuenvlsjenrivubdd" +
                            "qwsdkrjgsblieurvnlseiunrvlosienriosveirvdqvdfbdd" +
                            "kdjahriguhisdtrhsrtjruigkolgjhdsetgsjwjvn").data));
                    orig = garbage_pk.serialize();
                }
                uchar []dest = new uchar[orig.length];
                for (int i = 0; i < orig.length; i++) dest[i] = orig[i];
                pk1 = (PublicKey)ISerializable.deserialize(dest);
            }
            print_out(@"pk1 $(pk1)\n");
            assert(PublicKey.equal_func(pk0, pk1));
        }

        public void test_server_record_equal_func ()
        {
            PublicKey pk0 = new PublicKey(new SerializableBuffer((
                            "skjflvisejkrvolseuirnlvsejrnvlisuenvlsjenrivubdd" +
                            "qwsdkrjgsblieurvnlseiunrvlosienriosveirvdqvdfbdd" +
                            "kdjahriguhisdtrhsrtjruigkolgjhdsetgsjwjvn").data));
            PublicKey pk1 = new PublicKey(new SerializableBuffer((
                            "skjflvisejkrvolseuirnlvsejrnvlisuenvlsjenrivubdd" +
                            "qwsdkrjgsblieurvnlseiunrvlosienriosveirvdqvdfbdd" +
                            "kdjahriguhisdtrhsrtjruigkolgjhdsetgsjwjvn").data));

            AndnaServerRecord sar1 = new AndnaServerRecord(null, null, 555, 13, 1);
            AndnaServerRecord sar2 = new AndnaServerRecord("aa", pk0, 21, 1, 4);
            AndnaServerRecord sar3 = new AndnaServerRecord("aa", pk1, 21, 1, 4);
            AndnaServerRecord sar4 = new AndnaServerRecord(null, null, 555, 13, 1);
            assert(AndnaServerRecord.equal_func(sar1, sar4));
            assert(AndnaServerRecord.equal_func(sar2, sar3));
            assert(!AndnaServerRecord.equal_func(sar1, sar3));
            assert(!AndnaServerRecord.equal_func(sar2, sar4));

            AndnaServerRecord sar5 = (AndnaServerRecord)ISerializable.deserialize(sar1.serialize());
            assert(AndnaServerRecord.equal_func(sar1, sar5));
            AndnaServerRecord sar6 = (AndnaServerRecord)ISerializable.deserialize(sar2.serialize());
            assert(AndnaServerRecord.equal_func(sar3, sar6));

            AndnaServerRecord sar7 = new AndnaServerRecord(null, null, 22, 13, 1);
            assert(!AndnaServerRecord.equal_func(sar7, sar4));
        }

        public void test_ser_domain_record ()
        {
            {
                ISerializable i0;
                {
                    uchar[] orig;
                    {
                        /*
                        PublicKey alice_pub = new PublicKey(new SerializableBuffer((
                           "qwertyioplkjhgfdszxvbnmmedcjikmurgbvisindqweejfifvdvdfbdd" +
                           "kdjahriguhiluvhiukijebkuianlsjnvwopijwkmnvosidjjvn").data));
                        orig = alice_pub.serialize();
                        
                        AndnaServerRecord defaultfreedom = new AndnaServerRecord
                                (null, null, 1, 1, 1);
                        orig = defaultfreedom.serialize();
                        */
                        
                        PublicKey alice_pub = new PublicKey(new SerializableBuffer((
                           "qwertyioplkjhgfdszxvbnmmedcjikmurgbvisindqweejfifvdvdfbdd" +
                           "kdjahriguhiluvhiukijebkuianlsjnvwopijwkmnvosidjjvn").data));
                        
                        AndnaServerRecord rebelfreedom_1 = new AndnaServerRecord
                                (null, null, 555, 13, 1);
                        AndnaServerRecord rebelfreedom_2 = new AndnaServerRecord
                                ("aa", alice_pub, 21, 1, 4);
                        ArrayList<AndnaServerRecord> lst_alice_rebel =
                                new ArrayList<AndnaServerRecord>(AndnaServerRecord.equal_func);
                        lst_alice_rebel.add(rebelfreedom_1);
                        lst_alice_rebel.add(rebelfreedom_2);
                        
                        AndnaServerRecord webfreedom_1 = new AndnaServerRecord
                                ("aa", alice_pub, 222, 13, 1);
                        AndnaServerRecord webfreedom_2 = new AndnaServerRecord
                                (null, null, 333, 1, 4);
                        ArrayList<AndnaServerRecord> lst_alice_web =
                                new ArrayList<AndnaServerRecord>(AndnaServerRecord.equal_func);
                        lst_alice_web.add(webfreedom_1);
                        lst_alice_web.add(webfreedom_2);
                        
                        HashMap<AndnaServiceKey, ArrayList<AndnaServerRecord>> alice_services =
                                new HashMap<AndnaServiceKey, ArrayList<AndnaServerRecord>>
                                (AndnaServiceKey.hash_func, AndnaServiceKey.equal_func);
                        alice_services[new AndnaServiceKey("rebel", "tcp")] = lst_alice_rebel;
                        alice_services[new AndnaServiceKey("web", "tcp")] = lst_alice_web;
                        
                        AndnaDomainRecord arec = new AndnaDomainRecord(AndnaConst.MAX_TTL_ANDNA,
                            "freedom.riot",
                            alice_pub,
                            new NIP({1,2,3,4}),
                            alice_services);
                        
                        print_out(@"arec $(arec)\n");
                        orig = arec.serialize();
                    }
                    uchar []dest = new uchar[orig.length];
                    for (int i = 0; i < orig.length; i++) dest[i] = orig[i];
                    i0 = ISerializable.deserialize(dest);
                }
                AndnaDomainRecord another = (AndnaDomainRecord)i0;
                print_out(@"another $(another)\n");
            }
            AndnaServers servers;
            {
                uchar[] orig2;
                {
                    AndnaDomainRecord arec;
                    {
                        uchar[] orig;
                        {
                            PublicKey alice_pub = new PublicKey(new SerializableBuffer((
                                   "qwertyioplkjhgfdszxvbnmmedcjikmurgbvisindqweejfifvdvdfbdd" +
                                   "kdjahriguhiluvhiukijebkuianlsjnvwopijwkmnvosidjjvn").data));
                            NIP alice_nip = new NIP({1,0,3,2});
                            PublicKey bob_pub = new PublicKey(new SerializableBuffer((
                                   "jdfiwrjlieurvhkisejroaiuwnlcsinceujvnjkdfvbkiseubrvsjihbr" +
                                   "kjdsbklajwerblviuaenlvksjhgfsjhgfkbvkjdsrlujerlivu").data));
                            NIP bob_nip = new NIP({1,0,2,1});
                            assert(! PublicKey.equal_func(alice_pub, bob_pub));
                            // Alice owns domain "freedom.riot" and its current NIP is 2,3,0,1
                            // Bob owns domain "business.biz" and its current NIP is 1,2,0,1
                            // Bob is serving Alice with a hosting service.
                            // Alice wants that the requests for the website freedom.riot are
                            // served by a web server running on business.biz on port 123.
                            // Alice wants that the requests for the service "rebelgame" over
                            // procol "udp" (which is a online game) are served with a weight
                            // of 10 by her computer and with a weight of 20 (double the times)
                            // by Bob's one. The server of the game runs on port 555 on her
                            // computer and on port 1055 on Bob's computer.
                            // The rest of the services for domain "freedom.riot" are served by
                            // Alice's computer.
                            //
                            // 1 record for default (NULL_SERV_KEY)
                            // This record has dummy data for port_number
                            //
                            // 1 record:  _www._tcp.freedom.riot => business.biz:123
                            // This record goes to servicekey _www._tcp
                            //
                            // 2 records:  _rebelgame._udp.freedom.riot =>
                            //                      myself        w:10 555
                            //                      business.biz  w:20 1055
                            // These records go to servicekey _rebelgame._udp
                            //
                            AndnaServerRecord defaultfreedom = new AndnaServerRecord
                                    (null, null, 1, 1, 1);
                            ArrayList<AndnaServerRecord> lst_alice_default =
                                    new ArrayList<AndnaServerRecord>(AndnaServerRecord.equal_func);
                            lst_alice_default.add(defaultfreedom);
                            AndnaServerRecord webfreedom = new AndnaServerRecord
                                    ("business.biz", bob_pub, 123, 1, 1);
                            ArrayList<AndnaServerRecord> lst_alice_web =
                                    new ArrayList<AndnaServerRecord>(AndnaServerRecord.equal_func);
                            lst_alice_web.add(webfreedom);
                            AndnaServerRecord rebelfreedom_1 = new AndnaServerRecord
                                    (null, null, 555, 1, 10);
                            AndnaServerRecord rebelfreedom_2 = new AndnaServerRecord
                                    ("business.biz", bob_pub, 1055, 1, 20);
                            ArrayList<AndnaServerRecord> lst_alice_rebel =
                                    new ArrayList<AndnaServerRecord>(AndnaServerRecord.equal_func);
                            lst_alice_rebel.add(rebelfreedom_1);
                            lst_alice_rebel.add(rebelfreedom_2);
                            HashMap<AndnaServiceKey, ArrayList<AndnaServerRecord>> alice_services =
                                    new HashMap<AndnaServiceKey, ArrayList<AndnaServerRecord>>
                                    (AndnaServiceKey.hash_func, AndnaServiceKey.equal_func);
                            alice_services[AndnaServiceKey.NULL_SERV_KEY] = lst_alice_default;
                            alice_services[new AndnaServiceKey("www", "tcp")] = lst_alice_web;
                            alice_services[new AndnaServiceKey("rebelgame", "udp")] = lst_alice_rebel;
                            AndnaDomainRecord freedomriot = new AndnaDomainRecord
                                    (AndnaConst.MAX_TTL_ANDNA,
                                    "freedom.riot",
                                    alice_pub,
                                    alice_nip,
                                    alice_services);
                            print_out(@"freedomriot $(freedomriot)\n");
                            orig = freedomriot.serialize();
                        }
                        uchar []dest = new uchar[orig.length];
                        for (int i = 0; i < orig.length; i++) dest[i] = orig[i];
                        arec = (AndnaDomainRecord)ISerializable.deserialize(dest);
                    }
                    print_out(@"arec $(arec)\n");
                    assert(arec.hashed_domain == "freedom.riot");
                    assert(!arec.expires.is_expired());
                    AndnaServers in_servers = arec
                            .get_servers(new AndnaServiceKey("rebelgame", "udp"));
                    assert(!in_servers.expires.is_expired());
                    bool in_found1 = false;
                    bool in_found2 = false;
                    foreach (AndnaServer srv in in_servers.servers)
                    {
                        if (srv.alias_name == "business.biz")
                        {
                            in_found1 = true;
                            assert(srv.port_number == 1055);
                            assert(srv.registrar_nip == null);
                        }
                        if (PartialNIP.equal_func(srv.registrar_nip, new NIP({1,0,3,2})))
                        {
                            in_found2 = true;
                            assert(srv.port_number == 555);
                            assert(srv.alias_name == null);
                        }
                    }
                    assert(in_found1);
                    assert(in_found2);
                    
                    orig2 = in_servers.serialize();
                }
                uchar []dest2 = new uchar[orig2.length];
                for (int i = 0; i < orig2.length; i++) dest2[i] = orig2[i];
                servers = (AndnaServers)ISerializable.deserialize(dest2);
            }
            assert(!servers.expires.is_expired());
            bool found1 = false;
            bool found2 = false;
            foreach (AndnaServer srv in servers.servers)
            {
                if (srv.alias_name == "business.biz")
                {
                    found1 = true;
                    assert(srv.port_number == 1055);
                    assert(srv.registrar_nip == null);
                }
                if (PartialNIP.equal_func(srv.registrar_nip, new NIP({1,0,3,2})))
                {
                    found2 = true;
                    assert(srv.port_number == 555);
                    assert(srv.alias_name == null);
                }
            }
            assert(found1);
            assert(found2);
        }

        public void test_ser_get_cache ()
        {
            // test serialization of cached data from an hashnode
            AndnaGetCacheRecordsResponse cachequeue;
            {
                uchar[] orig;
                {
                    // this hashnode contains one hostname hashed "qwerty"
                    // there is the current owner and 2 queued requests.
                    AndnaDomainRecord record_owner;
                    PublicKey pubk_owner;
                    NIP nip_owner;
                    AndnaDomainRequest request_queued_a;
                    PublicKey pubk_a;
                    NIP nip_a;
                    SerializableBuffer signature_a;
                    RegisterHostnameArguments args_a;
                    AndnaDomainRequest request_queued_b;
                    PublicKey pubk_b;
                    NIP nip_b;
                    SerializableBuffer signature_b;
                    RegisterHostnameArguments args_b;
                    pubk_owner =
                        new PublicKey(
                            new SerializableBuffer(new uint8[] {1,2,3,4 /*fake pubk*/}));
                    nip_owner =
                        new NIP({23,12,45,6});
                    record_owner = 
                        new AndnaDomainRecord(
                            34589790 /*msec_ttl*/,
                            "qwerty" /*hashed_domain*/,
                            pubk_owner,
                            nip_owner);
                    pubk_a =
                        new PublicKey(
                            new SerializableBuffer(new uint8[] {31,42,23,54 /*fake pubk*/}));
                    nip_a =
                        new NIP({12,7,145,6});
                    signature_a =
                        new SerializableBuffer(new uint8[] {13,42,73,64 /*fake signature*/});
                    request_queued_a = 
                        new AndnaDomainRequest(
                            "qwerty" /*hashed_domain*/,
                            pubk_a,
                            nip_a,
                            AndnaDomainRequest.make_empty_services());
                    args_a =
                        new RegisterHostnameArguments(
                            request_queued_a, signature_a, new TimeCapsule(123));
                    pubk_b =
                        new PublicKey(
                            new SerializableBuffer(new uint8[] {6,2,4,54 /*fake pubk*/}));
                    nip_b =
                        new NIP({5,6,7,8});
                    signature_b =
                        new SerializableBuffer(new uint8[] {54,1,4,75 /*fake signature*/});
                    request_queued_b = 
                        new AndnaDomainRequest(
                            "qwerty" /*hashed_domain*/,
                            pubk_b,
                            nip_b,
                            AndnaDomainRequest.make_empty_services());
                    args_b =
                        new RegisterHostnameArguments(
                            request_queued_b, signature_b, new TimeCapsule(123));
                    var cache = new HashMap<string, AndnaDomainRecord>();
                    cache["qwerty"] = record_owner;
                    var request_queue = new HashMap<string, ArrayList<RegisterHostnameArguments>>();
                    request_queue["qwerty"] = new ArrayList<RegisterHostnameArguments>
                                    (RegisterHostnameArguments.equal_func_for_queue);
                    request_queue["qwerty"].add(args_a);
                    request_queue["qwerty"].add(args_b);
                    AndnaGetCacheRecordsResponse in_cachequeue =
                        new AndnaGetCacheRecordsResponse(cache, request_queue);
                    orig = in_cachequeue.serialize();
                }
                uchar []dest = new uchar[orig.length];
                for (int i = 0; i < orig.length; i++) dest[i] = orig[i];
                cachequeue = (AndnaGetCacheRecordsResponse)ISerializable.deserialize(dest);
            }
            assert(cachequeue.cache.keys.size == 1);
            assert(cachequeue.request_queue.keys.size == 1);
            assert(cachequeue.request_queue["qwerty"].size == 2);
            assert(! cachequeue.cache["qwerty"].expires.is_expired());
            assert(cachequeue.cache["qwerty"].nip.is_equal(new NIP({23,12,45,6})));
            assert(cachequeue.cache["qwerty"].services[AndnaServiceKey.NULL_SERV_KEY][0].hashed_alias == null);
            assert(cachequeue.request_queue["qwerty"][0].request.nip.is_equal(new NIP({12,7,145,6})));
            assert(cachequeue.request_queue["qwerty"][1].request.nip.is_equal(new NIP({5,6,7,8})));
        }

        public void test_ser_info_andna ()
        {
            InfoAndna info;
            {
                uchar[] orig;
                {
                    InfoAndna ret = new InfoAndna();
                    ret.will_participate = true;
                    ret.participating = false;
                    ret.hooked = false;
                    ret.register_ongoing = true;
                    ret.pubk = "A7C84D9F02AB";
                    InfoAndnaCache c = new InfoAndnaCache();
                    c.domain = "394icenorkv03orgIUGOb8ytfyi";
                    var pubk_fake =
                        new PublicKey(
                            new SerializableBuffer(new uint8[] {31,42,23,54 /*fake pubk*/}));
                    var nip_fake =
                        new NIP({12,7,145,6});
                    c.rec = 
                        new AndnaDomainRecord(
                            0,
                            "qwerty",
                            pubk_fake,
                            nip_fake);
                    ret.cache.add(c);
                    InfoAndnaRegistration r = new InfoAndnaRegistration();
                    r.domain = "PEGASUS";
                    r.registered = true;
                    r.ttl_before_request = new TimeCapsule(0);
                    ret.registrations.add(r);
                    orig = ret.serialize();
                }
                uchar []dest = new uchar[orig.length];
                for (int i = 0; i < orig.length; i++) dest[i] = orig[i];
                info = (InfoAndna)ISerializable.deserialize(dest);
            }
            print_out(@"pubk $(info.pubk)\n");
            assert(info.pubk == "A7C84D9F02AB");
            assert(info.will_participate);
        }

        public static int main(string[] args)
        {
            GLib.Test.init(ref args);
            Tasklet.init();
            GLib.Test.add_func ("/Andna/Timers", () => {
                // Andna module employs long TTLs.
                var x = new AndnaTester();
                x.set_up();
                x.test_timers_boundary();
                x.tear_down();
            });
            GLib.Test.add_func ("/Andna/SerializationServiceKey", () => {
                var x = new AndnaTester();
                x.set_up();
                x.test_ser_servkey();
                x.tear_down();
            });
            GLib.Test.add_func ("/Andna/SerializationPublicKey", () => {
                var x = new AndnaTester();
                x.set_up();
                x.test_ser_publickey();
                x.tear_down();
            });
            GLib.Test.add_func ("/Andna/DiscriminationServerRecord", () => {
                var x = new AndnaTester();
                x.set_up();
                x.test_server_record_equal_func();
                x.tear_down();
            });
            GLib.Test.add_func ("/Andna/SerializationDomainRecord", () => {
                var x = new AndnaTester();
                x.set_up();
                x.test_ser_domain_record();
                x.tear_down();
            });
            GLib.Test.add_func ("/Andna/SerializationGetCacheResponse", () => {
                var x = new AndnaTester();
                x.set_up();
                x.test_ser_get_cache();
                x.tear_down();
            });
            GLib.Test.add_func ("/Andna/SerializationInfoAndna", () => {
                var x = new AndnaTester();
                x.set_up();
                x.test_ser_info_andna();
                x.tear_down();
            });
            GLib.Test.run();
            Tasklet.kill();
            return 0;
        }
    }
}
