/*
 *  This file is part of Netsukuku.
 *  (c) Copyright 2011 Luca Dionisi aka lukisi <luca.dionisi@gmail.com>
 *
 *  Netsukuku is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  Netsukuku is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with Netsukuku.  If not, see <http://www.gnu.org/licenses/>.
 */

using Gee;
using zcd;
using Tasklets;

namespace Netsukuku
{
	public errordomain NeighbourError {
		DUPLICATE_INSERT,
		MISSING_REMOVE,
		MAX_EXCEED
	}

    /** Each instance of this class represents a neighbour as seen by a particular nic of this node.
      */
    public class Neighbour : Object
    {
        /*
        nip: neighbour's nip;
        nodeid: id of the node
              nip + nodeid = unique key.
        */
        public NIP nip {get; private set;}
        public int nodeid {get; private set;}
        /*
        levels, gsize, netid: attributes of the network
        is_primary, is_auxiliary: attributes of the neighbour's address
        */
        public int levels {get; private set;}
        public int gsize {get; private set;}
        public NetworkID netid {get; set;}
        public bool is_primary {get; set;}
        public bool is_auxiliary {get; private set;}
        /*
        for real neighbours:
            dev: the name of the nic
            macs: a sequence of string: the MAC of the nics of the neighbour that I can see through my nic "dev";
                there could be more than one.
            rem: the REM to the neighbour. Use one of the constructors with_deadrem, with_rtt, with_rem.
        */
        public string dev {get; set;}
        private ArrayList<string> _macs;
        public Gee.List<string> get_macs()
        {
            return _macs.read_only_view;
        }
        public REM rem {get; set;}
        /*
        for alternate addresses of this same host:
            rpcdispatcher: to use to rpc to this address of mine.
            Use the constructor with_local_dispatcher.
        */
        public RPCDispatcher? rpcdispatcher {get; private set; default=null;}

        private Neighbour.common(int levels, int gsize, NIP nip,
                         int nodeid, NetworkID netid, bool is_primary, bool is_auxiliary)
        {
            this.levels = levels;
            this.gsize = gsize;
            this.nip = nip;
            this.nodeid = nodeid;
            this.netid = netid;
            this.is_primary = is_primary;
            this.is_auxiliary = is_auxiliary;
        }

        public Neighbour.with_local_dispatcher(int levels, int gsize, NIP nip,
                         int nodeid, NetworkID netid, bool is_primary, bool is_auxiliary,
                         RPCDispatcher rpcdispatcher)
        {
            this.common(levels, gsize, nip, nodeid, netid, is_primary, is_auxiliary);
            this.rpcdispatcher = rpcdispatcher;
            this.rem = new NullREM();
        }

        public Neighbour.with_rtt(int levels, int gsize, NIP nip,
                         int nodeid, NetworkID netid, bool is_primary, bool is_auxiliary,
                         string dev, Gee.List<string> macs, int avg_rtt)
        {
            this.common(levels, gsize, nip, nodeid, netid, is_primary, is_auxiliary);
            this.dev = dev;
            _macs = new ArrayList<string>();
            foreach (string s in macs) _macs.add(s);
            this.rem = new RTT(avg_rtt);
        }

        public Neighbour.with_rem(int levels, int gsize, NIP nip,
                         int nodeid, NetworkID netid, bool is_primary, bool is_auxiliary,
                         string dev, Gee.List<string> macs, REM rem)
        {
            this.common(levels, gsize, nip, nodeid, netid, is_primary, is_auxiliary);
            this.dev = dev;
            _macs = new ArrayList<string>();
            foreach (string s in macs) _macs.add(s);
            this.rem = rem;
        }

        public Neighbour.with_deadrem(int levels, int gsize, NIP nip,
                         int nodeid, NetworkID netid, bool is_primary, bool is_auxiliary,
                         string dev, Gee.List<string> macs)
        {
            this.common(levels, gsize, nip, nodeid, netid, is_primary, is_auxiliary);
            this.dev = dev;
            _macs = new ArrayList<string>();
            foreach (string s in macs) _macs.add(s);
            this.rem = new DeadREM();
        }

        public bool is_local()
        {
            return rpcdispatcher != null;
        }

        public static bool equal_func(Neighbour? a, Neighbour? b)
        {
            if (a == b) return true;
            if (a == null || b == null) return false;
            if (a.nodeid != b.nodeid) return false;
            if (! PartialNIP.equal_func(a.nip, b.nip)) return false;
            return true;
        }

        public string to_string()
        {
            string ipstr = nip_to_str(levels, gsize, nip);
            string ret = @"<Neighbour($(ipstr), id $(nodeid) in $(netid)): ";
            if (is_local()) ret += "local>";
            else ret += @"$(rem) through $(dev)>";
            return ret;
        }
    }

    struct struct_helper_NeighbourManager_serialized_store_XXX_neighbour
    {
        public NeighbourManager self;
        public int function_code;
        // 1 : serialized_store_add_neighbour
        // 2 : serialized_store_delete_neighbour
        // 3 : serialized_store_changed_rem_neighbour
        // 4 : serialized_store_changed_attrib_neighbour
        public Neighbour val;
        public string key;
    }

    struct struct_helper_NeighbourManager_store
    {
        public NeighbourManager self;
        public int bouquet_numb;
        public HashMap<string, Neighbour> detected_neighbours;
    }

    /** By watching the previous statistics, it can deduces if a change occurred or
        not. If it is, one of the following events is fired:
               'NEIGHBOUR_NEW', 'NEIGHBOUR_DELETED', 'NEIGHBOUR_REM_CHGED'
        In this way, the other modules of ntk will be noticed.

        TODO: the current statistic is based on the RTT (Round Trip Time)
        of the packets. However, more refined way can be used and shall be
        implemented: first of all consider the number of bouquet packets lost, then
        see NTK_RFC 0002  http://lab.dyne.org/Ntk_bandwidth_measurement
      */
    public class NeighbourManager : Object
    {
        public Radar radar {get; protected set;}
        private int max_neighbours;
        private double rtt_variation_threshold;
        private HashMap<string, Neighbour> tnip_nodeid_table;
        private int number_of_scan_before_deleting;
        private HashMap<string, int> missing_neighbour_keys;
        private ArrayList<string> declared_dead;
        protected NeighbourManager.fake() {}  // to allow derivation of Glue
        public NeighbourManager(Radar radar, int max_neighbours=-1)
        {
            // max_neighbours: maximum number of neighbours we can have
            if (max_neighbours == -1) max_neighbours = Settings.MAX_NEIGHBOURS;
            this.max_neighbours = max_neighbours;
            // variation on neighbours' rtt greater than this will be notified
            // rtt_variation_threshold = 0.1; TODO changed to do less variations in netkit environment
            rtt_variation_threshold = 0.9;
            // This is a dict whose key is a pair (nip, nodeid), that is the unique
            //   identifier of a neighbour node. The values of this dict are instances of Neighbour.
            tnip_nodeid_table = new HashMap<string, Neighbour>();

            // To be certain, before deleting a neighbour, check a few times
            number_of_scan_before_deleting = 2;
            // This is a dict. The key is the neighbour id, the value is missing_scans.
            missing_neighbour_keys = new HashMap<string, int>();

            this.radar = radar;
            this.radar.scan_done.connect(store);
            this.radar.radar_stop.connect(empty);
            declared_dead = new ArrayList<string>();
        }

        private string tnip_nodeid_key(NIP key_nip, int key_nodeid)
        {
            string ret = "";
            int[] nip = key_nip.get_positions();
            foreach (int i in nip) ret += "%d_".printf(i);
            ret += "%d".printf(key_nodeid);
            return ret;
        }
        private void key_tnip_nodeid(string key, out NIP key_nip, out int key_nodeid)
        {
            string[] nums = key.split("_");
            int[] pos = new int[nums.length - 1];
            for (int i = 0; i < pos.length; i++) pos[i] = int.parse(nums[i]);
            key_nodeid = int.parse(nums[nums.length - 1]);
            key_nip = new NIP(pos);
        }

        public signal void neighbour_new(Neighbour neighbour);
        public signal void neighbour_deleted(Neighbour neighbour);
        public signal void neighbour_rem_chged(Neighbour neighbour, REM old_rem);
        public signal void neighbour_going_new(Neighbour neighbour);
        public signal void neighbour_going_deleted(Neighbour neighbour);
        public signal void neighbour_going_rem_chged(Neighbour neighbour);

        /** Returns the list of neighbours.
          */
        public virtual Gee.List<Neighbour> neighbour_list()
        {
            // ATTENTION: this method MUST NOT pass schedule while
            // gathering data from the structures.

            ArrayList<Neighbour> nlist = new ArrayList<Neighbour>(Neighbour.equal_func);
            foreach (string skey in tnip_nodeid_table.keys)
            {
                NIP key_nip;
                int key_nodeid;
                key_tnip_nodeid(skey, out key_nip, out key_nodeid);
                nlist.add(tnip_nodeid_table[skey]);
            }
            return nlist;
        }

        /** Inserts this neighbour in our data structures.
          */
        private void memorize(Neighbour val) throws NeighbourError
        {
            // key = (tuple(val.nip), val.nodeid)
            // key should not be already in tnip_nodeid_table.
            // If there is no more room, sends an exception.

            // ATTENTION: this method MUST NOT pass schedule until the end.

            string skey = tnip_nodeid_key(val.nip, val.nodeid);
            if (tnip_nodeid_table.has_key(skey))
                throw new NeighbourError.DUPLICATE_INSERT("Key was already present");
            // Check for Max Neighbours Exceeded
            if (tnip_nodeid_table.size > max_neighbours)
                throw new NeighbourError.MAX_EXCEED("Max Neighbours Exceeded");
            tnip_nodeid_table[skey] = val;
        }

        /** Removes this neighbour in our data structures.
          */
        private void unmemorize(string skey) throws NeighbourError
        {
            // key: pair tuple(nip), nodeid
            // key should be in tnip_nodeid_table.

            // ATTENTION: this method MUST NOT pass schedule until the end.

            if (! tnip_nodeid_table.has_key(skey))
                throw new NeighbourError.MISSING_REMOVE("Key was not present");
            tnip_nodeid_table.unset(skey);
        }

        /** return a Neighbour object from its nip and nodeid
          */
        public Neighbour? key_to_neighbour(NIP nip, int nodeid)
        {
            string skey = tnip_nodeid_key(nip, nodeid);
            if (! tnip_nodeid_table.has_key(skey)) return null;
            else return tnip_nodeid_table[skey];
        }

        private static int table_compare_func(Gee.Map.Entry<string, Neighbour> a, Gee.Map.Entry<string, Neighbour> b)
        {
            // reverse of REM ordering.
            return b.value.rem.compare_to(a.value.rem);
        }

        private static bool not_impl_equal_func (Object? a, Object? b)
        {
            error("NeighbourManager._truncate: equal_func not implemented");
        }
        public HashMap<string, Neighbour> _truncate(HashMap<string, Neighbour> old_tnip_nodeid_table)
        {
            // old_tnip_nodeid_table is an hashmap nip+nodeid => Neighbour.
            // Returns the best (with the lowest rtt) max_neighbours nodes only.

            HashMap<string, Neighbour> tnip_nodeid_table_trunc = new HashMap<string, Neighbour>();
            int counter = 0;

            // Sort elements (entries) of the dict.
            ArrayList<Gee.Map.Entry<string, Neighbour>> elements = new ArrayList<Gee.Map.Entry<string, Neighbour>>(not_impl_equal_func);
            foreach (Gee.Map.Entry<string, Neighbour> el in old_tnip_nodeid_table.entries) elements.add(el);
            elements.sort(table_compare_func);

            // we're cycling through tnip_nodeid_table, ordered by descending rem
            foreach (Gee.Map.Entry<string, Neighbour> el in elements)
            {
                // if we haven't still reached max_neighbours entries in the new tnip_nodeid_table_trunc
                if (counter < max_neighbours)
                {
                    // add the current row into tnip_nodeid_table_trunc
                    tnip_nodeid_table_trunc[el.key] = el.value;
                    counter++;
                }
                else break;

            }
            // return the new tnip_nodeid_table
            return tnip_nodeid_table_trunc;
        }

        private void serialized_store_add_neighbour(string key, Neighbour val) throws Error
        {
            // Check that it did NOT exist; otherwise exit.
            if (tnip_nodeid_table.has_key(key)) return;

            // First, we emit a signal to notify that we are
            //  gonna make a change. The handler shouldn't be a microfunc.
            going_add(val);

            // We update the data structures.
            memorize(val);

            // Then we emit a signal to notify the change.
            add_neighbour(key);
        }

        private void serialized_store_delete_neighbour(string key) throws Error
        {
            // Check that it did exist; otherwise exit.
            if (! tnip_nodeid_table.has_key(key)) return;

            // First, we emit a signal to notify that we are
            //  gonna make a change. The handler shouldn't be a microfunc.
            Neighbour old_val = tnip_nodeid_table[key];
            going_delete(old_val);

            // We update the data structures.
            unmemorize(key);

            // Then we emit a signal to notify the change.
            delete_neighbour(key, old_val);
        }

        private void serialized_store_changed_rem_neighbour(string key, Neighbour val) throws Error
        {
            // Check that it did exist; otherwise exit.
            if (! tnip_nodeid_table.has_key(key)) return;

            // First, we emit a signal to notify that we are
            //  gonna make a change. The handler shouldn't be a microfunc.
            Neighbour old_val = tnip_nodeid_table[key];
            going_rem_change(old_val);

            // We update the data structures.
            REM old_rem = old_val.rem;
            old_val.dev = val.dev;
            old_val.rem = val.rem;

            // Then we emit a signal to notify the change.
            rem_change_neighbour(key, old_rem);
        }

        private void serialized_store_changed_attrib_neighbour(string key, Neighbour val) throws Error
        {
            // Check that it did exist; otherwise exit.
            if (! tnip_nodeid_table.has_key(key)) return;

            // First, we emit a signal to notify that we are
            //  gonna make a change. The handler shouldn't be a microfunc.
            Neighbour old_val = tnip_nodeid_table[key];
            going_rem_change(old_val);

            // We update the data structures.
            REM old_rem = old_val.rem;
            old_val.netid = val.netid;
            old_val.is_primary = val.is_primary;

            // Then we emit a signal to notify the change.
            rem_change_neighbour(key, old_rem);
        }

        /* Decoration of several microfunc with one unique dispatcher */
        private static void * helper_serialized_store_XXX_neighbour(void *v) throws Error
        {
            Tasklet.declare_self("NeighbourManager.serialized_store dispatcher");
            struct_channel *ch_cont_p = (struct_channel *)v;
            // The caller function has to add a reference to the ref-counted instances
            Channel ch = ch_cont_p->self;
            // schedule back to the spawner; this will probably invalidate *v and *ch_cont_p.
            Tasklet.schedule_back();
            // The actual dispatcher
            while (true)
            {
                string? doing = null;
                struct_helper_NeighbourManager_serialized_store_XXX_neighbour tuple_p;
                try
                {
                    {
                        Value vv = ch.recv();
                        tuple_p = *((struct_helper_NeighbourManager_serialized_store_XXX_neighbour *)(vv.get_boxed()));
                    }
                    // The helper function should not need to copy values
                    if (tuple_p.function_code == 1)
                    {
                        doing = @"add_neighbour($(tuple_p.key))";
                        Tasklet.declare_self(doing);
                        tuple_p.self.serialized_store_add_neighbour(
                                tuple_p.key, tuple_p.val);
                    }
                    if (tuple_p.function_code == 2)
                    {
                        doing = @"delete_neighbour($(tuple_p.key))";
                        Tasklet.declare_self(doing);
                        tuple_p.self.serialized_store_delete_neighbour(
                                tuple_p.key);
                    }
                    if (tuple_p.function_code == 3)
                    {
                        doing = @"changed_rem_neighbour($(tuple_p.key))";
                        Tasklet.declare_self(doing);
                        tuple_p.self.serialized_store_changed_rem_neighbour(
                                tuple_p.key, tuple_p.val);
                    }
                    if (tuple_p.function_code == 4)
                    {
                        doing = @"changed_attrib_neighbour($(tuple_p.key))";
                        Tasklet.declare_self(doing);
                        tuple_p.self.serialized_store_changed_attrib_neighbour(
                                tuple_p.key, tuple_p.val);
                    }
                }
                catch (Error e)
                {
                    string part1 = "serialized_store_XXX_neighbour";
                    if (tuple_p.function_code == 1) part1 = "serialized_store_add_neighbour";
                    if (tuple_p.function_code == 2) part1 = "serialized_store_delete_neighbour";
                    if (tuple_p.function_code == 3) part1 = "serialized_store_changed_rem_neighbour";
                    if (tuple_p.function_code == 4) part1 = "serialized_store_changed_attrib_neighbour";
                    log_warn(@"NeighbourManager: $part1 reported an error: $(e.message)");
                }
                if (doing != null) Tasklet.declare_finished(doing);
            }
        }

        private void store_add_neighbour(string key, Neighbour val)
        {
            // Register (once) the spawnable function that is our dispatcher
            // and obtain a channel to drive it.
            Channel ch = TaskletDispatcher.get_channel_for_helper((Spawnable)helper_serialized_store_XXX_neighbour);
            struct_helper_NeighbourManager_serialized_store_XXX_neighbour arg = 
                        struct_helper_NeighbourManager_serialized_store_XXX_neighbour();
            arg.self = this;
            arg.function_code = 1;
            arg.key = key;
            arg.val = val;
            // send the struct
            ch.send_async(arg);
        }

        private void store_delete_neighbour(string key)
        {
            // Register (once) the spawnable function that is our dispatcher
            // and obtain a channel to drive it.
            Channel ch = TaskletDispatcher.get_channel_for_helper((Spawnable)helper_serialized_store_XXX_neighbour);
            struct_helper_NeighbourManager_serialized_store_XXX_neighbour arg = 
                        struct_helper_NeighbourManager_serialized_store_XXX_neighbour();
            arg.self = this;
            arg.function_code = 2;
            arg.key = key;
            // send the struct
            ch.send_async(arg);
        }

        private void store_changed_rem_neighbour(string key, Neighbour val)
        {
            // Register (once) the spawnable function that is our dispatcher
            // and obtain a channel to drive it.
            Channel ch = TaskletDispatcher.get_channel_for_helper((Spawnable)helper_serialized_store_XXX_neighbour);
            struct_helper_NeighbourManager_serialized_store_XXX_neighbour arg = 
                        struct_helper_NeighbourManager_serialized_store_XXX_neighbour();
            arg.self = this;
            arg.function_code = 3;
            arg.key = key;
            arg.val = val;
            // send the struct
            ch.send_async(arg);
        }

        private void store_changed_attrib_neighbour(string key, Neighbour val)
        {
            // Register (once) the spawnable function that is our dispatcher
            // and obtain a channel to drive it.
            Channel ch = TaskletDispatcher.get_channel_for_helper((Spawnable)helper_serialized_store_XXX_neighbour);
            struct_helper_NeighbourManager_serialized_store_XXX_neighbour arg = 
                        struct_helper_NeighbourManager_serialized_store_XXX_neighbour();
            arg.self = this;
            arg.function_code = 4;
            arg.key = key;
            arg.val = val;
            // send the struct
            ch.send_async(arg);
        }

        public virtual void declare_dead(NIP nip, int nodeid)
        {
            string key = tnip_nodeid_key(nip, nodeid);
            declared_dead.add(key);
        }

        /** Detects changes in our neighbourhood. Updates internal data structures
          * and notify about the changes.
          */
        private void impl_store(int bouquet_numb, HashMap<string, Neighbour> detected_neighbours) throws Error
        {
            Tasklet.declare_self("NeighbourManager.store");
            ArrayList<string> _declared_dead;
            _declared_dead = declared_dead;
            declared_dead = new ArrayList<string>();
            HashMap<string, Neighbour> new_tnip_nodeid_table = detected_neighbours;
            // remove tracks of nodes that are declared dead
            ArrayList<string> new_tnip_nodeid_table_keys = new ArrayList<string>();
            new_tnip_nodeid_table_keys.add_all(new_tnip_nodeid_table.keys);
            foreach (string k in new_tnip_nodeid_table_keys)
            {
                if (_declared_dead.contains(k)) new_tnip_nodeid_table.unset(k);
            }

            // The class Radar, on each scan, passes to this method a 
            // dict {key => Neighbour}.
            // Neighbour is a new complete instance. For new neighbours, it will be
            // added to NeighbourManager's internal data structures. For old neighbours
            // it will be discarded after updating values in the instance already
            // present in NeighbourManager's internal data structures.

            // Using _truncate, we remove the worst ones, if they exceed max_neighbours.
            new_tnip_nodeid_table = _truncate(new_tnip_nodeid_table);

            // Then, compare the dict against the data structures and summarize
            // the changes (e.g. new Neighbour, new Neighbour, changed Neighbour, ...)
            ArrayList<string> to_be_deleted = new ArrayList<string>();
            ArrayList<string> to_be_almost_deleted = new ArrayList<string>();
            ArrayList<string> to_be_added_key = new ArrayList<string>();
            ArrayList<Neighbour> to_be_added_value = new ArrayList<Neighbour>(Neighbour.equal_func);
            ArrayList<string> to_be_rem_changed_key = new ArrayList<string>();
            ArrayList<Neighbour> to_be_rem_changed_value = new ArrayList<Neighbour>(Neighbour.equal_func);
            ArrayList<string> to_be_attrib_changed_key = new ArrayList<string>();
            ArrayList<Neighbour> to_be_attrib_changed_value = new ArrayList<Neighbour>(Neighbour.equal_func);

            // remove from missing_neighbour_keys the detected neighbours
            //  and those declared dead
            foreach (string key in new_tnip_nodeid_table.keys)
            {
                if (missing_neighbour_keys.has_key(key))
                {
                    missing_neighbour_keys.unset(key);
                    to_be_rem_changed_key.add(key);
                    to_be_rem_changed_value.add(new_tnip_nodeid_table[key]);
                }
            }
            foreach (string key in _declared_dead)
            {
                if (missing_neighbour_keys.has_key(key))
                {
                    missing_neighbour_keys.unset(key);
                }
            }

            // We'll remove nodes that were 
            // present and now are missing, but only after <n> radar scans.
            // In the mean time, they are almost-dead.
            // Note: if we have reached the max neighbour, we must delete all missing
            // neighbours, never minding for how many scans they were missing.
            foreach (string key in tnip_nodeid_table.keys)
            {
                if (!new_tnip_nodeid_table.has_key(key) && !_declared_dead.contains(key))
                {
                    // It is a missing neighbour. For how many scan is it missing?
                    int times;
                    if (missing_neighbour_keys.has_key(key))
                    {
                        times = missing_neighbour_keys[key] + 1;
                    }
                    else
                    {
                        times = 1;
                        to_be_almost_deleted.add(key);
                    }
                    missing_neighbour_keys[key] = times;
                    if (max_neighbours <= new_tnip_nodeid_table.size || times > number_of_scan_before_deleting)
                    {
                        // now, we assume it is really dead.
                        missing_neighbour_keys.unset(key);
                        to_be_deleted.add(key);
                    }
                }
            }

            // now, we cycle through the new tnip_nodeid_table
            // looking for nodes that aren't in the old one.
            foreach (string key in new_tnip_nodeid_table.keys)
            {
                Neighbour val = new_tnip_nodeid_table[key];
                if (!tnip_nodeid_table.has_key(key) && ! _declared_dead.contains(key))
                {
                    // It is a new neighbour.
                    to_be_added_key.add(key);
                    to_be_added_value.add(val);
                }
            }

            // now we cycle through the new tnip_nodeid_table
            // looking for nodes whose rtt has sensibly changed
            foreach (string key in new_tnip_nodeid_table.keys)
            {
                Neighbour val_new = new_tnip_nodeid_table[key];
                // if the node is not new
                if (tnip_nodeid_table.has_key(key) && !_declared_dead.contains(key))
                {
                    // if we did not already add it because previously it was almost_dead
                    if (!to_be_rem_changed_key.contains(key))
                    {
                        Neighbour val = tnip_nodeid_table[key];
                        // it may have changed its network_id...
                        if (!val.netid.is_strictly_equal(val_new.netid))
                        {
                            to_be_attrib_changed_key.add(key);
                            to_be_attrib_changed_value.add(val_new);
                        }
                        else
                        {
                            // it may have changed its is_primary...
                            if (val.is_primary != val_new.is_primary)
                            {
                                to_be_attrib_changed_key.add(key);
                                to_be_attrib_changed_value.add(val_new);
                            }
                            else
                            {
                                // check if its rtt has changed more than rtt_variation
                                if (val_new.rem.get_type() == typeof(RTT) && val.rem.get_type() == typeof(RTT))
                                {
                                    int new_rtt = ((RTT)val_new.rem).delay;
                                    int old_rtt = ((RTT)val.rem).delay;
                                    // using the following algorithm, we accomplish this:
                                    //  e.g. rtt_variation_threshold = .5, we want to be warned 
                                    //       when rtt become more than double of previous 
                                    //       or less than half of previous.
                                    //  from 200 to 400+ we get warned, 
                                    //      because (400 - 200) / 400 = 0.5
                                    //  from 400 to 200- we get warned, 
                                    //      because (400 - 200) / 400 = 0.5
                                    double rtt_variation = 0;
                                    if (new_rtt > old_rtt) rtt_variation = (new_rtt - old_rtt) / (double)new_rtt;
                                    else rtt_variation = (old_rtt - new_rtt) / (double)old_rtt;
                                    if (rtt_variation > rtt_variation_threshold)
                                    {
                                        to_be_rem_changed_key.add(key);
                                        to_be_rem_changed_value.add(val_new);
                                    }
                                }
                            }
                        }
                    }
                }
            }

            // For each single change, call a "serialized" method.
            //
            // The called method will:
            //  .if it is a new, check that it does not exist; otherwise exit.
            //  .if it is a changed_rem, check that it does exist; otherwise exit.
            //  .if it is a dead, check that it does exist; otherwise exit.
            //  .notify the going-to change.
            //  .change data in internal structure.
            //  .notify the change.
            ListIterator<string> keys;
            bool hascur;
            ListIterator<Neighbour> values;

            foreach (string key in _declared_dead)
                store_delete_neighbour(key);

            foreach (string key in to_be_deleted)
                store_delete_neighbour(key);

            keys = to_be_added_key.list_iterator();
            hascur = keys.next();
            values = to_be_added_value.list_iterator();
            values.next();
            while (hascur)
            {
                string key = keys.get();
                Neighbour val = values.get();
                store_add_neighbour(key, val);
                hascur = keys.next();
                values.next();
            }

            keys = to_be_rem_changed_key.list_iterator();
            hascur = keys.next();
            values = to_be_rem_changed_value.list_iterator();
            values.next();
            while (hascur)
            {
                string key = keys.get();
                Neighbour val = values.get();
                store_changed_rem_neighbour(key, val);
                hascur = keys.next();
                values.next();
            }

            keys = to_be_attrib_changed_key.list_iterator();
            hascur = keys.next();
            values = to_be_attrib_changed_value.list_iterator();
            values.next();
            while (hascur)
            {
                string key = keys.get();
                Neighbour val = values.get();
                store_changed_attrib_neighbour(key, val);
                hascur = keys.next();
                values.next();
            }

            foreach (string key in to_be_almost_deleted)
            {
                Neighbour old_val = tnip_nodeid_table[key];
                Neighbour val = new Neighbour.with_rem(old_val.levels, old_val.gsize, old_val.nip,
                         old_val.nodeid, old_val.netid, old_val.is_primary, old_val.is_auxiliary,
                         old_val.dev, old_val.get_macs(), new AlmostDeadREM());
                store_changed_rem_neighbour(key, val);
            }
        }

        private static void * helper_store(void *v) throws Error
        {
            struct_helper_NeighbourManager_store *tuple_p = (struct_helper_NeighbourManager_store *)v;
            // The caller function has to add a reference to the ref-counted instances
            NeighbourManager self_save = tuple_p->self;
            HashMap<string, Neighbour> detected_neighbours_save = tuple_p->detected_neighbours;
            // The caller function has to copy the value of byvalue parameters
            int bouquet_numb_save = tuple_p->bouquet_numb;
            // schedule back to the spawner; this will probably invalidate *v and *tuple_p.
            Tasklet.schedule_back();
            // The actual call
            self_save.impl_store(bouquet_numb_save, detected_neighbours_save);
            // void method, return null
            return null;
        }

        public void store(int bouquet_numb, HashMap<string, Neighbour> detected_neighbours)
        {
            struct_helper_NeighbourManager_store arg = struct_helper_NeighbourManager_store();
            arg.self = this;
            arg.bouquet_numb = bouquet_numb;
            arg.detected_neighbours = detected_neighbours;
            Tasklet.spawn((Spawnable)helper_store, &arg);
        }

        /** No more radar. Deletes all neighbours. Updates internal data structures
          * and notify about the changes.
          */
        public void empty()
        {
            // Note: it is not a microfunc, because after the Radar has stopped
            //  the signals emitted by this NeighbourManager will not be listened to.
            ArrayList<string> keys = new ArrayList<string>();
            keys.add_all(tnip_nodeid_table.keys);
            foreach (string key in keys)
                store_delete_neighbour(key);
        }

        /** Emits signal BEFORE a new neighbour.
          */
        public void going_add(Neighbour neighbour)
        {
            string ipstr = nip_to_str(neighbour.levels, neighbour.gsize, neighbour.nip);
            // emit a signal notifying we are going to add a node
            neighbour_going_new(neighbour);
        }

        /** Emits signal BEFORE a dead neighbour.
          */
        public void going_delete(Neighbour neighbour)
        {
            string ipstr = nip_to_str(neighbour.levels, neighbour.gsize, neighbour.nip);
            // emit a signal notifying we are going to delete a node
            neighbour_going_deleted(neighbour);
        }

        /** Emits signal BEFORE a changed rem neighbour.
          */
        public void going_rem_change(Neighbour neighbour)
        {
            string ipstr = nip_to_str(neighbour.levels, neighbour.gsize, neighbour.nip);
            // emit a signal notifying we are going to chage rem of a node
            neighbour_going_rem_chged(neighbour);
        }

        /** Emits signal for a new neighbour.
          */
        public void add_neighbour(string key)
        {
            Neighbour neighbour = tnip_nodeid_table[key];
            string ipstr = nip_to_str(neighbour.levels, neighbour.gsize, neighbour.nip);
            log_debug(@"NeighbourManager: emit NEIGHBOUR_NEW for ip $(ipstr), netid $(neighbour.netid) dev $(neighbour.dev)");
            log_verbose(@"NeighbourManager: emit NEIGHBOUR_NEW for ip $(ipstr), netid $(neighbour.netid) dev $(neighbour.dev)");
            // emit a signal notifying we added a node
            neighbour_new(neighbour);
        }

        /** Emits signal for a dead neighbour.
          */
        public void delete_neighbour(string key, Neighbour old_val)
        {
            string ipstr = nip_to_str(old_val.levels, old_val.gsize, old_val.nip);
            log_debug(@"NeighbourManager: emit NEIGHBOUR_DELETED for ip $(ipstr), netid $(old_val.netid) dev $(old_val.dev)");
            log_verbose(@"NeighbourManager: emit NEIGHBOUR_DELETED for ip $(ipstr), netid $(old_val.netid) dev $(old_val.dev)");
            // emit a signal notifying we deleted a node
            neighbour_deleted(old_val);
        }

        /** Emits signal for a changed rem neighbour.
          */
        public void rem_change_neighbour(string key, REM old_rem)
        {
            Neighbour neighbour = tnip_nodeid_table[key];
            string ipstr = nip_to_str(neighbour.levels, neighbour.gsize, neighbour.nip);
            // emit a signal notifying the node's rtt changed
            neighbour_rem_chged(neighbour, old_rem);
        }

    }
}

