/*
 *  This file is part of Netsukuku.
 *  (c) Copyright 2011 Luca Dionisi aka lukisi <luca.dionisi@gmail.com>
 *
 *  Netsukuku is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  Netsukuku is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with Netsukuku.  If not, see <http://www.gnu.org/licenses/>.
 */

using Gee;
using zcd;

namespace Netsukuku
{
    /** A solution to reach a destination.
      */
    public class RouteSolution : Object, Comparable<RouteSolution>
    {
        public string gateway {get; private set;}
        public string dev {get; private set;}
        public int weight {get; private set;}
        public RouteSolution(string gateway, string dev, int weight)
        {
            this.gateway = gateway;
            this.dev = dev;
            this.weight = weight;
        }

        public static bool test_equality(RouteSolution? a, RouteSolution? b)
        {
            if (a == null && b == null) return true;
            if (a == null || b == null) return false;
            return a.equals(b);
        }

        public bool equals(RouteSolution b)
        {
            return this.gateway == b.gateway && this.dev == b.dev && this.weight == b.weight;
        }

        public int compare_to(RouteSolution b)
        {
            // The logical ordering is based on weight.
            if (weight != b.weight) return weight - b.weight;
            // We have to impose an ordering also when weight is identical
            //  in order to avoid that the TreeSet implementation discards
            //  a RouteSolution because it thinks it is already present.
            if (dev > b.dev) return 1;
            if (dev < b.dev) return -1;
            if (gateway > b.gateway) return 1;
            if (gateway < b.gateway) return -1;
            return 0;
        }
    }

    public class BaseRouteSolutions : Object
    {
        internal BaseRouteSolutions() {}
    }

    /** List of possible solutions to reach a destination
      */
    public class RouteSolutions : BaseRouteSolutions
    {
        private TreeSet<RouteSolution> paths;
        public string? pref_src {get; private set;}
        public RouteSolutions(Gee.List<string> gateways, Gee.List<string> devs, Gee.List<int> weights, string? pref_src)
        {
            paths = new TreeSet<RouteSolution>();
            assert(gateways.size == devs.size);
            assert(weights.size == devs.size);
            ListIterator<string> gateways_it = gateways.list_iterator();
            bool hascur = gateways_it.next();
            ListIterator<string> devs_it = devs.list_iterator();
            devs_it.next();
            ListIterator<int> weights_it = weights.list_iterator();
            weights_it.next();
            while (hascur)
            {
                RouteSolution rs = new RouteSolution(
                                    gateways_it.get(),
                                    devs_it.get(),
                                    weights_it.get());
                paths.add(rs);
                hascur = gateways_it.next();
                devs_it.next();
                weights_it.next();
            }
            this.pref_src = pref_src;
        }

        public Collection<RouteSolution> values {
            owned get {
                // TODO I would have preferred returning paths.read_only_view;
                //  but there is a problem in TreeSet implementation and this would
                //  cause GLib to emit a Critical that says:
                //  **: Read-only property 'read-only-view' on class 'GeeReadOnlyBidirSortedSet' has type 'GeeSortedSet' which is not equal to or more restrictive than the type 'GeeBidirSortedSet' of the property on the interface 'GeeBidirSortedSet'
                // So for now returning paths should have no issues.
                return paths;
            }
        }

        public RouteSolution best()
        {
            return paths.last();
        }

        public bool contains_gw(string gateway)
        {
            foreach (RouteSolution rs in values)
            {
                if (rs.gateway == gateway) return true;
            }
            return false;
        }

        public bool equals(RouteSolutions b)
        {
            bool ret = pref_src == b.pref_src && b.values.contains_all(values) && values.contains_all(b.values);
            return ret;
        }
    }

    /** This class represents a Tunnel. It is used in a ordered list in the abstract base class RouteSetter.
      */
    public class TunnelItem : Object
    {
        public string ipstr {get; private set;}
        public REM rem {get; private set;}
        public TunnelItem(string ipstr, REM rem)
        {
            this.rem = rem;
            this.ipstr = ipstr;
        }
        public string to_string()
        {
            return @"<Tunnel $ipstr ($rem)>";
        }
        public static bool equal_func(TunnelItem a, TunnelItem b)
        {
            return (a.rem.compare_to(b.rem) == 0) && (a.ipstr == b.ipstr);
        }
    }

    public abstract class RouteSetter : Object
    {
        private static CreateRouteSetterDelegate? registered_class_linux = null;
        public static void register_class(string k, CreateRouteSetterDelegate create_new_route)
        {
            if (k == "linux" && registered_class_linux == null) registered_class_linux = create_new_route;
        }

        private static RouteSetter create_instance()
        {
            string impl = Settings.NETWORK_IMPLEMENTATION;
            if (impl == "linux" && registered_class_linux != null) return registered_class_linux();
            error(@"No valid real implementation of class Route for $impl. Is your system supported?");
        }

        private static RouteSetter? _instance = null;
        public static RouteSetter get_instance()
        {
            if (_instance == null) _instance = create_instance();
            return _instance;
        }

        private BaseRouteSolutions UNREACHABLE;
        private BaseRouteSolutions DROP;

        /** routes_table[(null, v)] is present (and has value y)
          *   iff a RULE destination = v → gateway = y exists in table
          * routes_table[(null, v)] is NOT present
          *   iff no routes exist for destination = v → host/network is UNREACHABLE
          * routes_table[(w, v)] is present (and has value y)
          *   iff a RULE source w, destination = v → gateway = y exists in table
          * routes_table[(w, v)] is present and has value UNREACHABLE
          *   iff a RULE source w, destination = v → host/network UNREACHABLE exists in table
          * routes_table[(w, v)] is present and has value DROP
          *   iff a RULE source w, destination = v → host/network DROP (blackhole) exists in table
          * v is a ip/cidr string, eg '192.168.0.0/16';
          * w is a MAC string, eg '6a:b8:1e:cf:1d:4f';
          * the pair (w,v) is a Variant "(mss)"
          * y is a BaseRouteSolutions instance.
          */
        private HashMap<Variant, BaseRouteSolutions> routes_table;

        /** neighbours_table[ip] is present (and has value dev)
          *   iff a RULE neighbour = ip exists in table
          * ip and dev are strings, eg '192.168.1.1', 'eth1'
          */
        private HashMap<string, string> neighbours_table;

        /** IGS current status in kernel rules:
          * mode in ['NONE', 'SHARE', 'USE', 'BOTH']
          * announce_myself is a boolean
          * list_tunnels is a ordered list of TunnelItem
          */
        private string mode;
        private bool announce_myself;
        private ArrayList<TunnelItem> list_tunnels;

        protected RouteSetter()
        {
            UNREACHABLE = new BaseRouteSolutions();
            DROP = new BaseRouteSolutions();
            EqualDataFunc<Variant> variant_equal_func = (a, b) => a.equal(b);
            HashDataFunc<Variant> variant_hash_func = (a) => 0; // forces to use equal_func
            routes_table = new HashMap<Variant, BaseRouteSolutions>(variant_hash_func, variant_equal_func);
            neighbours_table = new HashMap<string, string>();
            mode = "NONE";
            announce_myself = false;
            EqualDataFunc<TunnelItem> tunnel_equal_func = (a, b) => TunnelItem.equal_func(a,b);
            list_tunnels = new ArrayList<TunnelItem>(tunnel_equal_func);
        }

        /** Returns a list of tuple containing str(IP), str(bits)
          */
        public Collection<Variant> get_known_destinations()
        {
            EqualDataFunc<Variant> variant_equal_func = (a, b) => a.equal(b);
            ArrayList<Variant> ret = new ArrayList<Variant>(variant_equal_func);
            foreach (Variant key in routes_table.keys)
            {
                string? w;
                string v;
                key.get("(mss)", out w, out v);
                if (w == null)
                {
                    string[] ipcidr = v.split("/");
                    Variant ipcidr_v = new Variant("(ss)", ipcidr[0], ipcidr[1]);
                    ret.add(ipcidr_v);
                }
            }
            return ret;
        }

        /** Returns a copy of the dict {ipstr: dev} of neighbours
          */
        public Gee.Map<string, string> get_known_neighbours()
        {
            HashMap<string, string> ret = new HashMap<string, string>();
            foreach (string k in neighbours_table.keys) ret[k] = neighbours_table[k];
            return ret;
        }

        /** We have no routes
          */
        public void reset_routes(string ip_whole_network, string cidr_whole_network) throws Error
        {
            if (_reset_routes(ip_whole_network, cidr_whole_network))
            {
                EqualDataFunc<Variant> variant_equal_func = (a, b) => a.equal(b);
                HashDataFunc<Variant> variant_hash_func = (a) => 0; // forces to use equal_func
                routes_table = new HashMap<Variant, BaseRouteSolutions>(variant_hash_func, variant_equal_func);
            }
            else
            {
                // use single remove
                // copy the current keys collection of routes_table
                EqualDataFunc<Variant> variant_equal_func = (a, b) => a.equal(b);
                ArrayList<Variant> initial_keys = new ArrayList<Variant>(variant_equal_func);
                initial_keys.add_all(routes_table.keys);
                foreach (Variant key in initial_keys)
                {
                    string? w;
                    string v;
                    key.get("(mss)", out w, out v);
                    BaseRouteSolutions y = routes_table[key];
                    string[] ipcidr = v.split("/");
                    string ip = ipcidr[0];
                    string cidr = ipcidr[1];
                    if (w == null)
                    {
                        remove_outgoing_route(ip, cidr);
                    }
                    else
                    {
                        if (y == UNREACHABLE)
                        {
                            remove_forwarding_route_was_unreachable(ip, cidr, w);
                        }
                        else if (y == DROP)
                        {
                            remove_forwarding_route_was_drop(ip, cidr, w);
                        }
                        else
                        {
                            remove_forwarding_route(ip, cidr, w);
                        }
                    }
                }
            }
        }

        /** Maintains this default routes for this destination.
          */
        public void outgoing_routes(string ip, string cidr, RouteSolutions? route_solutions) throws Error
        {
            /*
             * ip and cidr are strings.
             * Eg: '192.168.0.0', '16'
             *
             * route_solutions is an instance of RouteSolutions.
             * If route_solutions is null then there are no routes.
             */
            string? w = null;
            string v = ip + "/" + cidr;
            Variant tuple_mac_ipcidr = new Variant("(mss)", w, v);
            if (route_solutions != null)
            {
                if (routes_table.has_key(tuple_mac_ipcidr))
                {
                    if (!((RouteSolutions)routes_table[tuple_mac_ipcidr]).equals(route_solutions))
                    {
                        change_outgoing_route(ip, cidr, route_solutions);
                    }
                }
                else
                {
                    add_outgoing_route(ip, cidr, route_solutions);
                }
            }
            else
            {
                if (routes_table.has_key(tuple_mac_ipcidr))
                {
                    remove_outgoing_route(ip, cidr);
                }
                else
                {
                    // copy the current keys collection of routes_table
                    EqualDataFunc<Variant> variant_equal_func = (a, b) => a.equal(b);
                    ArrayList<Variant> initial_keys = new ArrayList<Variant>(variant_equal_func);
                    initial_keys.add_all(routes_table.keys);
                    foreach (Variant key in initial_keys)
                    {
                        string? w2;
                        string v2;
                        key.get("(mss)", out w2, out v2);
                        if (v2 == v)
                        {
                            remove_forwarding_route_whateverwas(ip, cidr, w2);
                        }
                    }
                }
            }
        }

        /** Maintains this routes for this destination when prev_hop is the
          * gateway from which we received the packet.
          */
        public void forwarding_routes(string ip, string cidr, string prev_hop, RouteSolutions? route_solutions) throws Error
        {
            /*
             * ip, cidr and prev_hop are strings. prev_hop is a MAC address.
             * Eg: '192.168.0.0', '16', '6a:b8:1e:cf:1d:4f'
             *
             * route_solutions is an instance of RouteSolutions.
             * If route_solutions is null then there are no routes (we shall use DROP!).
             */
            string? w = prev_hop;
            string v = ip + "/" + cidr;
            Variant tuple_mac_ipcidr = new Variant("(mss)", w, v);
            if (route_solutions != null)
            {
                if (routes_table.has_key(tuple_mac_ipcidr))
                {
                    BaseRouteSolutions old = routes_table[tuple_mac_ipcidr];
                    if (! old.get_type().is_a(typeof(RouteSolutions))
                        || !((RouteSolutions)old).equals(route_solutions))
                    {
                        change_forwarding_route_whateverwas(ip, cidr, prev_hop, route_solutions);
                    }
                }
                else
                {
                    add_forwarding_route(ip, cidr, prev_hop, route_solutions);
                }
            }
            else
            {
                if (routes_table.has_key(tuple_mac_ipcidr))
                {
                    change_forwarding_route_drop_whateverwas(ip, cidr, prev_hop);
                }
                else
                {
                    add_forwarding_route_drop(ip, cidr, prev_hop);
                }
            }
        }

        // Single routes management

        void add_outgoing_route(string ip, string cidr, RouteSolutions route_solutions) throws Error
        {
            _add_outgoing_route(ip, cidr, route_solutions);
            string? w = null;
            string v = ip + "/" + cidr;
            Variant tuple_mac_ipcidr = new Variant("(mss)", w, v);
            routes_table[tuple_mac_ipcidr] = route_solutions;
        }
        /** Add a route (that was non existent) towards ip/cidr
          * for packets generated by this host.
          * The gateways are route_solutions
          */
        protected abstract void _add_outgoing_route(string ip, string cidr, RouteSolutions route_solutions);

        void add_forwarding_route(string ip, string cidr, string prev_hop, RouteSolutions route_solutions) throws Error
        {
            _add_forwarding_route(ip, cidr, prev_hop, route_solutions);
            string? w = prev_hop;
            string v = ip + "/" + cidr;
            Variant tuple_mac_ipcidr = new Variant("(mss)", w, v);
            routes_table[tuple_mac_ipcidr] = route_solutions;
        }
        /** Add a route (that was non existent) towards ip/cidr
          * for packets coming from MAC prev_hop.
          * The gateways are route_solutions
          */
        protected abstract void _add_forwarding_route(string ip, string cidr, string prev_hop, RouteSolutions route_solutions) throws Error;

        void add_forwarding_route_unreachable(string ip, string cidr, string prev_hop) throws Error
        {
            _add_forwarding_route_unreachable(ip, cidr, prev_hop);
            string? w = prev_hop;
            string v = ip + "/" + cidr;
            Variant tuple_mac_ipcidr = new Variant("(mss)", w, v);
            routes_table[tuple_mac_ipcidr] = UNREACHABLE;
        }
        /** Add a route (that was non existent) towards ip/cidr
          * for packets coming from MAC prev_hop.
          * In fact this route is UNREACHABLE
          */
        protected abstract void _add_forwarding_route_unreachable(string ip, string cidr, string prev_hop) throws Error;

        void add_forwarding_route_drop(string ip, string cidr, string prev_hop) throws Error
        {
            _add_forwarding_route_drop(ip, cidr, prev_hop);
            string? w = prev_hop;
            string v = ip + "/" + cidr;
            Variant tuple_mac_ipcidr = new Variant("(mss)", w, v);
            routes_table[tuple_mac_ipcidr] = DROP;
        }
        /** Add a route (that was non existent) towards ip/cidr
          * for packets coming from MAC prev_hop.
          * In fact this route is DROP
          */
        protected abstract void _add_forwarding_route_drop(string ip, string cidr, string prev_hop) throws Error;

        void change_outgoing_route(string ip, string cidr, RouteSolutions route_solutions)
        {
            string? w = null;
            string v = ip + "/" + cidr;
            Variant tuple_mac_ipcidr = new Variant("(mss)", w, v);
            RouteSolutions old_route_solutions = (RouteSolutions)routes_table[tuple_mac_ipcidr];
            _change_outgoing_route(ip, cidr, route_solutions, old_route_solutions);
            routes_table[tuple_mac_ipcidr] = route_solutions;
        }
        /** Change a route (that was existent) towards ip/cidr
          * for packets generated by this host.
          * The gateways are route_solutions.
          * The old gateways were old_route_solutions
          */
        protected abstract void _change_outgoing_route(string ip, string cidr, RouteSolutions route_solutions, RouteSolutions old_route_solutions);

        void change_forwarding_route_whateverwas(string ip, string cidr, string prev_hop, RouteSolutions route_solutions) throws Error
        {
            string? w = prev_hop;
            string v = ip + "/" + cidr;
            Variant tuple_mac_ipcidr = new Variant("(mss)", w, v);
            BaseRouteSolutions old_route_solutions = routes_table[tuple_mac_ipcidr];
            if (old_route_solutions == UNREACHABLE)
            {
                change_forwarding_route_was_unreachable(ip, cidr, prev_hop, route_solutions);
            }
            else if (old_route_solutions == DROP)
            {
                change_forwarding_route_was_drop(ip, cidr, prev_hop, route_solutions);
            }
            else if (old_route_solutions.get_type().is_a(typeof(RouteSolutions)))
            {
                change_forwarding_route(ip, cidr, prev_hop, route_solutions);
            }
            else
            {
                error("garbage in routes_table");
            }
        }

        void change_forwarding_route_unreachable_whateverwas(string ip, string cidr, string prev_hop) throws Error
        {
            string? w = prev_hop;
            string v = ip + "/" + cidr;
            Variant tuple_mac_ipcidr = new Variant("(mss)", w, v);
            BaseRouteSolutions old_route_solutions = routes_table[tuple_mac_ipcidr];
            if (old_route_solutions == UNREACHABLE)
            {
                // pass
            }
            else if (old_route_solutions == DROP)
            {
                change_forwarding_route_unreachable_was_drop(ip, cidr, prev_hop);
            }
            else if (old_route_solutions.get_type().is_a(typeof(RouteSolutions)))
            {
                change_forwarding_route_unreachable(ip, cidr, prev_hop);
            }
            else
            {
                error("garbage in routes_table");
            }
        }

        void change_forwarding_route_drop_whateverwas(string ip, string cidr, string prev_hop) throws Error
        {
            string? w = prev_hop;
            string v = ip + "/" + cidr;
            Variant tuple_mac_ipcidr = new Variant("(mss)", w, v);
            BaseRouteSolutions old_route_solutions = routes_table[tuple_mac_ipcidr];
            if (old_route_solutions == UNREACHABLE)
            {
                change_forwarding_route_drop_was_unreachable(ip, cidr, prev_hop);
            }
            else if (old_route_solutions == DROP)
            {
                // pass
            }
            else if (old_route_solutions.get_type().is_a(typeof(RouteSolutions)))
            {
                change_forwarding_route_drop(ip, cidr, prev_hop);
            }
            else
            {
                error("garbage in routes_table");
            }
        }

        void change_forwarding_route(string ip, string cidr, string prev_hop, RouteSolutions route_solutions) throws Error
        {
            string? w = prev_hop;
            string v = ip + "/" + cidr;
            Variant tuple_mac_ipcidr = new Variant("(mss)", w, v);
            RouteSolutions old_route_solutions = (RouteSolutions)routes_table[tuple_mac_ipcidr];
            _change_forwarding_route(ip, cidr, prev_hop, route_solutions, old_route_solutions);
            routes_table[tuple_mac_ipcidr] = route_solutions;
        }
        /** Change a route (that was existent) towards ip/cidr
          * for packets coming from MAC prev_hop.
          * The gateways are route_solutions.
          * The old gateways were old_route_solutions
          */
        protected abstract void _change_forwarding_route(string ip, string cidr, string prev_hop, RouteSolutions route_solutions, RouteSolutions old_route_solutions) throws Error;

        void change_forwarding_route_unreachable(string ip, string cidr, string prev_hop) throws Error
        {
            string? w = prev_hop;
            string v = ip + "/" + cidr;
            Variant tuple_mac_ipcidr = new Variant("(mss)", w, v);
            RouteSolutions old_route_solutions = (RouteSolutions)routes_table[tuple_mac_ipcidr];
            _change_forwarding_route_unreachable(ip, cidr, prev_hop, old_route_solutions);
            routes_table[tuple_mac_ipcidr] = UNREACHABLE;
        }
        /** Change a route (that was existent) towards ip/cidr
          * for packets coming from MAC prev_hop.
          * In fact this route is UNREACHABLE
          * The old gateways were old_route_solutions
          */
        protected abstract void _change_forwarding_route_unreachable(string ip, string cidr, string prev_hop, RouteSolutions old_route_solutions) throws Error;

        void change_forwarding_route_drop(string ip, string cidr, string prev_hop) throws Error
        {
            string? w = prev_hop;
            string v = ip + "/" + cidr;
            Variant tuple_mac_ipcidr = new Variant("(mss)", w, v);
            RouteSolutions old_route_solutions = (RouteSolutions)routes_table[tuple_mac_ipcidr];
            _change_forwarding_route_drop(ip, cidr, prev_hop, old_route_solutions);
            routes_table[tuple_mac_ipcidr] = DROP;
        }
        /** Change a route (that was existent) towards ip/cidr
          * for packets coming from MAC prev_hop.
          * In fact this route is DROP
          * The old gateways were old_route_solutions
          */
        protected abstract void _change_forwarding_route_drop(string ip, string cidr, string prev_hop, RouteSolutions old_route_solutions) throws Error;

        void change_forwarding_route_was_unreachable(string ip, string cidr, string prev_hop, RouteSolutions route_solutions) throws Error
        {
            string? w = prev_hop;
            string v = ip + "/" + cidr;
            Variant tuple_mac_ipcidr = new Variant("(mss)", w, v);
            _change_forwarding_route_was_unreachable(ip, cidr, prev_hop, route_solutions);
            routes_table[tuple_mac_ipcidr] = route_solutions;
        }
        /** Change a route (that was existent) towards ip/cidr
          * for packets coming from MAC prev_hop.
          * The gateways are route_solutions.
          * Previously the route was UNREACHABLE
          */
        protected abstract void _change_forwarding_route_was_unreachable(string ip, string cidr, string prev_hop, RouteSolutions route_solutions) throws Error;

        void change_forwarding_route_drop_was_unreachable(string ip, string cidr, string prev_hop) throws Error
        {
            string? w = prev_hop;
            string v = ip + "/" + cidr;
            Variant tuple_mac_ipcidr = new Variant("(mss)", w, v);
            _change_forwarding_route_drop_was_unreachable(ip, cidr, prev_hop);
            routes_table[tuple_mac_ipcidr] = DROP;
        }
        /** Change a route (that was existent) towards ip/cidr
          * for packets coming from MAC prev_hop.
          * In fact now this route is DROP
          * Previously the route was UNREACHABLE
          */
        protected abstract void _change_forwarding_route_drop_was_unreachable(string ip, string cidr, string prev_hop) throws Error;

        void change_forwarding_route_was_drop(string ip, string cidr, string prev_hop, RouteSolutions route_solutions) throws Error
        {
            string? w = prev_hop;
            string v = ip + "/" + cidr;
            Variant tuple_mac_ipcidr = new Variant("(mss)", w, v);
            _change_forwarding_route_was_drop(ip, cidr, prev_hop, route_solutions);
            routes_table[tuple_mac_ipcidr] = route_solutions;
        }
        /** Change a route (that was existent) towards ip/cidr
          * for packets coming from MAC prev_hop.
          * The gateways are route_solutions.
          * Previously the route was DROP
          */
        protected abstract void _change_forwarding_route_was_drop(string ip, string cidr, string prev_hop, RouteSolutions route_solutions) throws Error;

        void change_forwarding_route_unreachable_was_drop(string ip, string cidr, string prev_hop) throws Error
        {
            string? w = prev_hop;
            string v = ip + "/" + cidr;
            Variant tuple_mac_ipcidr = new Variant("(mss)", w, v);
            _change_forwarding_route_unreachable_was_drop(ip, cidr, prev_hop);
            routes_table[tuple_mac_ipcidr] = UNREACHABLE;
        }
        /** Change a route (that was existent) towards ip/cidr
          * for packets coming from MAC prev_hop.
          * In fact now this route is UNREACHABLE
          * Previously the route was DROP
          */
        protected abstract void _change_forwarding_route_unreachable_was_drop(string ip, string cidr, string prev_hop) throws Error;

        void remove_outgoing_route(string ip, string cidr)
        {
            string? w = null;
            string v = ip + "/" + cidr;
            Variant tuple_mac_ipcidr = new Variant("(mss)", w, v);
            RouteSolutions old_route_solutions = (RouteSolutions)routes_table[tuple_mac_ipcidr];
            _remove_outgoing_route(ip, cidr, old_route_solutions);
            routes_table.unset(tuple_mac_ipcidr);
        }
        /** Remove a route (that was existent) towards ip/cidr
          * for packets generated by this host.
          * The old gateways were old_route_solutions
          */
        protected abstract void _remove_outgoing_route(string ip, string cidr, RouteSolutions old_route_solutions);

        void remove_forwarding_route_whateverwas(string ip, string cidr, string prev_hop) throws Error
        {
            string? w = prev_hop;
            string v = ip + "/" + cidr;
            Variant tuple_mac_ipcidr = new Variant("(mss)", w, v);
            BaseRouteSolutions old_route_solutions = routes_table[tuple_mac_ipcidr];
            if (old_route_solutions == UNREACHABLE)
            {
                remove_forwarding_route_was_unreachable(ip, cidr, prev_hop);
            }
            else if (old_route_solutions == DROP)
            {
                remove_forwarding_route_was_drop(ip, cidr, prev_hop);
            }
            else if (old_route_solutions.get_type().is_a(typeof(RouteSolutions)))
            {
                remove_forwarding_route(ip, cidr, prev_hop);
            }
            else
            {
                error("garbage in routes_table");
            }
        }

        void remove_forwarding_route(string ip, string cidr, string prev_hop) throws Error
        {
            string? w = prev_hop;
            string v = ip + "/" + cidr;
            Variant tuple_mac_ipcidr = new Variant("(mss)", w, v);
            RouteSolutions old_route_solutions = (RouteSolutions)routes_table[tuple_mac_ipcidr];
            _remove_forwarding_route(ip, cidr, prev_hop, old_route_solutions);
            routes_table.unset(tuple_mac_ipcidr);
        }
        /** Remove a route (that was existent) towards ip/cidr
          * for packets coming from MAC prev_hop.
          * The old gateways were old_route_solutions
          */
        protected abstract void _remove_forwarding_route(string ip, string cidr, string prev_hop, RouteSolutions old_route_solutions) throws Error;

        void remove_forwarding_route_was_unreachable(string ip, string cidr, string prev_hop) throws Error
        {
            string? w = prev_hop;
            string v = ip + "/" + cidr;
            Variant tuple_mac_ipcidr = new Variant("(mss)", w, v);
            _remove_forwarding_route_was_unreachable(ip, cidr, prev_hop);
            routes_table.unset(tuple_mac_ipcidr);
        }
        /** Remove a route (that was existent) towards ip/cidr
          * for packets coming from MAC prev_hop.
          * Previously the route was UNREACHABLE
          */
        protected abstract void _remove_forwarding_route_was_unreachable(string ip, string cidr, string prev_hop) throws Error;

        void remove_forwarding_route_was_drop(string ip, string cidr, string prev_hop) throws Error
        {
            string? w = prev_hop;
            string v = ip + "/" + cidr;
            Variant tuple_mac_ipcidr = new Variant("(mss)", w, v);
            _remove_forwarding_route_was_drop(ip, cidr, prev_hop);
            routes_table.unset(tuple_mac_ipcidr);
        }
        /** Remove a route (that was existent) towards ip/cidr
          * for packets coming from MAC prev_hop.
          * Previously the route was DROP
          */
        protected abstract void _remove_forwarding_route_was_drop(string ip, string cidr, string prev_hop) throws Error;

        public void forward_no_more_from(string prev_hop) throws Error
        {
            _forward_no_more_from(prev_hop);
            // copy the current keys collection of routes_table
            EqualDataFunc<Variant> variant_equal_func = (a, b) => a.equal(b);
            ArrayList<Variant> initial_keys = new ArrayList<Variant>(variant_equal_func);
            initial_keys.add_all(routes_table.keys);
            foreach (Variant key in initial_keys)
            {
                string? w;
                string v;
                key.get("(mss)", out w, out v);
                if (w == prev_hop)
                {
                    routes_table.unset(key);
                }
            }
        }
        /** Stop forwarding (in fact it means delete all rules for...)
          * packets coming from MAC prev_hop.
          */
        protected abstract void _forward_no_more_from(string prev_hop) throws Error;

        public void forward_no_more() throws Error
        {
            _forward_no_more();
            // copy the current keys collection of routes_table
            EqualDataFunc<Variant> variant_equal_func = (a, b) => a.equal(b);
            ArrayList<Variant> initial_keys = new ArrayList<Variant>(variant_equal_func);
            initial_keys.add_all(routes_table.keys);
            foreach (Variant key in initial_keys)
            {
                string? w;
                string v;
                key.get("(mss)", out w, out v);
                if (w != null)
                {
                    routes_table.unset(key);
                }
            }
        }
        /** Delete all specific rules for any prev_hop.
          */
        protected abstract void _forward_no_more() throws Error;

        // Neighbour management

        public void add_neighbour(string ip, string dev, string pref_src)
        {
            _add_neighbour(ip, dev, pref_src);
            neighbours_table[ip] = dev;
        }
        /** Adds a new neighbour with corresponding properties.
          */
        protected abstract void _add_neighbour(string ip, string dev, string pref_src);

        public void change_neighbour(string ip, string dev, string pref_src)
        {
            string old_dev = neighbours_table[ip];
            _change_neighbour(ip, dev, pref_src, old_dev);
            neighbours_table[ip] = dev;
        }
        /** Edits the neighbour with the corresponding properties.
          */
        protected abstract void _change_neighbour(string ip, string dev, string pref_src, string old_dev);

        public void delete_neighbour(string ip)
        {
            string old_dev = neighbours_table[ip];
            _delete_neighbour(ip, old_dev);
            neighbours_table.unset(ip);
        }
        /** Removes the neighbour with the corresponding properties.
          */
        protected abstract void _delete_neighbour(string ip, string old_dev);

        // IGS management

        public bool igs_update_rules(string mode, bool announce_myself, ArrayList<TunnelItem> list_tunnels, string pref_src) throws Error
        {
            if (this.mode != mode || this.announce_myself != announce_myself
                    || ! this.list_tunnels.contains_all(list_tunnels)
                    || ! list_tunnels.contains_all(this.list_tunnels))
            {
                if (_igs_update_rules(this.mode, this.announce_myself, this.list_tunnels,
                                mode, announce_myself, list_tunnels, pref_src))
                {
                    this.mode = mode;
                    this.announce_myself = announce_myself;
                    this.list_tunnels = list_tunnels;
                    return true;
                }
                else
                {
                    this.mode = "NONE";
                    return false;
                }
            }
            else return true;
        }

        // An implementor returns false iff it is unable to implement this feature.
        protected abstract bool _igs_update_rules(string prev_mode, bool prev_announce_myself, ArrayList<TunnelItem> prev_list_tunnels,
                    string mode, bool announce_myself, ArrayList<TunnelItem> list_tunnels, string pref_src) throws Error;

        protected abstract bool check_ping(string ipstr);

        // initializations

        // An implementor returns false iff it is unable to implement this feature.
        protected abstract bool _reset_routes(string ip_whole_network, string cidr_whole_network);

        public abstract void activate_multipath();
        public abstract void ip_forward(bool enable);
    }

    public delegate RouteSetter CreateRouteSetterDelegate();

    /** A class to retrieve info/statistics about active connections on the host
      */
    public abstract class Connections : Object
    {
        private static CreateConnectionsDelegate? registered_class_linux = null;
        public static void register_class(string k, CreateConnectionsDelegate create_new_connections)
        {
            if (k == "linux" && registered_class_linux == null) registered_class_linux = create_new_connections;
        }

        private static Connections create_instance()
        {
            string impl = Settings.NETWORK_IMPLEMENTATION;
            if (impl == "linux" && registered_class_linux != null) return registered_class_linux();
            error(@"No valid real implementation of class Connections for $impl. Is your system supported?");
        }

        private static Connections? _instance = null;
        public static Connections get_instance()
        {
            if (_instance == null) _instance = create_instance();
            return _instance;
        }

        /** Are there active TCP connections on this address?
          */
        public abstract bool active_tcp_connections(string ip);
    }

    public delegate Connections CreateConnectionsDelegate();

    public class DummyRoute : RouteSetter
    {
        public DummyRoute()
        {
            base();
        }

        protected override void _add_outgoing_route(string ip, string cidr, RouteSolutions route_solutions) {}
        protected override void _add_forwarding_route(string ip, string cidr, string prev_hop, RouteSolutions route_solutions) {}
        protected override void _add_forwarding_route_unreachable(string ip, string cidr, string prev_hop) {}
        protected override void _add_forwarding_route_drop(string ip, string cidr, string prev_hop) {}
        protected override void _change_outgoing_route(string ip, string cidr, RouteSolutions route_solutions, RouteSolutions old_route_solutions) {}
        protected override void _change_forwarding_route(string ip, string cidr, string prev_hop, RouteSolutions route_solutions, RouteSolutions old_route_solutions) {}
        protected override void _change_forwarding_route_unreachable(string ip, string cidr, string prev_hop, RouteSolutions old_route_solutions) {}
        protected override void _change_forwarding_route_drop(string ip, string cidr, string prev_hop, RouteSolutions old_route_solutions) {}
        protected override void _change_forwarding_route_was_unreachable(string ip, string cidr, string prev_hop, RouteSolutions route_solutions) {}
        protected override void _change_forwarding_route_drop_was_unreachable(string ip, string cidr, string prev_hop) {}
        protected override void _change_forwarding_route_was_drop(string ip, string cidr, string prev_hop, RouteSolutions route_solutions) {}
        protected override void _change_forwarding_route_unreachable_was_drop(string ip, string cidr, string prev_hop) {}
        protected override void _remove_outgoing_route(string ip, string cidr, RouteSolutions old_route_solutions) {}
        protected override void _remove_forwarding_route(string ip, string cidr, string prev_hop, RouteSolutions old_route_solutions) {}
        protected override void _remove_forwarding_route_was_unreachable(string ip, string cidr, string prev_hop) {}
        protected override void _remove_forwarding_route_was_drop(string ip, string cidr, string prev_hop) {}
        protected override void _forward_no_more_from(string prev_hop) {}
        protected override void _forward_no_more() {}
        protected override void _add_neighbour(string ip, string dev, string pref_src) {}
        protected override void _change_neighbour(string ip, string dev, string pref_src, string old_dev) {}
        protected override void _delete_neighbour(string ip, string old_dev) {}
        protected override bool _igs_update_rules(string prev_mode, bool prev_announce_myself, ArrayList<TunnelItem> prev_list_tunnels,
                    string mode, bool announce_myself, ArrayList<TunnelItem> list_tunnels, string pref_src)
        {
            // An implementor returns false iff it is unable to implement this feature.
            return false;
        }
        protected override bool check_ping(string ipstr) {return false;}
        protected override bool _reset_routes(string ip_whole_network, string cidr_whole_network)
        {
            // An implementor returns false iff it is unable to implement this feature.
            return false;
        }
        public override void activate_multipath() {}
        public override void ip_forward(bool enable) {}
    }
}

