/*
 *  This file is part of Netsukuku.
 *  (c) Copyright 2011 Luca Dionisi aka lukisi <luca.dionisi@gmail.com>
 *
 *  Netsukuku is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  Netsukuku is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with Netsukuku.  If not, see <http://www.gnu.org/licenses/>.
 */

using Gee;
using zcd;

namespace Netsukuku
{
    public class HookingSolution : Object, Comparable<HookingSolution>
    {
        public AggregatedNeighbour aggregated_neighbour {get; private set;}
        public int lvl {get; private set;}
        public int fn {get; private set;}

        public HookingSolution(AggregatedNeighbour aggregated_neighbour, int lvl, int fn)
        {
            this.aggregated_neighbour = aggregated_neighbour;
            this.lvl = lvl;
            this.fn = fn;
        }

        public int compare_to(HookingSolution b)
        {
            if (lvl > b.lvl) return 1;
            if (lvl < b.lvl) return -1;
            if (fn > b.fn) return 1;
            if (fn < b.fn) return -1;
            return 0;
        }

        public string to_string()
        {
            return @"<HookingSolution: neighbour $aggregated_neighbour has $fn places in level $lvl>";
        }
    }

    public class HookInformation : Object
    {
        public PartialNIP gnode;
        public CoordinatorReservation coordinator_reservation;
    }

    /** Class Hook
      *
      */
    public class Hook : Object, IHook
    {
        private MapRoute maproute;
        private AddressManager addr_man;

        public Hook(MapRoute maproute, AddressManager addr_man)
        {
            this.maproute = maproute;
            this.addr_man = addr_man;
        }

        private static bool not_impl_equal_func (Object? a, Object? b)
        {
            string astr = "null";
            if (a != null) astr = a.get_type().name();
            string bstr = "null";
            if (b != null) bstr = b.get_type().name();
            error(@"equal_func not implemented: compare between $(astr) and $(bstr)");
        }
        private static int reverse_compare(HookingSolution a, HookingSolution b)
        {
            return - a.compare_to(b);
        }
        /** The method returns a new valid address.
          * It could be in the current network or in another network.
          * We try to hook among the given neighbours.
          */
        public HookInformation hook(Gee.List<AggregatedNeighbour> neighbour_list) throws HookingError
        {
            ArrayList<HookingSolution> hfn = new ArrayList<HookingSolution>(not_impl_equal_func);
            // Find all the highest non saturated gnodes
            bool has_someone_responded = false;
            foreach (AggregatedNeighbour aggregated_neighbour in neighbour_list)
            {
                // We must consider that a node could not respond.
                try
                {
                    Gee.List<PairLvlNumberOfFreeNodes> resp = aggregated_neighbour.neighbour_client.hook.list_non_saturated_levels();
                    // returns a list of (lvl, fn)
                    has_someone_responded = true;
                    foreach (PairLvlNumberOfFreeNodes el in resp)
                    {
                        hfn.add(new HookingSolution(aggregated_neighbour, el.lvl, el.number_of_free_nodes));
                    }
                }
                catch (Error e)
                {
                    log_warn(@"Exception trying to get list_non_saturated_levels from a neighbour. $(e.message)");
                }
            }

            if (!has_someone_responded) throw new HookingError.GENERIC("False network collision was detected.");
            if (hfn.size == 0)
            {
                string str_records = "[";
                foreach (AggregatedNeighbour aggregated_neighbour in neighbour_list) str_records += @"$aggregated_neighbour,";
                str_records += "]";
                throw new HookingError.GENERIC(@"This network portion is full: $str_records");
            }

            hfn.sort(reverse_compare);

            // Contact the coordinator node, passing from one of the answering neighbours.
            foreach (HookingSolution hs in hfn)
            {
                AggregatedNeighbour neighbour_to_contact = hs.aggregated_neighbour;
                NIP nip = neighbour_to_contact.nip;
                int level_of_nodes = hs.lvl;
                int level_of_gnode = level_of_nodes + 1;
                PartialNIP partial_nip = nip.get_gnode_at_level(level_of_gnode);
                log_info(@"Hook: try entering gnode $partial_nip via $neighbour_to_contact");
                try
                {
                    CoordinatorReservation coordinator_reservation =
                        neighbour_to_contact.neighbour_client.coordinator.reserve_into(partial_nip);
                    log_info(@"Hooking to $partial_nip succeded");
                    HookInformation ret = new HookInformation();
                    ret.gnode = partial_nip;
                    ret.coordinator_reservation = coordinator_reservation;
                    return ret;
                }
                catch (HookingError er)
                {
                    log_info(@"Hooking to $partial_nip failed.");
                }
            }

            throw new HookingError.GENERIC("No good hooking solution.");
        }

        /** Returns a list of (lvl, number_of_free_nodes_at_lvl)
          */
        public Gee.List<PairLvlNumberOfFreeNodes> list_non_saturated_levels()
        {
            Gee.List<PairLvlNumberOfFreeNodes> ret = new ArrayList<PairLvlNumberOfFreeNodes>(PairLvlNumberOfFreeNodes.equal_func);
            if (! addr_man.is_mature) return ret;
            for (int lvl = maproute.levels-1; lvl >= 0; lvl--)
            {
                try
                {
                    int fn = maproute.free_nodes_nb(lvl);
                    if (fn > 0) ret.add(new PairLvlNumberOfFreeNodes(lvl, fn));
                }
                catch (RPCError e)
                {
                    error(@"shoud never happen when we call locally");
                }
            }
            return ret;
        }
    }
}
