/*
 *  This file is part of Netsukuku.
 *  (c) Copyright 2011 Luca Dionisi aka lukisi <luca.dionisi@gmail.com>
 *
 *  Netsukuku is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  Netsukuku is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with Netsukuku.  If not, see <http://www.gnu.org/licenses/>.
 */

using Gee;
using zcd;

namespace Netsukuku
{
    namespace RpcNtk
    {
        public void init()
        {
            // Register serializable types
            typeof(TimeCapsule).class_peek();
            typeof(PeerToPeerTracerPacketList).class_peek();
            typeof(SetOptionalServiceParticipants).class_peek();
            typeof(OptionalServiceParticipants).class_peek();
            typeof(PackedParticipantNodes).class_peek();
            typeof(ParticipantNode).class_peek();
            typeof(PairNipDistance).class_peek();
            typeof(HookReservation).class_peek();
            typeof(BookingRecord).class_peek();
            typeof(Bookings).class_peek();
            typeof(BnodeRecord).class_peek();
            typeof(BnodeList).class_peek();
            typeof(CoordinatorKnowledge).class_peek();
            typeof(CoordinatorKnowledgeSet).class_peek();
            typeof(PairLvlNumberOfFreeNodes).class_peek();
            typeof(HCoord).class_peek();
            typeof(PartialNIP).class_peek();
            typeof(NIP).class_peek();
            typeof(REM).class_peek();
            typeof(NullREM).class_peek();
            typeof(DeadREM).class_peek();
            typeof(AlmostDeadREM).class_peek();
            typeof(RTT).class_peek();
            typeof(TracerPacketList).class_peek();
            typeof(RouteInSet).class_peek();
            typeof(PositionInRoutesSetPerLevel).class_peek();
            typeof(RoutesSetPerLevel).class_peek();
            typeof(RoutesSet).class_peek();
            typeof(ExtendedTracerPacket).class_peek();
            typeof(GNodeID).class_peek();
            typeof(NetworkID).class_peek();
            typeof(InfoNeighbour).class_peek();
            typeof(InfoRoute).class_peek();
            typeof(InfoNode).class_peek();
            typeof(TaskletStats).class_peek();
            typeof(QspnStats).class_peek();
            typeof(InfoBorderNode).class_peek();
            typeof(InfoCoord).class_peek();
            typeof(InfoAndna).class_peek();
            typeof(InfoAndnaCache).class_peek();
            typeof(InfoAndnaRegistration).class_peek();
            typeof(DHTKey).class_peek();
            typeof(DHTRecord).class_peek();
            typeof(PublicKey).class_peek();
            typeof(AndnaServiceKey).class_peek();
            typeof(AndnaPrivateConfigurationServer).class_peek();
            typeof(AndnaPrivateConfiguration).class_peek();
            typeof(AndnaPrivateConfigurationList).class_peek();
            typeof(AndnaDomainRequest).class_peek();
            typeof(AndnaServerRequest).class_peek();
            typeof(AndnaServerRecord).class_peek();
            typeof(AndnaDomainRecord).class_peek();
            typeof(AndnaServer).class_peek();
            typeof(AndnaServers).class_peek();
            typeof(RegisterHostnameArguments).class_peek();
            typeof(CounterNipRecord).class_peek();
            typeof(CounterSetDataResponse).class_peek();
            typeof(CounterCheckHostnameResponse).class_peek();
            typeof(PairPublicKeyNIP).class_peek();
            typeof(CounterGetCacheRecordsResponse).class_peek();
            typeof(AndnaConfirmPubkResponse).class_peek();
            typeof(AndnaRegisterMainResponse).class_peek();
            typeof(AndnaRegisterSpreadResponse).class_peek();
            typeof(AndnaGetServersResponse).class_peek();
            typeof(AndnaGetRegistrarResponse).class_peek();
            typeof(AndnaGetCacheRecordsResponse).class_peek();
            typeof(BroadcastID).class_peek();
            typeof(UnicastID).class_peek();
        }
    }


    public errordomain QspnError {
        NOT_YOUR_GATEWAY,
        ALREADY_UP_TO_DATE,
        GENERIC
    }

    public errordomain BorderNodesError {
        WRONG_GNODE,
        NOT_BORDER_NODE,
        WILL_NOT_TUNNEL,
        TIMEOUT,
        GENERIC
    }

    public errordomain TunnelError {
        GENERIC
    }

    public errordomain HookingError {
        INEXISTENT_GNODE,
        GENERIC
    }

    public errordomain PeerRefuseServiceError {
        GENERIC
    }

    public errordomain AndnaError {
        GENERIC
    }

    /** This is the interface for a root-dispatcher class
      */
    public interface IAddressManagerRootDispatcher : Object
    {
        public IEtp etp {
            get {
                return this._etp_getter();
            }
        }
        public abstract unowned IEtp _etp_getter();

        public ITunnelManager tunnel_manager {
            get {
                return this._tunnel_manager_getter();
            }
        }
        public abstract unowned ITunnelManager _tunnel_manager_getter();

        public IHook hook {
            get {
                return this._hook_getter();
            }
        }
        public abstract unowned IHook _hook_getter();

        public IAggregatedNeighbourManager aggregated_neighbour_manager {
            get {
                return this._aggregated_neighbour_manager_getter();
            }
        }
        public abstract unowned IAggregatedNeighbourManager _aggregated_neighbour_manager_getter();

        public IMapRoute maproute {
            get {
                return this._maproute_getter();
            }
        }
        public abstract unowned IMapRoute _maproute_getter();

        public IBorderNodesManager border_nodes_manager {
            get {
                return this._border_nodes_manager_getter();
            }
        }
        public abstract unowned IBorderNodesManager _border_nodes_manager_getter();

        public IPeerToPeerAll peer_to_peer_all {
            get {
                return this._peer_to_peer_all_getter();
            }
        }
        public abstract unowned IPeerToPeerAll _peer_to_peer_all_getter();

        public ICoord coordnode {
            get {
                return this._coordnode_getter();
            }
        }
        public abstract unowned ICoord _coordnode_getter();

        public IAndna andna {
            get {
                return this._andna_getter();
            }
        }
        public abstract unowned IAndna _andna_getter();

        public abstract IPeerToPeer get_peer_to_peer_service(int pid);
        public abstract IOptionalPeerToPeer get_optional_peer_to_peer_service(int pid);

        // Test methods for peer_to_peer
        public abstract void dht_stor(DHTRecord rec) throws RPCError;
        public abstract DHTRecord? dht_retr(DHTKey k) throws RPCError;
        public abstract Gee.List<DHTRecord> dht_list() throws RPCError;

        public abstract Gee.List<TaskletStats> report_tasklets_stats(int minutes) throws RPCError;
        public abstract Gee.List<string> report_tasklet_logs(int id) throws RPCError;
        public abstract Gee.List<string> report_running_tasklets() throws RPCError;
    }

    public interface IEtp : Object
    {
        public abstract ExtendedTracerPacket request_etp(int prev_mod_seq_num, NIP nip_caller,
                                int nodeid_caller, NetworkID netid_caller,
                                Gee.List<string> macs_caller) throws QspnError, RPCError;
        public abstract int request_size() throws RPCError;
        public abstract void new_sequence_number(NIP nip_x, int nodeid_x, int seq_num_x) throws RPCError;
        public abstract void act_as_gateway(NIP nip_caller, int nodeid_caller,
                                NetworkID netid_caller, Gee.List<string> macs) throws RPCError;
        public abstract QspnStats report_qspn_stats() throws RPCError;
    }

    public interface ITunnelManager : Object
    {
        public abstract string? choose_tunnel_protocol(Gee.List<string> protocols) throws RPCError;
        public abstract void handshake(ISerializable mesg, int handler_id) throws TunnelError, RPCError;
        public abstract int request_tunnel(string protocol, int peer_handler_id, CallerInfo? _rpc_caller=null)
                throws TunnelError, RPCError;
        public abstract void close_tunnel(string nic_name, CallerInfo? _rpc_caller=null) throws RPCError;
    }

    public interface IAggregatedNeighbourManager : Object
    {
        public abstract void reply(int radar_id, NIP nip, int nodeid, NetworkID netid,
                        CallerInfo? _rpc_caller=null) throws RPCError;
        public abstract void time_register(int radar_id, int levels, int gsize, NIP nip, int nodeid,
                        NetworkID netid, string mac, bool is_primary, bool is_auxiliary) throws RPCError;
        public abstract void declare_dead(NIP nip, int nodeid) throws RPCError;
        public abstract Gee.List<InfoNeighbour> report_neighbours() throws RPCError;
    }

    public interface IBorderNodesManager : Object
    {
        public abstract Gee.List<PairNipDistance> get_distances(PartialNIP gnode, Gee.List<NIP> list_of_nips,
                        CallerInfo? _rpc_caller=null) throws BorderNodesError, RPCError;
        public abstract NIP get_new_address(PartialNIP gnode, NIP? peer_nip=null,
                        CallerInfo? _rpc_caller=null) throws BorderNodesError, RPCError;
        public abstract void assign_peer_nip(NIP nip_x_secondary, NIP nip_y_secondary) throws RPCError;
        public abstract Gee.List<InfoBorderNode> report_bordernodes_status() throws RPCError;
    }

    public interface ICoord : Object
    {
        public abstract void duplicate_all_knowledge(PartialNIP gnode, CoordinatorKnowledge coordinator_knowledge)
                        throws HookingError, RPCError;
        public abstract HookReservation? reserve(PartialNIP gnode) throws HookingError, RPCError;
        public abstract void free(PartialNIP gnode, int pos) throws HookingError, RPCError;
        public abstract void duplicate_free(PartialNIP gnode, int pos) throws HookingError, RPCError;
        public abstract bool register_bnode(PartialNIP gnode, NIP nip, bool is_border, bool has_tunnel,
                        bool is_willing) throws HookingError, RPCError;
        public abstract Gee.List<InfoCoord> report_status() throws RPCError;
    }

    public interface IPeerToPeer : Object
    {
        public abstract ISerializable msg_deliver(PeerToPeerTracerPacketList peer_to_peer_tpl, NIP sender_nip, NIP hip, RemoteCall rcdata) throws RPCError, Error;
        public abstract ISerializable msg_send(NIP sender_nip, NIP hip, RemoteCall rcdata) throws RPCError, Error;
        public abstract Gee.List<NIP> find_nearest(PeerToPeerTracerPacketList peer_to_peer_tpl, NIP hash_nip, int num_dupl, int lvl, int pos) throws RPCError;
        public abstract int number_of_participants(PeerToPeerTracerPacketList peer_to_peer_tpl, int lvl, int pos) throws RPCError;
    }
    public interface IOptionalPeerToPeer : Object
    {
        public abstract void participant_set(NIP nip, int lvl, int pos, ParticipantNode participant_node) throws RPCError;
        public abstract void participant_refresh(NIP nip, PackedParticipantNodes packed_nodes) throws RPCError;
    }
    public interface IPeerToPeerAll : Object
    {
        public abstract SetOptionalServiceParticipants get_optional_participants() throws RPCError;
    }

    public interface IMapRoute : Object
    {
        public abstract int free_nodes_nb(int lvl) throws RPCError;
        public abstract REM worst_internal_bestrem(int level_of_gnode) throws RPCError;
        public abstract void request_gid(int request_id, NetworkID gnetid, PartialNIP gprefix) throws RPCError;
        public abstract void answer_gid(NIP answering_nip, int request_id, GNodeID actual_gid) throws RPCError;
        public abstract Gee.List<InfoRoute> report_routes() throws RPCError;
        public abstract Gee.List<GNodeID> report_gid_list() throws RPCError;
        public abstract InfoNode report_yourself() throws RPCError;
    }

    public interface IHook : Object
    {
        public abstract Gee.List<PairLvlNumberOfFreeNodes> list_non_saturated_levels() throws RPCError;
    }

    public interface IAndna : Object
    {
        public abstract AndnaConfirmPubkResponse confirm_pubk(NIP yournip, PublicKey yourpubkey, int to_be_signed) throws RPCError;
        public abstract Gee.List<string> get_your_hostnames(NIP yournip) throws RPCError;
        public abstract InfoAndna report_status() throws RPCError;
        public abstract AndnaPrivateConfigurationList get_mynames() throws RPCError;
        public abstract void set_mynames(AndnaPrivateConfigurationList mynames) throws RPCError;
        public abstract PublicKey? retrieve_registrar_pubk(string hashed_domain) throws RPCError;
    }

    /** Serializables
      */

    /** Serializable class for "timeouts" or "timespans"
      */
    public class TimeCapsule : Tasklets.Timer, ISerializable
    {
        public TimeCapsule(int64 msec_ttl)
        {
            base(msec_ttl);
        }

        public Variant serialize_to_variant()
        {
            return Serializer.int64_to_variant(get_msec_ttl());
        }
        
        public void deserialize_from_variant(Variant v) throws SerializerError
        {
            int64 msec_ttl = Serializer.variant_to_int64(v);
            set_time(msec_ttl);
        }
    }

    /** A DataClass contains information regarding a node of the map.
      * Each Map.node[level, id] entry is a DataClass instance.
      */
    public abstract class DataClass : Object
    {
        public abstract bool is_free();
        public abstract void initialize(Object map, int lvl, int pos, bool its_me = false);
    }

    /** This value is big enough to last forever and
      *  not big enough to ever produce an overflow.
      */
    public TimeCapsule get_max_elder()
    {
        return new TimeCapsule(-999999999);
    }

    public class ParticipantNode : DataClass, ISerializable
    {
        public bool participant;
        public TimeCapsule tc;

        public ParticipantNode(bool participant=false, TimeCapsule? tc=null)
        {
            this.participant = participant;
            if (tc == null) this.tc = get_max_elder();
            else this.tc = tc;
        }

        /* This method will never get called in a MapPeerToPeer */
        public override void initialize(Object map, int lvl, int pos, bool its_me = false)
        {
        }

        /** Override the is_free() method of DataClass (see map.vala)
          */
        public override bool is_free()
        {
            return !participant;
        }

        public Variant serialize_to_variant()
        {
            Variant v0 = Serializer.int_to_variant(participant ? 1 : 0);
            Variant v1 = tc.serialize_to_variant();
            Variant vret = Serializer.tuple_to_variant(v0, v1);
            return vret;
        }
        
        public void deserialize_from_variant(Variant v) throws SerializerError
        {
            Variant v0;
            Variant v1;
            Serializer.variant_to_tuple(v, out v0, out v1);
            participant = Serializer.variant_to_int(v0) == 1;
            tc = (TimeCapsule)Object.new(typeof(TimeCapsule));
            tc.deserialize_from_variant(v1);
        }
    }

    public class PackedParticipantNodes : Object, ISerializable
    {
        public ArrayList<ArrayList<ParticipantNode>> packednodes;
        private static bool not_impl_equal_func (Object? a, Object? b)
        {
            error("PackedParticipantNodes: equal_func not implemented");
        }
        public PackedParticipantNodes(Gee.List<Gee.List<ParticipantNode>> packednodes)
        {
            this.packednodes = new ArrayList<ArrayList<ParticipantNode>>((EqualDataFunc)not_impl_equal_func);
            assert(packednodes.size > 0);
            foreach (Gee.List<ParticipantNode> lpn in packednodes)
            {
                assert(lpn.size > 0);
                ArrayList<ParticipantNode> my_lpn = new ArrayList<ParticipantNode>();
                my_lpn.add_all(lpn);
                this.packednodes.add(my_lpn);
            }
        }

        private Variant serialize_to_variant_list(ArrayList<ParticipantNode> list)
        {
            Variant[] vn = new Variant[list.size];
            for (int i = 0; i < list.size; i++)
                vn[i] = list[i].serialize_to_variant();
            return Serializer.variant_array_to_variant(vn);
        }

        public Variant serialize_to_variant()
        {
            Variant[] vn = new Variant[packednodes.size];
            for (int i = 0; i < packednodes.size; i++)
                vn[i] = serialize_to_variant_list(packednodes.get(i));
            return Serializer.variant_array_to_variant(vn);
        }

        private ArrayList<ParticipantNode> deserialize_from_variant_to_list(Variant v) throws SerializerError
        {
            Variant[] v_nested = Serializer.variant_to_variant_array(v);
            ArrayList<ParticipantNode> ret = new ArrayList<ParticipantNode>();
            for (int i = 0; i < v_nested.length; i++)
            {
                ParticipantNode temp = (ParticipantNode)Object.new(typeof(ParticipantNode));
                temp.deserialize_from_variant(v_nested[i]);
                ret.add(temp);
            }
            return ret;
        }

        public void deserialize_from_variant(Variant v) throws SerializerError
        {
            Variant[] v_nested = Serializer.variant_to_variant_array(v);
            this.packednodes = new ArrayList<ArrayList<ParticipantNode>>((EqualDataFunc)not_impl_equal_func);
            for (int i = 0; i < v_nested.length; i++)
            {
                this.packednodes.add(deserialize_from_variant_to_list(v_nested[i]));
            }
        }
    }

    public class OptionalServiceParticipants : Object, ISerializable
    {
        public int pid;
        public PackedParticipantNodes ppnodes;

        public Variant serialize_to_variant()
        {
            Variant v0 = Serializer.int_to_variant(pid);
            Variant v1 = ppnodes.serialize_to_variant();
            Variant vret = Serializer.tuple_to_variant(v0, v1);
            return vret;
        }
        
        public void deserialize_from_variant(Variant v) throws SerializerError
        {
            Variant v0;
            Variant v1;
            Serializer.variant_to_tuple(v, out v0, out v1);
            pid = Serializer.variant_to_int(v0);
            ppnodes = (PackedParticipantNodes)Object.new(typeof(PackedParticipantNodes));
            ppnodes.deserialize_from_variant(v1);
        }
    }

    public class SetOptionalServiceParticipants : Object, ISerializable
    {
        public ArrayList<OptionalServiceParticipants> o_s_participants;
        private static bool not_impl_equal_func (Object? a, Object? b)
        {
            error("SetOptionalServiceParticipants: equal_func not implemented");
        }
        public SetOptionalServiceParticipants(Gee.List<OptionalServiceParticipants> o_s_participants)
        {
            this.o_s_participants = new ArrayList<OptionalServiceParticipants>((EqualDataFunc)not_impl_equal_func);
            this.o_s_participants.add_all(o_s_participants);
        }

        public Variant serialize_to_variant()
        {
            Variant[] vn = new Variant[o_s_participants.size];
            for (int i = 0; i < o_s_participants.size; i++)
                vn[i] = o_s_participants[i].serialize_to_variant();
            Variant v = Serializer.variant_array_to_variant(vn);
            return v;
        }

        public void deserialize_from_variant(Variant v) throws SerializerError
        {
            Variant[] v_nested = Serializer.variant_to_variant_array(v);
            this.o_s_participants = new ArrayList<OptionalServiceParticipants>((EqualDataFunc)not_impl_equal_func);
            for (int i = 0; i < v_nested.length; i++)
            {
                OptionalServiceParticipants temp = (OptionalServiceParticipants)Object.new(typeof(OptionalServiceParticipants));
                temp.deserialize_from_variant(v_nested[i]);
                this.o_s_participants.add(temp);
            }
        }
    }

    /** Avoid loops, neverending stories, and the like.
      *
      * When a message is routed through the net by the peer_to_peer module,
      * we keep track of the path walked, through an instance of this class.
      * There are several functions in module peer_to_peer that try to
      * route messages, such as msg_deliver, find_nearest,
      * number_of_participants, and so on. Each function receives
      * as a parameter a PeerToPeerTracerPacketList instance,
      * calls its method execute (passing its own maproute)
      * and then passes it to the next hop.
      * The first caller instantiate it by passing no parameters or
      * can specify the timeout.
      */
    public class PeerToPeerTracerPacketList : Object, ISerializable
    {
        public ArrayList<ArrayList<int>>? tpl;
        public TimeCapsule tc;
        public PeerToPeerTracerPacketList(int timeout=120000)
        {
            tc = new TimeCapsule(timeout);
            tpl = null;
        }

        public Variant serialize_to_variant()
        {
            assert(tpl != null);
            Variant[] vn = new Variant[tpl.size];
            for (int i = 0; i < tpl.size; i++)
            {
                ArrayList<int> tpl_1 = tpl[i];
                int[] ai = new int[tpl_1.size];
                for (int j = 0; j < tpl_1.size; j++) ai[j] = tpl_1[j];
                vn[i] = Serializer.int_array_to_variant(ai);
            }
            Variant v0 = Serializer.variant_array_to_variant(vn);
            Variant v1 = tc.serialize_to_variant();
            Variant vret = Serializer.tuple_to_variant(v0, v1);
            return vret;
        }

        private static bool not_impl_equal_func (Object? a, Object? b)
        {
            error("PeerToPeerTracerPacketList: equal_func not implemented");
        }
        public void deserialize_from_variant(Variant v) throws SerializerError
        {
            Variant v0;
            Variant v1;
            Serializer.variant_to_tuple(v, out v0, out v1);
            Variant[] vn = Serializer.variant_to_variant_array(v0);
            tpl = new ArrayList<ArrayList<int>>((EqualDataFunc)not_impl_equal_func);
            for (int i = 0; i < vn.length; i++)
            {
                ArrayList<int> tpl_1 = new ArrayList<int>();
                int[] ai = Serializer.variant_to_int_array(vn[i]);
                for (int j = 0; j < ai.length; j++) tpl_1.add(ai[j]);
                tpl.add(tpl_1);
            }            
            tc = (TimeCapsule)Object.new(typeof(TimeCapsule));
            tc.deserialize_from_variant(v1);
        }

        public string to_string()
        {
            string ret_tpl = "null";
            if (tpl != null)
            {
                ret_tpl = "";
                string next = "";
                foreach (ArrayList<int> l1 in tpl)
                {
                    ret_tpl += next;
                    ret_tpl += "<";
                    string next1 = "";
                    foreach (int i in l1)
                    {
                        ret_tpl += next1;
                        ret_tpl += @"$i";
                        next1 = ", ";
                    }
                    ret_tpl += ">";
                    next = ", ";
                }
                ret_tpl = @"<$(ret_tpl)>";
            }
            return @"<PeerToPeerTracerPacketList TPL=$(ret_tpl) expiring in $(tc.get_string_msec_ttl()) msec>";
        }
    }

    /** The return value of the method get_distances
      * is a list of tuple (NIP nip, int distance)
      */
    public class PairNipDistance : Object, ISerializable
    {
        public NIP nip;
        public int distance; // in number of hops
        public PairNipDistance(NIP nip, int distance)
        {
            this.nip = nip;
            this.distance = distance;
        }

        public Variant serialize_to_variant()
        {
            Variant v0 = nip.serialize_to_variant();
            Variant v1 = Serializer.int_to_variant(distance);
            Variant vret = Serializer.tuple_to_variant(v0, v1);
            return vret;
        }

        public void deserialize_from_variant(Variant v) throws SerializerError
        {
            Variant v0;
            Variant v1;
            Serializer.variant_to_tuple(v, out v0, out v1);
            nip = (NIP)Object.new(typeof(NIP));
            nip.deserialize_from_variant(v0);
            distance = Serializer.variant_to_int(v1);
        }

        public static bool equal_func(PairNipDistance? a, PairNipDistance? b)
        {
            if (a == b) return true;
            if (a == null || b == null) return false;
            if (a.distance != b.distance) return false;
            if (! PartialNIP.equal_func(a.nip, b.nip)) return false;
            return true;
        }

        public string to_string()
        {
            return @"<$(nip) at $(distance) hops>";
        }
    }

    public class HookReservation : Object, ISerializable
    {
        public PartialNIP gnode;
        public int pos;
        public int elderliness;
        public GNodeID[] gids;
        public CoordinatorKnowledgeSet coordinator_knowledge_set;

        public Variant serialize_to_variant()
        {
            Variant[] _gids = new Variant[gids.length];
            for (int i = 0; i < gids.length; i++) _gids[i] = gids[i].serialize_to_variant();
            Variant v0 = gnode.serialize_to_variant();
            Variant v1 = Serializer.int_to_variant(pos);
            Variant v2 = Serializer.int_to_variant(elderliness);
            Variant v3 = Serializer.variant_array_to_variant(_gids);
            Variant v4 = coordinator_knowledge_set.serialize_to_variant();
            Variant vret = Serializer.tuple_to_variant_5(v0, v1, v2, v3, v4);
            return vret;
        }

        public void deserialize_from_variant(Variant v) throws SerializerError
        {
            Variant v0;
            Variant v1;
            Variant v2;
            Variant v3;
            Variant v4;
            Serializer.variant_to_tuple_5(v, out v0, out v1, out v2, out v3, out v4);
            gnode = (PartialNIP)Object.new(typeof(PartialNIP));
            gnode.deserialize_from_variant(v0);
            pos = Serializer.variant_to_int(v1);
            elderliness = Serializer.variant_to_int(v2);
            Variant[] _gids = Serializer.variant_to_variant_array(v3);
            gids = new GNodeID[_gids.length];
            for (int i = 0; i < _gids.length; i++)
            {
                GNodeID temp = (GNodeID)Object.new(typeof(GNodeID));
                temp.deserialize_from_variant(_gids[i]);
                gids[i] = temp;
            }
            coordinator_knowledge_set = (CoordinatorKnowledgeSet)Object.new(typeof(CoordinatorKnowledgeSet));
            coordinator_knowledge_set.deserialize_from_variant(v4);
        }

        public string to_string()
        {
            string str_gids = "[";
            foreach (GNodeID gid in gids) str_gids += @"$gid,";
            str_gids += "]";
            return @"<HookReservation: gnode $gnode, pos $pos, elderliness $elderliness, gids $str_gids, coordinator_knowledge_set TODO>";
        }
    }

    /** Represents a booking for a position in a given level of the map.
      */
    public class BookingRecord : Object, ISerializable
    {
        public int pos {get; private set;}
        public TimeCapsule tc_expire {get; private set;}
        public BookingRecord(int pos)
        {
            tc_expire = new TimeCapsule(300000); // 5 minutes
            this.pos = pos;
        }

        public Variant serialize_to_variant()
        {
            Variant v0 = Serializer.int_to_variant(pos);
            Variant v1 = tc_expire.serialize_to_variant();
            Variant vret = Serializer.tuple_to_variant(v0, v1);
            return vret;
        }

        public void deserialize_from_variant(Variant v) throws SerializerError
        {
            Variant v0;
            Variant v1;
            Serializer.variant_to_tuple(v, out v0, out v1);
            pos = Serializer.variant_to_int(v0);
            tc_expire = (TimeCapsule)Object.new(typeof(TimeCapsule));
            tc_expire.deserialize_from_variant(v1);
        }

        public static bool equal_func(BookingRecord? a, BookingRecord? b)
        {
            if (a == b) return true;
            if (a == null || b == null) return false;
            if (a.pos != b.pos) return false;
            return true;
        }

        public string to_string()
        {
            return @"<BookingRecord: pos $pos, $(tc_expire.get_string_msec_ttl()) milliseconds>";
        }
    }

    /** Collects the bookings of a certain gnode.
      */
    public class Bookings : Object, ISerializable
    {
        public PartialNIP gnode {get; private set;}
        private Gee.List<BookingRecord> records; // use iterator to access

        public Bookings(PartialNIP gnode)
        {
            /* PartialNIP of the gnode, e.g. with 4 levels: [-1,-1,-1,-1] is the whole network
             * [-1,-1,2,3] is the gnode 3,2
             * [0,1,2,3] is the node 3,2,1,0 which belongs to the gnode 3,2.
             */
            this.gnode = gnode;
            records = new ArrayList<BookingRecord>((EqualDataFunc)BookingRecord.equal_func);
        }

        public Variant serialize_to_variant()
        {
            Variant v0 = gnode.serialize_to_variant();
            ListISerializable lst = new ListISerializable.with_backer(records);
            Variant v1 = lst.serialize_to_variant();
            Variant vret = Serializer.tuple_to_variant(v0, v1);
            return vret;
        }

        public void deserialize_from_variant(Variant v) throws SerializerError
        {
            Variant v0;
            Variant v1;
            Serializer.variant_to_tuple(v, out v0, out v1);
            gnode = (PartialNIP)Object.new(typeof(PartialNIP));
            gnode.deserialize_from_variant(v0);
            ListISerializable lst = (ListISerializable)Object.new(typeof(ListISerializable));
            lst.deserialize_from_variant(v1);
            records = (Gee.List<BookingRecord>)lst.backed;
        }

        private void _purge()
        {
            Gee.List<BookingRecord> ret = new ArrayList<BookingRecord>((EqualDataFunc)BookingRecord.equal_func);
            foreach (BookingRecord r in records)
                if (! r.tc_expire.is_expired())
                    ret.add(r);
            records = ret;
        }

        public Iterator<BookingRecord> iterator()
        {
            _purge();
            ArrayList<BookingRecord> ret = new ArrayList<BookingRecord>((EqualDataFunc)BookingRecord.equal_func);
            ret.add_all(records);
            return ret.iterator();
        }

        public int records_size {
            get {
                _purge();
                return records.size;
            }
        }

        public bool is_booked(int pos)
        {
            _purge();
            foreach (BookingRecord r in records)
                if (r.pos == pos)
                    return true;
            return false;
        }

        public void new_book(int pos)
        {
            assert(! is_booked(pos));
            records.add(new BookingRecord(pos));
        }

        public void force_add(BookingRecord booking_record)
        {
            // we must ensure that there is at most 1 record for pos.
            force_remove(booking_record.pos);
            records.add(booking_record);
        }

        public void force_remove(int pos)
        {
            // we are sure that there is at most 1 record for pos.
            for (int i = 0; i < records.size; i++)
            {
                if (records.get(i).pos == pos)
                {
                    records.remove_at(i);
                    return;
                }
            }
        }

        public string to_string()
        {
            string str_records = "[";
            foreach (BookingRecord record in this) str_records += @"$record,";
            str_records += "]";
            return @"<Bookings: gnode $gnode has $str_records>";
        }

        public Bookings clone()
        {
            try
            {
                return (Bookings)ISerializable.deserialize(serialize());
            }
            catch (SerializerError e)
            {
                error(@"Bookings.clone(): Caught exception while deserializing members $(e.domain) code $(e.code): $(e.message)");
            }
        }
    }

    public class BnodeRecord : Object, ISerializable
    {
        public TimeCapsule tc_expire;
        public NIP nip {get; private set;}
        public bool has_tunnel;
        public bool is_willing;

        public BnodeRecord(NIP nip, bool has_tunnel, bool is_willing)
        {
            tc_expire = new TimeCapsule(300000); // 5 minutes
            this.nip = nip;
            this.has_tunnel = has_tunnel;
            this.is_willing = is_willing;
        }

        public Variant serialize_to_variant()
        {
            // Note: tc_expire is intended not to take part to serialization
            Variant v0 = nip.serialize_to_variant();
            Variant v1 = Serializer.int_to_variant(has_tunnel ? 1 : 0);
            Variant v2 = Serializer.int_to_variant(is_willing ? 1 : 0);
            Variant vret = Serializer.tuple_to_variant_3(v0, v1, v2);
            return vret;
        }

        public void deserialize_from_variant(Variant v) throws SerializerError
        {
            tc_expire = new TimeCapsule(300000); // 5 minutes
            Variant v0;
            Variant v1;
            Variant v2;
            Serializer.variant_to_tuple_3(v, out v0, out v1, out v2);
            nip = (NIP)Object.new(typeof(NIP));
            nip.deserialize_from_variant(v0);
            has_tunnel = Serializer.variant_to_int(v1) == 1;
            is_willing = Serializer.variant_to_int(v2) == 1;
        }

        public static bool equal_func(BnodeRecord? a, BnodeRecord? b)
        {
            if (a == b) return true;
            if (a == null || b == null) return false;
            if (a.has_tunnel != b.has_tunnel) return false;
            if (a.is_willing != b.is_willing) return false;
            if (! PartialNIP.equal_func(a.nip, b.nip)) return false;
            return true;
        }

        public string to_string()
        {
            string ret = @"<BnodeRecord: nip $nip";
            if (has_tunnel) ret += ", has_tunnel";
            if (is_willing) ret += ", is_willing";
            ret += @", $(tc_expire.get_string_msec_ttl()) milliseconds>";
            return ret;
        }
    }

    public const int MAX_BNODES = 50;

    /** Collects the bookings of a certain gnode.
      */
    public class BnodeList : Object, ISerializable
    {
        public PartialNIP gnode {get; private set;}
        public int max_bnodes {get; private set;}
        private Gee.List<BnodeRecord> records; // use iterator to access

        public BnodeList(PartialNIP gnode)
        {
            /* PartialNIP of the gnode, e.g. with 4 levels: [-1,-1,-1,-1] is the whole network
             * [-1,-1,2,3] is the gnode 3,2
             * [0,1,2,3] is the node 3,2,1,0 which belongs to the gnode 3,2.
             */
            this.gnode = gnode;
            max_bnodes = MAX_BNODES;
            records = new ArrayList<BnodeRecord>((EqualDataFunc)BnodeRecord.equal_func);
        }

        public Variant serialize_to_variant()
        {
            Variant v0 = gnode.serialize_to_variant();
            Variant v1 = Serializer.int_to_variant(max_bnodes);
            ListISerializable lst = new ListISerializable.with_backer(records);
            Variant v2 = lst.serialize_to_variant();
            Variant vret = Serializer.tuple_to_variant_3(v0, v1, v2);
            return vret;
        }

        public void deserialize_from_variant(Variant v) throws SerializerError
        {
            Variant v0;
            Variant v1;
            Variant v2;
            Serializer.variant_to_tuple_3(v, out v0, out v1, out v2);
            gnode = (PartialNIP)Object.new(typeof(PartialNIP));
            gnode.deserialize_from_variant(v0);
            max_bnodes = Serializer.variant_to_int(v1);
            ListISerializable lst = (ListISerializable)Object.new(typeof(ListISerializable));
            lst.deserialize_from_variant(v2);
            records = (Gee.List<BnodeRecord>)lst.backed;
        }

        private void _purge()
        {
            Gee.List<BnodeRecord> ret = new ArrayList<BnodeRecord>((EqualDataFunc)BnodeRecord.equal_func);
            foreach (BnodeRecord r in records)
                if (! r.tc_expire.is_expired())
                    ret.add(r);
            records = ret;
        }

        public Iterator<BnodeRecord> iterator()
        {
            _purge();
            ArrayList<BnodeRecord> ret = new ArrayList<BnodeRecord>((EqualDataFunc)BnodeRecord.equal_func);
            ret.add_all(records);
            return ret.iterator();
        }

        public int records_size {
            get {
                _purge();
                return records.size;
            }
        }

        public void delete_bnode(NIP nip)
        {
            for (int i = 0; i < records.size; i++)
            {
                if (records.get(i).nip.is_equal(nip))
                {
                    records.remove_at(i);
                    break;
                }
            }
        }

        public bool update_bnode(NIP nip, bool has_tunnel, bool is_willing)
        {
            bool adding = true;
            for (int i = 0; i < records.size; i++)
            {
                if (records.get(i).nip.is_equal(nip))
                {
                    records.remove_at(i);
                    adding = false;
                    break;
                }
            }
            if (adding && records.size >= max_bnodes)
            {
                // not interested
                return false;
            }
            records.add(new BnodeRecord(nip, has_tunnel, is_willing));
            return true;
        }

        public void copy_bnode(BnodeRecord bnode_record)
        {
            for (int i = 0; i < records.size; i++)
            {
                if (records.get(i).nip.is_equal(bnode_record.nip))
                {
                    records.get(i).tc_expire = bnode_record.tc_expire;
                    records.get(i).has_tunnel = bnode_record.has_tunnel;
                    records.get(i).is_willing = bnode_record.is_willing;
                    return;
                }
            }
            if (records.size < max_bnodes)
            {
                records.add(bnode_record);
            }
        }

        public string to_string()
        {
            string str_records = "[";
            foreach (BnodeRecord record in this) str_records += @"$record,";
            str_records += "]";
            return @"<BnodeList: gnode $gnode has $str_records>";
        }

        public BnodeList clone()
        {
            try
            {
                return (BnodeList)ISerializable.deserialize(serialize());
            }
            catch (SerializerError e)
            {
                error(@"BnodeList.clone(): Caught exception while deserializing members $(e.domain) code $(e.code): $(e.message)");
            }
        }
    }

    /** An instance of this class contains all the kind of knowledge that
      *  a Coordinator of a certain gnode may have.
      */
    public class CoordinatorKnowledge : Object, ISerializable
    {
        public PartialNIP gnode {get; private set;}
        public Bookings bookings {get; private set;}
        public int last_assigned_elderliness {get; private set;}
        public BnodeList bnode_list {get; private set;}

        public CoordinatorKnowledge(PartialNIP gnode,
                                    Bookings bookings,
                                    int last_assigned_elderliness,
                                    BnodeList bnode_list)
        {
            this.gnode = gnode;
            this.bookings = bookings;
            this.last_assigned_elderliness = last_assigned_elderliness;
            this.bnode_list = bnode_list;
        }

        public Variant serialize_to_variant()
        {
            Variant v0 = gnode.serialize_to_variant();
            Variant v1 = bookings.serialize_to_variant();
            Variant v2 = Serializer.int_to_variant(last_assigned_elderliness);
            Variant v3 = bnode_list.serialize_to_variant();
            Variant vret = Serializer.tuple_to_variant_4(v0, v1, v2, v3);
            return vret;
        }

        public void deserialize_from_variant(Variant v) throws SerializerError
        {
            Variant v0;
            Variant v1;
            Variant v2;
            Variant v3;
            Serializer.variant_to_tuple_4(v, out v0, out v1, out v2, out v3);
            gnode = (PartialNIP)Object.new(typeof(PartialNIP));
            gnode.deserialize_from_variant(v0);
            bookings = (Bookings)Object.new(typeof(Bookings));
            bookings.deserialize_from_variant(v1);
            last_assigned_elderliness = Serializer.variant_to_int(v2);
            bnode_list = (BnodeList)Object.new(typeof(BnodeList));
            bnode_list.deserialize_from_variant(v3);
        }
    }

    /** A CoordinatorKnowledgeSet is a dictionary<gnode, CoordinatorKnowledge>
      */
    public class CoordinatorKnowledgeSet : Object, ISerializable
    {
        public HashMap<PartialNIP, CoordinatorKnowledge> dict;
        public CoordinatorKnowledgeSet()
        {
            dict = new HashMap<PartialNIP, CoordinatorKnowledge>((HashDataFunc)PartialNIP.hash_func, (EqualDataFunc)PartialNIP.equal_func);
        }
        public void dict_set(PartialNIP k, CoordinatorKnowledge v)
        {
            dict[k] = v;
        }
        public void dict_unset(PartialNIP k)
        {
            dict.unset(k);
        }

        public Variant serialize_to_variant()
        {
            ListISerializable lk = new ListISerializable();
            ListISerializable lv = new ListISerializable();
            foreach (PartialNIP k in dict.keys)
            {
                lk.add(k);
                lv.add(dict[k]);
            }
            Variant v0 = lk.serialize_to_variant();
            Variant v1 = lv.serialize_to_variant();
            Variant vret = Serializer.tuple_to_variant(v0, v1);
            return vret;
        }

        public void deserialize_from_variant(Variant v) throws SerializerError
        {
            Variant v0;
            Variant v1;
            Serializer.variant_to_tuple(v, out v0, out v1);
            ListISerializable lk = (ListISerializable)Object.new(typeof(ListISerializable));
            lk.deserialize_from_variant(v0);
            ListISerializable lv = (ListISerializable)Object.new(typeof(ListISerializable));
            lv.deserialize_from_variant(v1);
            Gee.List<PartialNIP> typed_lk = (Gee.List<PartialNIP>)lk.backed;
            Gee.List<CoordinatorKnowledge> typed_lv = (Gee.List<CoordinatorKnowledge>)lv.backed;
            dict = new HashMap<PartialNIP, CoordinatorKnowledge>((HashDataFunc)PartialNIP.hash_func, (EqualDataFunc)PartialNIP.equal_func);
            for (int i = 0; i < typed_lk.size; i++)
            {
                dict[typed_lk[i]] = typed_lv[i];
            }
        }
    }

    /** The return value of the method list_non_saturated_levels
      * is a list of tuple (int lvl, int number_of_free_nodes_at_lvl)
      */
    public class PairLvlNumberOfFreeNodes : Object, ISerializable
    {
        public int lvl;
        public int number_of_free_nodes;
        public PairLvlNumberOfFreeNodes(int lvl, int number_of_free_nodes)
        {
            this.lvl = lvl;
            this.number_of_free_nodes = number_of_free_nodes;
        }

        public Variant serialize_to_variant()
        {
            Variant v0 = Serializer.int_to_variant(lvl);
            Variant v1 = Serializer.int_to_variant(number_of_free_nodes);
            Variant vret = Serializer.tuple_to_variant(v0, v1);
            return vret;
        }

        public void deserialize_from_variant(Variant v) throws SerializerError
        {
            Variant v0;
            Variant v1;
            Serializer.variant_to_tuple(v, out v0, out v1);
            lvl = Serializer.variant_to_int(v0);
            number_of_free_nodes = Serializer.variant_to_int(v1);
        }

        public static bool equal_func(PairLvlNumberOfFreeNodes? a, PairLvlNumberOfFreeNodes? b)
        {
            if (a == b) return true;
            if (a == null || b == null) return false;
            if (a.number_of_free_nodes != b.number_of_free_nodes) return false;
            if (a.lvl != b.lvl) return false;
            return true;
        }

        public string to_string()
        {
            return @"<$(number_of_free_nodes) free at level $(lvl)>";
        }
    }

    /** A Hierarchical topology Coordinates.
      * That is, level and position.
      * It is an immutable class.
      */
    public class HCoord : Object, ISerializable
    {
        private int _lvl;
        private int _pos;
        public HCoord(int lvl, int pos)
        {
            this._lvl = lvl;
            this._pos = pos;
        }

        public int lvl {
            get {
                return _lvl;
            }
        }

        public int pos {
            get {
                return _pos;
            }
        }

        public Variant serialize_to_variant()
        {
            Variant v0 = Serializer.int_to_variant(_lvl);
            Variant v1 = Serializer.int_to_variant(_pos);
            Variant vret = Serializer.tuple_to_variant(v0, v1);
            return vret;
        }

        public void deserialize_from_variant(Variant v) throws SerializerError
        {
            Variant v0;
            Variant v1;
            Serializer.variant_to_tuple(v, out v0, out v1);
            _lvl = Serializer.variant_to_int(v0);
            _pos = Serializer.variant_to_int(v1);
        }

        public bool is_equal(HCoord? other)
        {
            return HCoord.equal_func(this, other);
        }

        public static bool equal_func(HCoord? a, HCoord? b)
        {
            if (a == b) return true;
            if (a == null || b == null) return false;
            return a._lvl == b._lvl && a._pos == b._pos;
        }

        public PartialNIP get_partialnip_relative_to(NIP abs)
        {
            int[] nip = abs.get_positions();
            nip[_lvl] = _pos;
            for (int l = _lvl-1; l >= 0; l--)
            {
                nip[l] = -1;
            }
            return new PartialNIP(nip);
        }

        public string to_string()
        {
            return @"($lvl, $pos)";
        }
    }

    /** A Netsukuku IP address that could be partial.
      * That is, the positions at the various levels.
      * It is an immutable class.
      */
    public class PartialNIP : Object, ISerializable
    {
        protected int[] _positions;

        public PartialNIP(int[] positions)
        {
            this._positions = new int[positions.length];
            for (int i = 0; i < positions.length; i++)
                this._positions[i] = positions[i];
            // when one position is equal to -1 (because this object represents
            // a gnode) all the upper ones must be -1
            bool minusfound = false;
            for (int i = positions.length-1; i >= 0; i--)
            {
                assert(positions[i] >= -1);
                if (positions[i] == -1) minusfound = true;
                if (minusfound) assert(positions[i] == -1);
            }
        }

        public int[] get_positions()
        {
            int[] a = new int[_positions.length];
            for (int i = 0; i < _positions.length; i++)
                a[i] = _positions[i];
            return a;
        }

        public int levels {
            get {
                return _positions.length;
            }
        }

        public int position_at(int lvl)
        {
            assert(lvl >= 0);
            assert(lvl < _positions.length);
            return _positions[lvl];
        }

        public virtual Variant serialize_to_variant()
        {
            Variant vret = Serializer.int_array_to_variant(_positions);
            return vret;
        }

        public virtual void deserialize_from_variant(Variant v) throws SerializerError
        {
            _positions = Serializer.variant_to_int_array(v);
        }

        public bool is_equal(PartialNIP? other)
        {
            return PartialNIP.equal_func(this, other);
        }

        public static bool equal_func(PartialNIP? a, PartialNIP? b)
        {
            if (a == b) return true;
            if (a == null || b == null) return false;
            if (a._positions.length != b._positions.length) return false;
            for (int i = 0; i < a._positions.length; i++)
            {
                if (a._positions[i] != b._positions[i]) return false;
            }
            return true;
        }

        public static uint hash_func(PartialNIP a)
        {
            return a.to_string().hash();
        }

        public int compare(PartialNIP other)
        {
            assert(other != null);
            return NIP.nip_cmp(_positions, other._positions);
        }

        public string to_string()
        {
            string ret = "";
            if (get_type() == typeof(PartialNIP)) ret += "G";
            if (get_type() == typeof(NIP)) ret += "N";
            ret += "[";
            string add = "";
            for (int i = _positions.length-1; i >= 0; i--)
            {
                ret += add;
                add = ", ";
                if (_positions[i] == -1)
                {
                    ret += "*";
                    break;
                }
                ret += _positions[i].to_string();
            }
            return ret + "]";
        }

        public uchar[] hash_for_signature()
        {
            uchar[] ret = {};
            string tmp = to_string();
            foreach (uchar u in tmp.data)
                ret += u;
            return ret;
        }

        public PartialNIP get_gnode_at_level(int lvl)
        {
            assert(lvl <= _positions.length);
            assert(lvl >= level_of_gnode());
            int[] positions = get_positions();
            for (int l = 0; l < lvl; l++) positions[l] = -1;
            return new PartialNIP(positions);
        }

        /** This method is useful when this instance represents a g-node.
          */
        public int level_of_gnode()
        {
            for (int i = _positions.length; i > 0; i--)
            {
                if (_positions[i-1] == -1) return i;
            }
            return 0;
        }

        public bool belongs_to(PartialNIP gnode)
        {
            return gnode.is_equal(get_gnode_at_level(gnode.level_of_gnode()));
        }

        /** When this instance represents a g-node, this method returns the lowest NIP inside it.
          * This is useful, for instance, when we need a CIDR notation for a gnode, such as "10.123.0.0/16"
          */
        public NIP lowest_nip()
        {
            int levels = _positions.length;
            int[] full_pos = new int[levels];
            int given_up_to_level = level_of_gnode();
            for (int i = given_up_to_level; i < levels; i++) full_pos[i] = _positions[i];
            for (int l = given_up_to_level-1; l >= 0; l--) full_pos[l] = 0;
            return new NIP(full_pos);
        }

        public NIP to_nip()
        {
            return new NIP(get_positions());
        }

        public HCoord get_hcoord_relative_to(NIP abs)
        {
            int lvl = NIP.nip_cmp(get_positions(), abs.get_positions());
            assert(lvl >= 0);
            int pos = position_at(lvl);
            return new HCoord(lvl, pos);
        }
    }

    /** A Netsukuku IP address.
      * That is, the positions at the various levels.
      * It is an immutable class.
      */
    public class NIP : PartialNIP, ISerializable
    {
        public NIP(int[] positions)
        {
            base(positions);
            for (int i = 0; i < positions.length; i++)
                assert(positions[i] >= 0);
        }

        // is_equal is inherited from PartialNIP

        // We can use PartialNIP.equal_func and hash_func with NIP objects.

        /** Returns the first level where nipA and nipB differs. The search start from the end of the nip.
          */
        public static int nip_cmp(int[] nipA, int[] nipB)
        {
            assert(nipA.length == nipB.length);
            int levels = nipA.length;
            for (int lvl = levels-1; lvl >= 0; lvl--)
                if (nipA[lvl] != nipB[lvl]) return lvl;
            return -1;
        }
    }

    /** Route Efficiency Measure.
      *
      * This is a base class for different metrics (rtt, bandwidth, ...)
      */
    public abstract class REM : Object, ISerializable, Comparable<REM>
    {
        public abstract Variant serialize_to_variant();
        public abstract void deserialize_from_variant(Variant v) throws SerializerError;
        public abstract string to_string();
        
        /** The Comparable interface implemented here is useful when we
          * have a collection of REM objects and we want to sort them, or
          * to find the best or the worst of them.
          *
          * NOTE: a.compare_to(b) < 0   means   a < b
          * NOTE: A REM is a measure of efficiency, so a < b means a is less efficient
          */
        public abstract int compare_to(REM other);
        
        /** This method is used to calculate the efficiency of a path
          * x0...xi...xn when we know the efficiency of the path x0...xi
          * and the efficiency of the path xi...xn
          */
        public abstract REM add_segment(REM other);
    }


    /** Infinity.
      */
    public class NullREM : REM, ISerializable
    {
        public override Variant serialize_to_variant()
        {
            return 0;
        }

        public override void deserialize_from_variant(Variant v) throws SerializerError
        {
        }

        public override string to_string()
        {
            return "<NullREM>";
        }

        public override int compare_to(REM other)
        {
            if (other.get_type() == typeof(NullREM)) return 0;
            return 1;
        }

        public override REM add_segment(REM other)
        {
            return other;
        }
    }

    /** Zero.
      */
    public class DeadREM : REM, ISerializable
    {
        public override Variant serialize_to_variant()
        {
            return 0;
        }

        public override void deserialize_from_variant(Variant v) throws SerializerError
        {
        }

        public override string to_string()
        {
            return "<DeadREM>";
        }

        public override int compare_to(REM other)
        {
            if (other.get_type() == typeof(DeadREM)) return 0;
            return -1;
        }

        public override REM add_segment(REM other)
        {
            return this;
        }
    }

    /** Almost Zero.
      */
    public class AlmostDeadREM : REM, ISerializable
    {
        public override Variant serialize_to_variant()
        {
            return 0;
        }

        public override void deserialize_from_variant(Variant v) throws SerializerError
        {
        }

        public override string to_string()
        {
            return "<AlmostDeadREM>";
        }

        public override int compare_to(REM other)
        {
            if (other.get_type() == typeof(DeadREM)) return 1;
            if (other.get_type() == typeof(AlmostDeadREM)) return 0;
            return -1;
        }

        public override REM add_segment(REM other)
        {
            if (other.get_type() == typeof(DeadREM)) return other;
            return this;
        }
    }

    /** Round Trip Time.
      * This is a measure of latency. This class could well be used to
      * model the delay of the path from A to B. Instead, the current
      * implementation of netsukuku uses it to model the round-trip time
      * of a packet from A to B and back to A.
      */
    public class RTT : REM, ISerializable
    {
        private int _rtt;
        public RTT(int rtt)
        {
            _rtt = rtt;
        }

        public int delay {
            get {
                return _rtt;
            }
        }

        public override Variant serialize_to_variant()
        {
            return _rtt;
        }

        public override void deserialize_from_variant(Variant v) throws SerializerError
        {
            string vt = v.get_type_string();
            assert(vt == "i");
            _rtt = (int)v;
        }

        public override string to_string()
        {
            return @"<RTT $(_rtt) ms>";
        }

        public override int compare_to(REM other)
        {
            if (other.get_type() == typeof(NullREM)) return -1;
            if (other.get_type() == typeof(DeadREM)) return 1;
            if (other.get_type() == typeof(AlmostDeadREM)) return 1;
            if (other.get_type() != typeof(RTT)) return 0;
            RTT other_rtt = (RTT)other;
            if (_rtt > other_rtt._rtt) return -1;
            if (_rtt < other_rtt._rtt) return 1;
            return 0;
        }

        public override REM add_segment(REM other)
        {
            if (other.get_type() == typeof(NullREM)) return this;
            if (other.get_type() == typeof(DeadREM)) return other;
            if (other.get_type() == typeof(AlmostDeadREM)) return other;
            assert(other.get_type() == typeof(RTT));
            RTT other_rtt = (RTT)other;
            return new RTT(_rtt + other_rtt._rtt);
        }
    }

    /** Represents the tracer packet of the path covered until now by an ETP.
      *  This TP may have covered different levels. In general, TPL
      *  is a list of blocks. Each block is a (lvl, TP) pair, where lvl is
      *  the level of the block and TP is the tracer packet composed
      *  during the transit in the level `lvl'.
      *  TP is a list of (hop, rem) pairs.
      */
    public class TracerPacketList : Object, ISerializable
    {
        class TraceElement
        {
            public int hop;
            public REM rem;
            public TraceElement(int hop, REM rem)
            {
                this.hop = hop;
                this.rem = rem;
            }
            public Variant to_variant()
            {
                Variant v0 = Serializer.int_to_variant(hop);
                Variant v1 = Serializer.uchar_array_to_variant(rem.serialize());
                Variant vret = Serializer.tuple_to_variant(v0, v1);
                return vret;
            }
            public static TraceElement from_variant(Variant v) throws SerializerError
            {
                Variant v0;
                Variant v1;
                Serializer.variant_to_tuple(v, out v0, out v1);
                int _hop = Serializer.variant_to_int(v0);
                REM _rem = (REM)ISerializable.deserialize(Serializer.variant_to_uchar_array(v1));
                return new TraceElement(_hop, _rem);
            }
        }

        class Block
        {
            public int lvl;
            public LinkedList<TraceElement?> tp;
            public Block(int lvl, LinkedList<TraceElement?> tp)
            {
                this.lvl = lvl;
                this.tp = tp;
            }
            private Variant to_variant_tp()
            {
                Variant[] _tp = new Variant[tp.size];
                int j = 0;
                foreach (TraceElement x in tp)
                {
                    _tp[j] = x.to_variant();
                    j++;
                }
                return Serializer.variant_array_to_variant(_tp);
            }
            public Variant to_variant()
            {
                Variant v0 = Serializer.int_to_variant(lvl);
                Variant v1 = to_variant_tp();
                Variant vret = Serializer.tuple_to_variant(v0, v1);
                return vret;
            }
            private static LinkedList<TraceElement?> from_variant_tp(Variant v) throws SerializerError
            {
                LinkedList<TraceElement?> ret = new LinkedList<TraceElement?>();
                Variant[] _tp = Serializer.variant_to_variant_array(v);
                foreach (Variant v_x in _tp)
                {
                    TraceElement x = TraceElement.from_variant(v_x);
                    ret.offer_tail(x);
                }
                return ret;
            }
            public static Block from_variant(Variant v) throws SerializerError
            {
                Variant v0;
                Variant v1;
                Serializer.variant_to_tuple(v, out v0, out v1);
                int _lvl = Serializer.variant_to_int(v0);
                LinkedList<TraceElement?> _tp = from_variant_tp(v1);
                return new Block(_lvl, _tp);
            }
        }

        private LinkedList<Block?> tpl = new LinkedList<Block?>();

        public void append(int zero_pos, REM rem)
        {
            if (tpl.is_empty)
            {
                LinkedList<TraceElement?> tp = new LinkedList<TraceElement?>();
                tp.offer_tail(new TraceElement(zero_pos, rem));
                tpl.offer_tail(new Block(0, tp));
            }
            else
            {
                if (tpl.peek_tail().lvl != 0)
                {
                    // The last block isn't of level 0. Let's add a new block
                    LinkedList<TraceElement?> tp = new LinkedList<TraceElement?>();
                    tp.offer_tail(new TraceElement(zero_pos, rem));
                    tpl.offer_tail(new Block(0, tp));
                }
                else
                {
                    // The last block is of level 0. We can append our ID
                    tpl.peek_tail().tp.offer_tail(new TraceElement(zero_pos, rem));
                }
            }
        }

        public void group(int level, int pos)
        {
            foreach (Block block in tpl)
            {
                if (block.lvl < level)
                {
                    block.lvl = level;
                    REM blockrem = new NullREM();
                    foreach (TraceElement hoprem in block.tp)
                        blockrem = blockrem.add_segment(hoprem.rem);
                    block.tp = new LinkedList<TraceElement?>();
                    block.tp.offer_tail(new TraceElement(pos, blockrem));
                }
            }

            // Collapse groups
            LinkedList<Block?> tpl2 = new LinkedList<Block?>();
            if (tpl.size > 0) tpl2.offer_tail(tpl.get(0));
            if (tpl.size > 1) foreach (Block block in tpl.slice(1, tpl.size))
            {
                if (block.lvl == tpl2.peek_tail().lvl)
                    foreach (TraceElement x in block.tp)
                        tpl2.peek_tail().tp.offer_tail(x);
                else
                    tpl2.offer_tail(block);
            }
            tpl = tpl2;

            // Remove dups
            foreach (Block block in tpl)
            {
                LinkedList<TraceElement?> tp2 = new LinkedList<TraceElement?>();
                TraceElement prec = new TraceElement(-1, new NullREM());
                foreach (TraceElement x in block.tp)
                {
                    if (x.hop != prec.hop)
                    {
                        prec = x;
                        tp2.offer_tail(x);
                    }
                    else
                        prec.rem = prec.rem.add_segment(x.rem);
                }
                block.tp = tp2;
            }
            // The rem of the first block is useless.
            tpl.peek_head().tp.peek_head().rem = new NullREM();
        }

        /** Tests if this TPL already contains me.
          * The parameter nip has to be the NIP of this very node.
          * The TPL is already in the form pertinent to this node.
          */
        public bool contains(NIP nip)
        {
            foreach (Block block in tpl)
                foreach (TraceElement x in block.tp)
                    if (nip.position_at(block.lvl) == x.hop)
                        return true;
            return false;
        }

        public TracerPacketList clone()
        {
            try
            {
                return (TracerPacketList)ISerializable.deserialize(serialize());
            }
            catch (SerializerError e)
            {
                error(@"TracerPacketList.clone: Caught exception while deserializing members $(e.domain) code $(e.code): $(e.message)");
            }
        }

        public string to_string()
        {
            if (tpl.is_empty) return "<TPL: empty>";
            string ret = "<TPL: ";
            string coma = "";
            foreach (Block block in tpl)
            {
                ret += @"$(coma)level $(block.lvl) traversed ";
                coma = ", ";
                foreach (TraceElement x in block.tp)
                    ret += @"($(x.hop), $(x.rem))";
            }
            return ret + ">";
        }

        public Variant serialize_to_variant()
        {
            Variant[] _tpl = new Variant[tpl.size];
            int i = 0;
            foreach (Block block in tpl)
            {
                _tpl[i] = block.to_variant();
                i++;
            }
            return Serializer.variant_array_to_variant(_tpl);
        }

        public void deserialize_from_variant(Variant v) throws SerializerError
        {
            Variant[] _tpl = Serializer.variant_to_variant_array(v);
            tpl = new LinkedList<Block?>();
            foreach (Variant v0 in _tpl)
            {
                tpl.offer_tail(Block.from_variant(v0));
            }
        }
    }

    /** Represents a route to a given destination in a given level. Serializable.
      */
    public class RouteInSet : Object, ISerializable
    {
        public REM rem {get; private set;}
        public Gee.List<HCoord> hops {get; private set;}
        public GNodeID? gid {get; private set;}
        public RouteInSet(REM rem, Gee.List<HCoord> hops, GNodeID? gid)
        {
            this.rem = rem;
            this.hops = hops;
            this.gid = gid;
        }

        public static bool equal_func(RouteInSet a, RouteInSet b)
        {
            if (a == b) return true;
            if (a == null || b == null) return false;
            if (a.hops.size != b.hops.size) return false;
            ListIterator<HCoord> a_hops_it = a.hops.list_iterator();
            bool hascur = a_hops_it.next();
            ListIterator<HCoord> b_hops_it = b.hops.list_iterator();
            b_hops_it.next();
            while (hascur)
            {
                if (! HCoord.equal_func(a_hops_it.get(), b_hops_it.get())) return false;
                hascur = a_hops_it.next();
                b_hops_it.next();
            }
            if (a.rem.compare_to(b.rem) != 0) return false;
            if (! GNodeID.equal_func(a.gid, b.gid)) return false;
            return true;
        }
        public bool equals(RouteInSet other)
        {
            return RouteInSet.equal_func(this, other);
        }

        public string to_string()
        {
            if (rem.get_type() == typeof(DeadREM)) return "DEAD";
            string ret = @"$(rem), $(hops.size) hops";
            if (gid != null) ret += @", $(gid)";
            return @"<$(ret)>";
        }

        public Variant serialize_to_variant()
        {
            Variant v0 = Serializer.uchar_array_to_variant(rem.serialize());
            ListISerializable lst = new ListISerializable.with_backer(hops);
            Variant v1 = lst.serialize_to_variant();
            Variant vret = null;
            if (gid == null)
            {
                vret = Serializer.tuple_to_variant(v0, v1);
            }
            else
            {
                Variant v2 = gid.serialize_to_variant();
                vret = Serializer.tuple_to_variant_3(v0, v1, v2);
            }
            return vret;
        }

        public void deserialize_from_variant(Variant v) throws SerializerError
        {
            Variant v0;
            Variant v1;
            Variant v2;
            if (v.get_type_string() == "(vvv)")
            {
                Serializer.variant_to_tuple_3(v, out v0, out v1, out v2);
                gid = (GNodeID)Object.new(typeof(GNodeID));
                gid.deserialize_from_variant(v2);
            }
            else
            {
                Serializer.variant_to_tuple(v, out v0, out v1);
                gid = null;
            }
            rem = (REM)ISerializable.deserialize(Serializer.variant_to_uchar_array(v0));
            ListISerializable lst = (ListISerializable)Object.new(typeof(ListISerializable));
            lst.deserialize_from_variant(v1);
            hops = (Gee.List<HCoord>)lst.backed;
        }
    }

    /** Useful for serialization of RoutesSetPerLevel
      */
    public class PositionInRoutesSetPerLevel : Object, ISerializable
    {
        public int dst {get; private set;}
        public RouteInSet ris {get; private set;}
        public PositionInRoutesSetPerLevel(int dst, RouteInSet ris)
        {
            this.dst = dst;
            this.ris = ris;
        }

        public Variant serialize_to_variant()
        {
            Variant v0 = Serializer.int_to_variant(dst);
            Variant v1 = ris.serialize_to_variant();
            Variant vret = Serializer.tuple_to_variant(v0, v1);
            return vret;
        }

        public void deserialize_from_variant(Variant v) throws SerializerError
        {
            Variant v0;
            Variant v1;
            Serializer.variant_to_tuple(v, out v0, out v1);
            dst = Serializer.variant_to_int(v0);
            ris = (RouteInSet)Object.new(typeof(RouteInSet));
            ris.deserialize_from_variant(v1);
        }
    }

    /** Contains routes for a given level.
      * It is serializable.
      * It is iterable and gives the destinations (int).
      * When you have a destination you use get_value.
      */
    public class RoutesSetPerLevel : Object, ISerializable
    {
        private HashMap<int, RouteInSet> routes;

        public RoutesSetPerLevel()
        {
            routes = new HashMap<int, RouteInSet>();
        }

        public void add_replace(int dst, REM rem, Gee.List<HCoord> hops, GNodeID? gid)
        {
            if (routes.has_key(dst)) routes.unset(dst);
            routes[dst] = new RouteInSet(rem, hops, gid);
        }

        public void add_replace_no_route(int dst)
        {
            REM rem = new DeadREM();
            Gee.List<HCoord> hops = new ArrayList<HCoord>((EqualDataFunc)HCoord.equal_func);
            GNodeID? gid = null;
            add_replace(dst, rem, hops, gid);
        }

        public RouteInSet get_value(int dst)
        {
            assert(routes.has_key(dst)); // Use iterator or has_key before.
            return routes[dst];
        }

        public bool is_empty {
            get {
                return routes.is_empty;
            }
        }

        public Iterator<int> iterator()
        {
            return routes.keys.iterator();
        }

        public bool has_key(int dst)
        {
            return routes.has_key(dst);
        }

        public string to_string()
        {
            string ret = "";
            string next = "";
            foreach (int pos in this)
            {
                ret += next + @"$(pos)=$(routes[pos])";
                next = ", ";
            }
            return @"<$(ret)>";
        }

        public Variant serialize_to_variant()
        {
            ListISerializable lst = new ListISerializable();
            foreach (int dst in routes.keys) lst.add(new PositionInRoutesSetPerLevel(dst, routes[dst]));
            return lst.serialize_to_variant();
        }

        public void deserialize_from_variant(Variant v) throws SerializerError
        {
            routes = new HashMap<int, RouteInSet>();
            ListISerializable lst = (ListISerializable)Object.new(typeof(ListISerializable));
            lst.deserialize_from_variant(v);
            Gee.List<PositionInRoutesSetPerLevel> typed_lst = (Gee.List<PositionInRoutesSetPerLevel>)lst.backed;
            foreach (PositionInRoutesSetPerLevel psn in typed_lst) routes[psn.dst] = psn.ris;
        }
    }

    /** Represents a portion of the map_route of a node.
      */
    public class RoutesSet : Object, ISerializable
    {
        public RoutesSetPerLevel[] per_level;

        public RoutesSet(int levels)
        {
            per_level = new RoutesSetPerLevel[levels];
            for (int i = 0; i < levels; i++) per_level[i] = new RoutesSetPerLevel();
        }

        public void add_replace(int lvl, int dst, REM rem, Gee.List<HCoord> hops, GNodeID? gid)
        {
            assert(lvl < per_level.length);
            per_level[lvl].add_replace(dst, rem, hops, gid);
        }

        public void add_replace_no_route(int lvl, int dst)
        {
            assert(lvl < per_level.length);
            per_level[lvl].add_replace_no_route(dst);
        }

        public RouteInSet get_value(int lvl, int dst)
        {
            return per_level[lvl].get_value(dst);
        }

        public bool is_empty {
            get {
                foreach (RoutesSetPerLevel rspl in per_level)
                    if (! rspl.is_empty) return false;
                return true;
            }
        }

        public string to_string()
        {
            int levels = per_level.length;
            string ret = "";
            string next = "";
            for (int i = 0; i < levels; i++)
            {
                ret += next + @"lvl $(i) $(per_level[i])";
                next = ", ";
            }
            return @"<R: $(ret)>";
        }

        public Variant serialize_to_variant()
        {
            int levels = per_level.length;
            assert(levels > 0);
            Variant[] var_params = new Variant[levels];
            for (int i = 0; i < levels; i++)
                var_params[i] = per_level[i].serialize_to_variant();
            Variant v = Serializer.variant_array_to_variant(var_params);
            return v;
        }

        public void deserialize_from_variant(Variant v) throws SerializerError
        {
            Variant[] var_params = Serializer.variant_to_variant_array(v);
            int levels = var_params.length;
            per_level = new RoutesSetPerLevel[levels];
            for (int i = 0; i < levels; i++)
            {
                RoutesSetPerLevel rspl = (RoutesSetPerLevel)Object.new(typeof(RoutesSetPerLevel));
                rspl.deserialize_from_variant(var_params[i]);
                per_level[i] = rspl;
            }
        }
    }

    /** The return value of request_etp, contains R, TPL and sequential_number
      */
    public class ExtendedTracerPacket : Object, ISerializable
    {
        public RoutesSet r;
        public TracerPacketList tpl;
        public int seq_num;

        public string to_string()
        {
            return @"<ETP: $r, $tpl, $seq_num>";
        }

        public Variant serialize_to_variant()
        {
            Variant v0 = Serializer.int_to_variant(seq_num);
            Variant v1 = r.serialize_to_variant();
            Variant v2 = tpl.serialize_to_variant();
            Variant vret = Serializer.tuple_to_variant_3(v0, v1, v2);
            return vret;
        }

        public void deserialize_from_variant(Variant v) throws SerializerError
        {
            Variant v0;
            Variant v1;
            Variant v2;
            Serializer.variant_to_tuple_3(v, out v0, out v1, out v2);
            seq_num = Serializer.variant_to_int(v0);
            r = (RoutesSet)Object.new(typeof(RoutesSet));
            r.deserialize_from_variant(v1);
            tpl = (TracerPacketList)Object.new(typeof(TracerPacketList));
            tpl.deserialize_from_variant(v2);
        }
    }

    /** GNode I.D.
      * An instance is valid when we have all three values:
      *  #1 (elderliness) is the elderliness of the gnode inside its upper level gnode.
      *  #2 (eldest_internal_elderliness) is the elderliness of the eldest gnode inside me.
      *  #3 (eldest_internal_id) is the ID of the eldest gnode inside me. And it's also my gnode's ID.
      *
      * In more details:
      *  . for a gnode of level 0, values #2 has no meaning (will always be 0)
      *  . for a gnode of level 'levels', values #1 and #2 have no meaning. Its value #3 is the netid.
      */
    public class GNodeID : Object, ISerializable
    {
        public int elderliness {get; private set;}
        public int eldest_internal_elderliness {get; private set;}
        public int eldest_internal_id {get; private set;}
        public GNodeID(int elderliness, int eldest_internal_elderliness, int eldest_internal_id)
        {
            this.elderliness = elderliness;
            this.eldest_internal_elderliness = eldest_internal_elderliness;
            this.eldest_internal_id = eldest_internal_id;
        }

        public int ident {
            get {
                return eldest_internal_id;
            }
        }

        /** The elderliness of a gnode inside its upper level will never change.
          * The values #2 and #3 might change.
          */
        public bool change_values(GNodeID eldest_internal)
        {
            var new_eldest_internal_elderliness = eldest_internal.elderliness;
            var new_eldest_internal_id = eldest_internal.ident;
            bool ret = eldest_internal_elderliness != new_eldest_internal_elderliness;
            // TODO check: is it right that the control is done only with the elderliness?
            eldest_internal_elderliness = new_eldest_internal_elderliness;
            eldest_internal_id = new_eldest_internal_id;
            return ret;
        }

        /** This property is used as the key for a comparison when a node recalculates its gids.
          * The comparison is between 2 different gnodes of the same level. The one which results lower
          *  is the elder inside the upper level, so that the first one will pass its values #1 and #3
          *  to the upper level as values #2 and #3.
          */
        public int min_will_bubble {
            get {
                return elderliness;
            }
        }

        /** If we just want to know the one that "will bubble" we could do this
          */
        public GNodeID will_bubble(GNodeID? other)
        {
            if (other == null) return this;
            if (other.min_will_bubble >= this.min_will_bubble) return this;
            return other;
        }

        /** This property is used as the key for a comparison when a node verifies if the gnode has splitted (see route.get_eldest_gid()).
          * The comparison is between 2 instances that represent the same gnode. The one which results lower
          *  is the one which contains the elder gnode inside me, so that the first will not rehook.
          */
        public int min_will_not_rehook {
            get {
                return eldest_internal_elderliness;
            }
        }

        /** If we just want to know the one that "will not_rehook" we could do this
          */
        public GNodeID will_not_rehook(GNodeID? other)
        {
            if (other == null) return this;
            if (other.min_will_not_rehook >= this.min_will_not_rehook) return this;
            return other;
        }

        public static bool equal_func(GNodeID? a, GNodeID? b)
        {
            if (a == b) return true;
            if (a == null || b == null) return false;
            return a.elderliness == b.elderliness
                    && a.eldest_internal_elderliness == b.eldest_internal_elderliness
                    && a.eldest_internal_id == b.eldest_internal_id;
        }
        public bool equals(GNodeID? other)
        {
            return GNodeID.equal_func(this, other);
        }

        public string to_string()
        {
            return @"<GNodeID: $elderliness, $eldest_internal_elderliness, $eldest_internal_id>";
        }

        public Variant serialize_to_variant()
        {
            Variant v0 = Serializer.int_to_variant(elderliness);
            Variant v1 = Serializer.int_to_variant(eldest_internal_elderliness);
            Variant v2 = Serializer.int_to_variant(eldest_internal_id);
            Variant vret = Serializer.tuple_to_variant_3(v0, v1, v2);
            return vret;
        }

        public void deserialize_from_variant(Variant v) throws SerializerError
        {
            Variant v0;
            Variant v1;
            Variant v2;
            Serializer.variant_to_tuple_3(v, out v0, out v1, out v2);
            elderliness = Serializer.variant_to_int(v0);
            eldest_internal_elderliness = Serializer.variant_to_int(v1);
            eldest_internal_id = Serializer.variant_to_int(v2);
        }
    }

    /** Network I.D.
      * It is a set of GNodeid of uppermost level
      */
    public class NetworkID : Object, ISerializable
    {
        private Gee.List<GNodeID> gnodeids;
        public NetworkID(Gee.List<GNodeID> gnodeids)
        {
            this.gnodeids = new ArrayList<GNodeID>((EqualDataFunc)GNodeID.equal_func);
            this.gnodeids.add_all(gnodeids);
        }

        public Variant serialize_to_variant()
        {
            ListISerializable lst = new ListISerializable.with_backer(gnodeids);
            return lst.serialize_to_variant();
        }

        public void deserialize_from_variant(Variant v) throws SerializerError
        {
            ListISerializable lst = (ListISerializable)Object.new(typeof(ListISerializable));
            lst.deserialize_from_variant(v);
            gnodeids = (Gee.List<GNodeID>)lst.backed;
        }

        /** Compares two Networkid to see if they belong to the same network.
          *  True iff some items match. It is not needed to look for the "will_bubble"
          *  one; that would even be a risk in the early stage of a hook.
          */
        public bool is_same_network(NetworkID other)
        {
            if (other == null) return false;
            foreach (GNodeID gid_me in this.gnodeids)
                foreach (GNodeID gid_other in other.gnodeids)
                    if (gid_me.ident == gid_other.ident)
                        return true;
            return false;
        }

        /** When 2 NetworkID are not on the same network use this method
          * to see which one has to be preferred (the other one will rehook)
          */
        public bool is_preferred_over(NetworkID other)
        {
            if (other == null) return true;
            assert(!is_same_network(other));
            GNodeID bubble_me = null;
            foreach (GNodeID gnme in this.gnodeids)
                bubble_me = gnme.will_bubble(bubble_me);
            GNodeID bubble_other = null;
            foreach (GNodeID gnother in other.gnodeids)
                bubble_other = gnother.will_bubble(bubble_other);
            // Here we can choose to use ">" or "<".
            // Let's say that the major is the preferred.
            return bubble_me.ident > bubble_other.ident;
        }

        /** Use to see iff a netid has changed
          */
        public bool is_strictly_equal(NetworkID other)
        {
            if (other == null) return false;
            if (this.gnodeids.size != other.gnodeids.size) return false;
            ListIterator<GNodeID> this_it = this.gnodeids.list_iterator();
            bool hascur = this_it.next();
            ListIterator<GNodeID> other_it = other.gnodeids.list_iterator();
            other_it.next();
            while (hascur)
            {
                GNodeID gn_this = this_it.get();
                GNodeID gn_other = other_it.get();
                if (gn_this.elderliness != gn_other.elderliness) return false;
                if (gn_this.ident != gn_other.ident) return false;
                hascur = this_it.next();
                other_it.next();
            }
            return true;
        }

        public string to_string()
        {
            string ret = "Empty Networkid";
            if (gnodeids != null && gnodeids.size > 0)
            {
                ret = "Networkid:{";
                string next = "";
                foreach (GNodeID gnme in gnodeids)
                {
                    ret += next + gnme.to_string();
                    next = ",";
                }
                ret += "}";
            }
            return ret;
        }
    }

    /** Information about a neighbour node
      */
    public class InfoNeighbour : Object, ISerializable
    {
        public NetworkID net {get; private set;}
        public int levels {get; private set;}
        public int gsize {get; private set;}
        public NIP nip {get; private set;}
        public InfoNeighbour(NetworkID net, int levels, int gsize, NIP nip)
        {
            this.net = net;
            this.levels = levels;
            this.gsize = gsize;
            this.nip = nip;
        }

        public Variant serialize_to_variant()
        {
            Variant v0 = net.serialize_to_variant();
            Variant v1 = Serializer.int_to_variant(levels);
            Variant v2 = Serializer.int_to_variant(gsize);
            Variant v3 = nip.serialize_to_variant();
            Variant vret = Serializer.tuple_to_variant_4(v0, v1, v2, v3);
            return vret;
        }

        public void deserialize_from_variant(Variant v) throws SerializerError
        {
            Variant v0;
            Variant v1;
            Variant v2;
            Variant v3;
            Serializer.variant_to_tuple_4(v, out v0, out v1, out v2, out v3);
            net = (NetworkID)Object.new(typeof(NetworkID));
            net.deserialize_from_variant(v0);
            levels = Serializer.variant_to_int(v1);
            gsize = Serializer.variant_to_int(v2);
            nip = (NIP)Object.new(typeof(NIP));
            nip.deserialize_from_variant(v3);
        }
    }

    /** Information about a Route
      */
    public class InfoRoute : Object, ISerializable
    {
        public HCoord dest {get; private set;}
        public GNodeID gid {get; private set;}
        public NIP gwnip {get; private set;}
        public string gwdev {get; private set;}
        public Gee.List<HCoord> hops {get; private set;}
        public REM rem {get; private set;}
        public InfoRoute(HCoord dest, GNodeID gid, NIP gwnip, string gwdev, Gee.List<HCoord> hops, REM rem)
        {
            this.dest = dest;
            this.gid = gid;
            this.gwnip = gwnip;
            this.gwdev = gwdev;
            this.hops = new ArrayList<HCoord>((EqualDataFunc)HCoord.equal_func);
            this.hops.add_all(hops);
            this.rem = rem;
        }

        public Variant serialize_to_variant()
        {
            Variant v0 = dest.serialize_to_variant();
            Variant v1 = gid.serialize_to_variant();
            Variant v2 = gwnip.serialize_to_variant();
            Variant v3 = Serializer.string_to_variant(gwdev);
            ListISerializable lst_hops = new ListISerializable.with_backer(hops);
            Variant v4 = lst_hops.serialize_to_variant();
            Variant v5 = Serializer.uchar_array_to_variant(rem.serialize());
            Variant valpha = Serializer.tuple_to_variant_5(v0, v1, v2, v3, v4);
            Variant vret = Serializer.tuple_to_variant(valpha, v5);
            return vret;
        }

        public void deserialize_from_variant(Variant v) throws SerializerError
        {
            Variant v0;
            Variant v1;
            Variant v2;
            Variant v3;
            Variant v4;
            Variant v5;
            Variant valpha;
            Serializer.variant_to_tuple(v, out valpha, out v5);
            Serializer.variant_to_tuple_5(valpha, out v0, out v1, out v2, out v3, out v4);
            dest = (HCoord)Object.new(typeof(HCoord));
            dest.deserialize_from_variant(v0);
            gid = (GNodeID)Object.new(typeof(GNodeID));
            gid.deserialize_from_variant(v1);
            gwnip = (NIP)Object.new(typeof(NIP));
            gwnip.deserialize_from_variant(v2);
            gwdev = Serializer.variant_to_string(v3);
            ListISerializable lst_hops = (ListISerializable)Object.new(typeof(ListISerializable));
            lst_hops.deserialize_from_variant(v4);
            hops = (Gee.List<HCoord>)lst_hops.backed;
            rem = (REM)ISerializable.deserialize(Serializer.variant_to_uchar_array(v5));
        }
    }

    /** Basic information about this node
      */
    public class InfoNode : Object, ISerializable
    {
        public NetworkID net;
        public NIP nip;
        public PublicKey pubk;
        public int levels;
        public int gsize;
        public InfoNode(NetworkID net, NIP nip, PublicKey pubk, int levels, int gsize)
        {
            this.net = net;
            this.nip = nip;
            this.pubk = pubk;
            this.levels = levels;
            this.gsize = gsize;
        }

        public Variant serialize_to_variant()
        {
            Variant v0 = net.serialize_to_variant();
            Variant v1 = nip.serialize_to_variant();
            Variant v2 = pubk.serialize_to_variant();
            Variant v3 = Serializer.int_to_variant(levels);
            Variant v4 = Serializer.int_to_variant(gsize);
            Variant vret = Serializer.tuple_to_variant_5(v0, v1, v2, v3, v4);
            return vret;
        }

        public void deserialize_from_variant(Variant v) throws SerializerError
        {
            Variant v0;
            Variant v1;
            Variant v2;
            Variant v3;
            Variant v4;
            Serializer.variant_to_tuple_5(v, out v0, out v1, out v2, out v3, out v4);
            net = (NetworkID)Object.new(typeof(NetworkID));
            net.deserialize_from_variant(v0);
            nip = (NIP)Object.new(typeof(NIP));
            nip.deserialize_from_variant(v1);
            pubk = (PublicKey)Object.new(typeof(PublicKey));
            pubk.deserialize_from_variant(v2);
            levels = Serializer.variant_to_int(v3);
            gsize = Serializer.variant_to_int(v4);
        }
    }

    /** Statistics of tasklet usage
      */
    public class TaskletStats : Tasklets.Stat, ISerializable
    {
        /*Inherited
        public int id;
        public int parent;
        public string funcname = "";
        public Tasklets.Status status;
        public string crash_message = "";
        */
        public TimeCapsule? tasklet_started = null;
        public TimeCapsule? tasklet_ended = null;

        public Variant serialize_to_variant()
        {
            Variant v0 = Serializer.int_to_variant(id);
            Variant v1 = Serializer.int_to_variant(parent);
            Variant v2 = Serializer.string_to_variant(funcname);

            string str_status = "";
            if (status == Tasklets.Status.SPAWNED) str_status = "SPAWNED";
            if (status == Tasklets.Status.STARTED) str_status = "STARTED";
            if (status == Tasklets.Status.CRASHED) str_status = "CRASHED";
            if (status == Tasklets.Status.ENDED) str_status = "ENDED";
            if (status == Tasklets.Status.ABORTED) str_status = "ABORTED";
            Variant v3 = Serializer.string_to_variant(str_status);

            Variant v4;
            if (tasklet_started == null) v4 = Serializer.int_to_variant(0);
            else v4 = tasklet_started.serialize_to_variant();

            Variant v5;
            if (tasklet_ended == null) v5 = Serializer.int_to_variant(0);
            else v5 = tasklet_ended.serialize_to_variant();

            Variant v6 = Serializer.string_to_variant(crash_message);

            Variant vtemp = Serializer.tuple_to_variant_5(v0, v1, v2, v3, v4);
            Variant vret = Serializer.tuple_to_variant_3(vtemp, v5, v6);
            return vret;
        }

        public void deserialize_from_variant(Variant v) throws SerializerError
        {
            Variant v0;
            Variant v1;
            Variant v2;
            Variant v3;
            Variant v4;
            Variant v5;
            Variant v6;
            Variant vtemp;
            Serializer.variant_to_tuple_3(v, out vtemp, out v5, out v6);
            Serializer.variant_to_tuple_5(vtemp, out v0, out v1, out v2, out v3, out v4);

            id = Serializer.variant_to_int(v0);
            parent = Serializer.variant_to_int(v1);
            funcname = Serializer.variant_to_string(v2);

            string str_status = Serializer.variant_to_string(v3);
            if (str_status == "SPAWNED") status = Tasklets.Status.SPAWNED;
            if (str_status == "STARTED") status = Tasklets.Status.STARTED;
            if (str_status == "CRASHED") status = Tasklets.Status.CRASHED;
            if (str_status == "ENDED") status = Tasklets.Status.ENDED;
            if (str_status == "ABORTED") status = Tasklets.Status.ABORTED;

            string vt4 = v4.get_type_string();
            if (vt4 == "i") tasklet_started = null;
            else
            {
                tasklet_started = (TimeCapsule)Object.new(typeof(TimeCapsule));
                tasklet_started.deserialize_from_variant(v4);
            }

            string vt5 = v5.get_type_string();
            if (vt5 == "i") tasklet_ended = null;
            else
            {
                tasklet_ended = (TimeCapsule)Object.new(typeof(TimeCapsule));
                tasklet_ended.deserialize_from_variant(v5);
            }

            crash_message = Serializer.variant_to_string(v6);
        }
    }

    /** Statistics of QSPN
      */
    public class QspnStats : Object, ISerializable
    {
        public int etp_in {get; private set;}
        public int etp_out {get; private set;}
        public bool is_mature {get; private set;}
        public QspnStats(int etp_in, int etp_out, bool is_mature)
        {
            this.etp_in = etp_in;
            this.etp_out = etp_out;
            this.is_mature = is_mature;
        }

        public Variant serialize_to_variant()
        {
            Variant v0 = Serializer.int_to_variant(etp_in);
            Variant v1 = Serializer.int_to_variant(etp_out);
            Variant v2 = Serializer.int_to_variant(is_mature ? 1 : 0);
            Variant vret = Serializer.tuple_to_variant_3(v0, v1, v2);
            return vret;
        }

        public void deserialize_from_variant(Variant v) throws SerializerError
        {
            Variant v0;
            Variant v1;
            Variant v2;
            Serializer.variant_to_tuple_3(v, out v0, out v1, out v2);
            etp_in = Serializer.variant_to_int(v0);
            etp_out = Serializer.variant_to_int(v1);
            is_mature = Serializer.variant_to_int(v2) == 1;
        }
    }

    /** Information from BorderNodesManager
      */
    public class InfoBorderNode : Object, ISerializable
    {
        public PartialNIP gnode;
        public int number_messages;
        public bool has_tunnel;
        public bool is_willing;

        public Variant serialize_to_variant()
        {
            Variant v0 = gnode.serialize_to_variant();
            Variant v1 = Serializer.int_to_variant(number_messages);
            Variant v2 = Serializer.int_to_variant(has_tunnel ? 1 : 0);
            Variant v3 = Serializer.int_to_variant(is_willing ? 1 : 0);
            Variant vret = Serializer.tuple_to_variant_4(v0, v1, v2, v3);
            return vret;
        }

        public void deserialize_from_variant(Variant v) throws SerializerError
        {
            Variant v0;
            Variant v1;
            Variant v2;
            Variant v3;
            Serializer.variant_to_tuple_4(v, out v0, out v1, out v2, out v3);
            gnode = (PartialNIP)Object.new(typeof(PartialNIP));
            gnode.deserialize_from_variant(v0);
            number_messages = Serializer.variant_to_int(v1);
            has_tunnel = Serializer.variant_to_int(v2) == 1;
            is_willing = Serializer.variant_to_int(v3) == 1;
        }
    }

    /** Information from Coordinator module
      */
    public class InfoCoord : Object, ISerializable
    {
        public int level;
        public CoordinatorKnowledge knowledge;
        public bool is_main;
        public PartialNIP actual_coord;

        public Variant serialize_to_variant()
        {
            Variant v0 = Serializer.int_to_variant(level);
            Variant v1 = knowledge.serialize_to_variant();
            Variant v2 = Serializer.int_to_variant(is_main ? 1 : 0);
            Variant v3 = actual_coord.serialize_to_variant();
            Variant vret = Serializer.tuple_to_variant_4(v0, v1, v2, v3);
            return vret;
        }

        public void deserialize_from_variant(Variant v) throws SerializerError
        {
            Variant v0;
            Variant v1;
            Variant v2;
            Variant v3;
            Serializer.variant_to_tuple_4(v, out v0, out v1, out v2, out v3);
            level = Serializer.variant_to_int(v0);
            knowledge = (CoordinatorKnowledge)Object.new(typeof(CoordinatorKnowledge));
            knowledge.deserialize_from_variant(v1);
            is_main = Serializer.variant_to_int(v2) == 1;
            actual_coord = (PartialNIP)Object.new(typeof(PartialNIP));
            actual_coord.deserialize_from_variant(v3);
        }
    }

    /** Information from Andna module
      */
    public class InfoAndna : Object, ISerializable
    {
        public string pubk;
        public bool register_ongoing;
        public bool will_participate;
        public bool participating;
        public bool hooked;
        public ArrayList<InfoAndnaCache> cache {get; private set;}
        public ArrayList<InfoAndnaRegistration> registrations {get; private set;}
        public InfoAndna()
        {
            cache = new ArrayList<InfoAndnaCache>
                        (InfoAndnaCache.equal_func);
            registrations = new ArrayList<InfoAndnaRegistration>
                                (InfoAndnaRegistration.equal_func);
        }

        public Variant serialize_to_variant()
        {
            Variant v0 = Serializer.int_to_variant(register_ongoing ? 1 : 0);
            Variant v1 = Serializer.int_to_variant(will_participate ? 1 : 0);
            Variant v2 = Serializer.int_to_variant(participating ? 1 : 0);
            Variant v3 = Serializer.int_to_variant(hooked ? 1 : 0);
            Variant v4 = Serializer.string_to_variant(pubk);
            ListISerializable lst_cache = new ListISerializable.with_backer(cache);
            Variant v5 = lst_cache.serialize_to_variant();
            ListISerializable lst_registrations = new ListISerializable.with_backer(registrations);
            Variant v6 = lst_registrations.serialize_to_variant();
            Variant v7 = Serializer.int_to_variant(0 /*UNUSED*/);
            Variant v8 = Serializer.int_to_variant(0 /*UNUSED*/);

            Variant vtemp = Serializer.tuple_to_variant_5(v0, v1, v2, v3, v4);
            Variant vret = Serializer.tuple_to_variant_5(vtemp, v5, v6, v7, v8);
            return vret;
        }

        public void deserialize_from_variant(Variant v) throws SerializerError
        {
            cache = new ArrayList<InfoAndnaCache>
                        (InfoAndnaCache.equal_func);
            registrations = new ArrayList<InfoAndnaRegistration>
                                (InfoAndnaRegistration.equal_func);
            Variant v0;
            Variant v1;
            Variant v2;
            Variant v3;
            Variant v4;
            Variant v5;
            Variant v6;
            Variant v7;
            Variant v8;
            Variant vtemp;
            Serializer.variant_to_tuple_5(v, out vtemp, out v5, out v6, out v7, out v8);
            Serializer.variant_to_tuple_5(vtemp, out v0, out v1, out v2, out v3, out v4);

            register_ongoing = Serializer.variant_to_int(v0) == 1;
            will_participate = Serializer.variant_to_int(v1) == 1;
            participating = Serializer.variant_to_int(v2) == 1;
            hooked = Serializer.variant_to_int(v3) == 1;
            pubk = Serializer.variant_to_string(v4);
            ListISerializable lst_cache =
                    (ListISerializable)Object.new(typeof(ListISerializable));
            lst_cache.deserialize_from_variant(v5);
            cache.add_all((Gee.List<InfoAndnaCache>)
                        lst_cache.backed);
            ListISerializable lst_registrations =
                    (ListISerializable)Object.new(typeof(ListISerializable));
            lst_registrations.deserialize_from_variant(v6);
            registrations.add_all((Gee.List<InfoAndnaRegistration>)
                        lst_registrations.backed);
        }
    }

    /** Bits of information from Andna module
      */
    public class InfoAndnaCache : Object, ISerializable
    {
        public string domain;
        public AndnaDomainRecord rec;

        public Variant serialize_to_variant()
        {
            Variant v0 = Serializer.string_to_variant(domain);
            Variant v1 = rec.serialize_to_variant();
            Variant vret = Serializer.tuple_to_variant(v0, v1);
            return vret;
        }

        public void deserialize_from_variant(Variant v) throws SerializerError
        {
            Variant v0;
            Variant v1;
            Serializer.variant_to_tuple(v, out v0, out v1);
            domain = Serializer.variant_to_string(v0);
            rec = (AndnaDomainRecord)Object.new(typeof(AndnaDomainRecord));
            rec.deserialize_from_variant(v1);
        }

        public static bool equal_func(InfoAndnaCache a, InfoAndnaCache b)
        {
            return a.domain == b.domain;
        }
    }

    public class InfoAndnaRegistration : Object, ISerializable
    {
        public string domain;
        public bool registered;
        public TimeCapsule ttl_before_request;

        public Variant serialize_to_variant()
        {
            Variant v0 = Serializer.string_to_variant(domain);
            Variant v1 = Serializer.int_to_variant(registered ? 1 : 0);
            Variant v2 = ttl_before_request.serialize_to_variant();
            Variant vret = Serializer.tuple_to_variant_3(v0, v1, v2);
            return vret;
        }

        public void deserialize_from_variant(Variant v) throws SerializerError
        {
            Variant v0;
            Variant v1;
            Variant v2;
            Serializer.variant_to_tuple_3(v, out v0, out v1, out v2);
            domain = Serializer.variant_to_string(v0);
            registered = Serializer.variant_to_int(v1) == 1;
            ttl_before_request = (TimeCapsule)Object.new(typeof(TimeCapsule));
            ttl_before_request.deserialize_from_variant(v2);
        }

        public static bool equal_func(InfoAndnaRegistration a, InfoAndnaRegistration b)
        {
            return a.domain == b.domain
                    && a.registered == b.registered;
        }
    }

    /** Test classes for peer_to_peer
      */
    public class DHTKey : Object, ISerializable
    {
        public string key {get; private set;}
        public DHTKey(string key) {this.key = key;}

        public static bool equal_func(DHTKey a, DHTKey b)
        {
            return a.equals(b);
        }

        public bool equals(DHTKey a)
        {
            return a.key == key;
        }

        public static uint hash_func(DHTKey a)
        {
            return a.hash();
        }

        public uint hash()
        {
            return key.hash();
        }

        public Variant serialize_to_variant()
        {
            return Serializer.string_to_variant(key);
        }

        public void deserialize_from_variant(Variant v) throws SerializerError
        {
            key = Serializer.variant_to_string(v);
        }
    }

    public class DHTRecord : Object, ISerializable
    {
        public DHTKey key {get; private set;}
        public string @value {get; private set;}
        public TimeCapsule ttl {get; private set;}
        public DHTRecord(DHTKey key, string @value, owned TimeCapsule? ttl=null)
        {
            this.key = key;
            this.@value = @value;
            if (ttl == null) ttl = new TimeCapsule(60000);
            this.ttl = ttl;
        }

        public static bool key_equal_func(DHTRecord a, DHTRecord b)
        {
            return DHTKey.equal_func(a.key, b.key);
        }

        public Variant serialize_to_variant()
        {
            Variant v0 = Serializer.string_to_variant(@value);
            Variant v1 = key.serialize_to_variant();
            Variant v2 = ttl.serialize_to_variant();
            Variant vret = Serializer.tuple_to_variant_3(v0, v1, v2);
            return vret;
        }

        public void deserialize_from_variant(Variant v) throws SerializerError
        {
            Variant v0;
            Variant v1;
            Variant v2;
            Serializer.variant_to_tuple_3(v, out v0, out v1, out v2);
            @value = Serializer.variant_to_string(v0);
            key = (DHTKey)Object.new(typeof(DHTKey));
            key.deserialize_from_variant(v1);
            ttl = (TimeCapsule)Object.new(typeof(TimeCapsule));
            ttl.deserialize_from_variant(v2);
        }
    }

    /** Public key (used in conjunction with PublicKeyWrapper in module Crypto)
      */
    public class PublicKey : Object, ISerializable
    {
        public SerializableBuffer x {get; private set;}
        public PublicKey(SerializableBuffer buf)
        {
            x = buf;
        }

        public Variant serialize_to_variant()
        {
            return x.serialize_to_variant();
        }

        public void deserialize_from_variant(Variant v) throws SerializerError
        {
            x = (SerializableBuffer)Object.new(typeof(SerializableBuffer));
            x.deserialize_from_variant(v);
        }

        public static uint hash_func(PublicKey? a)
        {
            if (a == null) return 0;
            uint ret = 1;
            uint8[] a_buf = a.x.buffer;
            int pos = 0;
            while (pos < a_buf.length)
            {
                ret += @"$(a_buf[pos])".hash();
                pos++;
            }
            return ret;
        }

        public static bool equal_func(PublicKey? a, PublicKey? b)
        {
            if (a == b) return true;
            if (a == null || b == null) return false;
            uint8[] a_buf = a.x.buffer;
            uint8[] b_buf = b.x.buffer;
            if (a_buf.length != b_buf.length) return false;
            int pos = 0;
            while (pos < a_buf.length)
            {
                if (a_buf[pos] != b_buf[pos]) return false;
                pos++;
            }
            return true;
        }

        public string to_string()
        {
            return @"<PublicKey $(hash_func(this))>";
        }
    }

    /** Serializables of ANDNA
      */
    public class AndnaServiceKey : Object, ISerializable
    {
        // A Service Key has a name and a protocol. E.g. the
        // service key name="imaps" and protocol="tcp" is the
        // equivalent of the SRV record that identifies a host
        // serving as secure IMAP server for a certain domain.
        // The command "dig _imaps._tcp.gmail.com SRV"
        // will return that record for the domain gmail.com
        // The answer is imap.gmail.com.
        public string? name {get; private set;}
        public string? proto {get; private set;}

        public AndnaServiceKey(string name, string proto)
        {
            this.name = name;
            this.proto = proto;
        }

        private AndnaServiceKey.make_null()
        {
            this.name = null;
            this.proto = null;
        }

        // The serv_key to use for the main registration of the hostname
        private static AndnaServiceKey _NULL_SERV_KEY;

        // get the NULL_SERV_KEY
        public static AndnaServiceKey NULL_SERV_KEY {
            get {
                if (_NULL_SERV_KEY == null) _NULL_SERV_KEY = new AndnaServiceKey.make_null();
                return _NULL_SERV_KEY;
            }
        }

        public Variant serialize_to_variant()
        {
            Variant v0;
            if (name == null) v0 = Serializer.int_to_variant(0);
            else v0 = Serializer.string_to_variant(name);
            Variant v1;
            if (proto == null) v1 = Serializer.int_to_variant(0);
            else v1 = Serializer.string_to_variant(proto);
            return Serializer.tuple_to_variant(v0, v1);
        }
        
        public void deserialize_from_variant(Variant v)
        {
            Variant v0;
            Variant v1;
            Serializer.variant_to_tuple(v, out v0, out v1);
            string vt0 = v0.get_type_string();
            if (vt0 == "i") name = null;
            else name = Serializer.variant_to_string(v0);
            string vt1 = v1.get_type_string();
            if (vt1 == "i") proto = null;
            else proto = Serializer.variant_to_string(v1);
        }

        public static bool equal_func(AndnaServiceKey? a, AndnaServiceKey? b)
        {
            if (a == b) return true;
            if (a == null || b == null) return false;
            return a.name == b.name && a.proto == b.proto;
        }

        public static uint hash_func(AndnaServiceKey a)
        {
            if (a.name == null) return 0;
            return a.name.hash();
        }

        public string to_string()
        {
            if (name == null) return "<AndnaServiceKey NULL_SERV_KEY>";
            return @"<AndnaServiceKey _$(name)._$(proto)>";
        }
    }

    public class AndnaPrivateConfigurationServer : Object, ISerializable
    {
        public string? alias {get; private set;}
        public PublicKey? pubk {get; private set;}
        public int port_number {get; private set;}
        public int priority {get; private set;}
        public int weight {get; private set;}

        public AndnaPrivateConfigurationServer(string? alias, PublicKey? pubk,
                               int port_number, int priority, int weight)
        {
            if (alias == null) assert(pubk == null);
            if (pubk == null) assert(alias == null);
            this.alias = alias;
            this.pubk = pubk;
            this.port_number = port_number;
            this.priority = priority;
            this.weight = weight;
        }

        // The server record to use for requesting just basic resolution
        public static
        ArrayList<AndnaPrivateConfigurationServer>
        make_null_list()
        {
            var ret = make_empty_list();
            ret.add(new AndnaPrivateConfigurationServer(null, null, 0, 1, 1));
            return ret;
        }

        public static
        ArrayList<AndnaPrivateConfigurationServer>
        make_empty_list()
        {
            return new ArrayList<AndnaPrivateConfigurationServer>
                    (AndnaPrivateConfigurationServer.equal_func);
        }

        public static bool equal_func(AndnaPrivateConfigurationServer a,
                                               AndnaPrivateConfigurationServer b)
        {
            if (a.priority != b.priority) return false;
            if (a.weight != b.weight) return false;
            if (a.port_number != b.port_number) return false;
            if (a.alias != b.alias) return false;
            if (! PublicKey.equal_func(a.pubk, b.pubk)) return false;
            return true;
        }

        public string to_string()
        {
            string alias_str = "null";
            if (alias != null)
                alias_str = alias;
            string pubk_str = "null";
            if (pubk != null)
                pubk_str = pubk.to_string();
            return @"<AndnaPrivateConfigurationServer: alias $(alias_str), pubk $(pubk_str), " +
                @"port_number $(port_number), priority $(priority), weight $(weight)>";
        }

        public Variant serialize_to_variant()
        {
            Variant v0;
            if (alias == null)
            {
                v0 = Serializer.int_to_variant(0);
            }
            else
            {
                v0 = Serializer.string_to_variant(alias);
            }

            Variant v1;
            if (pubk == null)
            {
                v1 = Serializer.int_to_variant(0);
            }
            else
            {
                v1 = pubk.serialize_to_variant();
            }

            Variant v2 = Serializer.int_to_variant(port_number);

            Variant v3 = Serializer.int_to_variant(priority);

            Variant v4 = Serializer.int_to_variant(weight);

            Variant vret = Serializer.tuple_to_variant_5(v0, v1, v2, v3, v4);
            return vret;
        }
        
        public void deserialize_from_variant(Variant v)
        {
            Variant v0;
            Variant v1;
            Variant v2;
            Variant v3;
            Variant v4;
            Serializer.variant_to_tuple_5(v, out v0, out v1, out v2, out v3, out v4);

            if (v0.get_type_string() == "i")
            {
                alias = null;
            }
            else
            {
                alias = Serializer.variant_to_string(v0);
            }

            if (v1.get_type_string() == "i")
            {
                pubk = null;
            }
            else
            {
                pubk = (PublicKey)Object.new(typeof(PublicKey));
                pubk.deserialize_from_variant(v1);
            }

            port_number = Serializer.variant_to_int(v2);

            priority = Serializer.variant_to_int(v3);

            weight = Serializer.variant_to_int(v4);
        }
    }

    public class AndnaPrivateConfiguration : Object, ISerializable
    {
        public string domain;
        public HashMap<AndnaServiceKey,
                       ArrayList<AndnaPrivateConfigurationServer>>
                       services {get; private set;}

        public AndnaPrivateConfiguration
                (string domain,
                 HashMap<AndnaServiceKey, ArrayList<AndnaPrivateConfigurationServer>>? services=null)
        {
            this.domain = domain;
            if (services == null)
            {
                this.services = make_null_services();
            }
            else
            {
                this.services = make_empty_services();
                foreach (AndnaServiceKey k in services.keys)
                {
                    this.services[k] = AndnaPrivateConfigurationServer.make_empty_list();
                    this.services[k].add_all(services[k]);
                }
            }
        }

        public static
        HashMap<AndnaServiceKey, ArrayList<AndnaPrivateConfigurationServer>>
        make_null_services()
        {
            var ret = make_empty_services();
            ret[AndnaServiceKey.NULL_SERV_KEY] =
                    AndnaPrivateConfigurationServer.make_null_list();
            return ret;
        }

        public static
        HashMap<AndnaServiceKey, ArrayList<AndnaPrivateConfigurationServer>>
        make_empty_services()
        {
            return new HashMap<AndnaServiceKey, ArrayList<AndnaPrivateConfigurationServer>>
                            (AndnaServiceKey.hash_func, AndnaServiceKey.equal_func);
        }

        public static bool equal_func_for_list(AndnaPrivateConfiguration a,
                                               AndnaPrivateConfiguration b)
        {
            return a.domain == b.domain;
        }

        public string to_string()
        {
            string servers_str = "{";
            string nextel = "";
            foreach (AndnaServiceKey k in services.keys)
            {
                servers_str += nextel;
                nextel = ", ";
                servers_str += @"$(k): [";
                string next = "";
                foreach (AndnaPrivateConfigurationServer v in services[k])
                {
                    servers_str += next;
                    next = ", ";
                    servers_str += @"$(v)";
                }
                servers_str += "]";
            }
            servers_str += "}";
            return @"<AndnaPrivateConfiguration: domain $(domain)," +
                @" servers $(servers_str)>";
        }

        public Variant serialize_to_variant()
        {
            Variant v0 = Serializer.string_to_variant(domain);

            Variant v1;
            Variant v2;
            Variant v3;
            {
                int[] recordCounters = new int[services.keys.size];
                int pos = 0;
                ListISerializable lk = new ListISerializable();
                ListISerializable lv = new ListISerializable();
                foreach (AndnaServiceKey k in services.keys)
                {
                    lk.add(k);
                    ArrayList<AndnaPrivateConfigurationServer> v = services[k];
                    // how many records in this list?
                    recordCounters[pos++] = v.size;
                    foreach (AndnaPrivateConfigurationServer rec in v)
                    {
                        lv.add(rec);
                    }
                }
                v1 = lk.serialize_to_variant();
                v2 = Serializer.int_array_to_variant(recordCounters);
                v3 = lv.serialize_to_variant();
            }

            Variant vret = Serializer.tuple_to_variant_4(v0, v1, v2, v3);
            return vret;
        }
        
        public void deserialize_from_variant(Variant v)
        {
            Variant v0;
            Variant v1;
            Variant v2;
            Variant v3;
            Serializer.variant_to_tuple_4(v, out v0, out v1, out v2, out v3);

            domain = Serializer.variant_to_string(v0);

            {
                ListISerializable lk = (ListISerializable)Object.new(typeof(ListISerializable));
                lk.deserialize_from_variant(v1);
                Gee.List<AndnaServiceKey> typed_lk = (Gee.List<AndnaServiceKey>)lk.backed;
                int[] recordCounters = Serializer.variant_to_int_array(v2);
                ListISerializable lv = (ListISerializable)Object.new(typeof(ListISerializable));
                lv.deserialize_from_variant(v3);
                Gee.List<AndnaPrivateConfigurationServer> typed_lv =
                        (Gee.List<AndnaPrivateConfigurationServer>)lv.backed;
                services = make_empty_services();
                int pos = 0;
                for (int i = 0; i < typed_lk.size; i++)
                {
                    ArrayList<AndnaPrivateConfigurationServer> vlist =
                            AndnaPrivateConfigurationServer.make_empty_list();
                    for (int j = 0; j < recordCounters[i]; j++)
                    {
                        vlist.add(typed_lv[j + pos]);
                    }
                    pos += recordCounters[i];
                    services[typed_lk[i]] = vlist;
                }
            }
        }
    }

    public class AndnaPrivateConfigurationList : Object, ISerializable
    {
        public ArrayList<AndnaPrivateConfiguration> lst {get; private set;}

        public AndnaPrivateConfigurationList()
        {
            lst = new ArrayList<AndnaPrivateConfiguration>
                        (AndnaPrivateConfiguration.equal_func_for_list);
        }

        public AndnaPrivateConfiguration? get_domain(string domain)
        {
            foreach (AndnaPrivateConfiguration r in lst)
                if (r.domain == domain)
                    return r;
            return null;
        }

        public string to_string()
        {
            string cfg_str = "[";
            string next = "";
            foreach (AndnaPrivateConfiguration v in lst)
            {
                cfg_str += next;
                next = ", ";
                cfg_str += @"$(v)";
            }
            cfg_str += "]";
            return @"<AndnaPrivateConfigurationList: $(cfg_str)>";
        }

        public Variant serialize_to_variant()
        {
            Variant v0;
            {
                ListISerializable _lst = new ListISerializable();
                foreach (AndnaPrivateConfiguration o in lst) _lst.add(o);
                v0 = _lst.serialize_to_variant();
            }

            return v0;
        }
        
        public void deserialize_from_variant(Variant v)
        {
            Variant v0 = v;

            lst = new ArrayList<AndnaPrivateConfiguration>
                        (AndnaPrivateConfiguration.equal_func_for_list);
            {
                ListISerializable _lst = (ListISerializable)Object.new(typeof(ListISerializable));
                _lst.deserialize_from_variant(v0);
                Gee.List<AndnaPrivateConfiguration> typed_lst =
                        (Gee.List<AndnaPrivateConfiguration>)_lst.backed;
                lst.add_all(typed_lst);
            }
        }
    }

    public class AndnaServerRequest : Object, ISerializable
    {
        public string? hashed_alias {get; private set;}
        public PublicKey? pubk {get; private set;}
        public int port_number {get; private set;}
        public int priority {get; private set;}
        public int weight {get; private set;}

        public AndnaServerRequest(string? hashed_alias, PublicKey? pubk,
                               int port_number, int priority, int weight)
        {
            if (hashed_alias == null) assert(pubk == null);
            if (pubk == null) assert(hashed_alias == null);
            this.hashed_alias = hashed_alias;
            this.pubk = pubk;
            this.port_number = port_number;
            this.priority = priority;
            this.weight = weight;
        }

        public static
        ArrayList<AndnaServerRequest>
        make_empty_list()
        {
            return new ArrayList<AndnaServerRequest>(AndnaServerRequest.equal_func);
        }

        public static bool equal_func(AndnaServerRequest a,
                                               AndnaServerRequest b)
        {
            if (a.priority != b.priority) return false;
            if (a.weight != b.weight) return false;
            if (a.port_number != b.port_number) return false;
            if (a.hashed_alias != b.hashed_alias) return false;
            if (! PublicKey.equal_func(a.pubk, b.pubk)) return false;
            return true;
        }

        public string to_string()
        {
            string hashed_alias_str = "null";
            if (hashed_alias != null)
                hashed_alias_str = hashed_alias;
            string pubk_str = "null";
            if (pubk != null)
                pubk_str = pubk.to_string();
            return @"<AndnaServerRequest: hashed_alias $(hashed_alias_str), pubk $(pubk_str), " +
                @"port_number $(port_number), priority $(priority), weight $(weight)>";
        }

        public Variant serialize_to_variant()
        {
            Variant v0;
            if (hashed_alias == null)
            {
                v0 = Serializer.int_to_variant(0);
            }
            else
            {
                v0 = Serializer.string_to_variant(hashed_alias);
            }

            Variant v1;
            if (pubk == null)
            {
                v1 = Serializer.int_to_variant(0);
            }
            else
            {
                v1 = pubk.serialize_to_variant();
            }

            Variant v2 = Serializer.int_to_variant(port_number);

            Variant v3 = Serializer.int_to_variant(priority);

            Variant v4 = Serializer.int_to_variant(weight);

            Variant vret = Serializer.tuple_to_variant_5(v0, v1, v2, v3, v4);
            return vret;
        }
        
        public void deserialize_from_variant(Variant v)
        {
            Variant v0;
            Variant v1;
            Variant v2;
            Variant v3;
            Variant v4;
            Serializer.variant_to_tuple_5(v, out v0, out v1, out v2, out v3, out v4);

            if (v0.get_type_string() == "i")
            {
                hashed_alias = null;
            }
            else
            {
                hashed_alias = Serializer.variant_to_string(v0);
            }

            if (v1.get_type_string() == "i")
            {
                pubk = null;
            }
            else
            {
                pubk = (PublicKey)Object.new(typeof(PublicKey));
                pubk.deserialize_from_variant(v1);
            }

            port_number = Serializer.variant_to_int(v2);

            priority = Serializer.variant_to_int(v3);

            weight = Serializer.variant_to_int(v4);
        }
    }

    public class AndnaDomainRequest : Object, ISerializable
    {
        public string hashed_domain {get; private set;}
        public PublicKey pubk {get; private set;}
        public NIP nip {get; private set;}
        public HashMap<AndnaServiceKey,
                       ArrayList<AndnaServerRequest>>
                       services {get; private set;}

        public AndnaDomainRequest
                (string hashed_domain,
                 PublicKey pubk,
                 NIP nip,
                 HashMap<AndnaServiceKey,
                         ArrayList<AndnaServerRequest>>
                         services)
        {
            this.hashed_domain = hashed_domain;
            this.pubk = pubk;
            this.nip = nip;
            this.services = make_empty_services();
            foreach (AndnaServiceKey k in services.keys)
            {
                this.services[k] = AndnaServerRequest.make_empty_list();
                this.services[k].add_all(services[k]);
            }
        }

        public static
        HashMap<AndnaServiceKey, ArrayList<AndnaServerRequest>>
        make_empty_services()
        {
            return new HashMap<AndnaServiceKey, ArrayList<AndnaServerRequest>>
                            (AndnaServiceKey.hash_func, AndnaServiceKey.equal_func);
        }

        public uchar[] hash_for_signature()
        {
            string buf = @"$(hashed_domain),$(pubk),$(nip),";
            foreach (AndnaServiceKey k in services.keys)
            {
                buf += @"$(k),";
                foreach (AndnaServerRequest v in services[k])
                {
                    buf += @"$(v),";
                }
            }
            uchar[] ret = {};
            foreach (uchar u in buf.data)
                ret += u;
            return ret;
        }

        public Variant serialize_to_variant()
        {
            Variant v0 = Serializer.string_to_variant(hashed_domain);

            Variant v1 = pubk.serialize_to_variant();

            Variant v2 = nip.serialize_to_variant();

            Variant v3;
            Variant v4;
            Variant v5;
            {
                int[] recordCounters = new int[services.keys.size];
                int pos = 0;
                ListISerializable lk = new ListISerializable();
                ListISerializable lv = new ListISerializable();
                foreach (AndnaServiceKey k in services.keys)
                {
                    lk.add(k);
                    ArrayList<AndnaServerRequest> v = services[k];
                    // how many records in this list?
                    recordCounters[pos++] = v.size;
                    foreach (AndnaServerRequest rec in v)
                    {
                        lv.add(rec);
                    }
                }
                v3 = lk.serialize_to_variant();
                v4 = Serializer.int_array_to_variant(recordCounters);
                v5 = lv.serialize_to_variant();
            }

            Variant vtemp = Serializer.tuple_to_variant_5(v0, v1, v2, v3, v4);
            Variant vret = Serializer.tuple_to_variant(vtemp, v5);
            return vret;
        }
        
        public void deserialize_from_variant(Variant v)
        {
            Variant v0;
            Variant v1;
            Variant v2;
            Variant v3;
            Variant v4;
            Variant v5;
            Variant vtemp;
            Serializer.variant_to_tuple(v, out vtemp, out v5);
            Serializer.variant_to_tuple_5(vtemp, out v0, out v1, out v2, out v3, out v4);

            hashed_domain = Serializer.variant_to_string(v0);

            pubk = (PublicKey)Object.new(typeof(PublicKey));
            pubk.deserialize_from_variant(v1);

            nip = (NIP)Object.new(typeof(NIP));
            nip.deserialize_from_variant(v2);

            {
                ListISerializable lk = (ListISerializable)Object.new(typeof(ListISerializable));
                lk.deserialize_from_variant(v3);
                Gee.List<AndnaServiceKey> typed_lk = (Gee.List<AndnaServiceKey>)lk.backed;
                int[] recordCounters = Serializer.variant_to_int_array(v4);
                ListISerializable lv = (ListISerializable)Object.new(typeof(ListISerializable));
                lv.deserialize_from_variant(v5);
                Gee.List<AndnaServerRequest> typed_lv = (Gee.List<AndnaServerRequest>)lv.backed;
                services = make_empty_services();
                int pos = 0;
                for (int i = 0; i < typed_lk.size; i++)
                {
                    ArrayList<AndnaServerRequest> vlist = AndnaServerRequest.make_empty_list();
                    for (int j = 0; j < recordCounters[i]; j++)
                    {
                        vlist.add(typed_lv[j + pos]);
                    }
                    pos += recordCounters[i];
                    services[typed_lk[i]] = vlist;
                }
            }
        }
    }

    public class AndnaServerRecord : Object, ISerializable
    {
        // string hashed_alias = it can be null or an other hashed hostname
        //   If hashed_alias is null then this record refers to the NIP of
        //   the registrar. Otherwise it is an hostname, then this
        //   record refers to that hostname.
        public string? hashed_alias {get; private set;}
        // PublicKey pubk = public key of alias.
        //   If hashed_alias is an other hashed hostname, then pubk is the public
        //   key of the host registrar of that hostname. The value
        //   of pubk in this case is specified by the client and verified
        //   by the hash node.
        //   If hashed_alias is null, then pubk is null.
        public PublicKey? pubk {get; private set;}
        // int port_number = the port number of the service. This is meaningful
        //   only when this record is related to a valid AndnaServiceKey. When it is
        //   related to the NULL_SERV_KEY the client has not stated anything
        //   about the service it wants to contact. So port number is dummy.
        public int port_number {get; private set;}
        // int priority = the priority of the record (lower is better)
        public int priority {get; private set;}
        // int weight = the relative weight of the record, between records with
        //   the same priority.
        public int weight {get; private set;}

        public AndnaServerRecord(string? hashed_alias, PublicKey? pubk,
                              int port_number, int priority, int weight)
        {
            if (hashed_alias == null) assert(pubk == null);
            if (pubk == null) assert(hashed_alias == null);
            this.hashed_alias = hashed_alias;
            this.pubk = pubk;
            this.priority = priority;
            this.weight = weight;
            this.port_number = port_number;
        }

        // The server record to use for requesting just basic resolution
        public static
        ArrayList<AndnaServerRecord>
        make_null_serv_rec_list()
        {
            var ret = make_empty_serv_rec_list();
            ret.add(new AndnaServerRecord(null, null, 0, 1, 1));
            return ret;
        }

        public static
        ArrayList<AndnaServerRecord>
        make_empty_serv_rec_list()
        {
            return new ArrayList<AndnaServerRecord>(AndnaServerRecord.equal_func);
        }

        public Variant serialize_to_variant()
        {
            Variant v0;
            Variant v1;
            if (hashed_alias == null)
            {
                v0 = Serializer.int_to_variant(0);
                v1 = Serializer.int_to_variant(0);
            }
            else
            {
                v0 = Serializer.string_to_variant(hashed_alias);
                v1 = pubk.serialize_to_variant();
            }
            Variant v2 = Serializer.int_to_variant(priority);
            Variant v3 = Serializer.int_to_variant(weight);
            Variant v4 = Serializer.int_to_variant(port_number);
            Variant vret = Serializer.tuple_to_variant_5(v0, v1, v2, v3, v4);
            return vret;
        }

        public void deserialize_from_variant(Variant v) throws SerializerError
        {
            Variant v0;
            Variant v1;
            Variant v2;
            Variant v3;
            Variant v4;
            Serializer.variant_to_tuple_5(v, out v0, out v1, out v2, out v3, out v4);
            string vt = v0.get_type_string();
            if (vt == "i")
            {
                hashed_alias = null;
                pubk = null;
            }
            else
            {
                hashed_alias = Serializer.variant_to_string(v0);
                pubk = (PublicKey)Object.new(typeof(PublicKey));
                pubk.deserialize_from_variant(v1);
            }
            priority = Serializer.variant_to_int(v2);
            weight = Serializer.variant_to_int(v3);
            port_number = Serializer.variant_to_int(v4);
        }

        public static bool equal_func(AndnaServerRecord? a, AndnaServerRecord? b)
        {
            if (a == b) return true;
            if (a == null || b == null) return false;
            if (a.priority != b.priority) return false;
            if (a.weight != b.weight) return false;
            if (a.port_number != b.port_number) return false;
            if (a.hashed_alias != b.hashed_alias) return false;
            if (! PublicKey.equal_func(a.pubk, b.pubk)) return false;
            return true;
        }

        public string to_string()
        {
            string hashed_alias_str = "null";
            if (hashed_alias != null)
                hashed_alias_str = hashed_alias;
            string pubk_str = "null";
            if (pubk != null)
                pubk_str = pubk.to_string();
            return @"<AndnaServerRecord: hashed_alias $(hashed_alias_str), pubk $(pubk_str), " +
                @"port_number $(port_number), priority $(priority), weight $(weight)>";
        }
    }

    public class AndnaDomainRecord : Object, ISerializable
    {
        // time() of expiration
        public TimeCapsule expires {get; private set;}
        // Hostname hashed
        public string hashed_domain {get; set;}
        // public key of registrar
        public PublicKey pubk {get; private set;}
        // current nip of registrar
        public NIP nip {get; private set;}
        // for each service an array of records
        public HashMap<AndnaServiceKey, ArrayList<AndnaServerRecord>> services {get; private set;}

        public AndnaDomainRecord(int64 msec_ttl,
                               string hashed_domain,
                               PublicKey pubk,
                               NIP nip,
                               HashMap<AndnaServiceKey, ArrayList<AndnaServerRecord>>?
                                      services=null)
        {
            expires = new TimeCapsule(msec_ttl);
            this.hashed_domain = hashed_domain;
            this.pubk = pubk;
            this.nip = nip;
            this.services = new HashMap<AndnaServiceKey, ArrayList<AndnaServerRecord>>
                            (AndnaServiceKey.hash_func, AndnaServiceKey.equal_func);
            if (services == null)
            {
                this.services = make_null_services();
            }
            else
            {
                foreach (AndnaServiceKey s in services.keys)
                {
                    this.services[s] =
                            new ArrayList<AndnaServerRecord>
                                (AndnaServerRecord.equal_func);
                    this.services[s].add_all(services[s]);
                }
            }
        }

        public static
        HashMap<AndnaServiceKey, ArrayList<AndnaServerRecord>>
        make_null_services()
        {
            var ret = make_empty_services();
            ret[AndnaServiceKey.NULL_SERV_KEY] =
                                  AndnaServerRecord.make_null_serv_rec_list();
            return ret;
        }

        public static
        HashMap<AndnaServiceKey, ArrayList<AndnaServerRecord>>
        make_empty_services()
        {
            return new HashMap<AndnaServiceKey, ArrayList<AndnaServerRecord>>
                            (AndnaServiceKey.hash_func, AndnaServiceKey.equal_func);
        }

        public Variant serialize_to_variant()
        {
            Variant v0 = Serializer.string_to_variant(hashed_domain);
            Variant v1 = expires.serialize_to_variant();
            // Variant v2 = Serializer.int_to_variant(updates);
            Variant v3 = pubk.serialize_to_variant();
            Variant v4 = nip.serialize_to_variant();

            int[] recordCounters = new int[services.keys.size];
            int pos = 0;
            ListISerializable lk = new ListISerializable();
            ListISerializable lv = new ListISerializable();
            foreach (AndnaServiceKey k in services.keys)
            {
                lk.add(k);
                ArrayList<AndnaServerRecord> v = services[k];
                // how many records in this list?
                recordCounters[pos++] = v.size;
                foreach (AndnaServerRecord rec in v)
                {
                    lv.add(rec);
                }
            }
            Variant v5 = lk.serialize_to_variant();
            Variant v6 = Serializer.int_array_to_variant(recordCounters);
            Variant v7 = lv.serialize_to_variant();

            Variant vtemp = Serializer.tuple_to_variant_4(v0, v1, v3, v4);
            Variant vret = Serializer.tuple_to_variant_4(vtemp, v5, v6, v7);
            return vret;
        }
        
        public void deserialize_from_variant(Variant v)
        {
            Variant v0;
            Variant v1;
            // Variant v2;
            Variant v3;
            Variant v4;
            Variant v5;
            Variant v6;
            Variant v7;
            Variant vtemp;
            Serializer.variant_to_tuple_4(v, out vtemp, out v5, out v6, out v7);
            Serializer.variant_to_tuple_4(vtemp, out v0, out v1, out v3, out v4);

            hashed_domain = Serializer.variant_to_string(v0);
            expires = (TimeCapsule)Object.new(typeof(TimeCapsule));
            expires.deserialize_from_variant(v1);
            pubk = (PublicKey)Object.new(typeof(PublicKey));
            pubk.deserialize_from_variant(v3);
            nip = (NIP)Object.new(typeof(NIP));
            nip.deserialize_from_variant(v4);

            ListISerializable lk = (ListISerializable)Object.new(typeof(ListISerializable));
            lk.deserialize_from_variant(v5);
            ListISerializable lv = (ListISerializable)Object.new(typeof(ListISerializable));
            lv.deserialize_from_variant(v7);
            int[] recordCounters = Serializer.variant_to_int_array(v6);
            Gee.List<AndnaServiceKey> typed_lk = (Gee.List<AndnaServiceKey>)lk.backed;
            Gee.List<AndnaServerRecord> typed_lv = (Gee.List<AndnaServerRecord>)lv.backed;
            services = new HashMap<AndnaServiceKey, ArrayList<AndnaServerRecord>>
                       (AndnaServiceKey.hash_func, AndnaServiceKey.equal_func);
            int pos = 0;
            for (int i = 0; i < typed_lk.size; i++)
            {
                ArrayList<AndnaServerRecord> vlist = new ArrayList<AndnaServerRecord>
                                                  (AndnaServerRecord.equal_func);
                for (int j = 0; j < recordCounters[i]; j++)
                {
                    vlist.add(typed_lv[j + pos]);
                }
                pos += recordCounters[i];
                services[typed_lk[i]] = vlist;
            }
        }

        public string to_string()
        {
            string servers_str = "{";
            string nextel = "";
            foreach (AndnaServiceKey k in services.keys)
            {
                servers_str += nextel;
                nextel = ", ";
                servers_str += @"$(k): [";
                string next = "";
                foreach (AndnaServerRecord v in services[k])
                {
                    servers_str += next;
                    next = ", ";
                    servers_str += @"$(v)";
                }
                servers_str += "]";
            }
            servers_str += "}";
            return @"<AndnaDomainRecord: hashed_domain $(hashed_domain)," +
                @" pubk $(pubk)," +
                @" nip $(nip)," +
                @" ttl $(expires.get_string_msec_ttl())," +
                @" servers $(servers_str)>";
        }

        public uchar[] hash_for_signature()
        {
            string buf = @"$(hashed_domain),$(pubk),$(nip),";
            foreach (AndnaServiceKey k in services.keys)
            {
                buf += @"$(k),";
                foreach (AndnaServerRecord v in services[k])
                {
                    buf += @"$(v),";
                }
            }
            uchar[] ret = {};
            foreach (uchar u in buf.data)
                ret += u;
            return ret;
        }

        public void update(int64 new_msec_ttl, NIP new_nip,
                           HashMap<AndnaServiceKey, ArrayList<AndnaServerRecord>> new_services)
        {
            // Updates a registration
            expires = new TimeCapsule(new_msec_ttl);
            this.nip = new_nip;
            this.services = new HashMap<AndnaServiceKey, ArrayList<AndnaServerRecord>>
                            (AndnaServiceKey.hash_func, AndnaServiceKey.equal_func);
            foreach (AndnaServiceKey s in new_services.keys)
            {
                this.services[s] = new ArrayList<AndnaServerRecord>(AndnaServerRecord.equal_func);
                this.services[s].add_all(new_services[s]);
            }
        }

        public AndnaServers get_servers(AndnaServiceKey serv_key)
        {
            // Return the list of AndnaServer records of this domain for
            //  the given service key.
            AndnaServers ret;
            ArrayList<AndnaServer> servers =
                    new ArrayList<AndnaServer>(AndnaServer.equal_func);
            if (services.has_key(serv_key))
            {
                foreach (AndnaServerRecord srvrec in services[serv_key])
                {
                    string? hashed_alias_name = null;
                    NIP? registrar_nip = null;
                    if (srvrec.hashed_alias != null) hashed_alias_name = srvrec.hashed_alias;
                    else registrar_nip = nip;
                    AndnaServer srv =
                            new AndnaServer(registrar_nip,
                                            hashed_alias_name,
                                            srvrec.priority,
                                            srvrec.weight,
                                            srvrec.port_number);
                    servers.add(srv);
                }
                ret = new AndnaServers(expires, servers);
            }
            else
            {
                ret = new AndnaServers(expires, servers);
            }
            return ret;
        }

        public HashMap<AndnaServiceKey, AndnaServers> get_all()
        {
            // Return all the records of all the services
            HashMap<AndnaServiceKey, AndnaServers> ret;
            ret = new HashMap<AndnaServiceKey, AndnaServers>
                  (AndnaServiceKey.hash_func, AndnaServiceKey.equal_func);
            foreach (AndnaServiceKey s in services.keys)
                ret[s] = get_servers(s);
            return ret;
        }

        public static bool hashed_domain_equal_func(AndnaDomainRecord a, AndnaDomainRecord b)
        {
            return a.hashed_domain == b.hashed_domain;
        }
    }

    public class AndnaServer : Object, ISerializable
    {
        // this record can point to a further hostname...
        public string? alias_name {get; private set;}
        // ... otherwise has to be resolved in a nip
        public NIP? registrar_nip {get; private set;}
        // the priority of the record (lower is better)
        public int priority {get; private set;}
        // the relative weight of the record, between records with same priority.
        public int weight {get; private set;}
        // the port number
        public int port_number {get; private set;}

        public AndnaServer(NIP? registrar_nip,
                                  string? alias_name,
                                  int priority,
                                  int weight,
                                  int port_number)
        {
            assert(registrar_nip != null || alias_name != null);
            if (registrar_nip != null) assert(alias_name == null);
            this.registrar_nip = registrar_nip;
            this.alias_name = alias_name;
            this.priority = priority;
            this.weight = weight;
            this.port_number = port_number;
        }

        public void set_nip(NIP nip)
        {
            registrar_nip = nip;
            alias_name = null;
        }

        public static bool equal_func(AndnaServer? a, AndnaServer? b)
        {
            if (a == b) return true;
            if (a == null || b == null) return false;
            if (a.priority != b.priority) return false;
            if (a.weight != b.weight) return false;
            if (a.port_number != b.port_number) return false;
            if (a.alias_name != b.alias_name) return false;
            if (! PartialNIP.equal_func(a.registrar_nip, b.registrar_nip)) return false;
            return true;
        }

        public string to_string()
        {
            string str_server;
            if (registrar_nip != null) str_server = @"$(registrar_nip)";
            else str_server = alias_name;
            return @"<AndnaServer: (server $(str_server), " +
                   @"priority $(priority), " +
                   @"weight $(weight), " +
                   @"port_number $(port_number))>";
        }

        public Variant serialize_to_variant()
        {
            Variant v0;
            Variant v1;
            if (alias_name == null)
            {
                v0 = Serializer.int_to_variant(0);
                v1 = registrar_nip.serialize_to_variant();
            }
            else
            {
                v0 = Serializer.string_to_variant(alias_name);
                v1 = Serializer.int_to_variant(0);
            }
            Variant v2 = Serializer.int_to_variant(priority);
            Variant v3 = Serializer.int_to_variant(weight);
            Variant v4 = Serializer.int_to_variant(port_number);
            Variant vret = Serializer.tuple_to_variant_5(v0, v1, v2, v3, v4);
            return vret;
        }
        public void deserialize_from_variant(Variant v) throws SerializerError
        {
            Variant v0;
            Variant v1;
            Variant v2;
            Variant v3;
            Variant v4;
            Serializer.variant_to_tuple_5(v, out v0, out v1, out v2, out v3, out v4);
            string vt = v0.get_type_string();
            if (vt == "i")
            {
                alias_name = null;
                registrar_nip = (NIP)Object.new(typeof(NIP));
                registrar_nip.deserialize_from_variant(v1);
            }
            else
            {
                alias_name = Serializer.variant_to_string(v0);
                registrar_nip = null;
            }
            priority = Serializer.variant_to_int(v2);
            weight = Serializer.variant_to_int(v3);
            port_number = Serializer.variant_to_int(v4);
        }
    }

    public class AndnaServers : Object, ISerializable
    {
        /**
          * This class represent the resolution of an hostname and a serv_key.
          * It is normally returned by the hash node as the result of a request
          * for hostname+serv_key. But the same class can be used by a node to
          * cache the resolved queries. In this case an instance for each pair
          * (hostname, serv_key) has to be maintained.
          * This class does not contain the hostname and the serv_key themselves.
          */
        // Time To Live
        public TimeCapsule expires {get; private set;}
        // The records registered for this hostname+serv_key, maybe empty.
        public Gee.List<AndnaServer> servers {get; private set;}

        public AndnaServers(TimeCapsule expires, Gee.List<AndnaServer> servers)
        {
            this.expires = expires;
            this.servers = new ArrayList<AndnaServer>(AndnaServer.equal_func);
            this.servers.add_all(servers);
        }

        public AndnaServers.not_found(TimeCapsule expires)
        {
            this.expires = expires;
            this.servers = new ArrayList<AndnaServer>(AndnaServer.equal_func);
        }

        public bool is_not_found {
            get {
                return servers.is_empty;
            }
        }

        public Variant serialize_to_variant()
        {
            Variant v0 = expires.serialize_to_variant();
            ListISerializable lst = new ListISerializable.with_backer(servers);
            Variant v1 = lst.serialize_to_variant();
            Variant vret = Serializer.tuple_to_variant(v0, v1);
            return vret;
        }
        public void deserialize_from_variant(Variant v) throws SerializerError
        {
            Variant v0;
            Variant v1;
            Serializer.variant_to_tuple(v, out v0, out v1);
            expires = (TimeCapsule)Object.new(typeof(TimeCapsule));
            expires.deserialize_from_variant(v0);
            ListISerializable lst = (ListISerializable)Object.new(typeof(ListISerializable));
            lst.deserialize_from_variant(v1);
            servers = (Gee.List<AndnaServer>)lst.backed;
        }
    }

    public class RegisterHostnameArguments : Object, ISerializable
    {
        // Arguments for a request of register_main_for_pubk to service Andna.
        public AndnaDomainRequest request;
        public SerializableBuffer signature;
        public TimeCapsule expires;

        public RegisterHostnameArguments(AndnaDomainRequest request, 
                                         SerializableBuffer signature,
                                         TimeCapsule expires)
        {
            this.request = request;
            this.signature = signature;
            this.expires = expires;
        }

        public static bool equal_func_for_queue
                (RegisterHostnameArguments a,
                 RegisterHostnameArguments b)
        {
            return PublicKey.equal_func(a.request.pubk, b.request.pubk);
        }

        public Variant serialize_to_variant()
        {
            Variant v0 = request.serialize_to_variant();
            Variant v1 = signature.serialize_to_variant();
            Variant v2 = expires.serialize_to_variant();
            Variant vret = Serializer.tuple_to_variant_3(v0, v1, v2);
            return vret;
        }
        public void deserialize_from_variant(Variant v) throws SerializerError
        {
            Variant v0;
            Variant v1;
            Variant v2;
            Serializer.variant_to_tuple_3(v, out v0, out v1, out v2);
            request = (AndnaDomainRequest)Object.new(typeof(AndnaDomainRequest));
            request.deserialize_from_variant(v0);
            signature = (SerializableBuffer)Object.new(typeof(SerializableBuffer));
            signature.deserialize_from_variant(v1);
            expires = (TimeCapsule)Object.new(typeof(TimeCapsule));
            expires.deserialize_from_variant(v2);
        }
    }

    public class CounterNipRecord : Object, ISerializable
    {
        // This class represents the knowledge of the Counter hashnode
        //  relative to a node.
        public PublicKey pubk {get; private set;}
        public NIP nip {get; private set;}
        public ListString hashed_domains {get; private set;}
        public TimeCapsule expires {get; private set;}

        public CounterNipRecord(PublicKey pubk, NIP nip, 
                                ListString hashed_domains, TimeCapsule expires)
        {
            this.pubk = pubk;
            this.nip = nip;
            this.hashed_domains = hashed_domains;
            this.expires = expires;
        }

        public Variant serialize_to_variant()
        {
            Variant v0 = pubk.serialize_to_variant();
            Variant v1 = nip.serialize_to_variant();
            Variant v2 = hashed_domains.serialize_to_variant();
            Variant v3 = expires.serialize_to_variant();
            Variant vret = Serializer.tuple_to_variant_4(v0, v1, v2, v3);
            return vret;
        }
        public void deserialize_from_variant(Variant v) throws SerializerError
        {
            Variant v0;
            Variant v1;
            Variant v2;
            Variant v3;
            Serializer.variant_to_tuple_4(v, out v0, out v1, out v2, out v3);
            pubk = (PublicKey)Object.new(typeof(PublicKey));
            pubk.deserialize_from_variant(v0);
            nip = (NIP)Object.new(typeof(NIP));
            nip.deserialize_from_variant(v1);
            hashed_domains = (ListString)Object.new(typeof(ListString));
            hashed_domains.deserialize_from_variant(v2);
            expires = (TimeCapsule)Object.new(typeof(TimeCapsule));
            expires.deserialize_from_variant(v3);
        }
    }

    public class CounterSetDataResponse : Object, ISerializable
    {
        public string response {get; private set;}
        public TimeCapsule expires {get; private set;}

        public CounterSetDataResponse(string response, TimeCapsule expires)
        {
            this.response = response;
            this.expires = expires;
        }

        public Variant serialize_to_variant()
        {
            Variant v0 = Serializer.string_to_variant(response);
            Variant v1 = expires.serialize_to_variant();
            Variant vret = Serializer.tuple_to_variant(v0, v1);
            return vret;
        }
        public void deserialize_from_variant(Variant v) throws SerializerError
        {
            Variant v0;
            Variant v1;
            Serializer.variant_to_tuple(v, out v0, out v1);
            response = Serializer.variant_to_string(v0);
            expires = (TimeCapsule)Object.new(typeof(TimeCapsule));
            expires.deserialize_from_variant(v1);
        }
    }

    public class CounterCheckHostnameResponse : Object, ISerializable
    {
        public bool response {get; private set;}
        public TimeCapsule expires {get; private set;}

        public CounterCheckHostnameResponse(bool response, TimeCapsule expires)
        {
            this.response = response;
            this.expires = expires;
        }

        public Variant serialize_to_variant()
        {
            Variant v0 = Serializer.int_to_variant(response ? 1 : 0);
            Variant v1 = expires.serialize_to_variant();
            Variant vret = Serializer.tuple_to_variant(v0, v1);
            return vret;
        }
        public void deserialize_from_variant(Variant v) throws SerializerError
        {
            Variant v0;
            Variant v1;
            Serializer.variant_to_tuple(v, out v0, out v1);
            response = Serializer.variant_to_int(v0) == 1;
            expires = (TimeCapsule)Object.new(typeof(TimeCapsule));
            expires.deserialize_from_variant(v1);
        }
    }

    public class PairPublicKeyNIP : Object, ISerializable
    {
        public PublicKey pk {get; private set;}
        public NIP nip {get; private set;}

        public PairPublicKeyNIP(PublicKey pk, NIP nip)
        {
            this.pk = pk;
            this.nip = nip;
        }

        public Variant serialize_to_variant()
        {
            Variant v0 = pk.serialize_to_variant();
            Variant v1 = nip.serialize_to_variant();
            Variant vret = Serializer.tuple_to_variant(v0, v1);
            return vret;
        }
        public void deserialize_from_variant(Variant v) throws SerializerError
        {
            Variant v0;
            Variant v1;
            Serializer.variant_to_tuple(v, out v0, out v1);
            pk = (PublicKey)Object.new(typeof(PublicKey));
            pk.deserialize_from_variant(v0);
            nip = (NIP)Object.new(typeof(NIP));
            nip.deserialize_from_variant(v1);
        }

        public static uint hash_func(PairPublicKeyNIP? a)
        {
            if (a == null) return 0;
            uint ret = PublicKey.hash_func(a.pk);
            return ret;
        }

        public static bool equal_func(PairPublicKeyNIP? a, PairPublicKeyNIP? b)
        {
            if (a == b) return true;
            if (a == null || b == null) return false;
            if (! PublicKey.equal_func(a.pk, b.pk)) return false;
            if (! PartialNIP.equal_func(a.nip, b.nip)) return false;
            return true;
        }

        public string to_string()
        {
            return @"<$(nip) $(pk)>";
        }
    }

    public class CounterGetCacheRecordsResponse : Object, ISerializable
    {
        public HashMap<PairPublicKeyNIP, CounterNipRecord> cache {get; private set;}

        public CounterGetCacheRecordsResponse
                (HashMap<PairPublicKeyNIP, CounterNipRecord> cache)
        {
            this.cache = new HashMap<PairPublicKeyNIP, CounterNipRecord>(
                             PairPublicKeyNIP.hash_func, PairPublicKeyNIP.equal_func);
            foreach (PairPublicKeyNIP k in cache.keys)
            {
                this.cache[k] = cache[k];
            }
        }

        public Variant serialize_to_variant()
        {
            ListISerializable lk = new ListISerializable();
            ListISerializable lv = new ListISerializable();
            foreach (PairPublicKeyNIP k in cache.keys)
            {
                lk.add(k);
                lv.add(cache[k]);
            }

            Variant v0 = lk.serialize_to_variant();
            Variant v1 = lv.serialize_to_variant();

            Variant vret = Serializer.tuple_to_variant(v0, v1);
            return vret;
        }
        
        public void deserialize_from_variant(Variant v)
        {
            Variant v0;
            Variant v1;
            Serializer.variant_to_tuple(v, out v0, out v1);

            ListISerializable lk = (ListISerializable)Object.new(typeof(ListISerializable));
            lk.deserialize_from_variant(v0);
            ListISerializable lv = (ListISerializable)Object.new(typeof(ListISerializable));
            lv.deserialize_from_variant(v1);
            Gee.List<PairPublicKeyNIP> typed_lk = (Gee.List<PairPublicKeyNIP>)lk.backed;
            Gee.List<CounterNipRecord> typed_lv =
                    (Gee.List<CounterNipRecord>)lv.backed;
            cache = new HashMap<PairPublicKeyNIP, CounterNipRecord>(
                             PairPublicKeyNIP.hash_func, PairPublicKeyNIP.equal_func);
            for (int i = 0; i < typed_lk.size; i++)
            {
                cache[typed_lk[i]] = typed_lv[i];
            }
        }
    }

    public class AndnaConfirmPubkResponse : Object, ISerializable
    {
        public string response {get; private set;}
        public SerializableBuffer signature {get; private set;}

        public AndnaConfirmPubkResponse(string response, SerializableBuffer signature)
        {
            this.response = response;
            this.signature = signature;
        }

        public Variant serialize_to_variant()
        {
            Variant v0 = Serializer.string_to_variant(response);
            Variant v1 = signature.serialize_to_variant();
            Variant vret = Serializer.tuple_to_variant(v0, v1);
            return vret;
        }
        public void deserialize_from_variant(Variant v) throws SerializerError
        {
            Variant v0;
            Variant v1;
            Serializer.variant_to_tuple(v, out v0, out v1);
            response = Serializer.variant_to_string(v0);
            signature = (SerializableBuffer)Object.new(typeof(SerializableBuffer));
            signature.deserialize_from_variant(v1);
        }
    }

    public class AndnaRegisterMainResponse : Object, ISerializable
    {
        public string response {get; private set;}
        public TimeCapsule expires {get; private set;}

        public AndnaRegisterMainResponse(string response, TimeCapsule expires)
        {
            this.response = response;
            this.expires = expires;
        }

        public Variant serialize_to_variant()
        {
            Variant v0 = Serializer.string_to_variant(response);
            Variant v1 = expires.serialize_to_variant();
            Variant vret = Serializer.tuple_to_variant(v0, v1);
            return vret;
        }
        public void deserialize_from_variant(Variant v) throws SerializerError
        {
            Variant v0;
            Variant v1;
            Serializer.variant_to_tuple(v, out v0, out v1);
            response = Serializer.variant_to_string(v0);
            expires = (TimeCapsule)Object.new(typeof(TimeCapsule));
            expires.deserialize_from_variant(v1);
        }
    }

    public class AndnaRegisterSpreadResponse : Object, ISerializable
    {
        public bool response {get; private set;}
        public string message {get; private set;}

        public AndnaRegisterSpreadResponse(bool response, string message)
        {
            this.response = response;
            this.message = message;
        }

        public Variant serialize_to_variant()
        {
            Variant v0 = Serializer.int_to_variant(response ? 1 : 0);
            Variant v1 = Serializer.string_to_variant(message);
            Variant vret = Serializer.tuple_to_variant(v0, v1);
            return vret;
        }
        public void deserialize_from_variant(Variant v) throws SerializerError
        {
            Variant v0;
            Variant v1;
            Serializer.variant_to_tuple(v, out v0, out v1);
            response = Serializer.variant_to_int(v0) == 1;
            message = Serializer.variant_to_string(v1);
        }
    }

    public class AndnaGetServersResponse : Object, ISerializable
    {
        public AndnaServers response {get; private set;}
        public TimeCapsule expires {get; private set;}

        public AndnaGetServersResponse(AndnaServers response, TimeCapsule expires)
        {
            this.response = response;
            this.expires = expires;
        }

        public Variant serialize_to_variant()
        {
            Variant v0 = response.serialize_to_variant();
            Variant v1 = expires.serialize_to_variant();
            Variant vret = Serializer.tuple_to_variant(v0, v1);
            return vret;
        }
        public void deserialize_from_variant(Variant v) throws SerializerError
        {
            Variant v0;
            Variant v1;
            Serializer.variant_to_tuple(v, out v0, out v1);
            response = (AndnaServers)Object.new(typeof(AndnaServers));
            response.deserialize_from_variant(v0);
            expires = (TimeCapsule)Object.new(typeof(TimeCapsule));
            expires.deserialize_from_variant(v1);
        }
    }

    public class AndnaGetRegistrarResponse : Object, ISerializable
    {
        public PublicKey pubk {get; private set;}
        public NIP nip {get; private set;}
        public TimeCapsule expires {get; private set;}

        public AndnaGetRegistrarResponse(PublicKey pubk, NIP nip, TimeCapsule expires)
        {
            this.pubk = pubk;
            this.nip = nip;
            this.expires = expires;
        }

        public Variant serialize_to_variant()
        {
            Variant v0 = pubk.serialize_to_variant();
            Variant v1 = nip.serialize_to_variant();
            Variant v2 = expires.serialize_to_variant();
            Variant vret = Serializer.tuple_to_variant_3(v0, v1, v2);
            return vret;
        }
        public void deserialize_from_variant(Variant v) throws SerializerError
        {
            Variant v0;
            Variant v1;
            Variant v2;
            Serializer.variant_to_tuple_3(v, out v0, out v1, out v2);
            pubk = (PublicKey)Object.new(typeof(PublicKey));
            pubk.deserialize_from_variant(v0);
            nip = (NIP)Object.new(typeof(NIP));
            nip.deserialize_from_variant(v1);
            expires = (TimeCapsule)Object.new(typeof(TimeCapsule));
            expires.deserialize_from_variant(v2);
        }
    }

    public class AndnaGetCacheRecordsResponse : Object, ISerializable
    {
        public HashMap<string, AndnaDomainRecord> cache;
        public HashMap<string, ArrayList<RegisterHostnameArguments>> request_queue;

        public AndnaGetCacheRecordsResponse
                (HashMap<string, AndnaDomainRecord> cache,
                 HashMap<string, ArrayList<RegisterHostnameArguments>> request_queue)
        {
            this.request_queue = new HashMap<string, ArrayList<RegisterHostnameArguments>>();
            foreach (string s in request_queue.keys)
            {
                this.request_queue[s] = new ArrayList<RegisterHostnameArguments>
                            (RegisterHostnameArguments.equal_func_for_queue);
                this.request_queue[s].add_all(request_queue[s]);
            }
            this.cache = new HashMap<string, AndnaDomainRecord>();
            foreach (string s in cache.keys)
            {
                this.cache[s] = cache[s];
            }
        }

        public Variant serialize_to_variant()
        {
            int[] request_queue_value_lenghts = new int[request_queue.keys.size];
            int pos = 0;
            ListString lk_1 = new ListString();
            ListISerializable lv_1 = new ListISerializable();
            foreach (string k in request_queue.keys)
            {
                lk_1.add(k);
                ArrayList<RegisterHostnameArguments> v = request_queue[k];
                // how many records in this list?
                request_queue_value_lenghts[pos++] = v.size;
                foreach (RegisterHostnameArguments rec in v)
                {
                    lv_1.add(rec);
                }
            }

            ListString lk_2 = new ListString();
            ListISerializable lv_2 = new ListISerializable();
            foreach (string k in cache.keys)
            {
                lk_2.add(k);
                lv_2.add(cache[k]);
            }

            Variant v0 = lk_1.serialize_to_variant();
            Variant v1 = Serializer.int_array_to_variant(request_queue_value_lenghts);
            Variant v2 = lv_1.serialize_to_variant();
            Variant v3 = lk_2.serialize_to_variant();
            Variant v4 = lv_2.serialize_to_variant();

            Variant vret = Serializer.tuple_to_variant_5(v0, v1, v2, v3, v4);
            return vret;
        }
        
        public void deserialize_from_variant(Variant v)
        {
            Variant v0;
            Variant v1;
            Variant v2;
            Variant v3;
            Variant v4;
            Serializer.variant_to_tuple_5(v, out v0, out v1, out v2, out v3, out v4);

            ListString lk_1 = (ListString)Object.new(typeof(ListString));
            lk_1.deserialize_from_variant(v0);
            ListISerializable lv_1 = (ListISerializable)Object.new(typeof(ListISerializable));
            lv_1.deserialize_from_variant(v2);
            int[] request_queue_value_lenghts = Serializer.variant_to_int_array(v1);
            Gee.List<string> typed_lk_1 = (Gee.List<string>)lk_1.backed;
            Gee.List<RegisterHostnameArguments> typed_lv_1 =
                    (Gee.List<RegisterHostnameArguments>)lv_1.backed;
            request_queue = new HashMap<string, ArrayList<RegisterHostnameArguments>>();
            int pos = 0;
            for (int i = 0; i < typed_lk_1.size; i++)
            {
                ArrayList<RegisterHostnameArguments> vlist =
                        new ArrayList<RegisterHostnameArguments>
                            (RegisterHostnameArguments.equal_func_for_queue);
                for (int j = 0; j < request_queue_value_lenghts[i]; j++)
                {
                    vlist.add(typed_lv_1[j + pos]);
                }
                pos += request_queue_value_lenghts[i];
                request_queue[typed_lk_1[i]] = vlist;
            }

            ListString lk_2 = (ListString)Object.new(typeof(ListString));
            lk_2.deserialize_from_variant(v3);
            ListISerializable lv_2 = (ListISerializable)Object.new(typeof(ListISerializable));
            lv_2.deserialize_from_variant(v4);
            Gee.List<string> typed_lk_2 = (Gee.List<string>)lk_2.backed;
            Gee.List<AndnaDomainRecord> typed_lv_2 =
                    (Gee.List<AndnaDomainRecord>)lv_2.backed;
            cache = new HashMap<string, AndnaDomainRecord>();
            for (int i = 0; i < typed_lk_2.size; i++)
            {
                cache[typed_lk_2[i]] = typed_lv_2[i];
            }
        }
    }

    /** Identifies a group of nodes
      */
    public class BroadcastID : Object, ISerializable
    {
        public NetworkID? net = null;

        public Variant serialize_to_variant()
        {
            if (net == null) return Serializer.int_to_variant(0);
            return net.serialize_to_variant();
        }
        public void deserialize_from_variant(Variant v) throws SerializerError
        {
            net = null;
            string vt = v.get_type_string();
            if (vt != "i")
            {
                net = (NetworkID)Object.new(typeof(NetworkID));
                net.deserialize_from_variant(v);
            }
        }
    }

    /** Identifies a node
      */
    public class UnicastID : Object, ISerializable
    {
        public NIP nip;
        public int nodeid;

        public Variant serialize_to_variant()
        {
            Variant v0 = Serializer.int_to_variant(nodeid);
            Variant v1 = nip.serialize_to_variant();
            Variant vret = Serializer.tuple_to_variant(v0, v1);
            return vret;
        }
        public void deserialize_from_variant(Variant v) throws SerializerError
        {
            Variant v0;
            Variant v1;
            Serializer.variant_to_tuple(v, out v0, out v1);
            nodeid = Serializer.variant_to_int(v0);
            nip = (NIP)Object.new(typeof(NIP));
            nip.deserialize_from_variant(v1);
        }
    }
}

