/*
 *  This file is part of Netsukuku.
 *  (c) Copyright 2013 Luca Dionisi aka lukisi <luca.dionisi@gmail.com>
 *
 *  Netsukuku is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  Netsukuku is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with Netsukuku.  If not, see <http://www.gnu.org/licenses/>.
 */

using Gee;
using zcd;
using Netsukuku;

namespace Monitor
{
    public delegate G FuncReturning<G> ();
    public bool nap_until_condition(
            int total_msec,
            FuncReturning<bool> condition_func,
            int period_usec=2000)
    {
        Tasklets.Timer t = new Tasklets.Timer(total_msec);
        bool ret = false;
        while (! t.is_expired())
        {
            if (condition_func())
            {
                ret = true;
                break;
            }
            Thread.usleep(period_usec);
        }
        if (! ret) ret = condition_func();
        return ret;
    }

    /** This class does not wait too much to report an error in communications
      * with a remotable method.
      * It is not safe to use the same object for more than one operation,
      * because if the first operation takes longer than 2 seconds the object
      * will report an error but the operation actually is carried on.
      */
    public class QuickAddressManager : AddressManagerTCPClient
    {
        public QuickAddressManager(string tcpaddress)
        {
            base(tcpaddress);
            retry_connect = false;
        }

        public override ISerializable rmt (RemoteCall data) throws RPCError
        {
            thread_rmt_data = data;
            // start a thread
            thread_rmt_completed = false;
            new Thread<int>(null, thread_rmt);
            // wait a bit and then kill the thread if it didn't succeed
            if (! nap_until_condition(2000,
                () => {
                    return thread_rmt_completed;
                }))
                throw new RPCError.GENERIC("Timeout");

            // return what has been set by the thread
            if (thread_rmt_error_report != null)
                throw new RPCError.GENERIC(thread_rmt_error_report);
            return thread_rmt_ret;
        }

        private bool thread_rmt_completed;
        private RemoteCall thread_rmt_data;
        private ISerializable thread_rmt_ret;
        string? thread_rmt_error_report;
        private int thread_rmt()
        {
            thread_rmt_ret = new SerializableNone();
            thread_rmt_error_report = null;
            try {
                thread_rmt_ret = base.rmt(thread_rmt_data);
            } catch (RPCError e) {
                thread_rmt_error_report = e.message;
            }
            thread_rmt_completed = true;
            return 0;
        }
    }

    public class AddressManagerFakeRmtGetter : Object
    {
        private string tcpaddress;
        public AddressManagerFakeRmtGetter(string tcpaddress)
        {
            this.tcpaddress = tcpaddress;
        }
        public AddressManagerFakeRmt get_client()
        {
            AddressManagerTCPClient client = new QuickAddressManager(tcpaddress);
            return client;
        }
    }

    void main(string[] args)
    {
        OptionContext oc = new OptionContext("ip_address");
        OptionEntry[] entries = new OptionEntry[2];
        int index = 0;
        bool report_running = false;
        entries[index++] = {"report-running", 'r', 0, OptionArg.NONE, ref report_running, "Report currently running tasklets costantly", null};
        entries[index++] = { null };
        oc.add_main_entries(entries, null);
        try {
            oc.parse(ref args);
        }
        catch (OptionError e) {
            print(@"Error parsing options: $(e.message)\n");
            return;
        }

        string tcpaddress = args[1];

        // Initialize rpc library
        Serializer.init();

        // Register serializable types from sample model
        RpcNtk.init();

        AddressManagerFakeRmtGetter client_getter = new AddressManagerFakeRmtGetter(tcpaddress);

        // main thread waits forever and adds new threads
        int previous_max_id = 0;
        while (true)
        {
            Gee.List<string> str_ids = client_getter.get_client().report_running_tasklets();
            string output = "running: [";
            foreach (string str_id in str_ids)
            {
                output += str_id + " ";
                int id = int.parse(str_id);
                if (id > previous_max_id)
                {
                    TaskletLogger tl = new TaskletLogger(client_getter, id);
                    Thread<int> t_op = new Thread<int>(null, tl.run);
                    Thread.usleep(10000);
                    previous_max_id = id;
                }
            }
            output += "]\n";
            if (report_running) print(output);
            Thread.usleep(900000);
        }
    }

    public class TaskletLogger : Object
    {
        private AddressManagerFakeRmtGetter client_getter;
        private int id;
        private int last_log_pos;

        public TaskletLogger(AddressManagerFakeRmtGetter client_getter, int id)
        {
            this.client_getter = client_getter;
            this.id = id;
            last_log_pos = -1;
        }

        public int run()
        {
            while (true)
            {
                try
                {
                    Gee.List<string> logs = client_getter.get_client().report_tasklet_logs(id);
                    string name = logs.remove_at(0);
                    if (name.up().contains("TCPSERVER"))
                    {
                        return 0;
                    }
                    if (name == "STOPPED")
                    {
                        return 0;
                    }
                    if (! logs.is_empty)
                    {
                        int next_pos = int.parse(logs.remove_at(0));
                        if (next_pos + logs.size - 1 > last_log_pos)
                            print(@"$(name) [$(id)]\n");
                        int i = last_log_pos - next_pos + 1;
                        if (i < 0) i = 0;
                        for ( ; i < logs.size; i++)
                        {
                            if (i+1 < logs.size)
                            {
                                print(@" |\\__ $(logs[i])\n");
                            }
                            else
                            {
                                print(@"  \\__ $(logs[i])\n");
                            }
                        }
                        last_log_pos = next_pos + logs.size - 1;
                    }
                } catch (Error e) {
                    print(@"Error on id = $(id) $(e.message)\n");
                }
                Thread.usleep(700000);
            }
        }
    }
}
