/*
 *  This file is part of Netsukuku.
 *  (c) Copyright 2013 Luca Dionisi aka lukisi <luca.dionisi@gmail.com>
 *
 *  Netsukuku is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  Netsukuku is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with Netsukuku.  If not, see <http://www.gnu.org/licenses/>.
 */

using Gtk;
using Gee;
using zcd;
using Netsukuku;

namespace Monitor
{
    internal void log_debug(string msg)     {Posix.syslog(Posix.LOG_DEBUG,
                    "DEBUG "  + msg);}
    internal void log_info(string msg)      {Posix.syslog(Posix.LOG_INFO,
                    "INFO "   + msg);}
    internal void log_notice(string msg)    {Posix.syslog(Posix.LOG_NOTICE,
                    "INFO+ "  + msg);}
    internal void log_warn(string msg)      {Posix.syslog(Posix.LOG_WARNING,
                    "INFO++ " + msg);}
    internal void log_error(string msg)     {Posix.syslog(Posix.LOG_ERR,
                    "ERROR "  + msg);}
    internal void log_critical(string msg)  {Posix.syslog(Posix.LOG_CRIT,
                    "ERROR+ " + msg);}

    public delegate G FuncReturning<G> ();
    public bool nap_until_condition(
            int total_msec,
            FuncReturning<bool> condition_func,
            int period_usec=2000)
    {
        Tasklets.Timer t = new Tasklets.Timer(total_msec);
        bool ret = false;
        while (! t.is_expired())
        {
            if (condition_func())
            {
                ret = true;
                break;
            }
            Thread.usleep(period_usec);
        }
        if (! ret) ret = condition_func();
        return ret;
    }

    /** This class does not wait too much to report an error in communications
      * with a remotable method.
      * It is not safe to use the same object for more than one operation,
      * because if the first operation takes longer than 2 seconds the object
      * will report an error but the operation actually is carried on.
      */
    public class QuickAddressManager : AddressManagerTCPClient
    {
        public QuickAddressManager(string tcpaddress)
        {
            base(tcpaddress);
            retry_connect = false;
        }

        public override ISerializable rmt (RemoteCall data) throws RPCError
        {
            thread_rmt_data = data;
            // start a thread
            thread_rmt_completed = false;
            new Thread<int>(null, thread_rmt);
            // wait a bit and then kill the thread if it didn't succeed
            if (! nap_until_condition(2000,
                () => {
                    return thread_rmt_completed;
                }))
                throw new RPCError.GENERIC("Timeout");

            // return what has been set by the thread
            if (thread_rmt_error_report != null)
                throw new RPCError.GENERIC(thread_rmt_error_report);
            return thread_rmt_ret;
        }

        private bool thread_rmt_completed;
        private RemoteCall thread_rmt_data;
        private ISerializable thread_rmt_ret;
        string? thread_rmt_error_report;
        private int thread_rmt()
        {
            thread_rmt_ret = new SerializableNone();
            thread_rmt_error_report = null;
            try {
                thread_rmt_ret = base.rmt(thread_rmt_data);
            } catch (RPCError e) {
                thread_rmt_error_report = e.message;
            }
            thread_rmt_completed = true;
            return 0;
        }
    }

    public class AddressManagerFakeRmtGetter : Object
    {
        private string tcpaddress;
        public AddressManagerFakeRmtGetter(string tcpaddress)
        {
            this.tcpaddress = tcpaddress;
        }
        public AddressManagerFakeRmt get_client()
        {
            AddressManagerTCPClient client = new QuickAddressManager(tcpaddress);
            return client;
        }
    }

    public int main(string[] args)
    {
        OptionContext oc = new OptionContext("ip_address");
        OptionEntry[] entries = new OptionEntry[1];
        int index = 0;
        entries[index++] = { null };
        oc.add_main_entries(entries, null);
        try {
            oc.parse(ref args);
        }
        catch (OptionError e) {
            print(@"Error parsing options: $(e.message)\n");
            return 1;
        }

        Gtk.init (ref args);

        string tcpaddress = args[1];

        // Initialize rpc library
        Serializer.init();

        // Register serializable types from sample model
        RpcNtk.init();

        Posix.openlog("Ntk monitor", Posix.LOG_PID, Posix.LOG_USER);

        AddressManagerFakeRmtGetter client_getter = new AddressManagerFakeRmtGetter(tcpaddress);
        var x = new Node(tcpaddress, client_getter);

        Gtk.main ();
        Posix.exit(0);
        return 0;
    }

    string nip_to_str(int levels, int gsize, NIP nip)
    {
        uchar[] pip = nip_to_pip(levels, gsize, nip);
        return Netsukuku.InetUtils.address_v4_bytes_to_str(pip);
    }

    uchar[] nip_to_pip(int levels, int gsize, NIP nip)
    {
        uint64 ip = 0;
        for (int l = 0; l < levels; l++)
        {
            ip += nip.position_at(l) * (uint64)Math.pow(gsize, l);
        }
        return ip_to_pip(ip + 10 * (uint64)Math.pow(256, 3), 4);
    }

    uchar[] ip_to_pip(uint64 ip, int numbytes)
    {
        uchar[] ret = new uchar[numbytes];
        int j = 0;
        for (int i = numbytes-1; i >= 0; i--)
        {
            int c = (int)((ip % (uint64)Math.pow(256, i+1)) / (uint64)Math.pow(256, i));
            ret[j++] = (uchar)c;
        }
        return ret;
    }
}
