/*
 *  This file is part of Netsukuku.
 *  (c) Copyright 2013 Luca Dionisi aka lukisi <luca.dionisi@gmail.com>
 *
 *  Netsukuku is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  Netsukuku is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with Netsukuku.  If not, see <http://www.gnu.org/licenses/>.
 */

using Gtk;
using Gee;
using zcd;
using Netsukuku;

namespace Monitor
{
    // position of fields in treemodel
    enum LISTNEIGHBOURS {
        INFONEIGHBOUR,
        STR_NIP,
        NUMCOLUMNS
    }

    public class NodeNeighbours : Object
    {
        private AddressManagerFakeRmtGetter client_getter;
        private ListStore liststore_neighbours;

        public NodeNeighbours(Builder builder, Box box_parent, AddressManagerFakeRmtGetter client_getter)
        {
            this.client_getter = client_getter;
            liststore_neighbours = builder.get_object ("liststore_neighbours") as ListStore;
            builder.connect_signals (this);
            Widget widget_neighbours = builder.get_object ("widget_root") as Widget;
            widget_neighbours.reparent(box_parent);
        }

        private int impl_start_operations()
        {
            while (true)
            {
                try { refresh_neighbours();
                } catch (Error e) {}

                if (nap_until_condition(1000,
                    () => {
                        return t_op_aborting;
                    })) break;
            }
            return 0;
        }

        private Thread<int>? t_op;
        private bool t_op_aborting;
        public void start_operations()
        {
            if (t_op == null)
            {
                t_op_aborting = false;
                t_op = new Thread<int>(null, impl_start_operations);
            }
        }

        public void stop_operations()
        {
            if (t_op != null)
            {
                t_op_aborting = true;
                t_op.join();
                t_op = null;
            }
        }

        /** retrieve and display data
          */
        private void refresh_neighbours() throws Error
        {
            try
            {
                Gee.List<InfoNeighbour> ret = client_getter.get_client().aggregated_neighbour_manager.report_neighbours();

                MainContext.@default().invoke(() => {display_neighbours(ret); return false;});
            }
            catch (Error e)
            {
                string e_message = e.message;
                MainContext.@default().invoke(() => {display_error(e_message); return false;});
            }
        }

        void display_neighbours(Gee.List<InfoNeighbour> neighbours)
        {
            liststore_neighbours.clear();
            foreach (InfoNeighbour n in neighbours)
            {
                TreeIter iter;
                liststore_neighbours.prepend(out iter);
                liststore_neighbours.@set(iter, LISTNEIGHBOURS.INFONEIGHBOUR, n);
                string ipstr = nip_to_str(n.levels, n.gsize, n.nip);
                liststore_neighbours.@set(iter, LISTNEIGHBOURS.STR_NIP, @"$(n.nip) dev $(n.dev) rem $(n.rem)   \"$(ipstr)\"");
            }
        }

        void display_error(string e_message)
        {
            TreeIter iter;
            liststore_neighbours.clear();
            liststore_neighbours.append(out iter);
            liststore_neighbours.@set(iter, LISTNEIGHBOURS.STR_NIP, e_message);
        }
    }
}
