/*
 *  This file is part of Netsukuku.
 *  (c) Copyright 2014 Luca Dionisi aka lukisi <luca.dionisi@gmail.com>
 *
 *  Netsukuku is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  Netsukuku is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with Netsukuku.  If not, see <http://www.gnu.org/licenses/>.
 */

using Gtk;
using Gee;
using zcd;
using Netsukuku;

namespace Monitor
{
    // position of fields in treemodel
    enum LISTTASKLETUP {
        INT_ID,
        STR_FUNCNAME,
        STR_TIMESTART,
        NUMCOLUMNS
    }

    // position of fields in treemodel
    enum LISTTASKLETDOWN {
        INT_ID,
        STR_FUNCNAME,
        STR_TIMEEND,
        STR_COLOR,
        NUMCOLUMNS
    }

    public class NodeTasklets : Object
    {
        private AddressManagerFakeRmtGetter client_getter;
        private InfoNode info_node;
        private NIP nip;
        private ListStore liststore_taskletsup;
        private TreeView tv_taskletsup;
        private ListStore liststore_taskletsdown;
        private TreeView tv_taskletsdown;
        private Adjustment adj_detail_id;
        private SpinButton txt_detail_id;
        private Entry txt_detail_name;
        private Entry txt_detail_parent;
        private Label lbl_detail_status;
        private Label lbl_detail_info1;
        private Label lbl_detail_info2;
        private Label lbl_detail_info3;
        private Label lbl_detail_info4;
        private Gee.List<TaskletStats> stats;

        public NodeTasklets(Builder builder, Box box_parent, AddressManagerFakeRmtGetter client_getter, NIP nip)
        {
            this.client_getter = client_getter;
            this.nip = nip;
            builder.connect_signals (this);
            liststore_taskletsup = builder.get_object ("liststore_taskletsup") as ListStore;
            tv_taskletsup = builder.get_object ("tv_taskletsup") as TreeView;
            liststore_taskletsdown = builder.get_object ("liststore_taskletsdown") as ListStore;
            tv_taskletsdown = builder.get_object ("tv_taskletsdown") as TreeView;
            adj_detail_id = builder.get_object ("adj_detail_id") as Adjustment;
            txt_detail_id = builder.get_object ("txt_detail_id") as SpinButton;
            txt_detail_name = builder.get_object ("txt_detail_name") as Entry;
            txt_detail_parent = builder.get_object ("txt_detail_parent") as Entry;
            lbl_detail_status = builder.get_object ("lbl_detail_status") as Label;
            lbl_detail_info1 = builder.get_object ("lbl_detail_info1") as Label;
            lbl_detail_info2 = builder.get_object ("lbl_detail_info2") as Label;
            lbl_detail_info3 = builder.get_object ("lbl_detail_info3") as Label;
            lbl_detail_info4 = builder.get_object ("lbl_detail_info4") as Label;
            Widget widget_tasklets = builder.get_object ("widget_root") as Widget;
            widget_tasklets.reparent(box_parent);

            TreeSelection sel_taskletsup = tv_taskletsup.get_selection();
            sel_taskletsup.set_mode(SelectionMode.SINGLE);
            sel_taskletsup.changed.connect(() => {tv_taskletsup_selection_changed();});

            TreeSelection sel_taskletsdown = tv_taskletsdown.get_selection();
            sel_taskletsdown.set_mode(SelectionMode.SINGLE);
            sel_taskletsdown.changed.connect(() => {tv_taskletsdown_selection_changed();});

            info_node = client_getter.get_client().maproute.report_yourself();
        }

        private int impl_start_operations()
        {
            while (true)
            {
                try { refresh_tasklets();
                } catch (Error e) {}

                if (nap_until_condition(400,
                    () => {
                        return t_op_aborting;
                    })) break;
            }
            return 0;
        }

        private Thread<int>? t_op;
        private bool t_op_aborting;
        public void start_operations()
        {
            if (t_op == null)
            {
                t_op_aborting = false;
                t_op = new Thread<int>(null, impl_start_operations);
            }
        }

        public void stop_operations()
        {
            if (t_op != null)
            {
                t_op_aborting = true;
                t_op.join();
                t_op = null;
            }
        }

        /** retrieve and display data
          */
        private void refresh_tasklets() throws Error
        {
            try
            {
                stats = client_getter.get_client().report_tasklets_stats(4);
                MainContext.@default().invoke(() => {
                        adj_detail_id.set_upper(stats.size);
                        display_taskletsup();
                        display_taskletsdown();
                        return false;});
            }
            catch (Error e)
            {
                string e_message = e.message;
                MainContext.@default().invoke(() => {
                        lbl_detail_info1.label = "report_tasklets_stats:";
                        lbl_detail_info2.label = @" got $(e_message)";
                        lbl_detail_info3.label = "";
                        lbl_detail_info4.label = "";
                        return false;});
            }
        }

        private bool programatically_changing_selection;

        public void tv_taskletsup_selection_changed()
        {
            if (programatically_changing_selection) return;
            programatically_changing_selection = true;
            tv_taskletsdown.get_selection().unselect_all();
            programatically_changing_selection = false;
            TreePath? path;
            unowned TreeViewColumn? column;
            TreeIter iter;
            tv_taskletsup.get_cursor(out path, out column);
            if (path != null)
            {
                if (liststore_taskletsup.get_iter(out iter, path))
                {
                    int id;
                    liststore_taskletsup.@get(iter, LISTTASKLETUP.INT_ID, out id);
                    txt_detail_id.@value = id;
                }
            }
        }

        public void tv_taskletsdown_selection_changed()
        {
            if (programatically_changing_selection) return;
            programatically_changing_selection = true;
            tv_taskletsup.get_selection().unselect_all();
            programatically_changing_selection = false;
            TreePath? path;
            unowned TreeViewColumn? column;
            TreeIter iter;
            tv_taskletsdown.get_cursor(out path, out column);
            if (path != null)
            {
                if (liststore_taskletsdown.get_iter(out iter, path))
                {
                    int id;
                    liststore_taskletsdown.@get(iter, LISTTASKLETDOWN.INT_ID, out id);
                    txt_detail_id.@value = id;
                }
            }
        }

        [CCode (instance_pos = -1)]
        public void but_goto_parent_clicked(Button but)
        {
            if (displayed_stat != null)
                txt_detail_id.@value = displayed_stat.parent;
        }

        [CCode (instance_pos = -1)]
        public void txt_detail_id_value_changed(SpinButton but)
        {
            int id = txt_detail_id.get_value_as_int();
            // TODO start the processing
            //  only after the user has finished modifying the id.
            display_detail(get_stats(id));
        }

        TaskletStats? get_stats(int id)
        {
            foreach (TaskletStats t in stats)
            {
                if (t.id == id) return t;
            }
            return null;
        }

        TaskletStats? displayed_stat = null;
        void display_detail(TaskletStats? t)
        {
            displayed_stat = t;
            txt_detail_name.text = "";
            txt_detail_parent.text = "";
            lbl_detail_status.label = "";
            lbl_detail_info1.label = "";
            lbl_detail_info2.label = "";
            lbl_detail_info3.label = "";
            lbl_detail_info4.label = "";
            if (t != null)
            {
                txt_detail_name.text = @"$(t.funcname) ($(t.id))";
                TaskletStats? p = get_stats(t.parent);
                if (p == null)
                {
                    txt_detail_parent.text = @"<not in list> ($(t.parent))";
                }
                else
                {
                    txt_detail_parent.text = @"$(p.funcname) ($(t.parent))";
                }
                lbl_detail_status.label = @"$(t.status)";
                if (t.tasklet_started != null)
                {
                    int64 secs_start = t.tasklet_started.get_msec_ttl() / 1000;
                    lbl_detail_info1.label = @"Started $(secs_start) secs ago.";
                }
                if (t.tasklet_ended != null)
                {
                    int64 secs_end = t.tasklet_ended.get_msec_ttl() / 1000;
                    lbl_detail_info2.label = @"Ended $(secs_end) secs ago.";
                }
                if (t.status == Tasklets.Status.CRASHED)
                {
                    lbl_detail_info3.label = "Crashed";
                }
                if (t.crash_message != "")
                {
                    lbl_detail_info4.label = @"Error message $(t.crash_message)";
                }
            }
        }

        void display_taskletsup()
        {
            ArrayList<TaskletStats> disp = new ArrayList<TaskletStats>(Tasklets.Stat.equal_func);
            foreach (TaskletStats t in stats)
            {
                if (t.status == Tasklets.Status.STARTED) disp.add(t);
            }
            disp.sort(
                    (a, b) => {
                        // negative value if a < b; zero if a = b; positive value if a > b.
                        // sort puts from the smaller to the bigger.
                        // in order to have first the most recently started, a needs
                        // to result lesser than b (that is negative value) if a
                        // is_younger than b.
                        int ret;
                        // first the most recently started
                        if (a.tasklet_started.is_younger(b.tasklet_started)) ret = -1;
                        else if (b.tasklet_started.is_younger(a.tasklet_started)) ret = 1;
                        // in doubt, first the most recently created: a needs
                        // to result lesser than b (that is negative value) if a.id
                        // is bigger than b.id.
                        else ret = b.id - a.id;
                        return ret;
                    });
            // move, remove, add
            for (int i = 0; i < disp.size; i++)
            {
                TaskletStats t = disp[i];
                if (t.id == get_id(liststore_taskletsup, i, LISTTASKLETUP.INT_ID))
                {
                    update_taskletsup(i, t);
                }
                else
                {
                    // not the right one
                    bool found = false;
                    for (int j = i+1; j < getsize(liststore_taskletsup); j++)
                    {
                        if (t.id == get_id(liststore_taskletsup, j, LISTTASKLETUP.INT_ID))
                        {
                            found = true;
                            move(liststore_taskletsup, j, i);
                            update_taskletsup(i, t);
                            break;
                        }
                    }
                    if (! found) insert_into_taskletsup(i, t);
                }
            }
            remove_from_i_to_end(liststore_taskletsup, disp.size);
        }

        int get_id(ListStore liststore, int i, int column_for_id)
        {
            // returns the id of TaskletStats that is now in liststore at position i
            int id;
            TreePath path = new TreePath.from_indices(i);
            TreeIter iter;
            if (! liststore.get_iter(out iter, path)) return -1;  // NULL
            liststore.@get(iter, column_for_id, out id);
            return id;
        }

        void move(ListStore liststore, int j, int i)
        {
            // moves from position j to i an item of liststore
            TreePath path_i = new TreePath.from_indices(i);
            TreeIter iter_i;
            assert (liststore.get_iter(out iter_i, path_i));
            TreePath path_j = new TreePath.from_indices(j);
            TreeIter iter_j;
            assert (liststore.get_iter(out iter_j, path_j));
            liststore.move_before(ref iter_j, iter_i);
        }

        int getsize(ListStore liststore)
        {
            return liststore.iter_n_children(null);
        }

        void remove_from_i_to_end(ListStore liststore, int i)
        {
            // remove all items from position i to the end
            TreePath path = new TreePath.from_indices(i);
            TreeIter iter;
            while (true)
            {
                if (liststore.get_iter(out iter, path)) liststore.remove(iter);
                else break;
            }
        }

        void insert_into_taskletsup(int i, TaskletStats t)
        {
            // insert t at position i
            TreeIter iter;
            liststore_taskletsup.insert(out iter, i);
            liststore_taskletsup.@set(iter,
                    LISTTASKLETUP.INT_ID,
                    t.id);
            liststore_taskletsup.@set(iter,
                    LISTTASKLETUP.STR_FUNCNAME,
                    t.funcname);
            int64 secs = t.tasklet_started.get_msec_ttl() / 1000;
            liststore_taskletsup.@set(iter,
                    LISTTASKLETUP.STR_TIMESTART,
                    @"$(-secs)");
        }

        void update_taskletsup(int i, TaskletStats t)
        {
            TreePath path = new TreePath.from_indices(i);
            TreeIter iter;
            assert(liststore_taskletsup.get_iter(out iter, path));
            liststore_taskletsup.@set(iter,
                    LISTTASKLETUP.INT_ID,
                    t.id);
            liststore_taskletsup.@set(iter,
                    LISTTASKLETUP.STR_FUNCNAME,
                    t.funcname);
            int64 secs = t.tasklet_started.get_msec_ttl() / 1000;
            liststore_taskletsup.@set(iter,
                    LISTTASKLETUP.STR_TIMESTART,
                    @"$(-secs)");
        }

        void display_taskletsdown()
        {
            // process only recently ended and crashed tasklets
            ArrayList<TaskletStats> disp = new ArrayList<TaskletStats>(Tasklets.Stat.equal_func);
            Tasklets.Timer threshold = new Tasklets.Timer(-15000);
            foreach (TaskletStats t in stats)
            {
                if (t.status == Tasklets.Status.ENDED ||
                    t.status == Tasklets.Status.ABORTED ||
                    t.status == Tasklets.Status.CRASHED)
                {
                    if (t.status == Tasklets.Status.CRASHED) disp.add(t);
                    else if (t.tasklet_ended.is_younger(threshold)) disp.add(t);
                }
            }
            disp.sort(
                    (a, b) => {
                        // negative value if a < b; zero if a = b; positive value if a > b.
                        // sort puts from the smaller to the bigger.
                        // in order to have first the most recently finished, a needs
                        // to result lesser than b (that is negative value) if a
                        // is_younger than b.
                        int ret;
                        // first the most recently finished
                        if (a.tasklet_ended.is_younger(b.tasklet_ended)) ret = -1;
                        else if (b.tasklet_ended.is_younger(a.tasklet_ended)) ret = 1;
                        // in doubt, first the most recently created: a needs
                        // to result lesser than b (that is negative value) if a.id
                        // is bigger than b.id.
                        else ret = b.id - a.id;
                        return ret;
                    });
            // move, remove, add
            for (int i = 0; i < disp.size; i++)
            {
                TaskletStats t = disp[i];
                if (t.id == get_id(liststore_taskletsdown, i, LISTTASKLETDOWN.INT_ID))
                {
                    update_taskletsdown(i, t);
                }
                else
                {
                    // not the right one
                    bool found = false;
                    for (int j = i+1; j < getsize(liststore_taskletsdown); j++)
                    {
                        if (t.id == get_id(liststore_taskletsdown, j, LISTTASKLETDOWN.INT_ID))
                        {
                            found = true;
                            move(liststore_taskletsdown, j, i);
                            update_taskletsdown(i, t);
                            break;
                        }
                    }
                    if (! found) insert_into_taskletsdown(i, t);
                }
            }
            remove_from_i_to_end(liststore_taskletsdown, disp.size);
        }

        void insert_into_taskletsdown(int i, TaskletStats t)
        {
            // insert t at position i
            TreeIter iter;
            liststore_taskletsdown.insert(out iter, i);
            liststore_taskletsdown.@set(iter,
                    LISTTASKLETDOWN.INT_ID,
                    t.id);
            liststore_taskletsdown.@set(iter,
                    LISTTASKLETDOWN.STR_FUNCNAME,
                    t.funcname);
            int64 secs = t.tasklet_started.get_msec_ttl() / 1000;
            liststore_taskletsdown.@set(iter,
                    LISTTASKLETDOWN.STR_TIMEEND,
                    @"$(-secs)");
            string color = "green";
            if (t.status == Tasklets.Status.CRASHED) color = "red";
            liststore_taskletsdown.@set(iter,
                    LISTTASKLETDOWN.STR_COLOR,
                    color);
        }

        void update_taskletsdown(int i, TaskletStats t)
        {
            TreePath path = new TreePath.from_indices(i);
            TreeIter iter;
            assert(liststore_taskletsdown.get_iter(out iter, path));
            liststore_taskletsdown.@set(iter,
                    LISTTASKLETDOWN.INT_ID,
                    t.id);
            liststore_taskletsdown.@set(iter,
                    LISTTASKLETDOWN.STR_FUNCNAME,
                    t.funcname);
            int64 secs = t.tasklet_started.get_msec_ttl() / 1000;
            liststore_taskletsdown.@set(iter,
                    LISTTASKLETDOWN.STR_TIMEEND,
                    @"$(-secs)");
            string color = "green";
            if (t.status == Tasklets.Status.CRASHED) color = "red";
            liststore_taskletsdown.@set(iter,
                    LISTTASKLETDOWN.STR_COLOR,
                    color);
        }
    }
}
