/*
 *  This file is part of Netsukuku.
 *  (c) Copyright 2013 Luca Dionisi aka lukisi <luca.dionisi@gmail.com>
 *
 *  Netsukuku is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  Netsukuku is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with Netsukuku.  If not, see <http://www.gnu.org/licenses/>.
 */

using Gtk;
using Gee;
using zcd;
using Netsukuku;

namespace Monitor
{
    enum MODULE {
        COORDINATOR,
        ROUTES,
        GNODES,
        NEIGHBOURS,
        ANDNA,
        TASKLETS,
        PEERTOPEER,
        TASKLETLOGS,
        NONE
    }

    public class Node : Object
    {
        public int levels;
        public int gsize;
        public NIP nip;
        private string name;
        private AddressManagerFakeRmtGetter client_getter;
        private Notebook nb_main;
        private Notebook nb_basic;
        private Notebook nb_advanced;
        private Box box_neighbours;
        private NodeNeighbours neighbours;
        private Box box_gnodes;
        private NodeGnodes gnodes;
        private Box box_routes;
        private NodeRoutes routes;
        private Box box_coordinator;
        private NodeCoordinator coordinator;
        private Box box_andna;
        private NodeAndna andna;
        private Box box_tasklets;
        private NodeTasklets tasklets;
        private Box box_peertopeer;
        private NodePeerToPeer peertopeer;
        private Box box_taskletlogs;
        private NodeTaskletLogs taskletlogs;
        private Label l_pippovalue;
        private Label l_plutovalue;
        private Label l_paperinovalue;
        private Entry txt_value;
        private ComboBoxText cbo_key;
        private Label l_nip;
        private Label l_ip;
        private MODULE current_visible_module {
            get {
                Notebook in_main = nb_basic;
                if (nb_main.get_current_page() == 1) in_main = nb_advanced;
                Widget w = in_main.get_nth_page(in_main.get_current_page());
                if (w == box_neighbours) return MODULE.NEIGHBOURS;
                if (w == box_gnodes) return MODULE.GNODES;
                if (w == box_routes) return MODULE.ROUTES;
                if (w == box_coordinator) return MODULE.COORDINATOR;
                if (w == box_andna) return MODULE.ANDNA;
                if (w == box_tasklets) return MODULE.TASKLETS;
                if (w == box_peertopeer) return MODULE.PEERTOPEER;
                if (w == box_taskletlogs) return MODULE.TASKLETLOGS;
                return MODULE.NONE;
            }
        }
        private MODULE previous_active_module = MODULE.NONE;

        public Node(string name, AddressManagerFakeRmtGetter client_getter) throws RPCError
        {
            this.name = name;
            this.client_getter = client_getter;
            InfoNode n = client_getter.get_client().maproute.report_yourself();
            levels = n.levels;
            gsize = n.gsize;
            nip = n.nip;

            Builder builder = new Builder ();
            builder.add_from_resource ("/org/netsukuku/monitorradar/node.ui");
            builder.connect_signals (this);
            nb_main = builder.get_object ("nb_main") as Notebook;
            nb_basic = builder.get_object ("nb_basic") as Notebook;
            nb_advanced = builder.get_object ("nb_advanced") as Notebook;

            Builder builder_neighbours = new Builder ();
            builder_neighbours.add_from_resource ("/org/netsukuku/monitorradar/node-neighbours.ui");
            box_neighbours = builder.get_object ("box_neighbours") as Box;
            neighbours = new NodeNeighbours(builder_neighbours, box_neighbours, client_getter);

            Builder builder_gnodes = new Builder ();
            builder_gnodes.add_from_resource ("/org/netsukuku/monitorradar/node-gnodes.ui");
            box_gnodes = builder.get_object ("box_gnodes") as Box;
            gnodes = new NodeGnodes(builder_gnodes, box_gnodes, client_getter);

            Builder builder_routes = new Builder ();
            builder_routes.add_from_resource ("/org/netsukuku/monitorradar/node-routes.ui");
            box_routes = builder.get_object ("box_routes") as Box;
            routes = new NodeRoutes(builder_routes, box_routes, client_getter, nip);

            Builder builder_coordinator = new Builder ();
            builder_coordinator.add_from_resource ("/org/netsukuku/monitorradar/node-coordinator.ui");
            box_coordinator = builder.get_object ("box_coordinator") as Box;
            coordinator = new NodeCoordinator(builder_coordinator, box_coordinator, client_getter);

            Builder builder_andna = new Builder ();
            builder_andna.add_from_resource ("/org/netsukuku/monitorradar/node-andna.ui");
            box_andna = builder.get_object ("box_andna") as Box;
            andna = new NodeAndna(builder_andna, box_andna, client_getter, nip);

            Builder builder_tasklets = new Builder ();
            builder_tasklets.add_from_resource ("/org/netsukuku/monitorradar/node-tasklets.ui");
            box_tasklets = builder.get_object ("box_tasklets") as Box;
            tasklets = new NodeTasklets(builder_tasklets, box_tasklets, client_getter, nip);

            Builder builder_peertopeer = new Builder ();
            builder_peertopeer.add_from_resource ("/org/netsukuku/monitorradar/node-peertopeer.ui");
            box_peertopeer = builder.get_object ("box_peertopeer") as Box;
            peertopeer = new NodePeerToPeer(builder_peertopeer, box_peertopeer, client_getter);

            Builder builder_taskletlogs = new Builder ();
            builder_taskletlogs.add_from_resource ("/org/netsukuku/monitorradar/node-taskletlogs.ui");
            box_taskletlogs = builder.get_object ("box_taskletlogs") as Box;
            taskletlogs = new NodeTaskletLogs(builder_taskletlogs, box_taskletlogs, client_getter);

            l_nip = builder.get_object ("lbl_NIP") as Label;
            l_ip = builder.get_object ("lbl_IP") as Label;

            l_pippovalue = builder.get_object ("lbl_pippo") as Label;
            l_plutovalue = builder.get_object ("lbl_pluto") as Label;
            l_paperinovalue = builder.get_object ("lbl_paperino") as Label;
            cbo_key = builder.get_object ("cbo_key") as ComboBoxText;
            txt_value = builder.get_object ("txt_value") as Entry;

            var w2 = builder.get_object ("window1") as Window;
            w2.title = name;
            w2.show_all ();
            activate_module(MODULE.NEIGHBOURS, true);
            start_operations();
        }

        private int impl_start_operations()
        {
            while (true)
            {
                try { refresh_nip();
                } catch (Error e) {}

                if (nap_until_condition(1000,
                    () => {
                        return t_op_aborting;
                    })) break;
            }
            return 0;
        }

        private Thread<int>? t_op;
        private bool t_op_aborting;
        public void start_operations()
        {
            if (t_op == null)
            {
                t_op_aborting = false;
                t_op = new Thread<int>(null, impl_start_operations);
            }
        }

        public void stop_operations()
        {
            if (t_op != null)
            {
                t_op_aborting = true;
                t_op.join();
            }
        }

        /** retrieve and display data
          */
        private void refresh_nip() throws Error
        {
            try
            {
                InfoNode n = client_getter.get_client().maproute.report_yourself();
                levels = n.levels;
                gsize = n.gsize;
                nip = n.nip;

                MainContext.@default().invoke(() => {display_nip(); return false;});
            }
            catch (Error e)
            {
                string e_message = e.message;
                MainContext.@default().invoke(() => {display_error(e_message); return false;});
            }
        }

        void display_nip()
        {
            l_nip.label = nip.to_string();
            l_ip.label = nip_to_str(levels, gsize, nip);
        }

        void display_error(string e_message)
        {
            l_nip.label = e_message;
            l_ip.label = "";
        }

        private void activate_module(MODULE m=MODULE.NONE, bool make_visible=false)
        {
            if (m == MODULE.NONE) m = current_visible_module;
            if (make_visible)
            {
                if (m == MODULE.NEIGHBOURS)
                {
                    nb_main.set_current_page(0);
                    nb_basic.set_current_page(nb_basic.page_num(box_neighbours));
                }
                if (m == MODULE.GNODES)
                {
                    nb_main.set_current_page(0);
                    nb_basic.set_current_page(nb_basic.page_num(box_gnodes));
                }
                if (m == MODULE.ROUTES)
                {
                    nb_main.set_current_page(0);
                    nb_basic.set_current_page(nb_basic.page_num(box_routes));
                }
                if (m == MODULE.COORDINATOR)
                {
                    nb_main.set_current_page(0);
                    nb_basic.set_current_page(nb_basic.page_num(box_coordinator));
                }
                if (m == MODULE.ANDNA)
                {
                    nb_main.set_current_page(1);
                    nb_advanced.set_current_page(nb_advanced.page_num(box_andna));
                }
                if (m == MODULE.TASKLETS)
                {
                    nb_main.set_current_page(1);
                    nb_advanced.set_current_page(nb_advanced.page_num(box_tasklets));
                }
                if (m == MODULE.PEERTOPEER)
                {
                    nb_main.set_current_page(1);
                    nb_advanced.set_current_page(nb_advanced.page_num(box_peertopeer));
                }
                if (m == MODULE.TASKLETLOGS)
                {
                    nb_main.set_current_page(1);
                    nb_advanced.set_current_page(nb_advanced.page_num(box_taskletlogs));
                }
            }
            // start module
            if (m == MODULE.NEIGHBOURS) neighbours.start_operations();
            if (m == MODULE.GNODES) gnodes.start_operations();
            if (m == MODULE.ROUTES) routes.start_operations();
            if (m == MODULE.COORDINATOR) coordinator.start_operations();
            if (m == MODULE.ANDNA) andna.start_operations();
            if (m == MODULE.TASKLETS) tasklets.start_operations();
            if (m == MODULE.PEERTOPEER) peertopeer.start_operations();
            if (m == MODULE.TASKLETLOGS) taskletlogs.start_operations();
            // stop old module
            if (previous_active_module == MODULE.NEIGHBOURS) neighbours.stop_operations();
            if (previous_active_module == MODULE.GNODES) gnodes.stop_operations();
            if (previous_active_module == MODULE.ROUTES) routes.stop_operations();
            if (previous_active_module == MODULE.COORDINATOR) coordinator.stop_operations();
            if (previous_active_module == MODULE.ANDNA) andna.stop_operations();
            if (previous_active_module == MODULE.TASKLETS) tasklets.stop_operations();
            if (previous_active_module == MODULE.PEERTOPEER) peertopeer.stop_operations();
            if (previous_active_module == MODULE.TASKLETLOGS) taskletlogs.stop_operations();
            // now active is
            previous_active_module = m;
        }

        [CCode (instance_pos = -1)]
        public void nb_main_switch_page (Notebook n, Widget p, uint num)
        {
            Idle.add(() => {activate_module(); return false;});
        }

        [CCode (instance_pos = -1)]
        public void nb_basic_switch_page (Notebook n, Widget p, uint num)
        {
            Idle.add(() => {activate_module(); return false;});
        }

        [CCode (instance_pos = -1)]
        public void nb_advanced_switch_page (Notebook n, Widget p, uint num)
        {
            Idle.add(() => {activate_module(); return false;});
        }

        [CCode (instance_pos = -1)]
        public void close()
        {
            main_quit();
        }
    }
}
