/*
 *  This file is part of Netsukuku.
 *  (c) Copyright 2014 Luca Dionisi aka lukisi <luca.dionisi@gmail.com>
 *
 *  Netsukuku is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  Netsukuku is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with Netsukuku.  If not, see <http://www.gnu.org/licenses/>.
 */

namespace Netsukuku.InetUtils
{
    /** Given a IPv4 address in a dotted format, this function puts the
      *  address in 4 contiguous bytes, such as those in a Posix.InAddr
      *  or those in a AndnaResponseAnswer or AndnaQuery.
      */
    public void address_v4_str_to_bytes(string ipstr, uint8 *dst)
    {
        Posix.inet_pton(Posix.AF_INET, ipstr, dst);
    }

    /** Given a IPv6 address in a string (eg 2001:ffff::), this function puts the
      *  address in 16 contiguous bytes, such as those in a Posix.In6Addr
      *  or those in a AndnaResponseAnswer or AndnaQuery.
      */
    public void address_v6_str_to_bytes(string ipstr, uint8 *dst)
    {
        Posix.inet_pton(Posix.AF_INET6, ipstr, dst);
    }

    /** Given 4 contiguous bytes, such as those in a Posix.InAddr
      *  or those in a AndnaResponseAnswer or AndnaQuery, this function returns
      *  a dotted string, for displaying to the user or for a "ip" command.
      */
    public string address_v4_bytes_to_str(uint8[] addr)
    {
        assert(addr.length == 4);
        string ret;
        Posix.InAddr inaddr = {};
        void *p = &(inaddr.s_addr);
        uint8 *inaddr_p = (uint8 *)p;
        for (int i = 0; i < 4; i++) inaddr_p[i] = addr[i];

        // declaring an array with a define would not work due to vapi,
        // so instead of writing:
        //   uint8 dst[Posix.INET_ADDRSTRLEN];
        uint8 dst[16];
        assert(Posix.INET_ADDRSTRLEN == 16);

        weak string? w_ret = Posix.inet_ntop(Posix.AF_INET, inaddr_p, dst);
        ret = w_ret;
        return ret;
    }

    /** Given 16 contiguous bytes, such as those in a Posix.In6Addr
      *  or those in a AndnaResponseAnswer or AndnaQuery, this function returns
      *  a dotted string, for displaying to the user or for a "ip" command.
      */
    public string address_v6_bytes_to_str(uint8[] addr)
    {
        assert(addr.length == 16);
        string ret;
        Posix.In6Addr inaddr = {};
        for (int i = 0; i < 16; i++) inaddr.s6_addr[i] = addr[i];

        // declaring an array with a define would not work due to vapi,
        // so instead of writing:
        //   uint8 dst[Posix.INET6_ADDRSTRLEN];
        uint8 dst[46];
        assert(Posix.INET6_ADDRSTRLEN == 46);

        weak string? w_ret = Posix.inet_ntop(Posix.AF_INET6, &inaddr, dst);
        ret = w_ret;
        return ret;
    }

    /** Given 4 contiguous bytes, such as those in a Posix.InAddr
      *  or those in a AndnaResponseAnswer or AndnaQuery, this function returns
      *  a uint32. This is useful, as long as IPv4 is used, in order to make
      *  arithmetic computations with IP addresses.
      */
    public uint32 address_v4_bytes_to_int(uint8[] addr)
    {
        // we want a unsigned int at 32 bits that respects the order of bits,
        // so that doing arithmetic on it will have the same results on any
        // architecture (consider endianness)
        assert(addr.length == 4);
        uint32 ret;
        uint32 *p = (uint32 *)addr;
        ret = *p;
        ret = Posix.ntohl(ret);
        return ret;
    }

    /** Given a uint32 that represent a IPv4 address, this function puts the
      *  address in 4 contiguous bytes, such as those in a Posix.InAddr
      *  or those in a AndnaResponseAnswer or AndnaQuery.
      */
    public void address_v4_int_to_bytes(uint32 num, uint8 *dst)
    {
        // we consider endianness.
        uint32 num_h = Posix.htonl(num);
        uint32 *lp = &num_h;
        uint8 *p = (uint8 *)lp;
        for (int i = 0; i < 4; i++) dst[i] = p[i];
    }

    public string crypto_hash(string hostname)
    {
        // We want a domain name system that is case-insensitive.
        string ci_hname = hostname.up();
        // We want the hashnode not to know much about an hostname.
        // MD5 will output 128 bit = 16 octets.
        Bytes b = new Bytes(ci_hname.data);
        Checksum s = new Checksum(ChecksumType.MD5);
        s.update((uchar[])b.get_data(), b.get_size());
        uint8[] hash = new uint8[16];
        size_t digest_len = 16;
        s.get_digest(hash, ref digest_len);
        // We would like a printable sequence.
        // Base64 tranforms 16 octets in 22 character plus padding.
        string ret = Base64.encode((uchar[])hash).substring(0, 22);
        return ret;
    }
}


