/*
 *  This file is part of Netsukuku.
 *  (c) Copyright 2014 Luca Dionisi aka lukisi <luca.dionisi@gmail.com>
 *
 *  Netsukuku is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  Netsukuku is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with Netsukuku.  If not, see <http://www.gnu.org/licenses/>.
 */

using Nss;
using PosixExtra;
using Ntkresolv;
using Netsukuku.InetUtils;
using Gee;

string? andnaserver = null;
bool initialized = false;
const string LOG_LEAD = "nss_andna ";

internal void log_debug(string msg)     {Posix.syslog(Posix.LOG_DEBUG,
                LOG_LEAD + "DEBUG "  + msg);}
internal void log_info(string msg)      {Posix.syslog(Posix.LOG_INFO,
                LOG_LEAD + "INFO "   + msg);}
internal void log_notice(string msg)    {Posix.syslog(Posix.LOG_NOTICE,
                LOG_LEAD + "INFO+ "  + msg);}
internal void log_warn(string msg)      {Posix.syslog(Posix.LOG_WARNING,
                LOG_LEAD + "INFO++ " + msg);}
internal void log_error(string msg)     {Posix.syslog(Posix.LOG_ERR,
                LOG_LEAD + "ERROR "  + msg);}
internal void log_critical(string msg)  {Posix.syslog(Posix.LOG_CRIT,
                LOG_LEAD + "ERROR+ " + msg);}

void init()
{
    if (initialized) return;
    string fname = Config.SYSCONF_DIR + "/ntkresolv/ntkresolv.ini";
    KeyFile conf = new KeyFile();
    try
    {
        bool file_found = false;
        try
        {
            conf.load_from_file(fname, KeyFileFlags.NONE);
            file_found = true;
        }
        catch (KeyFileError e)
        {
            if (e is KeyFileError.NOT_FOUND)
            {
                // No configuration file. Ignore it.
            }
            else
            {
                log_warn(@"KeyFileError $(e.message)");
            }
        }
        if (file_found)
        {
            if (conf.has_group("NTKRESOLV") && conf.has_key("NTKRESOLV", "ANDNASERVER"))
            {
                andnaserver = conf.get_string("NTKRESOLV", "ANDNASERVER");
                log_debug(@"Initialized andnaserver $(andnaserver).");
            }
            else log_warn("Missing server in ini file.");
        }
        else
        {
            log_warn("Didn't read ini file.");
        }
    }
    catch (Error e)
    {
        log_warn(@"Error $(e.message)");
    }
    initialized = true;
}

internal Nss.Status return_unavail_eperm(int *h_errnop) {
    *h_errnop = Posix.EPERM;
    log_debug(@"returning unavail eperm");
    return Status.Unavailable;
}

internal Nss.Status return_unavail_esrch(int *h_errnop) {
    *h_errnop = Posix.ESRCH;
    log_debug(@"returning unavail esrch");
    return Status.Unavailable;
}

internal Nss.Status return_notfound_eperm(int *h_errnop) {
    *h_errnop = Posix.EPERM;
    log_debug(@"returning notfound eperm");
    return Status.NotFound;
}

internal Nss.Status return_unavail_enoent(int *h_errnop) {
    *h_errnop = Posix.ENOENT;
    log_debug(@"returning unavail enoent");
    return Status.Unavailable;
}

internal Nss.Status return_notfound_enoent(int *h_errnop) {
    *h_errnop = Posix.ENOENT;
    log_debug(@"returning notfound enoent");
    return Status.NotFound;
}

internal Nss.Status return_tryagain_erange(int *h_errnop) {
    *h_errnop = Posix.ERANGE;
    log_debug(@"returning tryagain erange");
    return Status.TryAgain;
}

internal Nss.Status return_tryagain_eagain(int *h_errnop) {
    *h_errnop = Posix.EAGAIN;
    log_debug(@"returning tryagain eagain");
    return Status.TryAgain;
}

internal Nss.Status return_success() {
    log_debug(@"returning success");
    return Status.Success;
}

errordomain NotAndnaError {
    GENERIC
}

errordomain NotFoundError {
    GENERIC
}

errordomain OnlyAliasesError {
    GENERIC
}

errordomain BufferError {
    GENERIC
}

/* Prepares for a success reply. The returned struct has a name and a ipv4 address.
 *
 */
void set_success
    (HostEnt *ret,
     char *buf,
     string name,
     Gee.List<NtkInetAddr> ips)
{
    ret->h_addrtype = Posix.AF_INET;
    ret->h_length = 4;
    char *h_name;
    char **h_aliases;
    char **h_addr_list = null;
    uint8 *buffer = (uint8 *)buf;

    // some variables
    char **x;
    char *p;
    uint pos = 0;
    uint[] pos_ip = new uint[1];

    p = (char *)buffer+pos;
    h_name = p;
    // put the official name
    foreach (uint8 o in name.data)
    {
        buffer[pos++] = o;
    }
    buffer[pos++] = '\0';
    // no aliases (list is NULL terminated)
    p = (char *)buffer+pos;
    x = (char **)p;
    *x = null;
    h_aliases = x;
    pos += (uint)sizeof(char*);
    // align to 32 bits
    pos += (uint)sizeof(void*) - pos % (uint)sizeof(void*);
    // foreach ip
    int ip_count = 0;
    foreach (NtkInetAddr ip in ips)
    {
        // put the ipv4 ip
        pos_ip[ip_count++] = pos;
        foreach (uint8 o in ip.addr)
        {
            buffer[pos++] = o;
        }
    }
    // foreach ip
    for (int ip_num = 0; ip_num < ip_count; ip_num++)
    {
        // put a pointer
        p = (char *)buffer+pos;
        x = (char **)p;
        *x = (char *)buffer+pos_ip[ip_num];
        if (ip_num == 0) h_addr_list = x;
        pos += (uint)sizeof(char*);
    }
    // list is NULL terminated
    p = (char *)buffer+pos;
    x = (char **)p;
    *x = null;
    pos += (uint)sizeof(char*);
    // All done.

    ret->h_name = (string)h_name;
    ret->h_aliases = (string[])h_aliases;
    ret->h_addr_list = (string[])h_addr_list;
}


public Nss.Status
_nss_andna_gethostbyname2_r
    (string name,
     int af,
     HostEnt *ret,
     char *buffer,
     size_t buflen,
     HostEnt **result,
     int *h_errnop)
{
    init();
    log_debug(@"name2: $(name), $(af)");
    if (af == Posix.AF_INET)
    {
        return _nss_andna_gethostbyname_r(
                name,
                ret,
                buffer,
                buflen,
                result,
                h_errnop);
    }
    else if (af == Posix.AF_INET6)
    {
        // IPv6 is not supported in netsukuku now.
        return return_unavail_esrch(h_errnop);
    }
    else
    {
        log_debug(@"name2: af is unknown");
        return return_unavail_esrch(h_errnop);
    }
}

public Nss.Status
_nss_andna_gethostbyname_r
    (string name,
     HostEnt *ret,
     char *buffer,
     size_t buflen,
     HostEnt **result,
     int *h_errnop)
{
    init();
    log_debug(@"name: $(name)");
    try
    {
        ArrayList<NtkInetAddr> ips = get_ips(name);
        uint ipsize = (uint)sizeof(uint8) * 4;
        if (buflen < 256 + (ipsize + (uint)sizeof(char *)) * ips.size) return return_tryagain_erange(h_errnop);
        set_success(ret, buffer, name, ips);
        // *result = ret;  
        // Odd behaviour: setting result (as per documentation of man gethostbyname)
        //  would determine failure in resolving names. Applications that try IPv6
        //  first and then fallback to IPv4 would report 'unknown host'.
        return return_success();
    }
    catch (NotAndnaError e)
    {
        return return_unavail_eperm(h_errnop);
    }
    catch (NotFoundError e)
    {
        return return_notfound_eperm(h_errnop);
    }
    catch (OnlyAliasesError e)
    {
        return return_notfound_eperm(h_errnop);
    }
    catch (BufferError e)
    {
        return return_tryagain_erange(h_errnop);
    }
}

ArrayList<NtkInetAddr>
get_ips(string origname) throws NotAndnaError, NotFoundError, OnlyAliasesError, BufferError
{
    string name = origname;
    // andna module is called before dns module. If the request is not for andna
    // immediately return UNAVAIL
    if (name.length < 5) throw new NotAndnaError.GENERIC("not for ANDNA");
    if (name.substring(name.length-1) == ".") name = name.substring(0,name.length-1);
    if (name.length < 5) throw new NotAndnaError.GENERIC("not for ANDNA");
    if (name.substring(name.length-4).up() != ".NTK") throw new NotAndnaError.GENERIC("not for ANDNA");
    name = name.substring(0,name.length-4);
    // request is for andna
    log_debug(@"name: is for andna...");
    ArrayList<NtkInetAddr> ips = new ArrayList<NtkInetAddr>();
    Gee.List<NtkAddrInfo> resp;
    try
    {
        if (andnaserver == null)
            resp = resolv(name, null, null);
        else
            resp = resolv(name, null, null, andnaserver);
    } catch (Error e) {
        throw new NotFoundError.GENERIC(e.message);
    }
    if (resp.size == 0)
    {
        throw new NotFoundError.GENERIC(@"$(name) not found.");
    }
    foreach (NtkAddrInfo ainfo in resp)
    {
        NtkInetAddr? addr = ainfo.address as NtkInetAddr;
        if (addr == null)
        {
            // print(" * Not supported address.\n");
        }
        else
        {
            // an IPv4 address.
            ips.add(addr);
            // ignore port.
        }
    }
    if (ips.size == 0)
    {
        throw new OnlyAliasesError.GENERIC("No IPv4 addresses.");
    }
    return ips;
}

bool
prepare_buffer
(Gee.List<NtkInetAddr> ips,
 string name,
 uint8 *buffer,
 uint buflen,
 out char *h_name,
 out char **h_aliases,
 out char **h_addr_list)
{
    // during debug, initialize the buffer to see better.
    //   for (uint i = 0; i < 1024; i++) buffer[i] = ' ';
    // check min size
    uint ipsize = (uint)sizeof(uint8) * 4;
    uint minima = name.length;
    minima += 1;
    minima += (uint)sizeof(char *);
    minima += (uint)sizeof(uint32); // alignment
    minima += (ipsize + (uint)sizeof(char *)) * ips.size;
    minima += (uint)sizeof(char *);
    if (buflen < minima) return false;
    // some variables
    char **x;
    char *p;
    uint pos = 0;
    uint[] pos_ip = new uint[ips.size];

    p = (char *)buffer+pos;
    h_name = p;
    // put the official name
    foreach (uint8 o in name.data)
    {
        buffer[pos++] = o;
    }
    buffer[pos++] = '\0';
    // no aliases (list is NULL terminated)
    p = (char *)buffer+pos;
    x = (char **)p;
    *x = null;
    h_aliases = x;
    pos += (uint)sizeof(char*);
    // align to 32 bits
    pos += (uint)sizeof(void*) - pos % (uint)sizeof(void*);
    // foreach ip
    int ip_count = 0;
    foreach (NtkInetAddr ip in ips)
    {
        // put the ipv4 ip
        pos_ip[ip_count++] = pos;
        foreach (uint8 o in ip.addr)
        {
            buffer[pos++] = o;
        }
    }
    // foreach ip
    for (int ip_num = 0; ip_num < ip_count; ip_num++)
    {
        // put a pointer
        p = (char *)buffer+pos;
        x = (char **)p;
        *x = (char *)buffer+pos_ip[ip_num];
        if (ip_num == 0) h_addr_list = x;
        pos += (uint)sizeof(char*);
    }
    // list is NULL terminated
    p = (char *)buffer+pos;
    x = (char **)p;
    *x = null;
    pos += (uint)sizeof(char*);
    // All done.
    return true;
}

public Nss.Status
_nss_andna_gethostbyaddr_r
    (void *addr,
     Posix.socklen_t len,
     int type,
     HostEnt *ret,
     char *buffer,
     size_t buflen,
     HostEnt **result,
     int *h_errnop)
{
    init();
    log_debug(@"addr: type = $(type)");
    if (type == Posix.AF_INET && len == sizeof(Posix.InAddr))
    {
        try
        {
            Posix.InAddr *in_addr = addr;
            set_host_ent_from_addr
                (in_addr,
                 ret,
                 buffer,
                 buflen);
            *result = ret;
            return return_success();
        }
        catch (NotAndnaError e)
        {
            return return_unavail_eperm(h_errnop);
        }
        catch (NotFoundError e)
        {
            return return_notfound_eperm(h_errnop);
        }
        catch (BufferError e)
        {
            return return_tryagain_erange(h_errnop);
        }
    }
    else
    {
        // Not ipv4
        return return_unavail_esrch(h_errnop);
    }
}

void
set_host_ent_from_addr
    (Posix.InAddr *addr,
     HostEnt *ret,
     char *buffer,
     size_t buflen,
     string? service=null,
     NtkAddrInfo? hints=null) throws NotAndnaError, NotFoundError, BufferError
{
    // andna module is called before dns module. If the request is not for andna
    // immediately return UNAVAIL

    // a pointer to the first byte
    uint32 *s_addr_p = &addr->s_addr;
    uint8 *addr_bytes = (uint8 *)s_addr_p;
    log_debug(@"addr: $(addr_bytes[0]).$(addr_bytes[1]).$(addr_bytes[2]).$(addr_bytes[3])");
    if (addr_bytes[0] != 10) throw new NotAndnaError.GENERIC("not for ANDNA");
    // request is for andna
    log_debug(@"addr: is for andna.");
    ret->h_addrtype = Posix.AF_INET;
    ret->h_length = 4;
    NtkInetAddr ntk_addr = new NtkInetAddr.from_pointer(addr_bytes, 0); // bytes are copied inside the constructor
    Gee.List<string> resp;
    if (addr_bytes[0] == 10 && addr_bytes[1] == 0 && addr_bytes[2] == 0 && addr_bytes[3] == 0)
    {
        throw new NotFoundError.GENERIC("IP address not found.");
    }
    if (addr_bytes[0] == 10 && addr_bytes[1] == 1 && addr_bytes[2] == 1 && addr_bytes[3] == 1)
    {
        resp = new ArrayList<string>();
        resp.add("presente");
    }
    else
    {
        try
        {
            if (andnaserver == null)
                resp = inverse(IpFamily.IPV4, ntk_addr);
            else
                resp = inverse(IpFamily.IPV4, ntk_addr, andnaserver);
        } catch (Error e) {
            throw new NotFoundError.GENERIC(e.message);
        }
        if (resp.size == 0)
        {
            throw new NotFoundError.GENERIC("IP address not found.");
        }
    }
    char *h_name;
    char **h_aliases;
    char **h_addr_list;
    for (int i = 0; i < resp.size; i++) resp[i] = resp[i] + ".NTK";
    if (!prepare_buffer_from_addr(resp, ntk_addr, (uint8 *)buffer, (uint)buflen,
                              out h_name,
                              out h_aliases,
                              out h_addr_list))
    {
        throw new BufferError.GENERIC("Not enough space.");
    }
    ret->h_name = (string)h_name;
    ret->h_aliases = (string[])h_aliases;
    ret->h_addr_list = (string[])h_addr_list;
}

bool
prepare_buffer_from_addr
(Gee.List<string> names,
 NtkInetAddr ip,
 uint8 *buffer,
 uint buflen,
 out char *h_name,
 out char **h_aliases,
 out char **h_addr_list)
{
    assert(names.size > 0);
    // during debug, initialize the buffer to see better.
    for (uint i = 0; i < 1024; i++) buffer[i] = ' ';
    // check min size
    uint ipsize = (uint)sizeof(uint8) * 4;
    uint minima = names[0].length; // canonical name
    minima += 1; // end of string
    for (int i = 1; i < names.size; i++)
    {
        // an alias name
        minima += names[i].length;
        minima += 1;
        minima += (uint)sizeof(char *);
    }
    minima += (uint)sizeof(char *);
    minima += (uint)sizeof(uint32); // alignment
    minima += ipsize + (uint)sizeof(char *);
    minima += (uint)sizeof(char *);
    if (buflen < minima) return false;
    // some variables
    char **x;
    char *p;
    uint pos = 0;
    uint[] pos_alias = new uint[names.size-1];

    p = (char *)buffer+pos;
    h_name = p;
    // put the canonical name
    string canonical_name = names[0];
    foreach (uint8 o in canonical_name.data)
    {
        buffer[pos++] = o;
    }
    buffer[pos++] = '\0';
    // aliases (list is NULL terminated)
    for (int i = 1; i < names.size; i++)
    {
        // put the name
        string alias_name = names[i];
        pos_alias[i-1] = pos;
        foreach (uint8 o in alias_name.data)
        {
            buffer[pos++] = o;
        }
        buffer[pos++] = '\0';
    }
    for (int i = 1; i < names.size; i++)
    {
        // put a pointer
        p = (char *)buffer+pos;
        x = (char **)p;
        *x = (char *)buffer+pos_alias[i-1];
        if (i == 1) h_addr_list = x;
        pos += (uint)sizeof(char*);
    }
    p = (char *)buffer+pos;
    x = (char **)p;
    *x = null;
    h_aliases = x;
    pos += (uint)sizeof(char*);
    // align to 32 bits
    pos += (uint)sizeof(void*) - pos % (uint)sizeof(void*);
    // the original ip
    uint pos_ip = pos;
    foreach (uint8 o in ip.addr)
    {
        buffer[pos++] = o;
    }
    // put the pointer to the original ip
    p = (char *)buffer+pos;
    x = (char **)p;
    *x = (char *)buffer+pos_ip;
    pos += (uint)sizeof(char*);
    // list is NULL terminated
    p = (char *)buffer+pos;
    x = (char **)p;
    *x = null;
    pos += (uint)sizeof(char*);
    // All done.
    return true;
}


